/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Properties;
import java.util.StringTokenizer;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.enumerations.ProjectType;
import org.eclipse.mtj.api.extension.AdminGuiProvider;
import org.eclipse.mtj.api.extension.MtjExtension;
import org.eclipse.mtj.api.extension.ProjectExtensionProvider;
import org.eclipse.mtj.exception.MtjException;

/**
 * MTJ Core Plug-ins functionalities are published by MtjServices class.
 * 
 * </p>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:50:23 $
 * <br>
 */
public class MtjServices {

	private static MtjServices instance;
	
	/**
	 * Method is used to get reference to the MtjServices -object.
	 * 
	 * @return
	 */
	public static MtjServices getInstance() {
		if (instance == null ) {
			instance = new MtjServices();
		}
		
		return instance;
	}
	
	/**
	 * Method returns all active Mtj Extensions that are the defined extensinon type and is implemented by the vendor.
	 * Also vendor's specifig version can be defined.
	 * Vendor and version attributes could have also null values.
	 * 
	 * @param type
	 * @param version
	 * @param vendor
	 * @return
	 */
	public MtjExtension[] getImplementations(ExtensionType extensionType,
			String version, String vendor) {
		return getImplementations(extensionType, version, vendor, true);
	}
	
	public MtjExtension[] getImplementations(ExtensionType extensionType,
			String version, String vendor, String project) {
		return getImplementations(extensionType, version, vendor, project, true);
	}
	
	public MtjExtension[] getImplementations(ExtensionType extensionType,
			String version, String vendor, boolean onlyActive) {
		return getImplementations(extensionType, version, vendor, null, onlyActive);
	}
	
	public MtjExtension[] getImplementations(ExtensionType extensionType,
			String version, String vendor, String project, boolean onlyActive) {
		
		ArrayList l = loadExtensions(MtjCorePlugin.getId(), extensionType);
				
		ArrayList r = new ArrayList();
		for ( int i = 0; i < l.size(); i++ ) {
			MtjExtension ex = (MtjExtension)l.get(i);
			
			boolean active = isActive(ex.getId(), extensionType, project);
			
			//
			// PersistentStoreProvider must be always Active
			// TODO: Check that at least one PersistentStoreProvider must be Active
			//
			if (extensionType.toString().equalsIgnoreCase( "PERSISTENT_STORE_PROVIDER") ){ //$NON-NLS-1$
				ex.setActive(true);
				active = true;
			} else {
				ex.setActive(active);
			}	
			
			if ( !onlyActive || active ) {
				if (vendor == null ||
					vendor.equals(ex.getVendor()) ) {

					if (version == null || version.equals(ex.getVersion()) ) {
						r.add(ex);
					}
				}
			}
		}
		
		// Create an MtjExtension array from the ArrayList
		MtjExtension[] ret = new MtjExtension[r.size()];
		for ( int i = 0; i < r.size(); i++ ) {
			ret[i] = (MtjExtension)r.get(i);
		}
		
		return ret;
	}

	/**
	 * Method returns all existing Mtj Extensions.
	 * 
	 * @return
	 */
	public MtjExtension[] getAllImplementations() {
		return getAllImplementations(null);
	}
	
	public MtjExtension[] getAllImplementations(String project) {
		ArrayList l = loadExtensions(MtjCorePlugin.getId(), null);
		MtjExtension[] ret = new MtjExtension[l.size()];
		for ( int i = 0; i < l.size(); i++ ) {
			ret[i] = (MtjExtension)l.get(i);
			ret[i].setActive(isActive(ret[i].getId(), ret[i].getType(), project));
		}
		
		return ret;
	}

	/**
	 * Method is used to verify if the extensino is active.
	 * 
	 * @param vendor
	 * @param version
	 * @param type
	 * @return
	 */
	public boolean isActive(String id, ExtensionType type) {
		return isActive(id, type, null);
	}
	
	/**
	 * Method is used to verify if the extensino is active.
	 * 
	 * @param vendor
	 * @param version
	 * @param type
	 * @param project
	 * @return
	 */
	public boolean isActive(String id, ExtensionType type, String project) {
		Properties props = getProperties();
		if ( props == null ) {
			return true;
		}
		
		String val = (String)props.get(getPropertyName(id, type, project));
		if ( val == null ) {
			if ( project != null ) {
				return isActive(id, type, null);
			}
			else {
				return true;
			}
		}
		
		return (new Boolean(val)).booleanValue();
	}
	
	/**
	 * Method is used to set the extension's activity.
	 * 
	 * @param vendor
	 * @param version
	 * @param type
	 * @param isActive
	 */
	public void setActive(String id, ExtensionType type, boolean isActive) 
			throws MtjException{
		setActive(id, type, null, isActive);
	}
	
	/**
	 * Method is used to set the extension's activity.
	 * 
	 * @param vendor
	 * @param version
	 * @param type
	 * @param project
	 * @param isActive
	 */
	public void setActive(String id, ExtensionType type, String project, boolean isActive) 
			throws MtjException{
		Properties props = getProperties();
		if ( props == null ) {
			throw new MtjException(Messages.MtjServices_ErrorGettingMtjProperties);
		}
		
		props.setProperty(getPropertyName(id, type, project), ""+isActive); //$NON-NLS-1$
		storeProperties();
	}
	
	private String getPropertyName(String id,
			ExtensionType type, String project) {
		return "plugin." + type.getName() + "_" + id + "." //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				+ (project != null ? project : "all") + ".active"; //$NON-NLS-1$ //$NON-NLS-2$
	}
	
	public ArrayList loadExtensions(String plugin_id, ExtensionType extensionType) {
		IExtensionRegistry r = Platform.getExtensionRegistry();
		// Testing 
		
		IExtensionPoint[] ps = r.getExtensionPoints(plugin_id);
		ArrayList l = new ArrayList();
		for (int j = 0; j < ps.length; j++) {
			IExtensionPoint p = ps[j];
			if (p.getSimpleIdentifier() != null
				&& ( extensionType == null || 
						p.getSimpleIdentifier().equalsIgnoreCase(capitalizeIdentifier(extensionType.toString())) ) ) {
				IConfigurationElement[] c = p.getConfigurationElements();
				if (c != null) {
					for (int i = 0; i < c.length; i++) {
						try {
							Object o = c[i].createExecutableExtension("class"); //$NON-NLS-1$
							if (o != null) {
								// Testing
								if (o instanceof MtjExtension) {
									// System.out.println(o.toString());
									l.add(o);
								}
							}
						} catch (CoreException x) {
//							System.out.println(">>>Error: " + p.getSimpleIdentifier() + " - " + x.getMessage());
						}
					}
				}
			}
		}
		return l;
	}

	/**
	 * 
     * CamelCase identifiers! 
     * Capitalize an identifier - i.e. Convert something like "Device_Management" to "deviceManagement"
     * or "DEVICE_MANAGEMENT" to "deviceManagement"
     * 
     * capitalizeIdentifier
     * @param value - The identifier to capitalize - e.g. "DEVICE_MANAGEMENT".
     * @return The capitalized identifier - e.g. "deviceManagement".
     */
    public String capitalizeIdentifier(String _value) {

    	if (_value == null) {
            return null;
        }
    	java.util.StringTokenizer _utokenizer = new StringTokenizer(_value, "_", true); //$NON-NLS-1$
        StringBuffer _uresult = new StringBuffer();
        while (_utokenizer.hasMoreTokens()) {
            String _word = _utokenizer.nextToken();
            String _tmp = _word;
            _word = _word.toLowerCase();
            
            if ((_word.equals(_tmp.toUpperCase())) || (!_utokenizer.hasMoreTokens())) {
                
            	if (!_word.equals("_")) { //$NON-NLS-1$
                    _uresult.append(_word);
                }
            } else {
                _uresult.append(_word + " "); //$NON-NLS-1$
            }
        }
        return capitalize(_uresult.toString());
    }
    /**
     * Capitalize a string - i.e. Convert something like "device management" to "Device Management".
     *
     * @param value The value to capitalize - e.g. "device management".
     * @return The capitalized value - e.g. "Device Management".
     */
    private String capitalize(String _value) {
    	
        if (_value == null) {
            return null;
        }

        java.util.StringTokenizer _tokenizer = new StringTokenizer(_value, " "); //$NON-NLS-1$
        StringBuffer _result = new StringBuffer();

        while (_tokenizer.hasMoreTokens()) {
            StringBuffer _word = new StringBuffer(_tokenizer.nextToken());

            /*
             * Upper case first character
             */ 
            _word.replace(0, 1, _word.substring(0, 1).toUpperCase());

            if (!_tokenizer.hasMoreTokens()) {
                _result.append(_word);
            } else {
                _result.append(_word + " "); //$NON-NLS-1$
            }
        }
        String _tmp = _result.toString();
        _tmp = _tmp.substring(0, 1).toLowerCase()+_tmp.substring(1);
        
        return replacePattern(_tmp, " ", ""); //$NON-NLS-1$ //$NON-NLS-2$
        
    }
    /**
     * Replaces all occurances of a given pattern with a replacement string.
     * Thus replacePattern("hey you!", " ","");
     * returns in "heyyou".
     * @param _original
     * @param _pattern
     * @param _replacement
     * @return
     */
    private String replacePattern(String _original, String _pattern, String _replacement) {
    	StringTokenizer _strtok = new StringTokenizer(_original, _pattern, true);
        StringBuffer _result = new StringBuffer();

        if (_replacement == null) {
            _replacement = ""; //$NON-NLS-1$
        }
        while (_strtok.hasMoreTokens()) {
            String _token = _strtok.nextToken();

            if (_token.equals(_pattern)) {
                _result.append(_replacement);
            } else {
                _result.append(_token);
            }
        }
        return _result.toString();
    }
		
    /**
     * Returns AdminGuiProvider implementation that belongs to the MtjExtension
     * 
     * @param extension
     * @return
     * @throws MtjException
     */
    public AdminGuiProvider getCorrespondingAdminGuiProvider(MtjExtension extension) throws MtjException {
    	MtjExtension[] exts = 
    		MtjServices.getInstance().getImplementations(ExtensionType.ADMIN_GUI_PROVIDER_LITERAL,null,null,false);
    	for (int i = 0; i < exts.length; i++) {
    		AdminGuiProvider a = (AdminGuiProvider)exts[i];
    		if ( a.getOwner() == extension ) {
    			return a;
    		}
    	}
    	return null;
    }

	public ProjectExtensionProvider getProjectExtension(ProjectType projectType) {
		Iterator it = loadExtensions("projectExtension").iterator(); //$NON-NLS-1$
		
		while (it.hasNext()) {
			ProjectExtensionProvider p = (ProjectExtensionProvider)it.next();
			if (p.getProjectType().toString().equals(projectType.getName())) {
				return p;
			}
		}
		
		return null;
	}
	
	public ArrayList loadExtensions(String extensionName) {
		IExtensionRegistry r = Platform.getExtensionRegistry();
		
		IExtensionPoint[] ps = r.getExtensionPoints(MtjCorePlugin.getId());
		ArrayList l = new ArrayList();
		for (int j = 0; j < ps.length; j++ ) {
			IExtensionPoint p = ps[j];
			if (p.getSimpleIdentifier() != null
				&& ( p.getSimpleIdentifier().equalsIgnoreCase(extensionName) ) ) {
				IConfigurationElement[] c = p.getConfigurationElements();
				if (c != null) {
					for (int i = 0; i < c.length; i++) {
						try {
							Object o = c[i].createExecutableExtension("class"); //$NON-NLS-1$
							if (o != null) {
								l.add(o);
							}
						} catch (CoreException x) {
							x.printStackTrace();
						}
					}
				}
			}
		}
		return l;
	}

	Properties properties = null;
	private Properties getProperties() {
		if ( properties == null ) {
			properties = new Properties();
			try {
				properties.load(new FileInputStream(getFileName()));
			} catch (FileNotFoundException e) {
				return properties;
			} catch (IOException e) {
				e.printStackTrace();
				return null;
			}
		}
		
		return properties;
	}
	
	private void storeProperties() throws MtjException {
		if ( properties != null ) {
			try {
				properties.store(new FileOutputStream(getFileName()), ""); //$NON-NLS-1$
			} catch (FileNotFoundException e) {
				try {
					File file = new File(getFileName());
					file.mkdirs();
					file.createNewFile();
					properties.store(new FileOutputStream(getFileName()), ""); //$NON-NLS-1$
				}
				catch (Exception ex) {
					throw new MtjException(Messages.MtjServices_ErrorStoringMtjProperties,e);
				}
			} catch (IOException e) {
				throw new MtjException(Messages.MtjServices_ErrorStoringMtjProperties,e);
			}
		}
	}
	
	private String getFileName() throws IOException {
		StringBuffer sb = new StringBuffer();
		
		IPath path = new Path(""); //$NON-NLS-1$
		String url = FileLocator.toFileURL(FileLocator.find(
				MtjCorePlugin.getDefault().getBundle(), path, null)).toString();
		if ( url.startsWith("file:/")) { //$NON-NLS-1$
			url = url.substring("file:/".length()); //$NON-NLS-1$
		}
		
		sb.append(url);
		sb.append("mtj.properties"); //$NON-NLS-1$
		
		return sb.toString();
	}

}
