/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 
package org.eclipse.mtj.internal.utils;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;

/**
 * A manager for handling temporary files and directories.
 * <p />
 * Copyright (c) 2003-2005 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:50:26 $
 * <br>
 * @author Craig Setera
 */
public class TemporaryFileManager {
	/** Singleton instance of the temporary file manager */
	public static final TemporaryFileManager instance = new TemporaryFileManager();
	
	// A Shutdown hook for cleaning up the temporary files
	private class CleanupThread extends Thread {
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run() {
			synchronized (tempFiles) {
				Iterator files = tempFiles.iterator();
				while (files.hasNext()) {
					File file = (File) files.next();
					doDelete(file);
				}
			}
		}

		/**
		 * Do the delete (recursively if necessary)
		 * @param file
		 */
		private void doDelete(File file) {
			if (file.isFile()) {
				try { file.delete(); } catch (Throwable t) { /* Be safe */ }
			} else {
				File[] files = file.listFiles();
				
				if (files != null) {
					// Recurse down the tree
					for (int i = 0; i < files.length; i++) {
						File child = files[i];
						doDelete(child);
					}
				}
				
				// Now, remove the directory
				try { file.delete(); } catch (Throwable t) { /* Be safe */ }
			}
		}
	}
	
	// The temp files that have been created
	private ArrayList tempFiles;
	
	/**
	 * Private constructor for singleton access.
	 */
	private TemporaryFileManager() {
		super();
		tempFiles = new ArrayList();
		
		// Wire up a shutdown hook to delete the files on
		// exit
		Runtime.getRuntime().addShutdownHook(new CleanupThread());
	}
	
    /**
     * Creates an empty directory in the default temporary-file directory, using
     * the given prefix and suffix to generate its name.  Invoking this method
     * is equivalent to invoking <code>{@link #createTempDirectory(java.lang.String,
     * java.lang.String, java.io.File)
     * createTempDirectory(prefix,&nbsp;suffix,&nbsp;null)}</code>.
     *
     * @param  prefix     The prefix string to be used in generating the file's
     *                    name; must be at least three characters long
     *
     * @param  suffix     The suffix string to be used in generating the file's
     *                    name; may be <code>null</code>, in which case the
     *                    suffix <code>".tmp"</code> will be used
     *
     * @return  An abstract pathname denoting a newly-created empty directory
     *
     * @throws  IllegalArgumentException
     *          If the <code>prefix</code> argument contains fewer than three
     *          characters
     *
     * @throws  IOException  If a file could not be created
     *
     * @throws  SecurityException
     *          If a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkWrite(java.lang.String)}</code>
     *          method does not allow a file to be created
     */
    public File createTempDirectory(String prefix, String suffix)
		throws IOException
    {
    	return createTempDirectory(prefix, suffix, null);
    }

    /**
     * <p> Creates a new empty directory in the specified directory, using the
     * given prefix and suffix strings to generate its name.  If this method
     * returns successfully then it is guaranteed that:
     *
     * <ol>
     * <li> The directory denoted by the returned abstract pathname did not exist
     *      before this method was invoked, and
     * <li> Neither this method nor any of its variants will return the same
     *      abstract pathname again in the current invocation of the virtual
     *      machine.
     * </ol>
     *
     * This method provides only part of a temporary-file facility.  To arrange
     * for a file created by this method to be deleted automatically, use the
     * <code>{@link #deleteOnExit}</code> method.
     *
     * <p> The <code>prefix</code> argument must be at least three characters
     * long.  It is recommended that the prefix be a short, meaningful string
     * such as <code>"hjb"</code> or <code>"mail"</code>.  The
     * <code>suffix</code> argument may be <code>null</code>, in which case the
     * suffix <code>".tmp"</code> will be used.
     *
     * <p> To create the new file, the prefix and the suffix may first be
     * adjusted to fit the limitations of the underlying platform.  If the
     * prefix is too long then it will be truncated, but its first three
     * characters will always be preserved.  If the suffix is too long then it
     * too will be truncated, but if it begins with a period character
     * (<code>'.'</code>) then the period and the first three characters
     * following it will always be preserved.  Once these adjustments have been
     * made the name of the new file will be generated by concatenating the
     * prefix, five or more internally-generated characters, and the suffix.
     *
     * <p> If the <code>directory</code> argument is <code>null</code> then the
     * system-dependent default temporary-file directory will be used.  The
     * default temporary-file directory is specified by the system property
     * <code>java.io.tmpdir</code>.  On UNIX systems the default value of this
     * property is typically <code>"/tmp"</code> or <code>"/var/tmp"</code>; on
     * Microsoft Windows systems it is typically <code>"c:\\temp"</code>.  A different
     * value may be given to this system property when the Java virtual machine
     * is invoked, but programmatic changes to this property are not guaranteed
     * to have any effect upon the the temporary directory used by this method.
     *
     * @param  prefix     The prefix string to be used in generating the file's
     *                    name; must be at least three characters long
     *
     * @param  suffix     The suffix string to be used in generating the file's
     *                    name; may be <code>null</code>, in which case the
     *                    suffix <code>".tmp"</code> will be used
     *
     * @param  directory  The directory in which the file is to be created, or
     *                    <code>null</code> if the default temporary-file
     *                    directory is to be used
     *
     * @return  An abstract pathname denoting a newly-created empty file
     *
     * @throws  IllegalArgumentException
     *          If the <code>prefix</code> argument contains fewer than three
     *          characters
     *
     * @throws  IOException  If a file could not be created
     *
     * @throws  SecurityException
     *          If a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkWrite(java.lang.String)}</code>
     *          method does not allow a file to be created
     */
    public File createTempDirectory(String prefix, String suffix, File directory)
        throws IOException
    {
    	// Create a new temporary file and register it for deletion
    	File file = null;
    	synchronized (tempFiles) {
    		// Create a temp file and then delete it to replace it
    		// with a created directory... Sort of an ugly hack
	    	file = File.createTempFile(prefix, suffix, directory);
	    	file.delete();
	    	file.mkdir();
	    	
	    	tempFiles.add(file);
    	}
    	
    	return file;
    }
	
    /**
     * Creates an empty file in the default temporary-file directory, using
     * the given prefix and suffix to generate its name.  Invoking this method
     * is equivalent to invoking <code>{@link #createTempFile(java.lang.String,
     * java.lang.String, java.io.File)
     * createTempFile(prefix,&nbsp;suffix,&nbsp;null)}</code>.
     *
     * @param  prefix     The prefix string to be used in generating the file's
     *                    name; must be at least three characters long
     *
     * @param  suffix     The suffix string to be used in generating the file's
     *                    name; may be <code>null</code>, in which case the
     *                    suffix <code>".tmp"</code> will be used
     *
     * @return  An abstract pathname denoting a newly-created empty file
     *
     * @throws  IllegalArgumentException
     *          If the <code>prefix</code> argument contains fewer than three
     *          characters
     *
     * @throws  IOException  If a file could not be created
     *
     * @throws  SecurityException
     *          If a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkWrite(java.lang.String)}</code>
     *          method does not allow a file to be created
     */
    public File createTempFile(String prefix, String suffix)
		throws IOException
    {
    	return createTempFile(prefix, suffix, null);
    }

    /**
     * <p> Creates a new empty file in the specified directory, using the
     * given prefix and suffix strings to generate its name.  If this method
     * returns successfully then it is guaranteed that:
     *
     * <ol>
     * <li> The file denoted by the returned abstract pathname did not exist
     *      before this method was invoked, and
     * <li> Neither this method nor any of its variants will return the same
     *      abstract pathname again in the current invocation of the virtual
     *      machine.
     * </ol>
     *
     * This method provides only part of a temporary-file facility.  To arrange
     * for a file created by this method to be deleted automatically, use the
     * <code>{@link #deleteOnExit}</code> method.
     *
     * <p> The <code>prefix</code> argument must be at least three characters
     * long.  It is recommended that the prefix be a short, meaningful string
     * such as <code>"hjb"</code> or <code>"mail"</code>.  The
     * <code>suffix</code> argument may be <code>null</code>, in which case the
     * suffix <code>".tmp"</code> will be used.
     *
     * <p> To create the new file, the prefix and the suffix may first be
     * adjusted to fit the limitations of the underlying platform.  If the
     * prefix is too long then it will be truncated, but its first three
     * characters will always be preserved.  If the suffix is too long then it
     * too will be truncated, but if it begins with a period character
     * (<code>'.'</code>) then the period and the first three characters
     * following it will always be preserved.  Once these adjustments have been
     * made the name of the new file will be generated by concatenating the
     * prefix, five or more internally-generated characters, and the suffix.
     *
     * <p> If the <code>directory</code> argument is <code>null</code> then the
     * system-dependent default temporary-file directory will be used.  The
     * default temporary-file directory is specified by the system property
     * <code>java.io.tmpdir</code>.  On UNIX systems the default value of this
     * property is typically <code>"/tmp"</code> or <code>"/var/tmp"</code>; on
     * Microsoft Windows systems it is typically <code>"c:\\temp"</code>.  A different
     * value may be given to this system property when the Java virtual machine
     * is invoked, but programmatic changes to this property are not guaranteed
     * to have any effect upon the the temporary directory used by this method.
     *
     * @param  prefix     The prefix string to be used in generating the file's
     *                    name; must be at least three characters long
     *
     * @param  suffix     The suffix string to be used in generating the file's
     *                    name; may be <code>null</code>, in which case the
     *                    suffix <code>".tmp"</code> will be used
     *
     * @param  directory  The directory in which the file is to be created, or
     *                    <code>null</code> if the default temporary-file
     *                    directory is to be used
     *
     * @return  An abstract pathname denoting a newly-created empty file
     *
     * @throws  IllegalArgumentException
     *          If the <code>prefix</code> argument contains fewer than three
     *          characters
     *
     * @throws  IOException  If a file could not be created
     *
     * @throws  SecurityException
     *          If a security manager exists and its <code>{@link
     *          java.lang.SecurityManager#checkWrite(java.lang.String)}</code>
     *          method does not allow a file to be created
     */
    public File createTempFile(String prefix, String suffix, File directory)
        throws IOException
    {
    	// Create a new temporary file and register it for deletion
    	File file = null;
    	synchronized (tempFiles) {
	    	file = File.createTempFile(prefix, suffix, directory);
	    	file.deleteOnExit();
	    	tempFiles.add(file);
    	}
    	
    	return file;
    }
}
