/*******************************************************************************
* Copyright (c) 2003,2006 IBM Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
* Contributors:
*IBM - initial API and implementation
*******************************************************************************/
package org.eclipse.mtj.jad.util;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;

/**
 * Class that can be used to represent property files in the key: value pair 
 * format.  It will currently fail(or behave very very badly) to handle files 
 * that contain duplicate keys.
 */
public class PropertyFile {

	/*Iterator constants*/
	private static final int KEYS = 0;
	private static final int VALUES = 1;
	private static final int COMMENTS = 2;
	
	/**
	 * removing properties 
	 */
     private boolean removingMidlets=true;
     private static final String[] digitSet = {"1", "2" ,"3", "4", "5", "6", "7", "8", "9", "0"};//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$ //$NON-NLS-8$ //$NON-NLS-9$ //$NON-NLS-10$
    
     
     
	/**
	 * Sequential property list.
	 */
	private List contents;

	/**
	 * Map of key->PropertyLine for fast searching.
	 */
	private Map lookupMap;

	private class PropertyFileIterator implements Iterator {
		private int typeOfInterest;
		
		private Iterator contentsIterator;
		
		PropertyFileIterator(int typeOfInterest) {
			this.typeOfInterest = typeOfInterest;			
			contentsIterator = lineIterator();
		}
		
		/**
		 * @see java.util.Iterator#hasNext()
		 */
		public boolean hasNext() {
			return contentsIterator.hasNext();
		}
		
		/**
		 * It may return null if the given portion of the PropertyLine is null.
		 * 
		 * @return the object
		 */
		public Object next() {
			String next = null;
			PropertyLine line = (PropertyLine)contentsIterator.next();
			switch (typeOfInterest) {
				case KEYS :
					next = line.getKey();
					break;
				case VALUES :
					next = line.getValue();
					break;
				case COMMENTS :
					next = line.getLeadingComment();
					break;
			}
			return next;
		}
				
		/**
		 * Not supported.
		 */
		public void remove() {
			throw new UnsupportedOperationException();
		}
	}

	/**
	 * Create an empty PropertyFile.
	 */
	public PropertyFile() {
		contents = new ArrayList();
		lookupMap = new HashMap();
	}

	/**
	 * An Iterator over the line collection.  The order of the elements in the 
	 * Iterator matches the order of the lines in the file.
	 * 
	 * @return the Iterator
	 */
	protected Iterator lineIterator() {
		return contents.iterator();
	}
	
	/**
	 * Iterator over the keys in this PropertyFile
	 * 
	 * @return the iterator
	 */
	public Iterator keyIterator() {
		return new PropertyFileIterator(KEYS);	
	}
	
	
	/**
	 * Get a Set containing all of the keys defined in this PropertyFile.
	 * 
	 * @return the Set
	 */
	public Set keySet() {
		Set set = new HashSet();
		for (Iterator i = keyIterator(); i.hasNext();) {
			set.add((String) i.next());			
		}
		return set;
	}
	
	/**
	 * Iterator over the values in this PropertyFile
	 * 
	 * @return the iterator
	 */
	public Iterator valueIterator() {
		return new PropertyFileIterator(VALUES);				
	}

	/**
	 * Iterator over the comments in this PropertyFile
	 * 
	 * @return the iterator
	 */
	public Iterator commentIterator() {
		return new PropertyFileIterator(COMMENTS);				
	}

	/**
	 * Determine the number of properties in this PropertyFile
	 * 
	 * @return the size
	 */
	public int size() {
		return contents.size();
	}
	
	/**
	 * Remove all properties from this PropertyFile.
	 */
	public void clear() {
		contents.clear();
		lookupMap.clear();
	}

	/**
	 * Remove the property with the given key from this PropertyFile
	 * 
	 * @param key the key
	 */
	public void removeProperty(String key) {
		PropertyLine lineToRemove = (PropertyLine)lookupMap.remove(key);
		if (lineToRemove != null) {
			for (Iterator i = lineIterator(); i.hasNext();) {
				PropertyLine line = (PropertyLine) i.next();
				if (lineToRemove == line) {
					i.remove();
					break;
				}
			}
		}		
	}

	/**
	 * Set a key/value/comment pair.
	 * 
	 * @param key the key
	 * @param value the value
	 * @param comment the comment
	 */
	public void setProperty(String key, String value, String comment) {
		if (key == null || value == null) {
			throw new IllegalArgumentException();
		}
		PropertyLine line = new PropertyLine();
		line.setKey(key);
		line.setValue(value);
		if (comment != null && comment.length() > 0) {
			// add the comment mark if the user hasn't already
			if (!comment.startsWith("#")) { //$NON-NLS-1$
				comment = "#" + comment; //$NON-NLS-1$
			}
			line.setLeadingComment(comment);
		}
		setPropertyLine(line);
	}

	/**
	 * Set a key/value pair.  Comment is left empty
	 * 
	 * @param key the key
	 * @param value the value
	 */		
	public void setProperty(String key, String value) {
		setProperty(key, value, null);
	}

	/**
	 * Set a property line.
	 * 
	 * @param line the PropertyLine
	 */
	protected void setPropertyLine(PropertyLine line) {
		int index = indexOf(line.key);
		if (index == contents.size()) {
			contents.add(line);
		}
		else {
			contents.set(index, line);
		}
		if (line.getKey() != null) { // dont try and add it to the lookup list if it's just a comment.	
			lookupMap.put(line.getKey(), line);
		}
	}	
	
	/**
	 * Determine the index in the content list where a PropertyLine with the 
	 * given key exists or should exist.
	 * 
	 * @return the index, or an index at the end of the list if it doesn't 
	 * exist
	 */
	private int indexOf(String key) {
		int indexOf = 0;		
		if (key == null) { // new comments go on the end of the list
			indexOf = contents.size();	
		}
		else {	
			indexOf = contents.size(); //assume it wont be found
			for (int i = 0; i < contents.size(); i++) {
				PropertyLine line = (PropertyLine) contents.get(i);
				if (line.getKey().equals(key)) {
					indexOf = i;									
					break;
				}
			}
		}
		return indexOf;
	}

	/**
	 * Determines whether this PropertyFile contains the given key.
	 * 
	 * @param key the key
	 * @return whether the key is present
	 */
	public boolean containsProperty(String key) {
		return getProperty(key) != null;
	}
	
	/**
	 * Determines whether this PropertyFile contains a set of keys.
	 * 
	 * @param keys the keys.  These should all be String objects.
	 * @return whether the keys are present
	 */
	public boolean containsAll(Set keys) {
		return lookupMap.keySet().containsAll(keys);
	}	

	/**
	 * Get a new PropertyFile that is a subset of this PropertyFile who's keys 
	 * match at least one of the prefixes in a given set.
	 * 
	 * @param keyPrefixSet the set of ket prefixes.
	 */
	public PropertyFile prefixedSubset(Set keyPrefixSet) {
		PropertyFile subset = new PropertyFile();
		for (Iterator i = keyIterator(); i.hasNext();) {
			String key = (String) i.next();
			for (Iterator j = keyPrefixSet.iterator(); j.hasNext();) {
				String keyPrefix = (String) j.next();
				if (key.startsWith(keyPrefix)) {
					subset.setProperty(key, getProperty(key), getComment(key));					
				}
			}
		}
		return subset;
	}

	/**
	 * Get the property value corresponding to a given key.
	 * 
	 * @param key the key
	 * @return the value, or null if it doesn't exist.
	 */
	public String getProperty(String key) {
		PropertyLine line = getPropertyLine(key);
		return line == null ? null : line.getValue();
	}
	
	/**
	 * Returns the String representing the property with this key.  Contains the
	 * leading comment, if any.
	 * 
	 * @param key the key
	 * @return the value with leading comment, or null if it doesn't exist.
	 */
	public String getPropertyLineString(String key) {
		PropertyLine line = getPropertyLine(key);
		return line == null ? null : line.toString();
	}
	
	/**
	 * Get the comment associated with a given property.
	 * 
	 * @param key the key
	 * @return the value.  Null if the key doesn't exist, empty String if the 
	 * key exists but there is no comment.
	 */
	public String getComment(String key) {
		PropertyLine line = getPropertyLine(key);
		return line == null ? null : line.getLeadingComment();		
	}

	/**
	 * Get the PropertyLine corresponding to a given key.
	 * 
	 * @param key the key
	 * @return the line
	 */
	protected PropertyLine getPropertyLine(String key) {
		return (PropertyLine) lookupMap.get(key);
	}

	/**
	 * Create a PropertyFile from properties contained within a file.
	 * 
	 * @param file the file
	 * @param retainComments include comments in the PropertyLines
	 * @return the PropertyFile
	 */
	public static PropertyFile load(IFile file, boolean retainComments)
		throws CoreException, IOException {
		return load(file.getContents(), retainComments);
	}

	/**
	 * Create a PropertyFile from properties contained within a stream.
	 * 
	 * @param stream the stream
	 * @param retainComments include comments in the PropertyLines
	 * @return the PropertyFile
	 */
	public static PropertyFile load(InputStream stream, boolean retainComments)
		throws IOException {
		return load(
			new BufferedReader(new InputStreamReader(stream, "utf8")), //$NON-NLS-1$
			retainComments);
	}

	/**
	 * Parse the given reader into a PropertyFile.
	 *
	 * @param reader the contents 
	 * @param retainComments include comments in the PropertyLines
	 * @return a new property file
	 */
	public static PropertyFile load(
		BufferedReader reader,
		boolean retainComments)
		throws IOException {
		PropertyFile fileContents = new PropertyFile();
		fileContents.loadInto(reader, retainComments);
		return fileContents;
	}
	

	/**
	 * Parse the given file and add its contents to this PropertyFile
	 * 
	 * @param file the file
	 * @param retainComments include comments in the PropertyLines
	 * @return the PropertyFile
	 */
	public void loadInto(IFile file, boolean retainComments)
		throws CoreException, IOException {
		loadInto(file.getContents(), retainComments);
	}

	/**
	 * Parse the given stream and add its contents to this PropertyFile
	 * 
	 * @param stream the stream
	 * @param retainComments include comments in the PropertyLines
	 * @return the PropertyFile
	 */
	public void loadInto(InputStream stream, boolean retainComments)
		throws IOException {
		loadInto(
			new BufferedReader(new InputStreamReader(stream, "utf8")), //$NON-NLS-1$
			retainComments);
	}
	
	
	/**
	 * Parse the given reader and add its contents to this PropertyFile
	 * 
	 * @param reader the contents 
	 * @param retainComments include comments in the PropertyLines
	 */
	public void loadInto(BufferedReader reader, boolean retainComments) throws IOException {
		PropertyLine propLine = new PropertyLine();

		for (String line = reader.readLine();
			line != null;
			line = reader.readLine()) {

			if (line.equals("") || line.startsWith("#")) { //$NON-NLS-1$ //$NON-NLS-2$
				//comment/whitespace line
				if (retainComments) {
					propLine.appendToLeadingComments(line);
					propLine.appendToLeadingComments("\n"); //$NON-NLS-1$
				}
			}
			else {
				String[] tokenizedLine = tokenizeLine(line);
				propLine.setKey(tokenizedLine[0]);
				propLine.setValue(tokenizedLine[1]);
				setPropertyLine(propLine);
				propLine = new PropertyLine();
			}
		}

		if (propLine.getLeadingCommentLength() > 0) {
			setPropertyLine(propLine);
		}
		reader.close();		
	}

	/**
	 * Serialize the contents of this file to the filesystem.
	 * 
	 * @param file the destination
	 * @param monitor progress monitor
	 */
	public void serialize(IFile file, IProgressMonitor monitor)
		throws CoreException, IOException {
		ByteArrayOutputStream slaveBytes = new ByteArrayOutputStream();
		int numProps = size();
		monitor.beginTask("", numProps + 1); //$NON-NLS-1$
		SubProgressMonitor subMonitor =
			new SubProgressMonitor(monitor, numProps);
		serialize(slaveBytes, subMonitor);
		ByteArrayInputStream slaveStream =
			new ByteArrayInputStream(slaveBytes.toByteArray());
		InputStream is = ConvertNativeToUTF8.convertNativeToUTF8(slaveStream);
		if (file.exists()) {
			file.setContents(is, true, true, monitor);
		}
		else {
			file.create(is, true, monitor);
		}
		monitor.worked(1);
		monitor.done();
	}

	/**
	 * Serialize the contents of this file.
	 * 
	 * @param slaveStream the destination
	 * @param monitor progress monitor
	 */
	public void serialize(OutputStream slaveStream, IProgressMonitor monitor)
		throws IOException {
		PrintWriter slaveWriter = new PrintWriter(slaveStream);

		for (Iterator i = lineIterator(); i.hasNext();) {
			PropertyLine line = (PropertyLine) i.next();
			if (line.getLeadingCommentLength() > 0) {
				slaveWriter.print(line.getLeadingComment());
			}
			if (line.key != null) {
				slaveWriter.print(line.getKey());
				slaveWriter.print(": "); //$NON-NLS-1$
				slaveWriter.print(ConvertNativeToUTF8.convertNativeToUTF8(line.getValue()));
				slaveWriter.println();
			}
			monitor.worked(1);
		}

		slaveWriter.flush();
	}
	
	/**
	 * Serialize the contents of this file and return an InputStream containing 
	 * it.
	 * 
	 * @return the contents of this file
	 */
	public InputStream serialize() {
		ByteArrayOutputStream stream =  new ByteArrayOutputStream();
		try {
			serialize(stream, new NullProgressMonitor());	
		}
		catch (IOException e) {
			//should never happen with a BAOS
		}
		InputStream bais = new ByteArrayInputStream(stream.toByteArray());
		InputStream is = ConvertNativeToUTF8.convertNativeToUTF8(bais);
		return is;
	}

	/**
	 * Merge this PropertyFile into another.
	 * 
	 * @param slave the file to merge into
	 * @return the merged PropertyFile
	 */
	public PropertyFile mergeInto(PropertyFile slave)
		throws CoreException, IOException {

		return mergeInto(lookupMap.keySet(), slave);
	}
	
	/**
	 * Merge the specified keys from this PropertyFile into another.
	 * 
	 * @param keys the keys to merge.
	 * @param slave the file to merge into
	 * @return the merged PropertyFile
	 * @throws IllegalArgumentException if keys is null
	 */	
	public PropertyFile mergeInto(Set keys, PropertyFile slave) {
		if (keys == null) {
			throw new IllegalArgumentException();
		}
		
		for (Iterator i = lineIterator(); i.hasNext();) {
			PropertyLine line = (PropertyLine) i.next();
			if (keys.contains(line.getKey())) {
				slave.mergeLine(line);
			}
		}

		return slave;		
	}
	/**
 	* Removes the extra Midlets entries in the Manifest file after meging it with the JAD descriptor
 	* 
 	* @param keys the keys to merge.
 	* @param slave the file to merge into
	 * @return the merged PropertyFile
 	* @throws IllegalArgumentException if keys is null
 	*/
	

public PropertyFile updateManifest(Set keys, PropertyFile slave) {
		if (keys == null) {
			throw new IllegalArgumentException();
		}
		Set digits= new HashSet(Arrays.asList(digitSet));
		setRemovingMidlets(true);
		while (isRemovingMidlets()){	
		for (Iterator j=slave.lineIterator(); j.hasNext();){
			setRemovingMidlets(false);
			PropertyLine line=null;
			line = (PropertyLine) j.next();
					if (!keys.contains(line.getKey())) {
					 if (line.getKey().startsWith("MIDlet-")){ //$NON-NLS-1$
					 	String digit1 = new String (line.getKey().substring(7,8));
					 	if (digits.contains(digit1)){
						slave.removeProperty(line.getKey());
						setRemovingMidlets(true);
						break;
					 	}
					 }
 	  				}
		}
		} //end of removing midlets 
		
		return slave;		
	}





	/**
	 * Replace the the value of a line matching masterLine in this PropertyFile 
	 * if it exists or adds it otherwise.
	 * 
	 * @param masterLine
	 */
	protected void mergeLine(PropertyLine masterLine) {
		PropertyLine slaveLine = getPropertyLine(masterLine.key);
		if (slaveLine != null) {
			slaveLine.value = masterLine.value;
		}
		else {
			setPropertyLine((PropertyLine) masterLine.clone());
		}
	}

	/**
	 * Break a String into a key: value pair if possible.
	 * 
	 * @param line
	 * @return String[] the key value pair
	 */
	public static String[] tokenizeLine(String line) throws IllegalArgumentException {
		String[] tokenizedLine = new String[2];
		int i = line.indexOf(':');

		if (i > 0) {
			tokenizedLine[0] = line.substring(0, i);

			tokenizedLine[1] = ""; //$NON-NLS-1$

			for (i = i + 1; i < line.length() && line.charAt(i) == ' '; i++);

			tokenizedLine[1] = i == line.length() ? "" : line.substring(i); //$NON-NLS-1$
		}
		else {
			throw new IllegalArgumentException();
		}
		return tokenizedLine;
	}
	
	/**
	 * Create a deep copy of this PropertyFile.
	 * 
	 * @return a copy
	 */
	public Object clone() {
		PropertyFile clone = new PropertyFile();
		for (Iterator i = lineIterator(); i.hasNext();) {
			PropertyLine line = (PropertyLine) i.next();
			clone.setPropertyLine((PropertyLine)line.clone());			
		}
		return clone;		
	}

	/**
	 * Determine whether two PropertyFiles are equal.  They are equal iff their
	 * sequential list of PropertyLines are equal.
	 * 
	 * @return equality
	 */
	public boolean equals(Object o) {
		boolean equals = false;
		if (o instanceof PropertyFile) {
			PropertyFile otherFile = (PropertyFile) o;
			if (size() == otherFile.size()) {
				equals = true; // assume equality until proven otherwise.
				for (Iterator i = lineIterator(), j = otherFile.lineIterator();
					i.hasNext();
					) {
					PropertyLine myLine = (PropertyLine) i.next();
					PropertyLine otherLine = (PropertyLine) j.next();
					if (!myLine.equals(otherLine)) {
						equals = false;
						break;
					}
				}
			}
		}
		return equals;
	}
	
	/**
	 * Expensive.  Effectivly serializes the file and converts to a String.  
	 * Primarily useful for debugging.
	 * 
	 * @return String representation of this file.
	 */
	public String toString() {
		String toString = "?"; //$NON-NLS-1$
		try {
			ByteArrayOutputStream stream =  new ByteArrayOutputStream();			
			serialize(stream, new NullProgressMonitor());
			toString = new String(stream.toByteArray());
		}
		catch (IOException e) {
		}
		return toString;
	}
	/**
	 * @return
	 */
	public boolean isRemovingMidlets() {
		return removingMidlets;
	}

	/**
	 * @param b
	 */
	public void setRemovingMidlets(boolean b) {
		removingMidlets = b;
	}
	
	

}
