/*******************************************************************************
* Copyright (c) 2003,2006 IBM Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
* Contributors:
*IBM - initial API and implementation
*******************************************************************************/
package org.eclipse.mtj.jad.util;

/**
 * Represents a key: value line in a PropertyFile.  Also contains a record to 
 * any #comments immediately before the line.  This is somewhat fuzzy - need to 
 * iron out what it means to have a 0 length comment.
 * 
 * @author IBM
 */
public class PropertyLine {
	protected String key;
	protected String value;
	protected StringBuffer leadingComments = new StringBuffer();

	/**
	 * Formats this PropertyLine as follows:<br>
	 * <code>
	 * #Comment
	 * Key: Value
	 * </code>
	 * 
	 * @return the formatted PropertyLine
	 */
	public String toString() {
		StringBuffer buffer = new StringBuffer();
		if (getLeadingCommentLength() > 0) {
			String comment = getLeadingComment();
			buffer.append(comment);
			if (!comment.endsWith("\n")) { //$NON-NLS-1$
				buffer.append('\n');
			}			
		}

		buffer.append(key).append(": ").append(value); //$NON-NLS-1$
		return buffer.toString();
	}

	/**
	 * Returns a copy of this PropertyLine.  key and value are copies of the 
	 * reference while leadingComments is a StringBuffer with a copy of the 
	 * contents of the original.
	 * 
	 * @return the copy
	 */
	public Object clone() {
		PropertyLine clone = new PropertyLine();
		clone.key = key;
		clone.value = value;
		clone.leadingComments.append(leadingComments.toString());
		return clone;
	}

	/**
	 * Equal iff o is a PropertyLine and key, value and comments are equal.
	 * 
	 * @return equality
	 */
	public boolean equals(Object o) {
		boolean equals = false;
		if (o instanceof PropertyLine) {
			PropertyLine otherLine = (PropertyLine) o;
			equals =
				otherLine.key.equals(key)
					&& otherLine.value.equals(value)
					&& otherLine.leadingComments.toString().equals(
						leadingComments.toString());
		}
		return equals;
	}	
	/**
	 * Returns the key.
	 * 
	 * @return String
	 */
	public String getKey() {
		return key;
	}

	/**
	 * Returns the leading comments of this PropertyLine.
	 * 
	 * @return String
	 */
	public String getLeadingComment() {
		return leadingComments.toString();
	}
	
	/**
	 * Set the leading comments of this PropertyLine.
	 * 
	 * @param comments the comments to set.
	 */
	public void setLeadingComment(String comments) {
		leadingComments.setLength(0);
		leadingComments.append(comments);
	}
	
	/**
	 * Add to the leading comments of this property line.
	 * 
	 * @param comments the comments to add.
	 */
	public void appendToLeadingComments(String comments) {
		leadingComments.append(comments);		
	}
	
	/**
	 * Get the length of the leading comment line.
	 * 
	 * @return the length
	 */
	public int getLeadingCommentLength() {
		return leadingComments.length();
	}

	/**
	 * Returns the value.
	 * 
	 * @return String
	 */
	public String getValue() {
		return value;
	}

	/**
	 * Sets the key.
	 * 
	 * @param key The key to set
	 */
	public void setKey(String key) {
		this.key = key;
	}

	/**
	 * Sets the value.
	 * 
	 * @param value The value to set
	 */
	public void setValue(String value) {
		this.value = value;
	}

}
