/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
 * Copyright (c) 2004 Kevin Hunter
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/

package org.eclipse.mtj.core.ui.errors;

import java.text.MessageFormat;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.swt.widgets.Shell;

import org.eclipse.mtj.core.ui.Messages;
import org.eclipse.mtj.core.ui.MtjCoreUiPlugin;
import org.eclipse.mtj.core.ui.constants.CoreConstants;

/**
 * This interface holds the error, warning codes for the org.eclipse.mtj.ui
 * package.
 * 
 * @author khunter
 * @author alaurila
 * 
 */
public class MtjUIErrors
{
	/**
	 * This string represents the prefix used in looking up messages for
	 * error and warning codes.
	 */
	
	public static final String MESSAGE_PREFIX = "EclipseMEUIError."; //$NON-NLS-1$

	
	/*
	 * The constants below represent errors that the eclipseme.ui plugin
	 * can generate.  Messages for these will be found in the EclipseMECoreStrings.properties
	 * file.
	 */
	public static final int UI_ERROR_BASE = 110000;
	public static final int UI_ERROR_EXCEPTION = UI_ERROR_BASE + 1;
	
	/*
	 * The constants below represent warnings that the eclipseme.core plugin
	 * can generate.  Messages for these will be found in the EclipseMECoreStrings.properties
	 * file.
	 */
	public static final int UI_WARNING_BASE = 120000;
	
	/*
	 * The constants below represent internal errors.  These indicate some kind of
	 * logic fault, as opposed to being something that could happen under normal
	 * conditions.  As such, these items do not have messages.  Instead, a "generic"
	 * message is generated.
	 */
	public static final int UI_INTERNAL_BASE = 190000;
	
	/**
	 * This routine returns the message associated with a particular error code
	 * or warning.  If there's a specific text message in the resource bundle,
	 * associated with this code, that one is returned.  Otherwise, a default
	 * message is used.  In either case, the error code is optionally (based
	 * on the string contents) substituted into the message.
	 * 
	 * @param nCode		The error code.
	 * @return
	 */
	
	public static final String getErrorMessage(int nCode)
	{
		String strTemplate;
		
		strTemplate = MtjUIErrorResources.getString(MESSAGE_PREFIX + nCode);
		if (strTemplate == null)
		{
			if (nCode < UI_INTERNAL_BASE)
			{
				strTemplate = MtjUIErrorResources.getString(MESSAGE_PREFIX+"Default"); //$NON-NLS-1$
			}
			else
			{
				strTemplate = MtjUIErrorResources.getString(MESSAGE_PREFIX+"InternalTemplate"); //$NON-NLS-1$
			}
		}
		
		String strMessage = MessageFormat.format(strTemplate, (Object[]) new Integer[] { new Integer(nCode) });
		return(strMessage);
	}
	
	/**
	 * This routine throws a CoreException with a status code of
	 * "ERROR", the specified error code, and a message that is
	 * internationalized.
	 * 
	 * @param code
	 * @throws CoreException
	 */
	public static void throwCoreExceptionError(int code) throws CoreException
	{
		IStatus status = new Status(IStatus.ERROR,
									CoreConstants.PLUGIN_ID,
									code,
									MtjUIErrors.getErrorMessage(code),
									null);
		throw new CoreException(status);
	}

	/**
	 * This routine throws a CoreException with a status code of
	 * "ERROR", the specified error code, and a message that is
	 * internationalized.
	 * 
	 * @param code
	 * @throws CoreException
	 */
	public static void throwCoreExceptionError(int code, Throwable e) throws CoreException
	{
		IStatus status = new Status(IStatus.ERROR,
									CoreConstants.PLUGIN_ID,
									code,
									MtjUIErrors.getErrorMessage(code),
									e);
		throw new CoreException(status);
	}
	
	/**
	 * This routine handles the problems of displaying internationalized error messages.
	 * 
	 * @param s					<code>Shell</code> on which to display.
	 * @param dialogTitleKey	Key in the resource file for dialog title
	 * @param messageKey		Key in the resource file for dialog message.
	 * @param status			Optional <code>IStatus</code> object that is the source of the error.
	 */
	
//	public static void displayError(Shell s, String dialogTitleKey, String messageKey, IStatus status)
//	{
//		if (dialogTitleKey == null)
//		{
//			dialogTitleKey = MESSAGE_PREFIX+"Default"; //$NON-NLS-1$
//		}
//		
//		if (messageKey == null)
//		{
//			messageKey = MESSAGE_PREFIX+"Default"; //$NON-NLS-1$
//		}
//		
//		int nCode = 0;
//		if (status != null)
//		{
//			nCode = status.getCode();
//		}
//		
//		Object[] substitutions = new Integer[] { new Integer(nCode) };
//		
//		String strTitle = MessageFormat.format(MtjUIStrings.getString(dialogTitleKey), substitutions);
//		String strMessage = MessageFormat.format(MtjUIStrings.getString(messageKey), substitutions);
//		
//		ErrorDialog.openError(	s,
//								strTitle,
//								strMessage,
//								status);
//	}
	
	public static void displayError(Shell s, String dialogTitleKey, String messageKey, Exception e)
	{
		if (dialogTitleKey == null)
		{
			dialogTitleKey = MESSAGE_PREFIX+"Default"; //$NON-NLS-1$
		}
		
		if (messageKey == null)
		{
			messageKey = MESSAGE_PREFIX+"Default"; //$NON-NLS-1$
		}
		
		String[] substitutions = new String[] { e.getMessage() };
		
		String message = e.getMessage();
		if (message == null) message = Messages.MtjUIErrors_NoMessage;
		Status status = 
			new Status(IStatus.ERROR, CoreConstants.PLUGIN_ID, UI_ERROR_EXCEPTION, message, e);
		
		//String strTitle = MessageFormat.format(MtjUIStrings.getString(dialogTitleKey), substitutions);
		//String strMessage = MessageFormat.format(MtjUIStrings.getString(messageKey), substitutions);
		
		MtjCoreUiPlugin.getDefault().getLog().log(status);
		//ErrorDialog.openError(	s,strTitle, strMessage, status);
		ErrorDialog.openError(	s,dialogTitleKey, messageKey, status);
	}
	
}

