/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.extension.dm.ui.util;

import java.io.File;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.enumerations.PlatformType;
import org.eclipse.mtj.api.extension.DeviceManagement;
import org.eclipse.mtj.api.extension.DevicePlatformProvider;
import org.eclipse.mtj.api.extension.MtjExtension;
import org.eclipse.mtj.api.project.TargetDevice;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.ImplementationRef;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.core.MtjServices;
import org.eclipse.mtj.core.toolkit.ToolkitSupport;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.dm.ui.preferences.DevicePlatformListContentProvider;
import org.eclipse.mtj.extension.dm.ui.preferences.DevicePlatformListLabelProvider;
import org.eclipse.mtj.extension.dm.ui.preferences.DevicePlatformListSorter;
import org.eclipse.mtj.extension.dm.ui.test.TestDeviceConfigurationImpl;
import org.eclipse.mtj.extension.dm.ui.test.TestDeviceImpl;
import org.eclipse.mtj.extension.dm.ui.test.TestDevicePlatformImpl;
import org.eclipse.mtj.extension.dm.ui.test.TestDeviceProfileImpl;
import org.eclipse.mtj.extension.dm.ui.test.TestRuntimePlatformDefinitionImpl;
import org.eclipse.mtj.extension.dm.ui.test.TestServiceApiImpl;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;


public class PreferenceUtils {
	
//	private CheckboxTableViewer deviceTableViewer;
//	private Table deviceTable;
	
	protected String[] variableTableColumnHeaders = { 
		"Default", //$NON-NLS-1$
		"Name", //$NON-NLS-1$
		"Version", //$NON-NLS-1$
		"Description", //$NON-NLS-1$
		"Vendor" //$NON-NLS-1$
	};

	protected ColumnLayoutData[] variableTableColumnLayouts = {
		new ColumnWeightData(35), 
		new ColumnWeightData(50),
		new ColumnWeightData(40),
		new ColumnWeightData(70),
		new ColumnWeightData(50) 
	};
	
	protected String[] variableDeviceTableColumnHeaders = {
			"Import", //$NON-NLS-1$
			"Device Name", //$NON-NLS-1$
			"Version", //$NON-NLS-1$
			"Description", //$NON-NLS-1$
			"Vendor" //$NON-NLS-1$
	};

	protected ColumnLayoutData[] variableDeviceTableColumnLayouts = {
			new ColumnWeightData(25), 
			new ColumnWeightData(70),
			new ColumnWeightData(40),
			new ColumnWeightData(150),
			new ColumnWeightData(50) 
	};

	
	public static Button createPushButton(Composite parent, String label) {
		
		Button button = new Button(parent, SWT.PUSH);
		button.setFont(parent.getFont());
		if (label != null){
			button.setText(label);
		}
//		initializeDialogUnits(button);
//		org.eclipse.jface.dialogs.DialogPage.initializeDialogUnits(button);
//		setButtonLayoutData(button);

		return button;	
	}
	
	public static Button createRadioButton(Group parent, String label) {
		
		Button button = new Button(parent, SWT.RADIO);
		button.setFont(parent.getFont());
		if (label != null){
			button.setText(label);
		}
//		initializeDialogUnits(button);
//		org.eclipse.jface.dialogs.DialogPage.initializeDialogUnits(button);
//		setButtonLayoutData(button);

		return button;	
	}
	
	public static Button createCheckButton(Composite parent, String label){
		
		Button button = new Button(parent, SWT.CHECK);
		button.setFont(parent.getFont());
		if (label != null){
			button.setText(label);
		}
		
		return button;
	}
	
	public Label createLabel(Composite parent, String text){
		
		Label label = new Label(parent, SWT.NONE);
		label.setText(text);
		label.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_BEGINNING));
		
		return label;
	}
	
	public Label createHorizontalSeparator(Composite parent){
		
		Label separator = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		separator.setLayoutData(gridData);
		
		return separator;
	}
	
	
	
	public Text createText(Composite parent, boolean editable){
		
		Text text = new Text(parent, SWT.SINGLE | SWT.BORDER);
		text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		text.setEditable(editable);
		
		return text;
	}
	
	public StyledText createStyledText(Composite parent, int height){
		
		StyledText text = new StyledText(parent, SWT.MULTI | SWT.WRAP | SWT.READ_ONLY);
		GridData gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, true);
		gridData.heightHint = height;
		gridData.widthHint = 100;
		text.setLayoutData(gridData);
		text.setBackground(parent.getBackground());
		text.setForeground(parent.getForeground());
		text.setEnabled(false);
		
		return text;
	}

	public static Object createDevicePlatformList() {

		Set<DevicePlatform> dpList = new HashSet<DevicePlatform>();
		
		DevicePlatform dp = new TestDevicePlatformImpl();
		
		dp.setType(PlatformType.JAVA_BASED_EMULATOR_LITERAL);
		dp.setName("DevicePlatform1");
		dp.setVersion("1.0");
		dp.setVendor("Vendor X");
		dp.setPreferencesDialogAvailable(false);
		dp.setUtilitesDialogAvailable(false);
		dp.setDebugEnabled(false);
		((TestDevicePlatformImpl)dp).setDevices(createTestDevices(4, "DeviceX", "1.1", "Vendor X"));
		((TestDevicePlatformImpl)dp).setRuntimePlatformDefinitions(createRpds(2, dp.getDevices()));
		dpList.add(dp);
		
		DevicePlatform dp2 = new TestDevicePlatformImpl();
		dp2.setType(PlatformType.JAVA_BASED_EMULATOR_LITERAL);
		dp2.setName("DevicePlatform2");
		dp2.setVersion("1.0");
		dp2.setVendor("Vendor Y");
		dp2.setPreferencesDialogAvailable(true);
		dp2.setUtilitesDialogAvailable(false);
		dp2.setDebugEnabled(true);
		((TestDevicePlatformImpl)dp2).setDevices(createTestDevices(3, "DeviceY", "1.2", "Vendor Y"));
		((TestDevicePlatformImpl)dp2).setRuntimePlatformDefinitions(createRpds(2, dp2.getDevices()));
		dpList.add(dp2);
		
		DevicePlatform dp3 = new TestDevicePlatformImpl();
		dp3.setType(PlatformType.JAVA_BASED_EMULATOR_LITERAL);
		dp3.setName("DevicePlatform3");
		dp3.setVersion("1.0");
		dp3.setVendor("Vendor Z");
		dp3.setPreferencesDialogAvailable(false);
		dp3.setUtilitesDialogAvailable(true);
		dp3.setDebugEnabled(true);
		((TestDevicePlatformImpl)dp3).setDevices(createTestDevices(2, "DeviceZ", "1.0", "Vendor Z"));
		((TestDevicePlatformImpl)dp3).setRuntimePlatformDefinitions(createRpds(2, dp3.getDevices()));
		dpList.add(dp3);
		
		return dpList;
	}
	
	private static Device[] createTestDevices(int amount, String name, String version, String vendor){
		
		Device[] devices = new TestDeviceImpl[amount];
		for(int i = 0; i < amount; i++){
			devices[i] = new TestDeviceImpl();
			devices[i].setName(name+"_"+i);
			devices[i].setVersion(version);
			devices[i].setVendor(vendor);
			devices[i].setDescription("Test device");
		}
		return devices;
	}
	
	private static RuntimePlatformDefinition[] createRpds(int amount, EList devices){
		
		RuntimePlatformDefinition[] rpds = new RuntimePlatformDefinition[amount];
		for(int i = 0; i < amount; i++){
			rpds[i] = new TestRuntimePlatformDefinitionImpl();
			rpds[i].setName("RPD_"+i);
			((TestRuntimePlatformDefinitionImpl)rpds[i]).setDevices(devices);
			((TestRuntimePlatformDefinitionImpl)rpds[i]).setProfiles(createTestProfiles());
			((TestRuntimePlatformDefinitionImpl)rpds[i]).setConfigurations(createTestConfigurations());
			((TestRuntimePlatformDefinitionImpl)rpds[i]).setServices(createTestServices());
		}	
		return rpds;
	}

	public static EList createTestRpds(){
		
		Object[] o = ((Set)createDevicePlatformList()).toArray();
		EList rpdList = null;
		
		DevicePlatform dp = null;
		for(int i = 0; i < o.length; i++){
			
			dp = (DevicePlatform)o[i];
			EList rpds = dp.getRuntimePlatformDefinitions();
			if(rpdList == null){
				rpdList = rpds;
			}
		}
		
		return rpdList;
	}
	
	public static EList createTestProfiles(){
		
		DeviceProfile profile = new TestDeviceProfileImpl();
		EList eList = ((TestDeviceProfileImpl)profile).getEmptyProfileList();
		
		for(int i = 0; i < 2; i++){
			
			profile = new TestDeviceProfileImpl();
			profile.setName("MIDP");
			profile.setVersion("1."+i);
			eList.add(profile);
		}
		
		return eList;
	}
	
	
	private static EList createTestConfigurations() {
		
		DeviceConfiguration config = new TestDeviceConfigurationImpl();
		EList eList = ((TestDeviceConfigurationImpl)config).getEmptyEList();
		
		for(int i = 0; i < 2; i++){
			
			config = new TestDeviceConfigurationImpl();
			config.setName("CLDC");
			config.setVersion("1."+i);
			eList.add(config);
		}
		
		return eList;
	}
	
	private static EList createTestServices() {
		
		ServiceApi service = new TestServiceApiImpl();
		EList eList = ((TestServiceApiImpl)service).getEmptyEList();
		
		for(int i = 0; i < 7; i++){
			
			service = new TestServiceApiImpl();
			service.setName("SAPI");
			service.setVersion("1."+i);
			eList.add(service);
		}
		
		return eList;
	}
	
	public TreeViewer createClassLibraryTree(Composite parent){
		
		//Create tree container
		final TreeViewer treeViewer = new TreeViewer(parent, SWT.NONE);
		treeViewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		treeViewer.setLabelProvider(new ClassLibraryLabelProvider());
		treeViewer.setContentProvider(new ClassLibraryContentProvider(treeViewer));
		return treeViewer;
	}
	
	public TreeViewer createDeviceTree(Composite parent) {
		
		//Create table container
		final TreeViewer treeViewer = new TreeViewer(parent, SWT.NONE);
		treeViewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		addListeners(treeViewer);
		treeViewer.setLabelProvider(new DevicePlatformListLabelProvider());
		treeViewer.setContentProvider(new DevicePlatformListContentProvider(treeViewer));
		treeViewer.setSorter(new DevicePlatformListSorter());
		return treeViewer;
		
	}
	
	public CheckboxTreeViewer createDevicePlatformCheckTree(Composite parent){
		
		CheckboxTreeViewer treeViewer = new CheckboxTreeViewer(parent);
		treeViewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		treeViewer.setLabelProvider(new DevicePlatformListLabelProvider());
		treeViewer.setContentProvider(new DevicePlatformImportContentProvider(treeViewer));
		treeViewer.setSorter(new DevicePlatformListSorter());
		
		return treeViewer;
	}
	
	public TreeViewer createTargetDeviceTree(Composite parent){
		
		//Create table container
		final TreeViewer treeViewer = new TreeViewer(parent, SWT.NONE);
		treeViewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		treeViewer.setLabelProvider(new TargetDeviceLabelProvider(this));
		treeViewer.setContentProvider(new TargetDeviceContentProvider(treeViewer));
		return treeViewer;
	}
	
	protected void addListeners(final TreeViewer treeViewer2) {
		
		treeViewer2.addPostSelectionChangedListener(new ISelectionChangedListener() {
			
			public void selectionChanged(SelectionChangedEvent event) {
				
			}
		});
		
		((Tree)treeViewer2.getControl()).addSelectionListener(new SelectionAdapter() {
			public void widgetDefaultSelected(final SelectionEvent event) {
				ISelection selection = treeViewer2.getSelection();
				if (selection.isEmpty())
					return;
				Object singleSelection = getSingleSelection(selection);
				boolean expanded = treeViewer2.getExpandedState(singleSelection);
				treeViewer2.setExpandedState(singleSelection, !expanded);
			}
		});
	}
	
	public Object getSingleSelection(ISelection selection) {
		
		if (!selection.isEmpty()) {
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof Device){
				return (Device)element;
			}
			else if(element instanceof TargetDevice){
				return (TargetDevice)element;
			}
			else if (element instanceof RuntimePlatformDefinition){
				return (RuntimePlatformDefinition)element;
			}
			else if(element instanceof ServiceApi){
				return (ServiceApi)element;
			}
			else if(element instanceof DeviceProfile){
				return (DeviceProfile)element;
			}
			else if(element instanceof DeviceConfiguration){
				return (DeviceConfiguration)element;
			}
		}
		return null;
	}
	
	public TreeViewer createConfigTable(Composite parent) {
		
		//Create table container
		final TreeViewer treeViewer = new TreeViewer(parent, SWT.NONE);
		treeViewer.getControl().setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		addListeners(treeViewer);
		treeViewer.setLabelProvider(new DeviceConfigurationLabelProvider());
		treeViewer.setContentProvider(new DeviceConfigurationContentProvider());
		treeViewer.setSorter(new DeviceConfigurationSorter());
		return treeViewer;
		
	}
	
	public DevicePlatform getSingleDevicePlatformSelection(ISelection selection) {
		
		if (!selection.isEmpty()) {
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof DevicePlatform){
				return (DevicePlatform)element;
			}
		}
		return null;
	}
	
	public Device getSingleDeviceSelection(ISelection selection) {
		
		if (!selection.isEmpty()) {
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof Device){
				return (Device)element;
			}
		}
		return null;
	}
	
	public RuntimePlatformDefinition getSingleRpdSelection(ISelection selection) {
		
		if (!selection.isEmpty()) {
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof RuntimePlatformDefinition){
				return (RuntimePlatformDefinition)element;
			}
		}
		return null;
	}

	public TargetDevice getSingleTargetDeviceSelection(ISelection selection) {
		if (!selection.isEmpty()) {
			IStructuredSelection structured = (IStructuredSelection) selection;
			Object element = structured.getFirstElement();
			if(element instanceof TargetDevice){
				return (TargetDevice)element;
			}
		}
		return null;
	}
	
	
	public Composite createDefaultComposite(Composite parent, int columns, boolean equalWidth) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout(columns, equalWidth);
		composite.setLayout(layout);

		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false);
		composite.setLayoutData(data);

		return composite;
	}
	
	public Group createDefaultGroup(Composite parent, int columns, boolean equalWidth) {
		
		Group group = new Group(parent, SWT.NONE);
		GridLayout layout = new GridLayout(columns, equalWidth);
		group.setLayout(layout);

		GridData data = new GridData(SWT.FILL, SWT.FILL, true, false);
		group.setLayoutData(data);

		return group;
	}
	
	public Composite createButtonComposite(Composite parent){
		
		Composite buttonComposite = new Composite(parent, SWT.NONE);
		GridLayout grid = new GridLayout();
		GridData gdata = new GridData(SWT.RIGHT, SWT.TOP, false, false);
		buttonComposite.setLayout(grid);
		buttonComposite.setLayoutData(gdata);
		buttonComposite.setFont(parent.getFont());

		return buttonComposite;
	}
	
	
	public static String getServiceApiListing(RuntimePlatformDefinition rpd){
		
		//ServiceAPI listing
		EList eList = rpd.getServiceApis();
		StringBuffer sb = new StringBuffer();
		ServiceApi sApi = null;
		for(int i = 0; i < eList.size(); i++){
			sApi = (ServiceApi)eList.get(i);
			sb.append(sApi.getName());
			sb.append("-");
			sb.append(sApi.getVersion());
			sb.append(", ");
		}
		//Remove the ", " from the last service api
		sb.delete((sb.length() - 2), (sb.length() - 1));
		
		return sb.toString();
	}
	
	public static String getCleanedServiceApiListing(EList serviceApis, EList profileApis){
		
		//ServiceAPI listing
		ServiceApi[] sApis = getCleanedServiceApis(serviceApis, profileApis);
		StringBuffer sb = new StringBuffer();
		for(int i = 0; i < sApis.length; i++){
			sb.append(sApis[i].getName());
			sb.append("-");
			sb.append(sApis[i].getVersion());
			if(i < sApis.length - 1){
				sb.append(", ");
			}
		}
		
		return sb.toString();
	}
	
	/**
	 * Table for holding device information.
	 * The checkbox is used to indicate the selected default device.
	 * @param amount TODO
	 */
	public CheckboxTableViewer createDeviceDefaultTable(Composite parent, int amount) {

		final CheckboxTableViewer deviceTableViewer = createDeviceTable(parent, amount);
		
		//Create columns
		TableLayout tableLayout2 = (TableLayout) deviceTableViewer.getTable().getLayout();
		for (int i = 0; i < variableTableColumnHeaders.length; i++) {
			
			tableLayout2.addColumnData(variableTableColumnLayouts[i]);
			TableColumn tableColumn = new TableColumn(deviceTableViewer.getTable(), SWT.NONE, i);
			tableColumn.setResizable(variableTableColumnLayouts[i].resizable);
			tableColumn.setText(variableTableColumnHeaders[i]);
		}
		
		return deviceTableViewer;
	}
	
	/**
	 * Table for holding device information.
	 * The checkbox is used to indicate devices to importe.
	 * @param amount TODO
	 */
	public CheckboxTableViewer createDeviceImportTable(Composite parent, int amount) {

		final CheckboxTableViewer deviceTableViewer = createDeviceTable(parent, amount);
		
		//Create columns
		TableLayout tableLayout2 = (TableLayout) deviceTableViewer.getTable().getLayout();
		for (int i = 0; i < variableDeviceTableColumnHeaders.length; i++) {
			
			tableLayout2.addColumnData(variableDeviceTableColumnLayouts[i]);
			TableColumn tableColumn = new TableColumn(deviceTableViewer.getTable(), SWT.NONE, i);
			tableColumn.setResizable(variableDeviceTableColumnLayouts[i].resizable);
			tableColumn.setText(variableDeviceTableColumnHeaders[i]);
		}
		
		return deviceTableViewer;
	}
	
	private CheckboxTableViewer createDeviceTable(Composite parent, int amount){
		
//		 Create table container
		Composite tableComposite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 1;
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, true);
		gridData.heightHint = amount * 20;
		gridData.widthHint = 400;
		tableComposite.setLayout(layout);
		tableComposite.setLayoutData(gridData);

	
		Table deviceTable = new Table(tableComposite, SWT.CHECK | SWT.BORDER
				| SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE | SWT.FULL_SELECTION);
		
		TableLayout tableLayout2 = new TableLayout();
		deviceTable.setLayout(tableLayout2);
		deviceTable.setHeaderVisible(true);
		deviceTable.setLinesVisible(true);
		deviceTable.setFont(parent.getFont());
		gridData = new GridData(GridData.FILL_BOTH);
		
		CheckboxTableViewer deviceTableViewer = new CheckboxTableViewer(deviceTable);
		deviceTableViewer.getControl().setLayoutData(gridData);
		deviceTableViewer.setContentProvider(new DeviceListContentProvider());
		deviceTableViewer.setLabelProvider(new DeviceListLabelProvider());
		deviceTableViewer.setSorter(new DeviceListSorter());
		
		return deviceTableViewer;
	}
	
	public static Object getDevicePlatforms() throws MtjException {
		
		Set<DevicePlatform> dpList = new HashSet<DevicePlatform>();

		DeviceManagement dm = (DeviceManagement)MtjServices
			.getInstance().getImplementations(
				ExtensionType.DEVICE_MANAGEMENT_LITERAL, null,
				null, true)[0];
		
		DevicePlatform[] dps = dm.getDevicePlatforms();
		
		for (int i = 0; i < dps.length; i++) {
			dpList.add(dps[i]);
		}
	
		return dpList;

/*
		PersistentStoreProvider pstore = (PersistentStoreProvider) MtjServices
				.getInstance().getImplementations(
						ExtensionType.PERSISTENT_STORE_PROVIDER_LITERAL, null,
						null)[0];

		EList list = null;
		try {
			list = (EList) pstore.get(PersistentStoreProvider.DATA_TYPE_EMF,
					"devices", "org.eclipse.mtj.extension.dpp",
					"DevicePlatform");
		} catch (MtjException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		if (list == null) { // TODO to be removed -> only for GUI testing
			return PreferenceUtils.createDevicePlatformList();
		} else {
			
			int count = list.size();
			for (int i = 0; i < count; i++) {
				dpList.add((DevicePlatform)list.get(i));
			}
		}
*/
	}
	
	public static String getObjectName(Object obj){
		
		String name = null;
		StringBuffer sb = null;
		
		if(obj instanceof DevicePlatform){
			name = ((DevicePlatform)obj).getName();
		}
		else if(obj instanceof RuntimePlatformDefinition){
			name = ((RuntimePlatformDefinition)obj).getName();
		}
		else if(obj instanceof Device){
			name = ((Device)obj).getName();
		}
		else if(obj instanceof TargetDevice){
			name = ((TargetDevice)obj).getDeviceName();
		}
		else if(obj instanceof DeviceProfile){
			DeviceProfile tmp = (DeviceProfile)obj;
			sb = new StringBuffer(tmp.getName());
			sb.append("-"); sb.append(tmp.getVersion());
			name = sb.toString();
		}
		else if(obj instanceof DeviceConfiguration){
			DeviceConfiguration tmp = (DeviceConfiguration)obj;
			sb = new StringBuffer(tmp.getName());
			sb.append("-"); sb.append(tmp.getVersion());
			name = sb.toString();
		}
		else if(obj instanceof ServiceApi){
			ServiceApi tmp = (ServiceApi)obj;
			sb = new StringBuffer(tmp.getName());
			sb.append("-"); sb.append(tmp.getVersion());
			name = sb.toString();
		}
		else if(obj instanceof ImplementationRef){
			ImplementationRef ref = (ImplementationRef)obj;
			sb = new StringBuffer(ref.getFileRef());
			name = sb.toString();
		}
		else if(obj instanceof DevicePlatformProvider){
			DevicePlatformProvider dpp = (DevicePlatformProvider)obj;
			sb = new StringBuffer(dpp.getVendor());
			sb.append(" (v.");
			sb.append(dpp.getVersion());
			sb.append(")");
			name = sb.toString();
		}
		else if(obj == null){
			name = "";
		}
		else {
			name = obj.toString();
		}
		return name;
	}
	
	public static String getObjectStyledName(Object obj){
		
		StringBuffer sb = new StringBuffer();
		
		if(obj instanceof DevicePlatform){
			sb.append("Device Platform: ");
			sb.append(((DevicePlatform)obj).getName());
		}
		else if(obj instanceof RuntimePlatformDefinition){
			sb.append("Runtime Platform: ");
			sb.append(((RuntimePlatformDefinition)obj).getName());
		}
		else if(obj instanceof Device){
			sb.append("Device: ");
			sb.append(((Device)obj).getName());
		}
		else if(obj instanceof TargetDevice){
			sb.append("Device: ");
			sb.append(((TargetDevice)obj).getDeviceName());
		}
		else if(obj instanceof DeviceProfile){
			sb.append("Profile: ");
			DeviceProfile tmp = (DeviceProfile)obj;
			sb.append(tmp.getName());
			sb.append("-"); sb.append(tmp.getVersion());
		}
		else if(obj instanceof DeviceConfiguration){
			sb.append("Configuration: ");
			DeviceConfiguration tmp = (DeviceConfiguration)obj;
			sb.append(tmp.getName());
			sb.append("-"); sb.append(tmp.getVersion());
		}
		else if(obj instanceof ServiceApi){
			sb.append("Service Api: ");
			ServiceApi tmp = (ServiceApi)obj;
			sb.append(tmp.getName());
			sb.append("-"); sb.append(tmp.getVersion());
		}
		else if(obj instanceof ImplementationRef){
			ImplementationRef ref = (ImplementationRef)obj;
			sb.append(ref.getFileRef());
		}
		else if(obj == null){
			//do nothing
		}
		else {
			sb.append(obj);
		}
		
		return sb.toString();
	}
	
	/** 
	 * Get a list of those ServiceApis that are not contained in the profile.
	 * 
	 * @param serviceApis list of RPD:s ServiceApis
	 * @param profileApis list of RPD:s DeviceProfile:s ServiceApis
	 * @return array of non-profile ServiceApis
	 */
	public static ServiceApi[] getCleanedServiceApis(EList serviceApis, EList profileApis){
		
		List l = new Vector<ServiceApi>();
		
		ServiceApi tmpApi = null;
		for(int i = 0; i < serviceApis.size(); i++){
			
			tmpApi = (ServiceApi)serviceApis.get(i);
			
			if(!profileApis.contains(tmpApi)){
				l.add(tmpApi);
			}
		}
		
		ServiceApi[] sapis = new ServiceApi[l.size()];
		
		return (ServiceApi[]) l.toArray(sapis);
		
//		Object[] o = v.toArray();
//		ServiceApi[] sapis = new ServiceApi[o.length];
//		for(int i = 0; i < o.length; i++){
//			
//			sapis[i] = (ServiceApi)o[i];
//		}
//		
//		return sapis;
	}
	
	public static String getPlatformTypeByName(String dpName){
		
		String typeName = "";
		
		//DP
		DeviceManagement dm =  (DeviceManagement)MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_MANAGEMENT_LITERAL, null, null)[0];
		try {
			DevicePlatform myDp = dm.getDevicePlatformByName(dpName);
			typeName = getPlatformTypeName(myDp);
		} catch (MtjException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		return typeName;
	}
	
	public static String getPlatformTypeName(DevicePlatform dp){
		
		if(dp != null){
			
			StringTokenizer st = new StringTokenizer(dp.getType().getName(), "_");
			StringBuffer sb = new StringBuffer();
			String s = null;
			while(st.hasMoreTokens()){
				
				s = st.nextToken();
				
				sb.append(s.charAt(0));
				sb.append(s.substring(1).toLowerCase());
				sb.append(" ");
			}
			
			return sb.toString().trim();
		}
		else{
			return new String();
		}
	}
	
	public DevicePlatformProvider[] getToolkitSupporters(){
		
		MtjExtension[] list = MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_PLATFORM_PROVIDER_LITERAL, null, null);
//		DevicePlatformProvider[] dppList = new DevicePlatformProvider[list.length];
		Vector v = new Vector();
		
		DevicePlatformProvider dpp = null;
		for(int i = 0; i < list.length; i++){
			
			dpp =  (DevicePlatformProvider)list[i];
			
			if(dpp instanceof ToolkitSupport){
				v.add(dpp);
			}
		}
		
		DevicePlatformProvider[] dppList = new DevicePlatformProvider[v.size()];
		
		return (DevicePlatformProvider[]) v.toArray(dppList);
	}
	
	public DevicePlatformProvider[] getDevicePlatformProviders(){
		
		MtjExtension[] list = MtjServices.getInstance().getImplementations(ExtensionType.DEVICE_PLATFORM_PROVIDER_LITERAL, null, null);
		DevicePlatformProvider[] dppList = new DevicePlatformProvider[list.length];
		
		for(int i = 0; i < list.length; i++){
			dppList[i] = (DevicePlatformProvider)list[i];
		}
		
		return dppList;
	}
	
	
	public DevicePlatform[] createDevicePlatformFromDirectory(ToolkitSupport toolkit, String rootLocation){
		
		File rootDir = new File(rootLocation);
		
		//TODO GUI testing -> remove these duplicates
		// DevicePlatformImporter importer = toolkit.getDevicePlatformImporter();
		// DevicePlatform[] dps = importer.getDevicePlatforms(rootDir);
		
		return toolkit.getDevicePlatformImporter().getDevicePlatforms(rootDir);
		
	}
	
}
