/*******************************************************************************
* Copyright (c) 2005,2006 Nokia Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
 * Additional Contributors -
 *  	Kevin Horowitz (IBM Corp) - Update javadoc
*******************************************************************************/
package org.eclipse.mtj.api.extension;

import org.eclipse.mtj.api.devices.Device;
import org.eclipse.mtj.api.devices.DeviceDescription;
import org.eclipse.mtj.api.devices.DeviceGroup;
import org.eclipse.mtj.api.devices.DevicePlatform;
import org.eclipse.mtj.api.enumerations.PlatformType;
import org.eclipse.mtj.api.extension.MtjExtension;
import org.eclipse.mtj.api.runtimes.DeviceConfiguration;
import org.eclipse.mtj.api.runtimes.DeviceProfile;
import org.eclipse.mtj.api.runtimes.RuntimePlatformDefinition;
import org.eclipse.mtj.api.runtimes.ServiceApi;
import org.eclipse.mtj.exception.MtjException;

/**
 * DeviceManagement provides services to fetch DevicePlatforms, Devices and Device Descriptions. 
 * It combines data from DevicePlatformProviders, DeviceDescriptionProviders and DeploymentProviders.
  */
public interface DeviceManagement extends MtjExtension {

	/**
	 * Returns all existing device platforms.
	 * 
	 * @return
	 */
	public DevicePlatform[] getDevicePlatforms() throws MtjException;

	/**
	 * Returns existing device platform by name.
	 * 
	 * @return
	 */
	public DevicePlatform getDevicePlatformByName(String name) throws MtjException;

	/**
	 * Returns the device platform provider that owns the device platform.
	 * 
	 * @param devicePlatform
	 * @return
	 * @throws MtjException
	 */
	public DevicePlatformProvider getDevicePlatformProvider(DevicePlatform devicePlatform) throws MtjException;

	/**
	 * Returns all devices of the device platform.
	 * 
	 * @param devicePlatformName
	 * @return
	 */
	public Device[] getDevices(String devicePlatformName) throws MtjException;

	/**
	 * Returns DeviceDescription object baded on the vendor and model.
	 * 
	 * @param vendor
	 * @param model
	 * @return
	 */
	public DeviceDescription getDeviceDescription(String vendor, String model) throws MtjException;

	/**
	 * Returns DeviceDescription objects that match to the device group.
	 * 
	 * @param group
	 * @return
	 */
	public DeviceDescription[] getMatchingDeviceDescriptions(DeviceGroup group) throws MtjException;

	/**
	 * Returns all defined device groups.
	 * 
	 * @return
	 */
	public DeviceGroup[] getDeviceGroups() throws MtjException;

	/**
	 * Returns all DeviceDescription objects that match to the configuration,
	 * profile and service api definitions.
	 * 
	 * @param deviceConfig
	 * @param deviceProfile
	 * @param apis
	 * @return
	 */
	public DeviceDescription[] getDevices(DeviceConfiguration deviceConfig,
			DeviceProfile deviceProfile, ServiceApi[] apis) throws MtjException;

	/**
	 * Returns all Device objects that match to the configuration,
	 * profile, service api and device platform type.
	 * 
	 * @param deviceConfig
	 * @param deviceProfile
	 * @param apis
	 * @param devicePlatformType
	 * @return
	 */
	public Device[] getTargetDevices(DeviceConfiguration deviceConfig,
			DeviceProfile deviceProfile, ServiceApi[] apis,
			PlatformType devicePlatformType) throws MtjException;

	/**
	 * Returns all the Runtime Platform Definition of the device platform's device.
	 * 
	 * @param devicePlatformName
	 * @param deviceName
	 * @return
	 */
	public RuntimePlatformDefinition[] getRuntimePlatforms(
			String devicePlatformName, String deviceName) throws MtjException;

	/**
	 * Returns all deployment target Device objects that match to the configuration,
	 * profile and service api definitions and device platform type.
	 * 
	 * @param deviceConfig
	 * @param deviceProfile
	 * @param apis
	 * @param devicePlatformType
	 * @return
	 */
	public Device[] getDeployableDevices(DeviceConfiguration deviceConfig,
			DeviceProfile deviceProfile, ServiceApi[] apis,
			PlatformType devicePlatformType) throws MtjException;

	/**
	 * Returns DevicePlatform's Device whose platform name and device name match.
	 * 
	 * @param devicePlatformName
	 * @param deviceName
	 * @return
	 * @throws MtjException
	 */
	public Device getDevice(String devicePlatformName, String deviceName) throws MtjException;
	
	/**
	 * Removes the DevicePlatform from the owner DevicePlatformProdiver if the provider 
	 * implements the ToolkitSupport interface.
	 * 
	 * @param devicePlatform
	 * @throws MtjException
	 */
	public void removeDevicePlatform(DevicePlatform devicePlatform) throws MtjException;
	
}
