/******************************************************************************* 
 * Copyright (c) 2005 Nokia Corporation                                         
 * Copyright (c) 2004 Craig Setera 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Nokia -  Initial API and implementation 
 * Craig Setera - partial implementation 
 *******************************************************************************/ 

package org.eclipse.mtj.core.internal.ota;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.internal.utils.Utils;
import org.mortbay.http.HttpException;
import org.mortbay.http.HttpRequest;
import org.mortbay.http.HttpResponse;
import org.mortbay.http.handler.AbstractHttpHandler;

/**
 * Jetty HttpHandler implementation for the OTA implementation.
 * <p />
 * Copyright (c) 2003-2005 Craig Setera<br>
 * All Rights Reserved.<br>
 * Licensed under the Eclipse Public License - v 1.0<p/>
 * <br>
 * $Revision: 1.1.1.1 $
 * <br>
 * $Date: 2006/08/30 18:50:23 $
 * <br>
 * @author Craig Setera
 */
public class OTAHandler extends AbstractHttpHandler {
	/**
	 * Provde default SerialVersionID
	 */
	private static final long serialVersionUID = 1L;

	// Internal map of the known MIME types
	private static final String[] KNOWN_MIME_TYPES = new String[] {
		"jad", "text/vnd.sun.j2me.app-descriptor", //$NON-NLS-1$ //$NON-NLS-2$
		"jar", "application/java-archive" //$NON-NLS-1$ //$NON-NLS-2$
	};

	/**
	 * Internal map of file extensions to MIME types.
	 */
	private static Map mimeTypeMap;

	/**
	 * 
	 * @uml.property name="mimeTypeMap"
	 */
	private static Map getMimeTypeMap() {
		if (mimeTypeMap == null) {
			mimeTypeMap = new HashMap();

			for (int i = 0; i < KNOWN_MIME_TYPES.length;) {
				mimeTypeMap.put(KNOWN_MIME_TYPES[i++], KNOWN_MIME_TYPES[i++]);
			}
		}

		return mimeTypeMap;
	}

	
	/**
	 * Default constructor
	 */
	public OTAHandler() {
		super();
	}

	/**
	 * @see org.mortbay.http.HttpHandler#getName()
	 */
	public String getName() {
		return "EclipseME Over the Air Handler";
	}

	/**
	 * @see org.mortbay.http.HttpHandler#handle(java.lang.String, java.lang.String, org.mortbay.http.HttpRequest, org.mortbay.http.HttpResponse)
	 */
	public void handle(
		String pathInContext, 
		String pathParams, 
		HttpRequest request, 
		HttpResponse response) 
			throws HttpException, IOException 
	{
		String project = null;
		String file = null;
		
		StringTokenizer st = new StringTokenizer(pathInContext, "/"); //$NON-NLS-1$
		if (st.hasMoreTokens()) project = st.nextToken();
		if (st.hasMoreTokens()) file = st.nextToken();
		
		if ((project != null) && (file != null)) {
			try {
				handle(project, file, pathParams, request, response);
			} catch (CoreException e) {
				throw new IOException(e.getMessage());
			}
		}
	}

	/**
	 * Get the appropriate content encoding for the specified
	 * file.
	 * 
	 * @param requestedFile
	 * @return
	 */
	private String getContentEncoding(IFile requestedFile) {
		return (requestedFile.getFileExtension().equals("jad")) ? //$NON-NLS-1$
			"UTF-8" : null; //$NON-NLS-1$
	}

	/**
	 * @param requestedFile
	 * @return
	 */
	private String getContentType(IFile requestedFile) {
		String contentType = 
			(String) getMimeTypeMap().get(requestedFile.getFileExtension());
			
		return (contentType != null) ?
			contentType : "application/octet-stream"; //$NON-NLS-1$
	}

	/**
	 * Return the IJavaProject for the specified name or
	 * <code>null</code> if no such project is found.
	 * 
	 * @param projectName
	 * @return
	 */
	private IJavaProject getJavaProject(String projectName) {
		IJavaProject javaProject = null;
		
		IWorkspaceRoot root = 
			MtjCorePlugin.getWorkspace().getRoot();
		IProject project = root.getProject(projectName);
		
		if (project != null) {
			javaProject = JavaCore.create(project);
		}
		
		return javaProject;
	}

	/**
	 * Handle sending the specified file in the specified Eclipse
	 * project.
	 * 
	 * @param project
	 * @param file
	 * @param pathParams
	 * @param request
	 * @param response
	 * @throws CoreException
	 * @throws IOException
	 */
	private void handle(
		String project, 
		String file, 
		String pathParams, 
		HttpRequest request, 
		HttpResponse response) 
			throws CoreException, IOException 
	{
		IJavaProject javaProject = getJavaProject(project);
		if (javaProject != null) {
			handle(javaProject, file, pathParams, request, response);
		}
	}

	/**
	 * Handle sending the specified file in the specified Eclipse
	 * Java Project.
	 * 
	 * @param javaProject
	 * @param file
	 * @param pathParams
	 * @param request
	 * @param response
	 * @throws CoreException
	 * @throws IOException
	 */
	private void handle(
		IJavaProject javaProject, 
		String file, 
		String pathParams, 
		HttpRequest request, 
		HttpResponse response) 
			throws CoreException, IOException 
	{
		// TODO: Get this from project extension
//		String deploymentFolderName = MtjCorePlugin.getDeploymentDirectoryName();
		String deploymentFolderName = ""; //$NON-NLS-1$
		IFolder deployedFolder = 
			javaProject.getProject().getFolder(deploymentFolderName);
			
		if (deployedFolder.exists()) {
			IFile requestedFile = deployedFolder.getFile(file);
			
			if (requestedFile.exists()) {
				sendFile(requestedFile, response);
			}
		}
	}

	/**
	 * Send the specified file back to the caller.
	 * 
	 * @param requestedFile
	 * @param response
	 * @throws CoreException
	 * @throws IOException
	 */
	private void sendFile(IFile requestedFile, HttpResponse response)
		throws CoreException, IOException 
	{
		String contentType = getContentType(requestedFile);
		String contentEncoding = getContentEncoding(requestedFile);
		
		File javaFile = new File(requestedFile.getLocation().toOSString());
		response.setCharacterEncoding(contentEncoding);
		response.setContentType(contentType);
		response.setContentLength((int) javaFile.length());
		
		InputStream is = requestedFile.getContents();
		OutputStream os = response.getOutputStream();
		Utils.copyInputToOutput(is, os);
		os.close();
		is.close();
		
		response.commit();
	}
}
