/*******************************************************************************
* Copyright (c) 2005 Nokia Corporation
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
*******************************************************************************/
package org.eclipse.mtj.core.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.core.JavaProject;
import org.eclipse.mtj.core.Messages;

/**
 *
 * This class houses static util methods related directly to Eclipse.
 */
public class EclipseUtil
{
	
	/**
	 * This method returns IFile object resolved from the given filesystem path 
	 * 
	 * @param filename the name of the file to be resolved. Must be absolute filename with path and does not have to belong to the project.
	 * @returns IFile object resolved from the given filename or null if the filename is not in any workspace projects.
	 */
	public static IFile getIFileForFilename (String fname)
	{
		String filename = fname;
		Path path = new Path(filename);
		IFile f = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
		if ( f.exists() ) {
			return f;
		}
		
		String canonicalPath = ""; //$NON-NLS-1$
		try {
			//given filename might have dubious case, resolve case-sensitivity to correct one:
			File file = new File(fname.startsWith("file:/") ? fname.substring("file:/".length()) : fname); //$NON-NLS-1$ //$NON-NLS-2$
			canonicalPath = file.getCanonicalPath();
		}
		catch (IOException ioe)
		{
			return null;
		}
		//NOTE: the given file must exist in some of the workspace project, and case-sensitivity is important also in the project name!!
		return ResourcesPlugin.getWorkspace().getRoot().getFileForLocation(new Path(canonicalPath));
	}

	public static IFolder getIFolderForPath (String fname)
	{
		String filename = fname;
		Path path = new Path(filename);
		IFolder f = ResourcesPlugin.getWorkspace().getRoot().getFolder(path);
		if ( f != null && f.exists() ) {
			return f;
		}
		
		return null;
	}

	/**
	 * @param builderId Id of the builder to add to the project
	 * @param project Project
	 * @throws CoreException
	 */
	public static void addBuilderToProject (String builderId, IProject project) throws CoreException
	{
		// closed projects can't be modified:
		if (!project.isOpen()) {
			return;
		}
		
		IProjectDescription projectDescription = project.getDescription();
		
		//is the builder already associated with the project:
		ICommand[] oldCommands = projectDescription.getBuildSpec();
		for (int i = 0; i < oldCommands.length; i++) {
			if (oldCommands[i].getBuilderName().equals(builderId)) {
				return;
			}
		}
		
		//associate builder with project:
		ICommand newCommand = projectDescription.newCommand();
		newCommand.setBuilderName(builderId);
		List newCommands = new ArrayList();
		newCommands.addAll(Arrays.asList(oldCommands));
		newCommands.add(newCommand);
		projectDescription.setBuildSpec((ICommand[])newCommands.toArray(new ICommand[newCommands.size()]) );
		project.setDescription(projectDescription, null);
	}
	
	/**
	 * @param builderId Id of the builder to remove from the project
	 * @param project Project
	 * @throws CoreException
	 */
	public static void removeBuilderFromProject (String builderId, IProject project) throws CoreException
	{
		// closed projects can't be modified:
		if (!project.isOpen()) {
			return;
		}
		
		IProjectDescription projectDescription = project.getDescription();
		
		//is the builder associated with the project:
		int idx = -1;
		ICommand[] oldCommands = projectDescription.getBuildSpec();
		for (int i = 0; i < oldCommands.length; i++) {
			if (oldCommands[i].getBuilderName().equals(builderId)) {
				idx = i;
				break;
			}
		}
		if (idx == -1) {
			return;
		}
		//remove builder from project:
		List newCommands = new ArrayList();
		newCommands.addAll(Arrays.asList(oldCommands));
		newCommands.remove(idx);
		projectDescription.setBuildSpec((ICommand[])newCommands.toArray(new ICommand[newCommands.size()]) );
		project.setDescription(projectDescription, null);
	}

	public static boolean hasProjectBuilder (String builderId, IProject project) throws CoreException
	{
		// closed projects can't be modified:
		if (!project.isOpen()) {
			return false;
		}
		
		IProjectDescription projectDescription = project.getDescription();
		
		ICommand[] oldCommands = projectDescription.getBuildSpec();
		for (int i = 0; i < oldCommands.length; i++) {
			if (oldCommands[i].getBuilderName().equals(builderId)) {
				return true;
			}
		}

		return false;
	}

	/**
	 * @param natureId Id of the nature to add/remove to the project
	 * @param project Project
	 * @throws CoreException
	 */
	public static void toggleProjectNature (String natureId, IProject project) throws CoreException
	{
		// closed projects can't be modified:
		if (!project.isOpen()) {
			return;
		}
		
		IProjectDescription projectDescription = project.getDescription();
		
		// get all natures for the project:
		List natureIds = new ArrayList();
		natureIds.addAll(Arrays.asList(projectDescription.getNatureIds()));
		
		//is the nature already associated with the project:
		int i = natureIds.indexOf(natureId);
		if (i == -1) {
			natureIds.add(natureId);
		} else {
			natureIds.remove(i);
		}
			
		projectDescription.setNatureIds((String[])natureIds.toArray(new String[0]));
		//projectDescription.setNatureIds((String[])natureIds.toArray(new String[natureIds.size()]));
		// project.setDescription(projectDescription, null);
		project.setDescription(projectDescription, IProject.AVOID_NATURE_CONFIG, null);
		
	}
	
	/**
	 * @param natureId
	 * @param project
	 * @throws CoreException
	 */
	public static void addProjectNature (String natureId, IProject project) throws CoreException
	{
		// closed projects can't be modified:
		if (!project.isOpen()) {
			return;
		}
		IProjectDescription projectDescription = project.getDescription();
		
		// get all natures for the project:
		List natureIds = new ArrayList();
		natureIds.addAll(Arrays.asList(projectDescription.getNatureIds()));

		//is the nature already associated with the project:
		int i = natureIds.indexOf(natureId);
		if (i == -1) {
			natureIds.add(natureId);
			projectDescription.setNatureIds((String[])natureIds.toArray(new String[0]));
			project.setDescription(projectDescription, null);
		} 
			
		
	}
	
	/**
	 * @param natureId
	 * @param project
	 * @throws CoreException
	 */
	public static void removeProjectNature (String natureId, IProject project) throws CoreException
	{
		// closed projects can't be modified:
		if (!project.isOpen()) {
			return;
		}
		IProjectDescription projectDescription = project.getDescription();
		
		// get all natures for the project:
		List natureIds = new ArrayList();
		natureIds.addAll(Arrays.asList(projectDescription.getNatureIds()));

		//is the nature already associated with the project:
		int i = natureIds.indexOf(natureId);
		if (i != -1) {
			natureIds.remove(i);
			projectDescription.setNatureIds((String[])natureIds.toArray(new String[0]));
			project.setDescription(projectDescription, null);
		}		
		
	}
	
	/**
	 * This method returns an array of open projects which has the given Nature.
	 * @param natureId Id of the nature the project
	 * @return array of projects which has given nature
	 * @throws CoreException
	 */
	public static IProject[] getProjectsForNature (String natureId) throws CoreException
	{
		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
		ArrayList projectsOfNature = new ArrayList();
		
		for (int i = 0; i < projects.length; i++) {
			IProject project = projects[i];
			
			if (project.isOpen()) {		// closed projects can't be used
				IProjectDescription projectDescription = project.getDescription();
				// get all natures for the project:
				List natureIds = new ArrayList();
				natureIds.addAll(Arrays.asList(projectDescription.getNatureIds()));
				
				//is the nature associated with the project:
				int j = natureIds.indexOf(natureId);
				if (j != -1) {
					projectsOfNature.add(project);
				}
			}
		}
			
		return (IProject[])projectsOfNature.toArray(new IProject[projectsOfNature.size()]);
	}

	
	/**
	 * This method tells if the project is of given nature.
	 * @param projectId Id of the project
	 * @param natureId Id of the nature
	 * @return true if it is of given nature
	 * @throws CoreException
	 */
	public static boolean isProjectOfNature (String projectId, String natureId) throws CoreException
	{
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectId);

		if (project.isOpen()) {
			IProjectDescription projectDescription = project.getDescription();
			// get all natures for the project:
			List natureIds = new ArrayList();
			natureIds.addAll(Arrays.asList(projectDescription.getNatureIds()));
					
			//is the nature associated with the project:
			if (natureIds.indexOf(natureId) != -1) {
				return true;
			} else {
				return false;
			}
		} else {
			return false;	// we must return false, although the closed project might have requested nature
		}
	}

	
	/**
	 * This method tells if the project is open.
	 * @param projectId Id of the project
	 * @return true if it is open
	 * @throws CoreException
	 */
	public static boolean isProjectOpen (String projectId) throws CoreException
	{
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectId);

		if (project.isOpen()) {
			return true;
		} else {
			return false;
		}
	}
	
	/**
	 * This method returns the project for the given project-id.
	 * @param projectId Id of the project
	 * @return project object
	 * @throws CoreException
	 */
	public static IProject getProject (String projectId) throws CoreException
	{
		return ResourcesPlugin.getWorkspace().getRoot().getProject(projectId);
	}
	
	/**
	 * This method creates a 'Simple' project into the given path
	 * @param projectName name of the project to be created
	 * @param projectPath name of the path where the project will be created
	 * @throws CoreException
	 */
	public static IProject createSimpleProject (String projectName, String projectPath) throws CoreException
	{
		return createProject(projectName, projectPath, null, null);
	}
	
	/**
	 * This method creates a project into the given path using given natures and builders
	 * @param projectName name of the project to be created
	 * @param projectPath name of the path where the project will be created
	 * @param natures array of project natures (Strings) or null for Simple project
	 * @param builders array of project builders or null for Simple project
	 * @throws CoreException
	 */
	public static IProject createProject (String projectName, String projectPath, String[] natures, ICommand[] builders) throws CoreException
	{
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);  // used in creation too
		IProjectDescription projectDescription = ResourcesPlugin.getWorkspace().newProjectDescription(projectName);
		if (projectPath != null) {
			projectDescription.setLocation(new Path(projectPath));
		}
		if (natures == null) {
			projectDescription.setNatureIds(new String[] { });
		} else {
			projectDescription.setNatureIds(natures);
		}
		if (builders == null) {
			projectDescription.setBuildSpec(new ICommand[] { });
		} else {
			projectDescription.setBuildSpec(builders);
		}
		project.create(projectDescription, new NullProgressMonitor());
		return project;
	}
	
	/**
	 * This method deletes the given project completely (include the files)
	 * @param projectName name of the project to be deleted
	 * @throws CoreException
	 */
	public static void deleteProjectCompletely (String projectName) throws CoreException
	{
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		project.delete(true, true, new NullProgressMonitor());
	}
	
	/**
	 * This method creates a task marker which will be seen in the Tasks -view.
	 * @param file
	 * @param type
	 * @param priority
	 * @param message
	 * @param line
	 * @param col
	 * @return
	 */
	public static IMarker makeTaskMarker (IFile file, String type, int priority, String message, int line, int col)
	{
		IMarker marker = null;
		try {			
			marker = file.createMarker(IMarker.TASK);	// using 'type' causes that IMarker.PROBLEM is not deletable in the tasks list  (IMarker.TASK)
			marker.setAttribute(IMarker.DONE, false);
			marker.setAttribute(IMarker.MESSAGE, message);
			//marker.setAttribute(IMarker.USER_EDITABLE, false); //default is true, with false the row cannot be deleted.
			marker.setAttribute(IMarker.PRIORITY, new Integer(priority));
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR); //new
			if (line != -1)
				marker.setAttribute(IMarker.LINE_NUMBER, new Integer(line));
			if (col != -1)
				marker.setAttribute(IMarker.LOCATION, "column " + col); //$NON-NLS-1$
				
		} catch (Exception e) {}
		return marker;
	}
	

	/**
	 * getJavaProjectSourceDirectories
	 * @param projectName
	 * @return array of java source directories for this project (absolute full path for the directory)
	 * @throws CoreException
	 */
	public static String[] getJavaProjectSourceDirectories (String projectName) throws CoreException, JavaModelException
	{
		ArrayList paths = new ArrayList();
		
		IProject project = getProject(projectName);
		if (project.isOpen() && JavaProject.hasJavaNature(project)) {
			IJavaProject javaProject = JavaCore.create(getProject(projectName));
		
			IClasspathEntry[] classpathEntries = null;
			classpathEntries = javaProject.getResolvedClasspath(true);
			
			for (int i = 0; i < classpathEntries.length; i++) {
				IClasspathEntry entry = classpathEntries[i];
				if (entry.getContentKind() == IPackageFragmentRoot.K_SOURCE)
				{
					// now we have IPath, which contains path in format /{projectName}/src/java and it is changed to c:\\proj\\projectdir\\src\\java
					String relativePath = entry.getPath().toString();	
					String absolutePath = null;
					if (relativePath.charAt(0) != '/') {		// Whatta heck is this? Try to handle it decently
						absolutePath = relativePath;
					} else {
						int j = relativePath.substring(1).indexOf('/');		// skip the first slash, after second slash its all ours
						if (j != -1) {
							absolutePath = relativePath.substring(j+1);  
						}
					}
					String javadir = null;
					
					if (absolutePath != null) {
						javadir = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName).getFile(absolutePath).getLocation().toString();  // gives out system file path
					} else {
						javadir = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName).getLocation().toString();  // no subdirs, sources are straight under the project.
					}
					paths.add(javadir);
					
				}
			}
		}
		
		return (String[])paths.toArray(new String[0]);
	}
	

    public static String readFile(String filename) throws Exception {
		IFile iFile = EclipseUtil.getIFileForFilename(filename);
		if ( iFile != null ) {
			try {
				iFile.refreshLocal(IResource.DEPTH_ONE, null);
			} catch (Throwable e1) { }
			try {
				InputStream is = iFile.getContents();
				int s = is.available();
				byte[] b = new byte[s];
				is.read(b);
				is.close();
				return new String(b);
			}
			catch (Throwable ex) { }
		}
		try {
			return readFileContent(filename);
		}
		catch (Throwable ex2) {
			throw new Exception(Messages.EclipseUtil_ErrorReadingFileContents,ex2);
		}
    }
    
    public static void deleteFile(String filename) throws CoreException {
    	IFile iFile = EclipseUtil.getIFileForFilename(filename);
    	if ( iFile != null ) {
    		iFile.delete(true, true, null);	// force deletion, and keep local history just in case
    	}
    	else {
		    String froot = filename.startsWith("file:/") ? filename.substring("file:/".length()) : filename; //$NON-NLS-1$ //$NON-NLS-2$
		    File file = new File(froot);
		    file.delete();
    	}
    }
    
    public static void writeFile(String filename, String filecontent) throws Exception {
    	writeFile( filename,  filecontent, null, -1);
    }
    
    public static void writeFile(String filename, String filecontent, String parentFolder, int depth) throws Exception {
		IFile iFile = EclipseUtil.getIFileForFilename(filename);
		if ( iFile != null ) {
			try {
				String fname = filename.replace('\\','/');
				if ( fname.indexOf("/") != -1 ) { //$NON-NLS-1$
					fname = fname.substring(0,fname.lastIndexOf("/")); //$NON-NLS-1$
				}
				File f = new File(fname);
				f.mkdirs();
				if ( parentFolder != null ) {
					IFile parent = EclipseUtil.getIFileForFilename(parentFolder);
					parent.refreshLocal(depth,null);
				}
				else {
					iFile.refreshLocal(1,null);
				}
			} catch (CoreException ex1) {
				throw new Exception(Messages.EclipseUtil_ErrorRefreshingFile,ex1);
			}
			try {
				try {
					InputStream input = new BufferedInputStream(new ByteArrayInputStream(filecontent.getBytes()));
					iFile.setContents(input, IResource.KEEP_HISTORY, null);
				}
				catch (Exception ex) {
					InputStream input = new BufferedInputStream(new ByteArrayInputStream(filecontent.getBytes()));
					iFile.create(input, IResource.KEEP_HISTORY, null);
				}
			}
			catch (Exception ex) {
				throw new Exception(Messages.EclipseUtil_ErrorSaviningFileContents,ex);
			}
		}
		else {
		    String fileroot = filename.startsWith("file:/") ? filename.substring("file:/".length()) : filename; //$NON-NLS-1$ //$NON-NLS-2$
		    FileOutputStream fos = new FileOutputStream(new File(fileroot));
		    fos.write(filecontent.getBytes());
		    fos.close();
		}
    }
    
	public static String[] getLinkedResources(IProject project) {
		ArrayList paths = new ArrayList();
		
		String path = project.getLocation().toString() + "/.project"; //$NON-NLS-1$
		File file = new File(path);
		if ( file.exists() ) {
			try {
				String content = readFile(file.toString());
				String[] s = parseContentBetweenTokens(content, "<location>", "</location>"); //$NON-NLS-1$ //$NON-NLS-2$
				while ( s[1].length() > 0 ) {
					paths.add(s[1]);
					s = parseContentBetweenTokens(s[2], "<location>", "</location>"); //$NON-NLS-1$ //$NON-NLS-2$
				}
			} catch (Exception e) {
				// error while reading file contents
			}
		}
		
		return (String[])paths.toArray(new String[0]);
	}

	
	public static String findFileRecursively (IResource resource, String filename) throws CoreException
	{
		IResource[] resources;

		if (resource instanceof IFile) {
			int fpos;
			String filepath = ((IFile)resource).getLocation().toString();
			if ((fpos = filepath.lastIndexOf(filename)) != -1) {
				if (fpos + filename.length() == filepath.length()) {	// must be the last one in the path
					return filepath;
				}
			}
		} else if (resource instanceof IProject) {
			resources = ((IProject)resource).members();
			for (int i = 0; i < resources.length; i++) {
				IResource resource2 = resources[i];
				String ret = findFileRecursively(resource2, filename);
				if (ret != null) {
					return ret;
				}
			}
		} else if (resource instanceof IFolder) {
			resources = ((IFolder)resource).members();
			for (int i = 0; i < resources.length; i++) {
				IResource resource3 = resources[i];
				String ret = findFileRecursively(resource3, filename);
				if (ret != null) {
					return ret;
				}
			}
		}

		return null;
	}

	
	public static ArrayList findDirectories (IProject project, String dirNameEnd) throws CoreException
	{
		ArrayList dirNameList = new ArrayList();
		IProject baseProject = project;
		
		findDirectoriesRecursively (baseProject, baseProject, dirNameEnd, dirNameList);
		
		return dirNameList;
	}
	
	private static void findDirectoriesRecursively (IProject project, IResource resource, String dirNameEnd, ArrayList dirNameList) throws CoreException
	{
		IResource[] resources;

		if (resource instanceof IProject) {
			resources = ((IProject)resource).members();
			for (int i = 0; i < resources.length; i++) {
				IResource resource2 = resources[i];
				findDirectoriesRecursively(project, resource2, dirNameEnd, dirNameList);
			}
		} else if (resource instanceof IFolder) {
			int fpos;
			String path = project.getFile(resource.getProjectRelativePath().toString()).getLocation().toString();  // gives out system file path
			if ((fpos = path.lastIndexOf(dirNameEnd)) != -1) {
				if (fpos + dirNameEnd.length() == path.length()) {		
					try {
						path = new File(path).getCanonicalPath();	// Eclipse has severe problems with case-sensitivy, here we restore the right case back.
					} catch (IOException ioe) {}
					path = path != null ? path.replace('\\', '/') : path;	// convert \\ to /
					dirNameList.add(path);
				}
			}
			resources = ((IFolder)resource).members();
			for (int i = 0; i < resources.length; i++) {
				IResource resource3 = resources[i];
				findDirectoriesRecursively(project, resource3, dirNameEnd, dirNameList);
			}
		}
	}
	
	private static String[] parseContentBetweenTokens(String content, String startTag, String endTag) {
		ArrayList c = new ArrayList();
		
		if ( content.indexOf(startTag) != -1 ) {
			String a = content.substring(0,content.indexOf(startTag));
			String s = content.substring(content.indexOf(startTag) + startTag.length());
			if ( s.indexOf(endTag) != -1 ) {
				String l = s.substring(s.indexOf(endTag) + endTag.length());
				s = s.substring(0,s.indexOf(endTag));
				c.add(a);
				c.add(s);
				c.add(l);
			}
			else {
				c.add(a);
				c.add(s);
				c.add(""); //$NON-NLS-1$
			}
		}
		else {
			c.add(content);
			c.add(""); //$NON-NLS-1$
			c.add(""); //$NON-NLS-1$
		}

		return (String[])c.toArray(new String[0]);
	}

	private static String readFileContent(String froot) throws Exception {
		    String fileroot = froot.startsWith("file:/") ? froot.substring("file:/".length()) : froot; //$NON-NLS-1$ //$NON-NLS-2$
		    File f = new File(fileroot);
		    FileInputStream bais = new FileInputStream(f);
		    int len = bais.available();
		    byte[] b = new byte[len];
		    bais.read(b);
		    bais.close();
		    return new String(b);

	}

	/**
	 * Recursively set the resources in the specified container and all resources
	 * within that container as derived.
	 * 
	 * @param container
	 * @throws CoreException
	 */
	public static void setResourcesAsDerived(IContainer container)
		throws CoreException 
	{
		if (container.exists()) {
			// Mark this folder first...
			container.setDerived(true);
					
			// Recursively handle the members of the directory
			IResource[] resources = container.members();
			for (int i = 0; i < resources.length; i++) {
				IResource resource = resources[i];
				if (resource instanceof IContainer) {
					setResourcesAsDerived((IContainer) resource);
				} else {
					resource.setDerived(true);
				}
			}
		}
	}

}
