/*******************************************************************************
* Copyright (c) 2005,2006 Nokia Corporation and others.
* All rights reserved. This program and the accompanying materials
* are made available under the terms of the Eclipse Public License v1.0
* which accompanies this distribution, and is available at
* http://www.eclipse.org/legal/epl-v10.html
 *
 * Additional Contributors - 
 * 		Kevin Horowitz (IBM Corp.) - Add the zip of a resource directory to the packaging
*******************************************************************************/
package org.eclipse.mtj.extension.pp;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Properties;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.mtj.api.deployment.Deployment;
import org.eclipse.mtj.api.enumerations.DeploymentType;
import org.eclipse.mtj.api.enumerations.ExtensionType;
import org.eclipse.mtj.api.extension.PackagingProvider;
import org.eclipse.mtj.api.extension.impl.BuildExtensionImpl;
import org.eclipse.mtj.api.model.IMtjProject;
import org.eclipse.mtj.api.model.packaging.MidpPackagingResources;
import org.eclipse.mtj.api.project.Parameter;
import org.eclipse.mtj.api.project.Project;
import org.eclipse.mtj.core.MtjCorePlugin;
import org.eclipse.mtj.core.jad.IJADConstants;
import org.eclipse.mtj.core.version.Version;
import org.eclipse.mtj.exception.MtjException;
import org.eclipse.mtj.extension.devide.project.MtjProject;
import org.eclipse.mtj.extension.devide.utils.FilteringClasspathEntryVisitor;
import org.eclipse.mtj.internal.utils.ColonDelimitedProperties;
import org.eclipse.mtj.internal.utils.EntryTrackingJarOutputStream;
import org.eclipse.mtj.internal.utils.Utils;
import org.eclipse.mtj.jad.util.SynchronizeManifestFileToJad;

public class MidpPackagingProviderImpl extends BuildExtensionImpl implements PackagingProvider {
	public static final String EXT_JAR_NAME = "packager.jar"; //$NON-NLS-1$
	public static final String RESOURCE_DIRECTORY_NAME = "res"; //$NON-NLS-1$
	
	public MidpPackagingProviderImpl() {
		super();
		
		setId("org.eclipse.mtj.extension.pp.MIDLET"); //$NON-NLS-1$
		setDescription(Messages.MidpPackagingProviderImpl_PluginDescription);
		setVendor(Messages.MidpPackagingProviderImpl_PluginVendor);
		setVersion(Messages.MidpPackagingProviderImpl_PluginVersion);
		setType(ExtensionType.PACKAGING_PROVIDER_LITERAL);
		
		setExtJar(EXT_JAR_NAME);
	}

	public DeploymentType[] getSupportedTypes() throws MtjException {
		DeploymentType[] ret = { DeploymentType.DEPLOYMENT_TYPE_MIDLET_LITERAL };
		return ret;
	}

	Properties midletProperties;
	
	public void setMidletProperties(Properties midletProperties) {
		this.midletProperties = midletProperties;
	}
	
	private void deploy(String projectName, Project projectData, IResource[] resources,
			IFolder deploymentFolder, DeploymentType type, String natureId, IProgressMonitor monitor) throws CoreException, IOException {
		
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		IProject project = workspace.getRoot().getProject(projectName);
		IMtjProject mtjProject = null;
		try {
			mtjProject = MtjProject.getMtjProject(project);
		} catch (MtjException e) {}
		
		IFolder projectFolder = ResourcesPlugin.getWorkspace().getRoot().getFolder(project.getLocation());
				
		IFile finalJarFile = getJarFile(projectName, deploymentFolder, ""); //$NON-NLS-1$
		if (finalJarFile.exists()) finalJarFile.delete(true, monitor);

		// If the user wants auto versioning to occur, do it now
		boolean autoVersion = projectData.getPackagingDetails() != null ?
			projectData.getPackagingDetails().isIncrementVersionAutomatically() : false;
		if (autoVersion) {
			updateJADVersion(projectName, projectFolder, monitor);
		}
		
		MidpPackagingResources midpPackagingResources = new MidpPackagingResources(resources);
		IFolder verifiedClassesOutputFolder = midpPackagingResources.getVerifiedClassesOutputFolder();
		IFolder verifiedLibrariesOutputFolder = midpPackagingResources.getVerifiedLibrariesOutputFolder();
			
		IFolder resourcesFolder = null;
		if (mtjProject != null)
			resourcesFolder = mtjProject.getFolder(RESOURCE_DIRECTORY_NAME, monitor);
		// Do the actual build
		File deployedJar = createDeployedJarFile(projectName, projectData, 
				verifiedLibrariesOutputFolder, verifiedClassesOutputFolder,
				resourcesFolder,
				monitor, deploymentFolder, projectFolder, natureId);
		copyAndUpdateJadFile(projectName, deploymentFolder, deployedJar);
	
		// Let Eclipse know about changes and mark
		// everything as derived.
		deploymentFolder.refreshLocal(IResource.DEPTH_ONE, monitor);		
	}
	
	public Deployment createDeployment(String projectName, Project projectData, IResource[] resources,
			IFolder deploymentFolder, DeploymentType type, String natureId, IProgressMonitor monitor) throws MtjException {
		try {
			deploy(projectName, projectData, resources, deploymentFolder, type, natureId, monitor);
			return null;
		} catch (CoreException ex) {
			throw new MtjException();
		}
		catch (IOException ex) {
			throw new MtjException();
		}
	}

	/**
	 * Update the JAD version in the manifest properties.
	 * @throws IOException
	 * @throws CoreException
	 */
	private void updateJADVersion(String projectName, IFolder folder, IProgressMonitor monitor) 
		throws IOException, CoreException 
	{
		// Read the source jad file and update the jar file
		// length property.
		ColonDelimitedProperties jadProperties = getSourceJADProperties(projectName);
		
		// Calculate the updated version string
		String versionString = 
			jadProperties.getProperty(IJADConstants.JAD_MIDLET_VERSION, "0.0.0"); //$NON-NLS-1$
		Version version = new Version(versionString);
		
		StringBuffer newVersion = new StringBuffer();
		
		String major = version.getMajor();
		newVersion.append((major != null) ? major : "0").append("."); //$NON-NLS-1$ //$NON-NLS-2$
		
		String minor = version.getMinor();
		newVersion.append((minor != null) ? minor : "0").append("."); //$NON-NLS-1$ //$NON-NLS-2$
		
		int secondaryInt = 0;
		String secondary = version.getSecondary();
		if (secondary == null) secondary = "0"; //$NON-NLS-1$
		try { secondaryInt = Integer.parseInt(secondary); } catch (NumberFormatException e) {}
		secondaryInt++;
		newVersion.append(secondaryInt);
		
		// Update the JAD
		jadProperties.setProperty(IJADConstants.JAD_MIDLET_VERSION, newVersion.toString());
		IFile jadSourceFile = getJadFile(projectName);
		writeJad(projectName, folder, jadProperties);
		jadSourceFile.refreshLocal(IResource.DEPTH_ONE, monitor);
	}	
	
	/**
	 * Get the IFile instance into which the JAR will be
	 * written.
	 * 
	 * @return
	 */
	private IFile getJarFile(String projectName, IFolder deploymentFolder, String extension) {
		// TODO: This is wrong.  This should come from the jad file
		// KMH:
		String jarFileName = projectName.replace(' ', '_') + extension + ".jar"; //$NON-NLS-1$

		return deploymentFolder.getFile(jarFileName);
	}

	/**
	 * Create the deployed JAR file.
	 * 
	 * @param deployedJarFile
	 * @throws IOException
	 */
	private File createDeployedJarFile(
			String projectName, 
			Project projectData, 
			IFolder verifiedLibrariesOutputFolder,
			IFolder verifiedClassesOutputFolder,
			IFolder resourcesFolder,
			IProgressMonitor monitor, 
			IFolder deploymentFolder,
			IFolder projectFolder,
			String natureId)
		throws CoreException, IOException 
	{
		// The jar file
		IFile jarIFile = getJarFile(projectName, deploymentFolder, ""); //$NON-NLS-1$
		
		IFile jadIFile = getJadFile(projectName);
		SynchronizeManifestFileToJad info = new SynchronizeManifestFileToJad (jadIFile);
		try {
			info.run(monitor);
		} catch (InvocationTargetException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		IFile manifestIFile = info.getManifestFile();
		
		
		Manifest jarManifest = new Manifest (manifestIFile.getContents());

		File jarFile = resourceToFile(jarIFile);
		
		// Open up the JAR output stream to which the contents
		// will be added.
		FileOutputStream fos = new FileOutputStream(jarFile);
		EntryTrackingJarOutputStream jarOutputStream = 
			new EntryTrackingJarOutputStream(fos, jarManifest);
		
		// Add the contents
		addVerifiedClasspathContentsToJar(jarOutputStream, 
				 projectName,
				 verifiedLibrariesOutputFolder,
				 verifiedClassesOutputFolder,
				 natureId,
				 monitor);
		
		if (resourcesFolder != null)
			addFilteredResourcesToJar(jarOutputStream, new IncludeAllResourceFilter(),
				resourcesFolder.getFullPath(), resourcesFolder);
		// All done with the initial jar.
		jarOutputStream.close();
		
		return resourceToFile(getJarFile(projectName, deploymentFolder, "")); //$NON-NLS-1$
	}

	/**
	 * Add the contents of the verified classpath to the jar.
	 * 
	 * @param jarOutputStream
	 * @param monitor
	 */
	private void addVerifiedClasspathContentsToJar(
		EntryTrackingJarOutputStream jarOutputStream, 
		String projectName,
		IFolder verifiedLibrariesOutputFolder,
		IFolder verifiedClassesOutputFolder,
		String natureId,
		IProgressMonitor monitor) 
			throws CoreException, IOException
	{   IJavaProject javaProject = getJavaProject(projectName);
		PackagerClasspathEntryVisitor visitor =
			new PackagerClasspathEntryVisitor(jarOutputStream, verifiedLibrariesOutputFolder,
					verifiedClassesOutputFolder);
		visitor.getRunner().run(javaProject, visitor, natureId, monitor);
	}

	private IJavaProject getJavaProject(String projectName) {
		return JavaCore.create(getProject(projectName));
	}
	
	private IProject getProject(String projectName) {
		return ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
	}
	
	/**
	 * Filter out the attributes as requested by the user.
	 * 
	 * @param attributes
	 */
	private void filterManifestAttributes(Project projectData, Properties attributes) {
		ArrayList excluded = new ArrayList();
		if ( projectData.getPackagingDetails() != null &&
				projectData.getPackagingDetails().getExculedManifestEntries() != null ) {
			Iterator it = projectData.getPackagingDetails().getExculedManifestEntries().iterator();
			while (it.hasNext()) {
				excluded.add(((Parameter)it.next()).getName());
			}
			
			for (int i = 0; i < excluded.size(); i++) {
				String excludedName = (String)excluded.get(i);
				if (attributes.containsKey(excludedName)) {
					attributes.remove(excludedName);
				}
			}
		}
	}

	/**
	 * Get a Manifest instance on the contents of the
	 * JAD source file.
	 * 
	 * @return
	 * @throws IOException
	 */
	private ColonDelimitedProperties getJadSourceManifest(String projectName, IFolder deploymentFolder) 
		throws IOException, CoreException
	{
		ColonDelimitedProperties properties = new ColonDelimitedProperties();
		properties.load(getJadSourceStream(projectName, deploymentFolder));
		
		return properties;
	}
	
	/**
	 * Get an input stream on the JAD source file.
	 * 
	 * @return
	 * @throws IOException
	 */
	private InputStream getJadSourceStream(String projectName, IFolder folder)
		throws IOException, CoreException 
	{
		return getJadFile(projectName, folder).getContents(true);	
	}
	
	/**
	 * Get the File instance into which the JAD will be
	 * written.
	 * 
	 * @return
	 */
	private IFile getJadFile(String projectName) {
		return getJadFile(projectName, null);
	}
	
	private IFile getJadFile(String projectName, IFolder deploymentFolder) {
		if ( deploymentFolder == null ) {
			IProject iproject = getProject(projectName);
			return iproject.getFile(getJadSourceFile(projectName));
		}
		else {
			return deploymentFolder.getFile(getJadSourceFile(projectName));
		}
	}
	
	/**
	 * Get the source JAD file instance.
	 * 
	 * @return
	 */
	private String getJadSourceFile(String projectName) {
		return projectName.replace(' ', '_') + ".jad"; //$NON-NLS-1$
	}

	/**
	 * Convert the specified resource handle to a File handle.
	 * 
	 * @param resource
	 * @return
	 */
	private File resourceToFile(IResource resource) {
		return new File(resource.getLocation().toOSString());
	}
	
	/**
	 * Copy the source jad file and update the contents.
	 * 
	 * @param deploymentFolder
	 * @param deployedJarFile
	 * @throws IOException
	 * @throws CoreException
	 */
	private void copyAndUpdateJadFile(String projectName, IFolder deploymentFolder, File deployedJarFile) 
		throws IOException, CoreException 
	{
		// Read the source jad file and update the jar file
		// length property.
		ColonDelimitedProperties jadProperties = getSourceJADProperties(projectName);
		
		// Update the size of the jar file
		jadProperties.setProperty(
			IJADConstants.JAD_MIDLET_JAR_SIZE,
			(new Long(deployedJarFile.length())).toString());

		int count = 1;
		if(midletProperties != null) {
			Enumeration e = midletProperties.elements();
			while (e.hasMoreElements()) {
				String m = (String) e.nextElement();
				jadProperties.setProperty("MIDlet-"+count, m); //$NON-NLS-1$
				count ++;
			}
		}
		
		// write the JAD file in case signing blows up - at least we'll have
		// something non-bogus.
		
		// By (re)writing the deployed jad in this fashion, all signing information 
		// is lost from the jad file. Now, signing data stored in the project
		// is out of sync with the jad file. Signing provider is now forced to somehow
		// get back in sync. 
		writeJad(projectName, deploymentFolder, jadProperties);
		
	}

	/**
	 * Write out the JAD properties to the specified folder.
	 * 
	 * @param outputFolder
	 * @param jadProperties
	 * @throws IOException
	 */
	private void writeJad(String projectName, IFolder outputFolder, ColonDelimitedProperties jadProperties) 
		throws IOException 
	{
		// Write the new version
		IFile jadFile = getJadFile(projectName, outputFolder);
		File outputFile = resourceToFile(jadFile);
		FileOutputStream stream = new FileOutputStream(outputFile);
		jadProperties.store(stream, ""); //$NON-NLS-1$
		stream.close();
	}

	/**
	 * Get the JAD source properties as a Properties.
	 * 
	 * @return
	 * @throws IOException
	 */
	private ColonDelimitedProperties getSourceJADProperties(String projectName)
		throws IOException, CoreException 
	{
		ColonDelimitedProperties jadProperties = new ColonDelimitedProperties();
		
		InputStream jadStream = getJadSourceStream(projectName, null);
		jadProperties.load(jadStream);
		jadStream.close();
		
		return jadProperties;
	}

	/**
	 * Add the resources to the output jar stream after filtering based on the
	 * specified resource filter.
	 * 
	 * @param jarOutputStream
	 * @param resourceFilter
	 * @param rootPath
	 * @param container
	 * @throws CoreException
	 * @throws IOException
	 */
	private void addFilteredResourcesToJar(
		EntryTrackingJarOutputStream jarOutputStream,
		IResourceFilter resourceFilter,
		IPath rootPath,
		IContainer container)
			throws CoreException, IOException
	{
		IResource[] members = container.members();
		for (int i = 0; i < members.length; i++) {
			IResource resource = members[i];
			if (resource instanceof IContainer) {
				IContainer cont = (IContainer) resource;
				if (resourceFilter.shouldTraverseContainer(cont)) {
					addFilteredResourcesToJar(jarOutputStream, resourceFilter, rootPath, cont);
				}
			} else if (resource.getType() == IResource.FILE) {
				if (resourceFilter.shouldBeIncluded((IFile) resource)) {
					addResourceToJar(jarOutputStream, rootPath, resource);
				}
			}
		}
	}

	/**
	 * 
	 * Add the specified resources to the JAR file.
	 * 
	 * @param jarOutputStream
	 * @param rootPath
	 * @param resource
	 * @throws IOException
	 */
	private void addResourceToJar(
		EntryTrackingJarOutputStream jarOutputStream, 
		IPath rootPath, 
		IResource resource) 
			throws IOException
	{
		// Figure the path of the JAR file entry
		IPath resourcePath = resource.getFullPath();
		int commonSegments = resourcePath.matchingFirstSegments(rootPath);
		IPath entryPath = resourcePath.removeFirstSegments(commonSegments);
		
		// Add the new entry
		File file = new File(resource.getLocation().toOSString());
		addFileToJar(jarOutputStream, entryPath.toString(), file);
	}

	/**
	 * Add the specified file contents to the jar file.
	 * 
	 * @param jarOutputStream
	 * @param string
	 * @param file
	 * @throws IOException
	 */
	private void addFileToJar(
		EntryTrackingJarOutputStream jarOutputStream, 
		String entryName, 
		File file) 
			throws IOException
	{
		// Create the ZipEntry to represent the file
		ZipEntry entry = new ZipEntry(entryName);
		entry.setSize(file.length());
		entry.setTime(file.lastModified());
		createJarEntry(jarOutputStream, entry, new FileInputStream(file));
	}

	/**
	 * Create a new Jar file entry given the specified information.
	 * 
	 * @param jarOutputStream
	 * @param zipEntry
	 * @param is
	 * @throws IOException
	 */
	private void createJarEntry(
		EntryTrackingJarOutputStream jarOutputStream, 
		ZipEntry zipEntry, 
		InputStream is)
			throws IOException
	{
		// Copy the zip entry before adding it to the output stream
		ZipEntry newEntry = new ZipEntry(zipEntry.getName());
		newEntry.setTime(zipEntry.getTime());
		if (zipEntry.getComment() != null) {
			newEntry.setComment(zipEntry.getComment());
		}
		if (zipEntry.getExtra() != null) {
			newEntry.setExtra(zipEntry.getExtra());
		}
		
		if (!jarOutputStream.alreadyAdded(zipEntry)) {
			// Add the new ZipEntry to the stream
			jarOutputStream.putNextEntry(newEntry);
			Utils.copyInputToOutput(is, jarOutputStream);
			
			// Close this entry
			jarOutputStream.closeEntry();
		}
	}

	/**
	 * IClasspathEntryVisitor that does the work necessary to
	 * package up the information based on the classpath.
	 */
	private class PackagerClasspathEntryVisitor extends FilteringClasspathEntryVisitor {
		private EntryTrackingJarOutputStream jarStream;
		private boolean visitedSource;
		private IFolder verifiedLibrariesOutputFolder;
		private IFolder verifiedClassesOutputFolder;
		
		/** Constructor */
		private PackagerClasspathEntryVisitor(EntryTrackingJarOutputStream jarStream, 
				IFolder verifiedLibrariesOutputFolder,
				IFolder verifiedClassesOutputFolder) {
			this.jarStream = jarStream;
			this.verifiedLibrariesOutputFolder = verifiedLibrariesOutputFolder;
			this.verifiedClassesOutputFolder = verifiedClassesOutputFolder;
			visitedSource = false;
		}
		
		public void visitLibraryEntry(
			IClasspathEntry entry,
			IJavaProject javaProject, 
			IProgressMonitor monitor)
				throws CoreException 
		{
			File entryFile = Utils.getResolvedClasspathEntryFile(entry);
			if ((entryFile != null) && entryFile.isFile()) {
				
				IFile lib = verifiedLibrariesOutputFolder.getFile(entry.getPath().lastSegment());
				if (lib.exists()) {
					try {
						copyLibContentsToJar(jarStream, lib, monitor);
					} catch (IOException e) {
						MtjCorePlugin.throwCoreException(IStatus.ERROR, -999, e);
					}
				}
			}
		}
		
		public void visitSourceEntry(
			IClasspathEntry entry,
			IJavaProject javaProject, 
			IProgressMonitor monitor)
				throws CoreException 
		{
			if (!visitedSource) {
				visitedSource = true;
				
				try {
					addFilteredResourcesToJar(
						jarStream, 
						new IncludeAllResourceFilter(), 
						verifiedClassesOutputFolder.getFullPath(), 
						verifiedClassesOutputFolder);
				} catch (IOException e) {
					MtjCorePlugin.throwCoreException(IStatus.ERROR, -999, e);
				}
			}
		}

		/**
		 * Copy the contents of a library file to the jar output.
		 * 
		 * @param jarOutputStream
		 * @param lib
		 * @param monitor
		 */
		private void copyLibContentsToJar(
			EntryTrackingJarOutputStream jarOutputStream, 
			IFile lib, 
			IProgressMonitor monitor) 
				throws IOException
		{
			File libFile = lib.getLocation().toFile();
			ZipInputStream zis = new ZipInputStream(new FileInputStream(libFile));
			
			ZipEntry zipEntry = null;
			do {
				zipEntry = zis.getNextEntry();
				if (zipEntry != null) {
					createJarEntry(jarOutputStream, zipEntry, zis);
				}
			} while (zipEntry != null);

			zis.close();
		}
	}

	/**
	 * Resource filter that always includes everything.
	 */
	private class IncludeAllResourceFilter implements IResourceFilter {
		/**
		 * @see eclipseme.core.model.impl.Packager.IResourceFilter#shouldTraverseContainer(org.eclipse.core.resources.IContainer)
		 */
		public boolean shouldTraverseContainer(IContainer container) {
			return true;
		}

		/**
		 * @see eclipseme.core.model.impl.Packager.IResourceFilter#shouldBeIncluded(org.eclipse.core.resources.IFile)
		 */
		public boolean shouldBeIncluded(IFile file) {
			return true;
		}
	}

	/**
	 * This interface represents a filter controlling
	 * inclusion of resources into the packaged output.
	 * <p>
	 * <b>Note:</b> This class/interface is part of an interim API that is still under development and expected to
	 * change before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost 
	 * certainly be broken as the API evolves.
	 * </p>
	 * Copyright (c) 2004 Craig Setera<br>
	 * All Rights Reserved.<br>
	 * Licensed under the Eclipse Public License - v 1.0<p/>
	 * <br>
	 * $Revision: 1.2 $
	 * <br>
	 * $Date: 2006/10/31 16:51:46 $
	 * <br>
	 * @author Craig Setera
	 */
	public interface IResourceFilter {
		/**
		 * Return a boolean indicating whether the specified
		 * container should be traversed.
		 * 
		 * @param container the container to be traversed
		 * @return whether the container should be traversed
		 */
		public boolean shouldTraverseContainer(IContainer container);
		
		/**
		 * Return a boolean indicating whether the specified
		 * file should be included.
		 * 
		 * @param file the file to be tested
		 * @return whether the file should be included
		 */
		public boolean shouldBeIncluded(IFile file);
  }
	
}
