/*******************************************************************************
 * Copyright (c) 2000, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

/**
 * Copyright (c) 2008 Motorola.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Diego Sandin (Motorola) - Adapted code available in JDT
 */
package org.eclipse.mtj.internal.ui.wizards.dialogfields;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Table;

/**
 * A list with check boxes and a button bar. Typical buttons are 'Check All' and
 * 'Uncheck All'. List model is independent of widget creation. DialogFields
 * controls are: Label, List and Composite containing buttons.
 * 
 * @since 0.9.1
 */
public class CheckedListDialogField extends ListDialogField {

    private int fCheckAllButtonIndex;
    private List<Object> fCheckedElements;

    private List<Object> fGrayedElements;
    private int fUncheckAllButtonIndex;

    /**
     * @param adapter
     * @param customButtonLabels
     * @param lprovider
     */
    public CheckedListDialogField(IListAdapter adapter,
            String[] customButtonLabels, ILabelProvider lprovider) {
        super(adapter, customButtonLabels, lprovider);
        fCheckedElements = new ArrayList<Object>();
        fGrayedElements = new ArrayList<Object>();

        fCheckAllButtonIndex = -1;
        fUncheckAllButtonIndex = -1;
    }

    /**
     * Sets the check state of all elements
     * 
     * @param state
     */
    public void checkAll(boolean state) {
        if (state) {
            fCheckedElements = getElements();
        } else {
            fCheckedElements.clear();
        }
        if (isOkToUse(fTableControl)) {
            ((CheckboxTableViewer) fTable).setAllChecked(state);
        }
        checkStateChanged();
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.dialogfields.ListDialogField#dialogFieldChanged()
     */
    @Override
    public void dialogFieldChanged() {
        for (int i = fCheckedElements.size() - 1; i >= 0; i--) {
            if (!fElements.contains(fCheckedElements.get(i))) {
                fCheckedElements.remove(i);
            }
        }
        super.dialogFieldChanged();
    }

    /**
     * Gets the checked elements.
     * 
     * @return
     */
    public List<Object> getCheckedElements() {
        if (isOkToUse(fTableControl)) {
            // workaround for bug 53853
            Object[] checked = ((CheckboxTableViewer) fTable)
                    .getCheckedElements();
            ArrayList<Object> res = new ArrayList<Object>(checked.length);
            for (Object element : checked) {
                res.add(element);
            }
            return res;
        }

        return new ArrayList<Object>(fCheckedElements);
    }

    /**
     * Returns the number of checked elements.
     * 
     * @return
     */
    public int getCheckedSize() {
        return fCheckedElements.size();
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.dialogfields.ListDialogField#getListControl(org.eclipse.swt.widgets.Composite)
     */
    @Override
    public Control getListControl(Composite parent) {
        Control control = super.getListControl(parent);
        if (parent != null) {
            ((CheckboxTableViewer) fTable).setCheckedElements(fCheckedElements
                    .toArray());
            ((CheckboxTableViewer) fTable).setGrayedElements(fGrayedElements
                    .toArray());
        }
        return control;
    }

    /**
     * Returns true if the element is checked.
     * 
     * @param obj
     * @return
     */
    public boolean isChecked(Object obj) {
        if (isOkToUse(fTableControl)) {
            return ((CheckboxTableViewer) fTable).getChecked(obj);
        }

        return fCheckedElements.contains(obj);
    }

    /**
     * @param obj
     * @return
     */
    public boolean isGrayed(Object obj) {
        if (isOkToUse(fTableControl)) {
            return ((CheckboxTableViewer) fTable).getGrayed(obj);
        }

        return fGrayedElements.contains(obj);
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.dialogfields.ListDialogField#refresh()
     */
    @Override
    public void refresh() {
        super.refresh();
        if (isOkToUse(fTableControl)) {
            ((CheckboxTableViewer) fTable).setCheckedElements(fCheckedElements
                    .toArray());
            ((CheckboxTableViewer) fTable).setGrayedElements(fGrayedElements
                    .toArray());
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.dialogfields.ListDialogField#replaceElement(java.lang.Object, java.lang.Object)
     */
    @Override
    public void replaceElement(Object oldElement, Object newElement)
            throws IllegalArgumentException {
        boolean wasChecked = isChecked(oldElement);
        super.replaceElement(oldElement, newElement);
        setChecked(newElement, wasChecked);
    }

    /**
     * Sets the index of the 'check' button in the button label array passed in
     * the constructor. The behavior of the button marked as the check button
     * will then be handled internally. (enable state, button invocation
     * behavior)
     * 
     * @param checkButtonIndex
     */
    public void setCheckAllButtonIndex(int checkButtonIndex) {
        Assert.isTrue(checkButtonIndex < fButtonLabels.length);
        fCheckAllButtonIndex = checkButtonIndex;
    }

    /**
     * Sets the checked state of an element.
     * 
     * @param object
     * @param state
     */
    public void setChecked(Object object, boolean state) {
        setCheckedWithoutUpdate(object, state);
        checkStateChanged();
    }

    /**
     * Sets the checked elements.
     * 
     * @param list
     */
    public void setCheckedElements(Collection<Object> list) {
        fCheckedElements = new ArrayList<Object>(list);
        if (isOkToUse(fTableControl)) {
            ((CheckboxTableViewer) fTable).setCheckedElements(list.toArray());
        }
        checkStateChanged();
    }

    /**
     * Sets the checked state of an element. No dialog changed listener is
     * informed.
     * 
     * @param object
     * @param state
     */
    public void setCheckedWithoutUpdate(Object object, boolean state) {
        if (state) {
            if (!fCheckedElements.contains(object)) {
                fCheckedElements.add(object);
            }
        } else {
            fCheckedElements.remove(object);
        }
        if (isOkToUse(fTableControl)) {
            ((CheckboxTableViewer) fTable).setChecked(object, state);
        }
    }

    /**
     * @param object
     * @param state
     */
    public void setGrayedWithoutUpdate(Object object, boolean state) {
        if (state) {
            if (!fGrayedElements.contains(object)) {
                fGrayedElements.add(object);
            }
        } else {
            fGrayedElements.remove(object);
        }
        if (isOkToUse(fTableControl)) {
            ((CheckboxTableViewer) fTable).setGrayed(object, state);
        }
    }

    /**
     * Sets the index of the 'uncheck' button in the button label array passed
     * in the constructor. The behavior of the button marked as the uncheck
     * button will then be handled internally. (enable state, button invocation
     * behavior)
     */
    public void setUncheckAllButtonIndex(int uncheckButtonIndex) {
        Assert.isTrue(uncheckButtonIndex < fButtonLabels.length);
        fUncheckAllButtonIndex = uncheckButtonIndex;
    }

    /**
     * 
     */
    private void checkStateChanged() {
        // call super and do not update check model
        super.dialogFieldChanged();
    }

    /**
     * @param e
     */
    private void doCheckStateChanged(CheckStateChangedEvent e) {
        if (e.getChecked()) {
            fCheckedElements.add(e.getElement());
        } else {
            fCheckedElements.remove(e.getElement());
        }
        checkStateChanged();
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.dialogfields.ListDialogField#createTableViewer(org.eclipse.swt.widgets.Composite)
     */
    @Override
    protected TableViewer createTableViewer(Composite parent) {
        Table table = new Table(parent, SWT.CHECK + getListStyle());
        table.setFont(parent.getFont());
        CheckboxTableViewer tableViewer = new CheckboxTableViewer(table);
        tableViewer.addCheckStateListener(new ICheckStateListener() {
            public void checkStateChanged(CheckStateChangedEvent e) {
                doCheckStateChanged(e);
            }
        });
        return tableViewer;
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.dialogfields.ListDialogField#getManagedButtonState(org.eclipse.jface.viewers.ISelection, int)
     */
    @Override
    protected boolean getManagedButtonState(ISelection sel, int index) {
        if (index == fCheckAllButtonIndex) {
            return !fElements.isEmpty();
        } else if (index == fUncheckAllButtonIndex) {
            return !fElements.isEmpty();
        }
        return super.getManagedButtonState(sel, index);
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.internal.ui.wizards.dialogfields.ListDialogField#managedButtonPressed(int)
     */
    @Override
    protected boolean managedButtonPressed(int index) {
        if (index == fCheckAllButtonIndex) {
            checkAll(true);
        } else if (index == fUncheckAllButtonIndex) {
            checkAll(false);
        } else {
            return super.managedButtonPressed(index);
        }
        return true;
    }

}
