/**
 * Copyright (c) 2004,2008 Craig Setera and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Craig Setera (EclipseME) - Initial implementation
 *     Diego Sandin (Motorola)  - Refactoring package name to follow eclipse 
 *                                standards
 *     Diego Sandin (Motorola)  - Fixed nullPointExcepiton on 
 *                                getPreverificationConfigurationVersion
 *     Hugo Raniere (Motorola)  - Including preference to store the default preverifier
 *     Gang Ma      (Sybase)    - Add getting debug level from preference
 */
package org.eclipse.mtj.core.internal;

import java.util.StringTokenizer;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.mtj.core.IMTJCoreConstants;
import org.eclipse.mtj.core.model.Version;
import org.eclipse.mtj.core.model.device.IDevice;
import org.eclipse.mtj.core.model.jad.ApplicationDescriptor;
import org.eclipse.mtj.core.model.library.ILibrary;
import org.eclipse.mtj.core.model.library.api.API;
import org.eclipse.mtj.core.model.project.IMidletSuiteProject;
import org.eclipse.mtj.core.model.project.MidletSuiteFactory;

/**
 * A helper wrapper around the more complex preferences supported by MTJ.
 * 
 * @author Craig Setera
 */
public class PreferenceAccessor {

	/**
	 * The singleton instance of the Obfuscation Preferences
	 */
	public static final PreferenceAccessor instance = new PreferenceAccessor();

	/**
	 * Separator for multi-valued preferences
	 */
	public static final String MULTI_VALUE_SEPARATOR = "|";

	// Scope search contexts
	private IEclipsePreferences defaultPreferences;
	private IEclipsePreferences instancePreferences;

	/**
	 * Construct a new preference wrapper.
	 * 
	 * @param preferenceStore
	 */
	private PreferenceAccessor() {
		super();

		defaultPreferences = new DefaultScope()
				.getNode(IMTJCoreConstants.PLUGIN_ID);
		instancePreferences = new InstanceScope()
				.getNode(IMTJCoreConstants.PLUGIN_ID);
	}

	/**
	 * Return the default Proguard keep expressions from the preferences.
	 * 
	 * @return
	 */
	public String[] getDefaultProguardKeepExpressions() {
		String keepString = getPreferences().getDefaultString(
				IMTJCoreConstants.PREF_PROGUARD_KEEP);
		return getProguardKeepExpressions(keepString);
	}

	/**
	 * @return Returns the defaultProguardOptions.
	 */
	public String getDefaultProguardOptions() {
		return getPreferences().getDefaultString(
				IMTJCoreConstants.PREF_PROGUARD_OPTIONS);
	}

	/**
	 * Return the excluded manifest properties based on the specified project.
	 * 
	 * @param project
	 * @return
	 */
	public String[] getExcludedManifestProperties(IProject project) {
		IEclipsePreferences preferences = getProjectPreferences(project,
				IMTJCoreConstants.PREF_PKG_USE_PROJECT,
				IMTJCoreConstants.PREF_PKG_EXCLUDED_PROPS);
		String value = preferences.get(
				IMTJCoreConstants.PREF_PKG_EXCLUDED_PROPS,
				MTJCorePreferenceInitializer.PREF_DEF_PKG_EXCLUDED_PROPS);

		return parseMultiValuedPreferenceValue(value);
	}

	/**
	 * Return a boolean indicating whether or not automatic versioning should be
	 * done while packaging.
	 * 
	 * @param project
	 * @return
	 */
	public boolean getAutoversionPackage(IProject project) {
		IEclipsePreferences preferences = getProjectPreferences(project,
				IMTJCoreConstants.PREF_PKG_USE_PROJECT,
				IMTJCoreConstants.PREF_PKG_AUTOVERSION);

		return preferences.getBoolean(IMTJCoreConstants.PREF_PKG_AUTOVERSION,
				MTJCorePreferenceInitializer.PREF_DEF_PKG_AUTOVERSION);
	}

	/**
	 * Return the configuration to be used for preverification.
	 * 
	 * @param project
	 * @return
	 * @throws CoreException
	 */
	public Version getPreverificationConfigurationVersion(IProject project)
			throws CoreException {
		Version version = null;

		IJavaProject javaProject = JavaCore.create(project);
		IMidletSuiteProject suite = MidletSuiteFactory
				.getMidletSuiteProject(javaProject);

		// Figure out which preference store to use
		IEclipsePreferences preferences = getProjectPreferences(project,
				IMTJCoreConstants.PREF_PREVERIFY_USE_PROJECT,
				IMTJCoreConstants.PREF_PREVERIFY_CONFIG_LOCATION);

		// Figure out where to extract the configuration from
		String location = preferences
				.get(
						IMTJCoreConstants.PREF_PREVERIFY_CONFIG_LOCATION,
						MTJCorePreferenceInitializer.PREF_DEF_PREVERIFY_CONFIG_LOCATION);

		// Pull the configuration from the specified place
		if (IMTJCoreConstants.PREF_PREVERIFY_CONFIG_LOCATION_JAD
				.equals(location)) {
			ApplicationDescriptor jad = suite.getApplicationDescriptor();
			version = jad.getConfigurationSpecificationVersion();
		} else if (IMTJCoreConstants.PREF_PREVERIFY_CONFIG_LOCATION_PLATFORM
				.equals(location)) {
			version = new Version("1.0");

			IDevice device = suite.getDevice();
			if (device != null) {
				ILibrary configurationLibrary = device
						.getConfigurationLibrary();

				API api = null;
				if (configurationLibrary != null) {
					api = configurationLibrary.getConfiguration();
					if (api != null) {
						version = api.getVersion();
					}
				}
			}
		} else {
			String identifierOrVersion = preferences
					.get(
							IMTJCoreConstants.PREF_PREVERIFY_CONFIG_VALUE,
							MTJCorePreferenceInitializer.PREF_DEF_PREVERIFY_CONFIG_VALUE);

			// This handles the transition to the case where configurations
			// have been dropped from the plug-in extension points.
			if (identifierOrVersion.startsWith("CLDC")) {
				String[] split = identifierOrVersion.split("-");
				if (split.length == 2) {
					identifierOrVersion = split[1];
				}
			}

			version = new Version(identifierOrVersion);
		}

		// Fall back if we don't have a specification yet
		if (version == null) {
			version = new Version("1.0");
		}

		return version;
	}

	/**
	 * Return the Proguard keep expressions from the preferences.
	 * 
	 * @param project
	 * @return
	 */
	public String[] getProguardKeepExpressions(IProject project) {
		IEclipsePreferences preferences = getProjectPreferences(project,
				IMTJCoreConstants.PREF_OBFUSCATION_USE_PROJECT,
				IMTJCoreConstants.PREF_PROGUARD_KEEP);
		String value = preferences.get(IMTJCoreConstants.PREF_PROGUARD_KEEP,
				MTJCorePreferenceInitializer.PREF_DEF_PROGUARD_KEEP);

		return parseMultiValuedPreferenceValue(value);
	}

	/**
	 * Return the specified proguard options.
	 * 
	 * @param project
	 * @return Returns the specifiedOptions.
	 */
	public String getSpecifiedProguardOptions(IProject project) {
		IEclipsePreferences preferences = getProjectPreferences(project,
				IMTJCoreConstants.PREF_OBFUSCATION_USE_PROJECT,
				IMTJCoreConstants.PREF_PROGUARD_OPTIONS);
		return preferences.get(IMTJCoreConstants.PREF_PROGUARD_OPTIONS,
				MTJCorePreferenceInitializer.PREF_DEF_PROGUARD_OPTIONS);
	}

	/**
	 * Return whether to use the specified proguard options.
	 * 
	 * @param project
	 * @return Returns the useSpecifiedOptions.
	 */
	public boolean isUseSpecifiedProguardOptions(IProject project) {
		IEclipsePreferences preferences = getProjectPreferences(project,
				IMTJCoreConstants.PREF_OBFUSCATION_USE_PROJECT,
				IMTJCoreConstants.PREF_PROGUARD_USE_SPECIFIED);

		return preferences.getBoolean(
				IMTJCoreConstants.PREF_PROGUARD_USE_SPECIFIED,
				MTJCorePreferenceInitializer.PREF_DEF_PROGUARD_USE_SPECIFIED);
	}

	/**
	 * Get the preferences store from the core.
	 * 
	 * @return
	 */
	private Preferences getPreferences() {
		return MTJCorePlugin.getDefault().getPluginPreferences();
	}

	/**
	 * Return the correct preferences based on the specified project and any
	 * groups of keys that tell whether the preference should be pulled from the
	 * project or instance preferences.
	 * 
	 * @param project
	 * @param selectionKeys
	 * @return
	 */
	private IEclipsePreferences getProjectPreferences(IProject project,
			String projectSpecificKey, String preferenceKey) {
		ProjectScope projectScope = new ProjectScope(project);
		IEclipsePreferences prefNode = projectScope
				.getNode(IMTJCoreConstants.PLUGIN_ID);

		boolean useProjectSpecific = prefNode.getBoolean(projectSpecificKey,
				false);
		prefNode = useProjectSpecific ? prefNode : instancePreferences;

		if (prefNode.get(preferenceKey, null) == null) {
			prefNode = defaultPreferences;
		}

		return prefNode;
	}

	/**
	 * Return the Proguard keep expressions from the specified value
	 * 
	 * @return
	 */
	private String[] getProguardKeepExpressions(String expressionsString) {
		return getMultiValuedPreference(expressionsString);
	}

	/**
	 * Return the property names that are excluded during packaging
	 * from the preferences.
	 * 
	 * @return
	 */
	public String[] getMultiValuedPreference(String preferenceName) {
		Preferences prefs = MTJCorePlugin.getDefault().getPluginPreferences();
		String propNames = prefs.getString(preferenceName);
		return parseMultiValuedPreferenceValue(propNames);
	}

	/**
	 * Return the parsed multi-value preference.
	 * 
	 * @param value
	 * @return
	 */
	public String[] parseMultiValuedPreferenceValue(String value) {
		StringTokenizer st = new StringTokenizer(value, MULTI_VALUE_SEPARATOR);

		int count = st.countTokens();
		String[] names = new String[count];
		for (int i = 0; i < count; i++) {
			names[i] = st.nextToken();
		}

		return names;
	}
	
	/**
         * Return the Preprocess debug level from the preferences.
         * 
         * @param project
         * @return
         */
        public String getPreprecessorDebuglevel(IProject project) {
                IEclipsePreferences preferences = getProjectPreferences(project,
                                IMTJCoreConstants.PREF_PREPROCESS_USE_PROJECT,
                                IMTJCoreConstants.PREF_PREPROCESS_DEBUG_LEVEL);
                String value = preferences.get(IMTJCoreConstants.PREF_PREPROCESS_DEBUG_LEVEL,
                                MTJCorePreferenceInitializer.PREF_DEF_PREPROCESS_DEBUGLEVEL);

                return value;
        }
}
