/**
 * Copyright (c) 2008 Motorola.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     David Marques (Motorola) - Initial version
 *     David Marques (Motorola) - Fixing propeties folder path
 */
package org.eclipse.mtj.internal.core.l10n;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.mtj.core.internal.MTJCorePlugin;
import org.eclipse.mtj.core.internal.utils.Utils;
import org.eclipse.mtj.core.model.ReplaceableParametersProcessor;
import org.eclipse.mtj.internal.core.text.IDocumentElementNode;
import org.eclipse.mtj.internal.core.text.l10n.L10nEntry;
import org.eclipse.mtj.internal.core.text.l10n.L10nLocale;
import org.eclipse.mtj.internal.core.text.l10n.L10nModel;
import org.eclipse.osgi.util.NLS;

/**
 *  L10nApi Class contains utilitary classes for management
 *  of the Localization Client API.
 */
public class L10nApi {

    public static final String LOCALIZATION_DATA_FILE = "Localization Data"; //$NON-NLS-1$
    public static final String DESTINATION = "destination"; //$NON-NLS-1$
    public static final String PACKAGE = "package"; //$NON-NLS-1$

    public static final String L10N_RESOURCES_CLASS = "L10nResources.java"; //$NON-NLS-1$
    public static final String L10N_CONSTANTS_CLASS = "L10nConstants.java"; //$NON-NLS-1$
    
    /**
     * Creates the Localization API classes.
     * 
     * @param project Target project
     * @param _package Class package name.
     * @param _properties Resources destination.
     * @return true upon success false otherwise.
     * @throws CoreException If any error occurs.
     */
    public static boolean createLocalizationApi(IProject project,
            IPackageFragment _package, IPath _properties) throws CoreException {
        ICompilationUnit resourcesClass = null;
        ICompilationUnit constantsClass = null;

        IPackageFragment targetPackage = (IPackageFragment) _package;

        String code = buildL10nResourcesClass(_package, _properties);
        resourcesClass = targetPackage.createCompilationUnit(
                L10N_RESOURCES_CLASS, code, true, null);

        code = buildL10nConstantsClass(project, _package);
        constantsClass = targetPackage.createCompilationUnit(
                L10N_CONSTANTS_CLASS, code, true, null);

        return resourcesClass != null && constantsClass != null;
    }

    /**
     * Creates the L10nConstant Class code.
     * 
     * @param project Target project
     * @param _package Class package name.
     * @return the class code.
     * @throws CoreException If any error occurs.
     */
    public static String buildL10nConstantsClass(IProject project,
            IPackageFragment _package) throws CoreException {
        InputStream stream = MTJCorePlugin.getResourceAsStream(new Path(
                "templates/L10nConstants.java.template"));
        if (stream == null) {
            MTJCorePlugin.throwCoreException(IStatus.ERROR, -999,
                    "Unable to load templates/L10nResources.java.template.");
        }

        StringBuffer localesBuffer = new StringBuffer();
        List<String> keys = new ArrayList<String>();        
        L10nModel model = loadL10nModel(project);
        if (!model.isValid()) {
            MTJCorePlugin.throwCoreException(IStatus.ERROR, -999,
                    Messages.L10nBuilder_InvalidModel);
        }

        List<IDocumentElementNode> localeNodes = model.getLocales()
                .getChildren();
        for (IDocumentElementNode localeElement : localeNodes) {
            L10nLocale locale = (L10nLocale) localeElement;
            localesBuffer.append(writeStringConstantDeclaration(locale
                    .getName().toUpperCase(), locale.getName()));

            IDocumentElementNode[] entryNodes = locale.getChildNodes();
            for (int i = 0; i < entryNodes.length; i++) {
                L10nEntry entry = (L10nEntry) entryNodes[i];
                if (!keys.contains(entry.getKey())) {
                    keys.add(entry.getKey());
                }
            }
        }
        
        StringBuffer constntBuffer = new StringBuffer();
        for (String key : keys) {
            constntBuffer.append(writeStringConstantDeclaration(key, key));
        }
        
        Map<String, String> map = new HashMap<String, String>();
        String packName = _package.getElementName();
        if (packName != null && packName.length() > 0x00) {
            map.put(PACKAGE, NLS.bind("package {0};", packName));
        } else {
            map.put(PACKAGE, Utils.EMPTY_STRING);
        }
        map.put("locales", localesBuffer.toString());
        map.put("keys", constntBuffer.toString());

        StringBuffer buffer = new StringBuffer(Utils.getStreamContent(stream));
        String code = ReplaceableParametersProcessor.processReplaceableValues(
                buffer.toString(), map);
        return code;
    }

    /**
     * Creates the L10nResources Class code.
     * 
     * @param _package Class package name.
     * @param _properties The properties folder path.
     * @return the class code.
     * @throws CoreException If any error occurs.
     */
    public static String buildL10nResourcesClass(IPackageFragment _package,
            IPath _properties) throws CoreException {
        InputStream stream = MTJCorePlugin.getResourceAsStream(new Path(
                "templates/L10nResources.java.template"));
        if (stream == null) {
            MTJCorePlugin.throwCoreException(IStatus.ERROR, -999,
                    "Unable to load templates/L10nResources.java.template.");
        }

        Map<String, String> map = new HashMap<String, String>();
        map.put(DESTINATION, _properties.removeFirstSegments(1).toString());
        String packName = _package.getElementName();
        if (packName != null && packName.length() > 0x00) {
            map.put(PACKAGE, NLS.bind("package {0};", packName));
        } else {
            map.put(PACKAGE, Utils.EMPTY_STRING);
        }

        StringBuffer buffer = new StringBuffer(Utils.getStreamContent(stream));
        String code = ReplaceableParametersProcessor.processReplaceableValues(
                buffer.toString(), map);
        return code;
    }

    /**
     * Loads the L10n Model from the specified project.
     * 
     * @param project target project.
     * @return the model.
     * @throws CoreException - If could not load model.
     */
    public static L10nModel loadL10nModel(IProject project)
            throws CoreException {
        L10nModel model = null;
        IFile l10nData = project.getFile(LOCALIZATION_DATA_FILE);
        if (l10nData.exists()) {
            IPath l10nDataPath = l10nData.getLocation();
            if (l10nDataPath != null) {
                model = new L10nModel(Utils.getTextDocument(l10nData
                        .getContents()), false);
                model.load();
            } else {
                throw new CoreException(
                        MTJCorePlugin
                                .newStatus(
                                        IStatus.ERROR,
                                        -999,
                                        Messages.L10nBuilder_loadL10nModel_CanNotRealizeLocation));
            }
        } else {
            throw new CoreException(MTJCorePlugin.newStatus(IStatus.ERROR,
                    -999, Messages.L10nBuilder_LocalizationDataDoesNotExist));
        }
        return model;
    }

    /**
     * Writes a constant declaration.
     * 
     * @param name constant name.
     * @param value constant value.
     * @return the declaration.
     */
    private static String writeStringConstantDeclaration(String name,
            String value) {
        return NLS.bind("\t\tpublic static final String {0} = \"{1}\";\n",
                new String[] { name.replace("-", "_"), value });
    }
}
