/**
 * Copyright (c) 2003,2008 Craig Setera and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Craig Setera (EclipseME) - Initial implementation
 *     Diego Sandin (Motorola)  - Refactoring package name to follow eclipse 
 *                                standards
 *     Hugo Raniere (Motorola)  - Removing Preprocessor code
 *     Diego Sandin (Motorola)  - Re-enabling Preprocessor code
 */
package org.eclipse.mtj.core.internal;

import java.io.File;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.preferences.DefaultScope;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IPreferencesService;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IStatusHandler;
import org.eclipse.mtj.core.IMTJCoreConstants;
import org.eclipse.mtj.core.hook.sourceMapper.SourceMapperAccess;
import org.eclipse.mtj.core.internal.preprocessor.PreprocessedSourceMapper;
import org.eclipse.mtj.core.model.Version;
import org.eclipse.mtj.core.model.library.LibrarySpecification;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;

/**
 * The main plug-in class to be used in the workbench.
 * 
 * @author Craig Setera
 */
public class MTJCorePlugin extends Plugin implements IMTJCoreConstants {

    /**
     * This job is responsible for walking through the workspace projects and
     * migrates the preprocessor natures and builders enabling that the
     * preprocessor works within the project rather than requiring the secondary
     * project
     */
    private class MigrationJob extends Job {

        /** Constructor */
        MigrationJob() {
            super("MTJ Migration");
            setPriority(SHORT);
            setSystem(true);
        }

        /*
         * (non-Javadoc)
         * 
         * @see org.eclipse.core.runtime.jobs.Job#run(org.eclipse.core.runtime.IProgressMonitor)
         */
        @Override
        protected IStatus run(IProgressMonitor monitor) {
            IStatus status = null;

            try {
                doMigration(monitor);
                status = OK_STATUS;
            } catch (CoreException e) {
                status = e.getStatus();
            }

            return status;
        }
    }

    /**
     * Status code for which a UI prompter is registered.
     */
    public static final IStatus OK_STATUS = new Status(IStatus.OK,
            IMTJCoreConstants.PLUGIN_ID, 0, "OK", null);

    // The shared instance.
    private static MTJCorePlugin plugin;

    /**
     * Status code for which a UI prompter is registered.
     */
    private static final IStatus PROMPTER_STATUS = new Status(IStatus.INFO,
            "org.eclipse.debug.ui", // TODO This should probably not be done....
            200, "", null);

    /**
     * Return the configuration specifications provided by the plugin extension
     * points.
     * 
     * @return the configuration specifications
     * @throws CoreException
     */
    public static LibrarySpecification[] getConfigurationSpecifications()
            throws CoreException {
        MTJCorePlugin plugin = getDefault();
        if (plugin.configSpecs == null) {
            IExtensionPoint point = Platform.getExtensionRegistry()
                    .getExtensionPoint(IMTJCoreConstants.PLUGIN_ID,
                            IMTJCoreConstants.J2ME_CONFIGURATIONS_ID);
            IConfigurationElement[] configElements = point
                    .getConfigurationElements();
            plugin.configSpecs = new LibrarySpecification[configElements.length];

            for (int i = 0; i < configElements.length; i++) {
                plugin.configSpecs[i] = createComponentSpecification(configElements[i]);
            }
        }

        return plugin.configSpecs;
    }

    /**
     * Returns the shared instance.
     */
    public static MTJCorePlugin getDefault() {
        return plugin;
    }

    /**
     * Get the deployment directory name the user has specified in the
     * preferences.
     * 
     * @return
     */
    public static String getDeploymentDirectoryName() {
        return getDefault().getPluginPreferences().getString(
                PREF_DEPLOYMENT_DIR);
    }

    /**
     * Return the current version associated with this plug-in.
     * 
     * @return the plug-in version
     */
    public static String getPluginVersion() {
        Bundle bundle = MTJCorePlugin.getDefault().getBundle();
        return (String) bundle.getHeaders().get(Constants.BUNDLE_VERSION);
    }

    /**
     * Return the profile specifications provided by the plugin extension
     * points.
     * 
     * @return the profile specifications
     * @throws CoreException
     */
    public static LibrarySpecification[] getProfileSpecifications()
            throws CoreException {
        MTJCorePlugin plugin = getDefault();
        if (plugin.profileSpecs == null) {
            IExtensionPoint point = Platform.getExtensionRegistry()
                    .getExtensionPoint(IMTJCoreConstants.PLUGIN_ID,
                            IMTJCoreConstants.J2ME_PROFILES_ID);
            IConfigurationElement[] configElements = point
                    .getConfigurationElements();
            plugin.profileSpecs = new LibrarySpecification[configElements.length];

            for (int i = 0; i < configElements.length; i++) {
                plugin.profileSpecs[i] = createComponentSpecification(configElements[i]);
            }
        }

        return plugin.profileSpecs;
    }

    /**
     * Return the File instance representing the Proguard implementation jar
     * file as defined by the user preferences. This File is not guaranteed to
     * exist.
     * 
     * @return
     */
    public static File getProguardJarFile() {
        String proguardDirPref = getDefault().getPluginPreferences().getString(
                PREF_PROGUARD_DIR);
        File proguardDir = new File(proguardDirPref);
        File proguardLibDir = new File(proguardDir, "lib");
        File proguardJar = new File(proguardLibDir, PROGUARD_JAR);

        return proguardJar;
    }

    /**
     * Return a boolean preference scoped to the specified project where
     * possible, otherwise falling back to instance scope.
     * 
     * @param project
     * @param key
     * @return
     */
    public static boolean getProjectBooleanPreference(IProject project,
            String key) {
        IScopeContext[] searchContexts = new IScopeContext[] {
                new ProjectScope(project), new InstanceScope(),
                new DefaultScope(), };

        IPreferencesService service = Platform.getPreferencesService();
        return service.getBoolean(IMTJCoreConstants.PLUGIN_ID, key, false,
                searchContexts);
    }

    /**
     * Return the preferences that are specific to the project and plugin.
     * 
     * @param context
     * @return
     */
    public static IEclipsePreferences getProjectPreferences(IProject context) {
        ProjectScope projectScope = new ProjectScope(context);
        return projectScope.getNode(IMTJCoreConstants.PLUGIN_ID);
    }

    /**
     * Return a String preference scoped to the specified project where
     * possible, otherwise falling back to instance scope.
     * 
     * @param project
     * @param key
     * @return
     */
    public static String getProjectStringPreference(IProject project, String key) {
        IScopeContext[] searchContexts = new IScopeContext[] {
                new ProjectScope(project), new InstanceScope(),
                new DefaultScope(), };

        IPreferencesService service = Platform.getPreferencesService();
        return service.getString(IMTJCoreConstants.PLUGIN_ID, key, null,
                searchContexts);
    }

    /**
     * Get the resources directory name the user has specified in the
     * preferences. This directory will automatically be added to the emulator
     * classpath when running MIDlets from source.
     * 
     * @return
     */
    public static String getResourcesDirectoryName() {
        return getDefault().getPluginPreferences()
                .getString(PREF_RESOURCES_DIR);
    }

    /**
     * Get the verified output directory name the user has specified in the
     * preferences.
     * 
     * @return
     */
    public static String getVerifiedOutputDirectoryName() {
        return getDefault().getPluginPreferences().getString(PREF_VERIFIED_DIR);
    }

    /**
     * Returns the workspace instance.
     */
    public static IWorkspace getWorkspace() {
        return ResourcesPlugin.getWorkspace();
    }

    /**
     * Log the specified message.
     * 
     * @param severity
     * @param message
     */
    public static void log(int severity, String message) {
        log(severity, message, null);
    }

    /**
     * Log the specified message and exception
     * 
     * @param severity
     * @param message
     * @param throwable
     */
    public static void log(int severity, String message, Throwable throwable) {
        if (message == null) {
            message = throwable.getMessage();
        }
        if (message == null) {
            message = "No Message";
        }

        MTJCorePlugin plugin = MTJCorePlugin.getDefault();
        String id = IMTJCoreConstants.PLUGIN_ID;
        Status status = new Status(severity, id, IStatus.OK, message, throwable);
        plugin.getLog().log(status);
    }

    /**
     * Log the specified exception.
     * 
     * @param severity
     * @param throwable
     */
    public static void log(int severity, Throwable throwable) {
        log(severity, throwable.getMessage(), throwable);
    }

    /**
     * Creates a new status object for our plug-in. The created status has no
     * children.
     * 
     * @param severity the severity; one of <code>OK</code>,
     *                <code>ERROR</code>, <code>INFO</code>, or
     *                <code>WARNING</code>
     * @param code the plug-in-specific status code, or <code>OK</code>
     * @param message a human-readable message, localized to the current locale
     */
    public static IStatus newStatus(int severity, int code, String message) {
        return newStatus(severity, code, message, null);
    }

    /**
     * Creates a new status object for our plug-in. The created status has no
     * children.
     * 
     * @param severity the severity; one of <code>OK</code>,
     *                <code>ERROR</code>, <code>INFO</code>, or
     *                <code>WARNING</code>
     * @param code the plug-in-specific status code, or <code>OK</code>
     * @param message a human-readable message, localized to the current locale
     * @param exception a low-level exception, or <code>null</code> if not
     *                applicable
     */
    public static IStatus newStatus(int severity, int code, String message,
            Throwable exception) {
        return new Status(severity, PLUGIN_ID, code, message, exception);
    }

    /**
     * Recursively set the resources in the specified container and all
     * resources within that container as derived.
     * 
     * @param container
     * @throws CoreException
     */
    public static void setResourcesAsDerived(IContainer container)
            throws CoreException {
        if (container.exists()) {
            // Mark this folder first...
            container.setDerived(true);

            // Recursively handle the members of the directory
            IResource[] resources = container.members();
            for (IResource resource : resources) {
                if (resource instanceof IContainer) {
                    setResourcesAsDerived((IContainer) resource);
                } else {
                    resource.setDerived(true);
                }
            }
        }
    }

    /**
     * Attempt to prompt on a status object. If prompting fails, a CoreException
     * will be thrown.
     * 
     * @param status
     * @param source
     * @return
     * @throws CoreException
     */
    public static Object statusPrompt(IStatus status, Object source)
            throws CoreException {
        Object result = null;

        IStatusHandler prompterStatus = DebugPlugin.getDefault()
                .getStatusHandler(PROMPTER_STATUS);

        if (prompterStatus == null) {
            // if there is no handler, throw the exception
            throw new CoreException(status);
        } else {
            result = prompterStatus.handleStatus(status, source);
        }

        return result;
    }

    /**
     * Throw a new CoreException wrapped around the specified String.
     * 
     * @param severity
     * @param code
     * @param message
     * @throws CoreException
     */
    public static void throwCoreException(int severity, int code, String message)
            throws CoreException {
        if (message == null) {
            message = "[No Message]";
        }

        IStatus status = new Status(severity, IMTJCoreConstants.PLUGIN_ID,
                code, message, null);
        throw new CoreException(status);
    }

    /**
     * Throw a new CoreException wrapped around the specified exception.
     * 
     * @param severity
     * @param code
     * @param exception
     * @throws CoreException
     */
    public static void throwCoreException(int severity, int code,
            Throwable exception) throws CoreException {
        // Make sure we create a valid status object
        String message = null;
        if (exception != null) {
            message = exception.getMessage();
        }
        if (message == null) {
            message = "[No Message]";
        }

        IStatus status = new Status(severity, IMTJCoreConstants.PLUGIN_ID,
                code, message, exception);
        throw new CoreException(status);
    }

    /**
     * Create a component specification for the specified configuration element
     * 
     * @param element
     * @return
     * @throws CoreException
     */
    private static LibrarySpecification createComponentSpecification(
            IConfigurationElement element) throws CoreException {
        String id = element.getAttribute("id");
        String name = element.getAttribute("name");
        String versionString = element.getAttribute("version");
        Version version = new Version(versionString);

        LibrarySpecification specification = new LibrarySpecification();
        specification.setIdentifier(id);
        specification.setName(name);
        specification.setVersion(version);

        return specification;
    }

    // Resource bundle.
    private ResourceBundle resourceBundle;

    // The configuration specifications provided by extension points
    private LibrarySpecification[] configSpecs;

    // The profile specifications provided by extension points
    private LibrarySpecification[] profileSpecs;

    /**
     * The constructor.
     */
    public MTJCorePlugin() {
        super();

        if (plugin == null) {
            plugin = this;
        }

        try {
            resourceBundle = ResourceBundle
                    .getBundle("org.eclipse.mtj.core.MTJPluginResources");
        } catch (MissingResourceException x) {
            resourceBundle = null;
        }
    }

    /**
     * Returns the plugin's resource bundle,
     */
    public ResourceBundle getResourceBundle() {
        return resourceBundle;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.core.runtime.Plugin#start(org.osgi.framework.BundleContext)
     */
    @Override
    public void start(BundleContext context) throws Exception {
        super.start(context);

        // Install the preprocessor source mapper
        SourceMapperAccess.setSourceMapper(new PreprocessedSourceMapper());

        // Do version to version migration
        (new MigrationJob()).schedule(5000L);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.core.runtime.Plugin#stop(org.osgi.framework.BundleContext)
     */
    @Override
    public void stop(BundleContext context) throws Exception {
        super.stop(context);
    }

    /**
     * Execute the necessary migration steps.
     * 
     * @param monitor
     * @throws CoreException
     */
    private void doMigration(IProgressMonitor monitor) throws CoreException {
        ResourcesPlugin.getWorkspace().run(
                new PreprocessedProjectMigrationRunnable(), monitor);
    }
}
