/**
 * Copyright (c) 2003,2009 Craig Setera and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Craig Setera (EclipseME) - Initial implementation
 *     Diego Sandin (Motorola)  - Refactoring package name to follow eclipse
 *                                standards
 *     Gang Ma      (Sybase)	- Refactoring the page to add expansibilities
 *     Diego Sandin (Motorola)  - Use Eclipse Message Bundles [Bug 255874]
 *     Jon Dearden  (Research In Motion) - Move fields from JADOTAPropertiesEditorPage 
 *                                         and JADPushRegistryEditorPage onto this page 
 *                                         [Bug 284452]                      
 */
package org.eclipse.mtj.internal.ui.editors.jad.form.pages;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.preference.FieldEditor;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.IntegerFieldEditor;
import org.eclipse.jface.preference.StringFieldEditor;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.DialogCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.mtj.core.project.midp.DescriptorPropertyDescription;
import org.eclipse.mtj.core.project.midp.IMidletSuiteProject;
import org.eclipse.mtj.internal.core.project.midp.IJADConstants;
import org.eclipse.mtj.internal.core.project.midp.JADAttributesRegistry;
import org.eclipse.mtj.internal.core.project.midp.MidletSuiteFactory;
import org.eclipse.mtj.internal.core.project.midp.MidletSuiteProject;
import org.eclipse.mtj.internal.core.sdk.device.midp.PackagingModel;
import org.eclipse.mtj.internal.core.util.Utils;
import org.eclipse.mtj.internal.core.util.log.MTJLogger;
import org.eclipse.mtj.internal.ui.MTJUIMessages;
import org.eclipse.mtj.internal.ui.MTJUIPluginImages;
import org.eclipse.mtj.internal.ui.editors.FormLayoutFactory;
import org.eclipse.mtj.internal.ui.editors.jad.form.JADFormEditor;
import org.eclipse.mtj.internal.ui.editors.jad.form.pages.OverviewEditorPage.PackagingModelChangeListener;
import org.eclipse.mtj.internal.ui.util.MidletSelectionDialogCreator;
import org.eclipse.mtj.ui.editors.jad.JADPropertiesEditorPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.SelectionDialog;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.ExpandableComposite;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;

/**
 * JAD editor page for handling the optional properties.
 * 
 * @author Craig Setera
 */
public class JADOptionalPropertiesEditorPage extends JADPropertiesEditorPage implements PackagingModelChangeListener{
    
    /**
     * The unique page identifier.
     */
    public static final String ID = "optional"; //$NON-NLS-1$
        
    private Composite optionalActivePanel;
    private Composite optionalMidletPanel;
    private Composite optionalLibletPanel;
    private Composite otaActivePanel;
    private Composite otaMidletPanel;
    private Composite otaLibletPanel;

    public static final String[][] MIDLET_LIBLET_KEYS = {
    	{IJADConstants.JAD_MIDLET_DESCRIPTION, IJADConstants.JAD_LIBLET_DESCRIPTION},
    	{IJADConstants.JAD_MIDLET_DELETE_NOTIFY, IJADConstants.JAD_LIBLET_DELETE_NOTIFY},
    	{IJADConstants.JAD_MIDLET_INSTALL_NOTIFY, IJADConstants.JAD_LIBLET_INSTALL_NOTIFY}
    };
    
    private Map<String, FieldEditor> midletFieldEditorsMap = new HashMap<String, FieldEditor>();
    private Map<String, FieldEditor> libletFieldEditorsMap = new HashMap<String, FieldEditor>();
    
    private PackagingModel packagingModel = PackagingModel.MIDLET;
    
    /**
     * Creates the <b>Optional</b> JAD Properties EditorPage.
     */
    public JADOptionalPropertiesEditorPage() {
        super(ID, MTJUIMessages.JADOptionalPropertiesEditorPage_title);
        pushRegEntries = new ArrayList<PushRegEntry>();
    }

    /**
     * Creates the <b>Optional</b> JAD Properties EditorPage.
     * 
     * @param editor the parent editor
     */
    public JADOptionalPropertiesEditorPage(JADFormEditor editor) {
        super(editor, ID, MTJUIMessages.JADOptionalPropertiesEditorPage_title);
        pushRegEntries = new ArrayList<PushRegEntry>();
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#getTitle()
     */
    @Override
    public String getTitle() {
        return MTJUIMessages.JADOptionalPropertiesEditorPage_title; 
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.JADPropertiesEditorPage#addContextHelp(org.eclipse.swt.widgets.Composite)
     */
    @Override
    protected void addContextHelp(Composite c) {
        PlatformUI.getWorkbench().getHelpSystem().setHelp(c,
                "org.eclipse.mtj.ui.help_JADOptionalPropertiesEditorPage"); //$NON-NLS-1$
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#getHelpResource()
     */
    @Override
    protected String getHelpResource() {
        return "/org.eclipse.mtj.doc.user/html/reference/editors/jad_editor/optional.html"; //$NON-NLS-1$
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.JADPropertiesEditorPage#getSectionDescription()
     */
    @Override
    protected String getSectionDescription() {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.JADPropertiesEditorPage#getSectionTitle()
     */
    @Override
    protected String getSectionTitle() {
        return null;
    }
    
    @Override
    protected void createFormContent( IManagedForm managedForm ) {
        final ScrolledForm form = managedForm.getForm();
        FormToolkit toolkit = managedForm.getToolkit();
        form.setText( getTitle() );
        toolkit.decorateFormHeading( form.getForm() );
        createErrorMessageHandler( managedForm );
        /*
         * launch the help system UI, displaying the documentation identified by
         * the href parameter.
         */
        final String href = getHelpResource();
        if (href != null) {
            IToolBarManager manager = form.getToolBarManager();
            Action helpAction = new Action("help") { //$NON-NLS-1$
                @Override
                public void run() {
                    PlatformUI.getWorkbench().getHelpSystem()
                    .displayHelpResource(href);
                }
            };
            helpAction.setImageDescriptor(MTJUIPluginImages.DESC_LINKTOHELP);
            manager.add(helpAction);
        }
        form.updateToolBar();
        createSectionContent( managedForm, managedForm.getForm().getBody(), this );
    }
    
    @Override
    protected void createSectionContent(IManagedForm managedForm,
            Composite composite, IPropertyChangeListener propertyChangeListener) {

        FormToolkit toolkit = managedForm.getToolkit();
        Composite body = managedForm.getForm().getBody();
        body.setLayoutData(new GridData(GridData.FILL_BOTH));
        body.setLayout(new GridLayout(2, true));

        new Label(body, SWT.NONE);
        new Label(body, SWT.NONE);

        Section optionalSection = createStaticBasicSection(toolkit, body,
                MTJUIMessages.JADOptionalPropertiesEditorPage_SectionTitle,
                MTJUIMessages.JADOptionalPropertiesEditorPage_description);
        Composite optionalSectionClient = createStaticSectionClient(toolkit,
                optionalSection, "optional"); //$NON-NLS-1$
        optionalSection.setLayoutData(new GridData(
                GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
        optionalSection.setLayout(new GridLayout());

        Section otaSection = createStaticBasicSection(toolkit, body,
                MTJUIMessages.JADOTAPropertiesEditorPage_SectionTitle,
                MTJUIMessages.JADOTAPropertiesEditorPage_SectionDescription);
        Composite otaSectionClient = createStaticSectionClient(toolkit,
                otaSection, "ota"); //$NON-NLS-1$
        otaSection.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING
                | GridData.FILL_HORIZONTAL));
        otaSection.setLayout(new GridLayout());
        
        optionalActivePanel = new Composite(optionalSectionClient, SWT.NONE);
        optionalMidletPanel = new Composite(optionalActivePanel, SWT.NONE);
        optionalLibletPanel = new Composite(optionalActivePanel, SWT.NONE);
        
        otaActivePanel = new Composite(otaSectionClient, SWT.NONE);
        otaMidletPanel = new Composite(otaActivePanel, SWT.NONE);
        otaLibletPanel = new Composite(otaActivePanel, SWT.NONE);
        
        createSectionContent(managedForm, optionalSectionClient, optionalActivePanel,
        		optionalMidletPanel, optionalLibletPanel, this);
        createSectionContent(managedForm, otaSectionClient, otaActivePanel,
        		otaMidletPanel, otaLibletPanel, this);
        
        List<FieldEditor> fieldEditorsList = new ArrayList<FieldEditor>();
        fieldEditorsList.addAll(midletFieldEditorsMap.values());
        fieldEditorsList.addAll(libletFieldEditorsMap.values());
        fieldEditors = fieldEditorsList.toArray(new FieldEditor[0]);
        
        changePackagingModel(packagingModel);
        
        updateEditComponents();

        optionalSection.setClient(optionalSectionClient);
        otaSection.setClient(otaSectionClient);

        new Label(body, SWT.NONE);
        new Label(body, SWT.NONE);

        Section pushSection = toolkit.createSection(body, Section.DESCRIPTION
                | ExpandableComposite.TITLE_BAR);
        pushSection.setText(MTJUIMessages.JADPushRegistryEditorPage_sectionTitle);
        pushSection.setDescription(MTJUIMessages.JADPushRegistryEditorPage_sectionDescription);

        GridData gd = new GridData(GridData.FILL_BOTH);
        gd.horizontalSpan = 2;
        pushSection.setLayoutData(gd);
        pushSection.setLayout(new GridLayout());

        Composite pushSectionClient = toolkit.createComposite(pushSection, SWT.NONE);
        pushSectionClient.setLayout(new GridLayout(2, false));
        pushSection.setClient(pushSectionClient);

        createTableViewer(toolkit, pushSectionClient);
        createButtons(toolkit, pushSectionClient);
        addContextHelp(composite);
    }
    
    /**
     * This method creates all sections on the page at one time 
     * so that the fieldEditors array is populated only once.
     */
    private void createSectionContent(IManagedForm managedForm,
        Composite section, Composite activePanel, Composite midletPanel,
        Composite libletPanel, IPropertyChangeListener propertyChangeListener) {
        
        section.setLayout(FormLayoutFactory.createSectionClientGridLayout(false, 1));
         
        activePanel.setLayout(new StackLayout());
        GridData gridData = new GridData();
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalAlignment = SWT.FILL;
        activePanel.setLayoutData(gridData);
        midletPanel.setLayout(FormLayoutFactory.createSectionClientGridLayout(false, 2));
        libletPanel.setLayout(FormLayoutFactory.createSectionClientGridLayout(false, 2));
        
        new Label(midletPanel, SWT.NONE);
        new Label(midletPanel, SWT.NONE);
        new Label(libletPanel, SWT.NONE);
        new Label(libletPanel, SWT.NONE);
        
        DescriptorPropertyDescription[] theDescriptors = getDescriptors();        
        FormToolkit toolkit = managedForm.getToolkit();

        Composite sectionComp = null;
        for (int i = 0; i < theDescriptors.length; i++) {

        	sectionComp = theDescriptors[i].isLibletDescriptor() ? libletPanel : midletPanel;
        	
            if (!theDescriptors[i].getSectionId().equals((String) section.getData())) {
                continue;
            }

            FieldEditor editor = null;
            switch (theDescriptors[i].getDataType()) {
                case DescriptorPropertyDescription.DATATYPE_INT:
                	editor = createIntegerFieldEditor(toolkit,
                            sectionComp, theDescriptors[i]);
                    break;
                case DescriptorPropertyDescription.DATATYPE_LIST:
                	editor = createComboFieldEditor(toolkit, sectionComp,
                            theDescriptors[i]);
                    break;
                case DescriptorPropertyDescription.DATATYPE_URL:
                case DescriptorPropertyDescription.DATATYPE_STRING:
                default:
                	editor = createStringFieldEditor(toolkit, sectionComp,
                            theDescriptors[i]);
                    break;
            }

            if (theDescriptors[i].isLibletDescriptor()) {
                libletFieldEditorsMap.put(theDescriptors[i].getPropertyName(), editor);
            } else {
            	midletFieldEditorsMap.put(theDescriptors[i].getPropertyName(), editor);
            }
            
            Label label = editor.getLabelControl(sectionComp);
            toolkit.adapt(label, false, false);

            // Listen for property change events on the editor
            editor.setPropertyChangeListener(propertyChangeListener);
        }

        // Adapt the Combo instances...
        Control[] children = midletPanel.getChildren();
        for (Control control : children) {
            if (control instanceof Combo) {
                toolkit.adapt(control, false, false);
            }
        }
        
        children = libletPanel.getChildren();
        for (Control control : children) {
            if (control instanceof Combo) {
                toolkit.adapt(control, false, false);
            }
        }
    }
    
    @Override
    protected DescriptorPropertyDescription[] doGetDescriptors() {
        DescriptorPropertyDescription[] optionalDescriptors;
        optionalDescriptors = JADAttributesRegistry
                .getJADAttrDescriptorsByPage(getId() + ".optional"); //$NON-NLS-1$
        DescriptorPropertyDescription[] otaDescriptors;
        otaDescriptors = JADAttributesRegistry
                .getJADAttrDescriptorsByPage(getId() + ".ota"); //$NON-NLS-1$
        DescriptorPropertyDescription[] result 
            = new DescriptorPropertyDescription[optionalDescriptors.length + otaDescriptors.length];
        System.arraycopy(optionalDescriptors, 0, result, 0,
                optionalDescriptors.length);
        System.arraycopy(otaDescriptors, 0, result, optionalDescriptors.length,
                otaDescriptors.length);
        return result;
    }
   
    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#isManagingProperty(java.lang.String)
     */
    @Override
    public boolean isManagingProperty(String property) {
        String[] managedProperties = new String[] { IJADConstants.JAD_MIDLET_DATA_SIZE,
                IJADConstants.JAD_MIDLET_DESCRIPTION, IJADConstants.JAD_MIDLET_ICON, IJADConstants.JAD_MIDLET_INFO_URL,
                IJADConstants.JAD_MIDLET_DELETE_CONFIRM, IJADConstants.JAD_MIDLET_DELETE_NOTIFY,
                IJADConstants.JAD_MIDLET_INSTALL_NOTIFY, IJADConstants.JAD_LIBLET_DELETE_NOTIFY,
                IJADConstants.JAD_LIBLET_INSTALL_NOTIFY, IJADConstants.JAD_LIBLET_DESCRIPTION};
        
        for (String managedProperty : managedProperties) {
            if (managedProperty.equals(property)) {
                return true;
            }
        }
        
        boolean manages = property.startsWith(PUSH_REGISTRY_PREFIX);
        if (manages) {
            String value = property.substring(PUSH_REGISTRY_PREFIX.length());
            try {
                Integer.parseInt(value);
            } catch (NumberFormatException e) {
                manages = false;
            }
        }

        return manages;
    }
    
    

    /* (non-Javadoc)
     * @see org.eclipse.ui.forms.editor.FormPage#setFocus()
     */
    @Override
    public void setFocus() {
        tableViewer.getTable().setFocus();
    }

    /**
     * Add a new item to the table.
     */
    private void addItem() {
        PushRegEntry midletDefinition = new PushRegEntry(
                MTJUIMessages.JADPushRegistryEditorPage_new_pushReg,
                Utils.EMPTY_STRING, "*"); //$NON-NLS-1$
        pushRegEntries.add(midletDefinition);
        tableViewer.refresh();
        setDirty(true);
    }

    /**
     * Create the add and remove buttons to the composite.
     * 
     * @param toolkit the Eclipse Form's toolkit
     * @param parent
     */
    private void createButtons(FormToolkit toolkit, Composite parent) {
        Composite composite = toolkit.createComposite(parent);
        FillLayout layout = new FillLayout();
        layout.type = SWT.VERTICAL;
        composite.setLayout(layout);

        addButton = toolkit
                .createButton(composite,
                        MTJUIMessages.JADPushRegistryEditorPage_add_btn_label,
                        SWT.PUSH);
        addButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent se) {
                addItem();
            }
        });

        toolkit.createLabel(composite, Utils.EMPTY_STRING);

        removeButton = toolkit.createButton(composite,
                MTJUIMessages.JADPushRegistryEditorPage_remove_btn_label,
                SWT.PUSH);
        removeButton.setEnabled(false);
        removeButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent se) {
                removeSelectedItems();
            }
        });
    }

    /**
     * Create the table viewer for this editor.
     * 
     * @param toolkit The Eclipse form's toolkit
     * @param parent
     */
    private void createTableViewer(FormToolkit toolkit, Composite parent) {
        
        String[] columns = new String[] {
                MTJUIMessages.JADPushRegistryEditorPage_connection_column,
                MTJUIMessages.JADPushRegistryEditorPage_class_column,
                MTJUIMessages.JADPushRegistryEditorPage_sender_column, };

        // Setup the table
        int styles = SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER
                | SWT.FULL_SELECTION;
        Table table = toolkit.createTable(parent, styles);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);
        
        table.setLayoutData(new GridData(GridData.FILL_BOTH));
        
        table.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                TableItem selected = (TableItem) e.item;
                removeButton.setEnabled(selected.getParent()
                        .getSelectionCount() > 0);
            }
        });
        tableViewer = new TableViewer(table);

        // Set the table layout on the table
        TableLayout layout = new TableLayout();

        int width = 100 / columns.length;
        for (String element : columns) {
            TableColumn column = new TableColumn(table, SWT.NONE);
            column.setText(element);
            layout.addColumnData(new ColumnWeightData(width));
        }
        table.setLayout(layout);
        
        // Set the content providers
        tableViewer.setContentProvider(new TableContentProvider());
        tableViewer.setLabelProvider(new TableLabelProvider());

        // Wire up the cell modification handling
        tableViewer.setCellModifier(new CellModifier());
        tableViewer.setColumnProperties(PROPERTIES);
        tableViewer.setCellEditors(new CellEditor[] {
                new TextCellEditor(table), new MidletCellEditor(table),
                new TextCellEditor(table), });

        // Get some data into the viewer
        tableViewer.setInput(getEditorInput());
        tableViewer.refresh();
    }

    /**
     * Load the Push Registry Entries from the current preference store.
     */
    private void loadPushRegistryEntries() {
        pushRegEntries.clear();
        IPreferenceStore store = getPreferenceStore();

        // This is sort of ugly, but IPreferenceStore does not
        // allow getting the complete list of preference keys
        for (int i = 1; i < 1000; i++) {
            String propName = PUSH_REGISTRY_PREFIX + i;

            if (store.contains(propName)) {
                String propValue = store.getString(propName);
                pushRegEntries.add(new PushRegEntry(propValue));
            } else {
                break;
            }
        }

        storedEntriesCount = pushRegEntries.size();
        if (tableViewer != null) {
            tableViewer.refresh();
        }
    }

    /**
     * Remove the items currently selected within the table.
     */
    private void removeSelectedItems() {
        int[] indices = tableViewer.getTable().getSelectionIndices();

        for (int i = indices.length; i > 0; i--) {
            int index = indices[i - 1];
            pushRegEntries.remove(index);
        }

        setDirty(true);
        tableViewer.refresh();
    }    
    
    /** The prefix of all push registry definition properties */
    public static final String PUSH_REGISTRY_PREFIX = "MIDlet-Push-"; //$NON-NLS-1$

    private static final String PROP_CLASS = "class"; //$NON-NLS-1$

    // Column property names
    private static final String PROP_CONNSTR = "connection string"; //$NON-NLS-1$

    private static final String PROP_SENDER = "allowed sender"; //$NON-NLS-1$

    // All of the properties in order
    private static final String[] PROPERTIES = new String[] { PROP_CONNSTR,
            PROP_CLASS, PROP_SENDER };

    private static final List<String> PROPERTY_LIST = Arrays.asList(PROPERTIES);

    // Buttons
    private Button addButton;

    // The collections of entries representing the MIDlets to be registered
    private ArrayList<PushRegEntry> pushRegEntries;
    private Button removeButton;

    // The number of MIDlets registered
    private int storedEntriesCount;

    // The table viewer in use
    private TableViewer tableViewer;
    
    /* (non-Javadoc)
     * @see org.eclipse.ui.part.EditorPart#setInput(org.eclipse.ui.IEditorInput)
     */
    @Override
    protected void setInput(IEditorInput input) {
        super.setInput(input);
        if (tableViewer != null) {
            tableViewer.setInput(input);
        }

        setDirty(false);
        loadPushRegistryEntries();
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.ui.forms.editor.FormPage#doSave(org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void doSave(IProgressMonitor monitor) {
        // Save push registry entries
        monitor.setTaskName(getTitle());
        IPreferenceStore store = getPreferenceStore();
        // Add the push registry entries to the store
        int i;
        int currentEntriesCount = pushRegEntries.size();

        List<FieldEditor> integerFieldEditorsToExclude = new ArrayList<FieldEditor>();
        
		IMidletSuiteProject midletProject = MidletSuiteFactory
				.getMidletSuiteProject(getJavaProject());

		if (midletProject instanceof MidletSuiteProject) {
			Collection<FieldEditor> elementsList = null;
			switch (packagingModel) {
			case LIBLET:
				elementsList = midletFieldEditorsMap.values();
				break;
			case MIDLET:
				elementsList = libletFieldEditorsMap.values();
				break;			
			}
			
			for (FieldEditor element : elementsList) {
			    if (element instanceof IntegerFieldEditor) {
                    integerFieldEditorsToExclude.add(element);
			    } else if (element instanceof StringFieldEditor) {
    				((StringFieldEditor) element).setStringValue("");
    			}
			}
		}
        
        for (i = 0; i < currentEntriesCount; i++) {
            PushRegEntry def = pushRegEntries.get(i);
            store.setValue(PUSH_REGISTRY_PREFIX + (i + 1), def.toString());
        }
        // Removing deleted entries
        for (; i < storedEntriesCount; i++) {
            store.setToDefault(PUSH_REGISTRY_PREFIX + (i + 1));
        }
        storedEntriesCount = currentEntriesCount;
        // Save other fields
        super.doSave(monitor);
        
        for (FieldEditor fieldEditor : integerFieldEditorsToExclude) {
            store.setToDefault(fieldEditor.getPreferenceName());
        }
    }    
    
    /**
     * Implementation of the ICellModifier interface.
     */
    private class CellModifier implements ICellModifier {

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.ICellModifier#canModify(java.lang.Object, java.lang.String)
         */
        public boolean canModify(Object element, String property) {
            // All columns are modifiable
            return true;
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.ICellModifier#getValue(java.lang.Object, java.lang.String)
         */
        public Object getValue(Object element, String property) {
            Object value = null;

            if (element instanceof PushRegEntry) {
                PushRegEntry entry = (PushRegEntry) element;

                int fieldIndex = getFieldIndex(property);
                if (fieldIndex != -1) {
                    value = entry.fields[fieldIndex];
                }
            }

            return value;
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.ICellModifier#modify(java.lang.Object, java.lang.String, java.lang.Object)
         */
        public void modify(Object element, String property, Object value) {
            if (element instanceof TableItem) {
                Object data = ((TableItem) element).getData();
                String newValue = (String) value;

                if (data instanceof PushRegEntry) {
                    int fieldIndex = getFieldIndex(property);
                    PushRegEntry entry = (PushRegEntry) data;

                    if (fieldIndex != -1) {
                        updateField(entry, property, fieldIndex, newValue);
                    }
                }
            }
        }

        /**
         * Return the field index to match the specified property name.
         * 
         * @param property property name to search for
         * @return the index that matchs the specified property name.
         *         <code>-1</code> if the property is not recognized.
         */
        private int getFieldIndex(String property) {
            return PROPERTY_LIST.indexOf(property);
        }

        /**
         * Update the specified field as necessary.
         * 
         * @param entry the Push Registry entry to be updated
         * @param property property of entry to be updated
         * @param fieldIndex the index of the field to be updated in entry
         * @param newValue the new value to be set
         */
        private void updateField(PushRegEntry entry, String property,
                int fieldIndex, String newValue) {
            if (!entry.fields[fieldIndex].equals(newValue)) {
                entry.fields[fieldIndex] = newValue;
                setDirty(true);
                tableViewer.update(entry, new String[] { property });
            }
        }
    }

    /**
     * A cell editor implementation that allows for selection of a midlet class.
     */
    private class MidletCellEditor extends DialogCellEditor {
        /** Construct a new cell editor */
        MidletCellEditor(Composite parent) {
            super(parent);
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.DialogCellEditor#openDialogBox(org.eclipse.swt.widgets.Control)
         */
        @Override
        protected Object openDialogBox(Control cellEditorWindow) {
            Object value = null;

            try {
                IJavaProject javaProject = getJavaProject();
                SelectionDialog dialog = MidletSelectionDialogCreator
                        .createMidletSelectionDialog(
                                cellEditorWindow.getShell(),
                                getSite().getPage().getWorkbenchWindow(),
                                javaProject,
                                false,
                                MTJUIMessages.JADPushRegistryEditorPage_choose_MIDlet);

                if (dialog.open() == Window.OK) {
                    Object[] results = dialog.getResult();
                    if ((results != null) && (results.length > 0)) {
                        IType type = (IType) results[0];
                        if (type != null) {
                            value = type.getFullyQualifiedName();
                        }
                    }
                }
            } catch (JavaModelException e) {
                MTJLogger.log(IStatus.ERROR, "openDialogBox", e); //$NON-NLS-1$
            }

            return value;
        }
    }

    /**
     * Implementation of the table's content provider.
     */
    private class TableContentProvider implements IStructuredContentProvider {

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.IContentProvider#dispose()
         */
        public void dispose() {
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
         */
        public Object[] getElements(Object inputElement) {
            return pushRegEntries.toArray(new Object[pushRegEntries.size()]);
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
         */
        public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        }
    }

    /**
     * Implementation of the table's label provider.
     */
    private static class TableLabelProvider extends LabelProvider implements
            ITableLabelProvider {

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
         */
        public Image getColumnImage(Object element, int columnIndex) {
            return null;
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
         */
        public String getColumnText(Object element, int columnIndex) {
            return ((PushRegEntry) element).fields[columnIndex];
        }
    }

    static class PushRegEntry {
        public String[] fields;

        PushRegEntry(String entryString) {
            fields = new String[3];
            String[] tokens = entryString.split(","); //$NON-NLS-1$

            for (int i = 0; i < 3; i++) {
                fields[i] = (i > tokens.length) ? Utils.EMPTY_STRING
                        : tokens[i];
            }
        }

        PushRegEntry(String connStr, String className, String allowedSender) {
            fields = new String[3];

            fields[0] = connStr;
            fields[1] = className;
            fields[2] = allowedSender;
        }

        /* (non-Javadoc)
         * @see java.lang.Object#toString()
         */
        @Override
        public String toString() {
            StringBuffer sb = new StringBuffer();
            if (fields != null) {
                for (int i = 0; i < fields.length; i++) {
                    if (i != 0) {
                        sb.append(","); //$NON-NLS-1$
                    }
                    sb.append(fields[i]);
                }
            }
            return sb.toString();
        }

    }

	public void packagingModelChanged(PackagingModel newPackagingModel) {
		changePackagingModel(newPackagingModel);
	}
	
	private void changePackagingModel(PackagingModel newPackagingModel) {		
		if (newPackagingModel != packagingModel) {
			for (int i = 0; i < MIDLET_LIBLET_KEYS.length; i++) {
				StringFieldEditor midletField = (StringFieldEditor) midletFieldEditorsMap
						.get(MIDLET_LIBLET_KEYS[i][0]);
				StringFieldEditor libletField = (StringFieldEditor) libletFieldEditorsMap
						.get(MIDLET_LIBLET_KEYS[i][1]);
				if (midletField != null && libletField != null) {
				    if (newPackagingModel == PackagingModel.MIDLET) {
					    midletField.setStringValue(libletField.getStringValue());
    				} else {
	    				libletField.setStringValue(midletField.getStringValue());
		    		}
				}
			}
		}
		
		packagingModel = newPackagingModel;
		
		if (optionalActivePanel != null) {
		    ((StackLayout) optionalActivePanel.getLayout()).topControl =
        	    	(newPackagingModel == PackagingModel.LIBLET)
        		    ? optionalLibletPanel : optionalMidletPanel;
		    optionalActivePanel.layout();
		}
		
		if (otaActivePanel != null) {
            ((StackLayout) otaActivePanel.getLayout()).topControl = 
        	    	(newPackagingModel == PackagingModel.LIBLET) 
        		    ? otaLibletPanel : otaMidletPanel;
		
		    otaActivePanel.layout();
		}       
	}
}
