/**
 * Copyright (c) 2003,2008 Craig Setera and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Craig Setera (EclipseME) - Initial implementation
 *     Diego Sandin (Motorola)  - Refactoring package name to follow eclipse
 *                                standards
 *     Gang Ma      (Sybase)	- Refactoring the page to add expansibilities
 *     Diego Sandin (Motorola)  - Fixed error in  new MIDlet entry name generation
 *     David Marques(Motorola)  - Enabling Up and Down buttons.
 *     Diego Sandin (Motorola)  - Use Eclipse Message Bundles [Bug 255874]
 *     Edwin Carlo  (Motorola)  - Fixing MIDlets tab model synchronization.      
 */
package org.eclipse.mtj.internal.ui.editors.jad.form.pages;

import java.util.*;
import java.util.Map.Entry;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.*;
import org.eclipse.mtj.internal.core.util.Utils;
import org.eclipse.mtj.internal.ui.MTJUIMessages;
import org.eclipse.mtj.internal.ui.MTJUIPluginImages;
import org.eclipse.mtj.internal.ui.editors.IModelListener;
import org.eclipse.mtj.internal.ui.editors.jad.form.JADFormEditor;
import org.eclipse.mtj.internal.ui.editors.jad.form.pages.MidletsModel.Direction;
import org.eclipse.mtj.internal.ui.forms.blocks.*;
import org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage;
import org.eclipse.mtj.ui.editors.jad.IManifestPreferenceStore;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Button;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.IDetailsPage;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;

/**
 * A property page editor for the MIDlets in the MIDlet suite.
 * 
 * @author Craig Setera
 */
public class MidletsEditorPage extends AbstractJADEditorPage implements
        IModelListener {

    /**
     * Content providers for structured viewers.
     */
    class MasterContentProvider implements IStructuredContentProvider {

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.IContentProvider#dispose()
         */
        public void dispose() {
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
         */
        public Object[] getElements(Object inputElement) {
            return model.getContents();
        }

        /* (non-Javadoc)
         * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
         */
        public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
        }
    }

    /**
     * A label provider implementation.
     */
    class MidletMasterLabelProvider extends MasterLabelProvider {

        /* (non-Javadoc)
         * @see org.eclipse.mtj.internal.ui.forms.blocks.MasterLabelProvider#getColumnImage(java.lang.Object, int)
         */
        @Override
        public Image getColumnImage(Object obj, int index) {

            return MTJUIPluginImages.DESC_MIDLET_ICON.createImage();
        }

        /* (non-Javadoc)
         * @see org.eclipse.mtj.internal.ui.forms.blocks.MasterLabelProvider#getColumnText(java.lang.Object, int)
         */
        @Override
        public String getColumnText(Object obj, int index) {
            return ((MidletType) obj).getName();
        }
    }

    /**
     * The prefix of all MIDlet definition properties
     */
    public static final String MIDLET_PREFIX = "MIDlet-"; //$NON-NLS-1$

    /**
     * The page unique identifier
     */
    public static final String MIDLETS_PAGEID = "midlets"; //$NON-NLS-1$

    /**
     * The Add MIDlets button
     */
    private Button add;

    private ScrolledPropertiesBlock block;

    private Button down;

    /**
     * The amount of MIDlets listed.
     */
    private int midletcount = 0;

    /**
     * The model based on the MIDlets listed.
     */
    private MidletsModel model;

    /**
     * The Remove MIDlets button
     */
    private Button remove;

    /** The original MIDlet count as stored */
    private int storedMidletCount;

    private Button up;

    List<DetailPage> list = new ArrayList<DetailPage>();

    /**
     * Creates a new page for MIDlets edition, with default description. The
     * parent editor need to be passed in the <code>initialize</code> method if
     * this constructor is used.
     */
    public MidletsEditorPage() {
        super(MIDLETS_PAGEID, MTJUIMessages.MidletsEditorPage_title);
        model = new MidletsModel(null);
        model.addModelListener(this);
    }

    /**
     * Creates a new page for MIDlets edition.
     * 
     * @param editor The parent editor
     * @param title the page title
     */
    public MidletsEditorPage(JADFormEditor editor, String title) {
        super(editor, MIDLETS_PAGEID, title);
        model = new MidletsModel(null);
        model.addModelListener(this);
    }

    /* (non-Javadoc)
     * @see org.eclipse.ui.forms.editor.FormPage#doSave(org.eclipse.core.runtime.IProgressMonitor)
     */
    @Override
    public void doSave(IProgressMonitor monitor) {

        IPreferenceStore store = getPreferenceStore();

        int i;

        Object midlets[] = model.getContents();

        int currentMidletCount = midlets.length;

        for (i = 0; i < currentMidletCount; i++) {
            store.setValue(MIDLET_PREFIX + (i + 1), ((MidletType) midlets[i])
                    .toString());
        }

        for (; i < storedMidletCount; i++) {
            store.setToDefault(MIDLET_PREFIX + (i + 1));
        }

        midletcount = currentMidletCount;
        storedMidletCount = currentMidletCount;
        setDirty(false);
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#editorInputChanged()
     */
    @Override
    public void editorInputChanged() {
        // if the file has been updated external to Eclipse, the model must be
        // reseted
        if (model != null) {
            model.clear();
        }

        updateMidletProperties();
        midletcount = 0;
        if (block != null) {
            block.refresh();
        }

        for (DetailPage page : this.list) {
            if (page.getObjectClass() == MidletType.class) {
                IDetailsPage detailsPage = page.getDetailsPage();
                detailsPage.selectionChanged(null, block.getCurrentSelectedItem());
            }
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#getTitle()
     */
    @Override
    public String getTitle() {

        return MTJUIMessages.MidletsEditorPage_title;
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#isManagingProperty(java.lang.String)
     */
    @Override
    public boolean isManagingProperty(String property) {
        boolean manages = property.startsWith(MIDLET_PREFIX);
        if (manages) {
            String value = property.substring(MIDLET_PREFIX.length());
            try {
                Integer.parseInt(value);
            } catch (NumberFormatException e) {
                manages = false;
            }
        }

        return manages;
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.models.IModelListener#modelChanged(java.lang.Object[], java.lang.String, java.lang.String)
     */
    public void modelChanged(Object[] objects, String type, String property) {
        block.refresh();
        setDirty(true);
    }

    /* (non-Javadoc)
     * @see org.eclipse.ui.forms.editor.FormPage#setFocus()
     */
    @Override
    public void setFocus() {

    }

    /**
     * Adds mouse listeners to the add and remove buttons.
     */
    private void createButtonsListeners() {
        add.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                midletcount = model.getContents().length + 1;
                String midletName = MIDLET_PREFIX + String.valueOf(midletcount);

                MidletType midlet = new MidletType(midletName, midletName, Utils.EMPTY_STRING, Utils.EMPTY_STRING);
                model.add(new NamedObject[] { midlet }, true);
                block.setCurrentSelectedItem(new StructuredSelection(midlet));
                block.refresh();
                updateButtonsState();
            }
        });
        
        remove.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                IStructuredSelection ssel = (IStructuredSelection) block.getCurrentSelectedItem();

                if (ssel.size() == 1) {
                    MidletType type = (MidletType) ssel.getFirstElement();
                    int indexToSelect = Math.min(Arrays.asList(model.getContents()).indexOf(type) + 1, midletcount - 1) - 1;
                    
                    model.remove(new MidletType[] { type }, true);
                    Object[] contents = model.getContents();
                    midletcount = contents.length;
                    
                    if (indexToSelect >= 0) {
                        block.setCurrentSelectedItem(new StructuredSelection(contents[indexToSelect]));
                    }
                    
                    updateButtonsState();
                }
            }
        });

        up.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (up.isEnabled()) {
                    IStructuredSelection seletion = (IStructuredSelection) block.getCurrentSelectedItem();

                    if (seletion.size() == 1) {
                        MidletType type = (MidletType) seletion.getFirstElement();
                        model.reorder(type, Direction.UP);
                        
                        updateButtonsState();
                    }
                }
            }
        });
        
        down.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (down.isEnabled()) {
                    IStructuredSelection seletion = (IStructuredSelection) block.getCurrentSelectedItem();

                    if (seletion.size() == 1) {
                        MidletType type = (MidletType) seletion.getFirstElement();
                        model.reorder(type, Direction.DOWN);
                        
                        updateButtonsState();
                    }
                }
            }
        });
    }
    
    private void updateButtonsState() {
        ISelection selection = block.getCurrentSelectedItem();
        int selectionCount = 0;

        if (selection != null && !selection.isEmpty() && (selection instanceof IStructuredSelection)) {
            selectionCount = ((IStructuredSelection) selection).size();
        }

        remove.setEnabled(selectionCount > 0);
        
        Object[] contents = model.getContents();
        boolean enabled = selectionCount == 1;
        
        if (enabled) {
            Object selectedItem = ((IStructuredSelection) selection).getFirstElement();
            
            up.setEnabled(contents[0] != selectedItem);
            down.setEnabled(contents[contents.length - 1] != selectedItem);
        } else {
            up.setEnabled(false);
            down.setEnabled(false);
        }
    }

    /**
     * Update the MIDlet properties from the current preference store.
     */
    private void updateMidletProperties() {
        List<MidletType> midlets = new ArrayList<MidletType>();

        LinkedHashMap<String, String> midletsMap = collectAndNormalizePropertiesWithPrefix(MIDLET_PREFIX);
        
        for (Entry<String, String> midletEntry : midletsMap.entrySet()) {
            String propName = midletEntry.getKey();
            String propValue = midletEntry.getValue();

            MidletType midletType = new MidletType(propName, propValue);
            midletType.setName(midletType.getMidletName(), false);

            midlets.add(midletType);
        }

        midletcount = storedMidletCount = midletsMap.size();

        if (!midlets.isEmpty()) {
            model.add(midlets.toArray(new NamedObject[0]), false);
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#createFormContent(org.eclipse.ui.forms.IManagedForm)
     */
    @Override
    protected void createFormContent(IManagedForm managedForm) {

        list.add(new DetailPage(MidletType.class, new MidletTypeDetailsPage(
                getJavaProject())));
        block = new ScrolledPropertiesBlock(this, new MasterContentProvider(),
                new MidletMasterLabelProvider(), list);
        block.addSelectionChangedListener(new ISelectionChangedListener() {
            public void selectionChanged(SelectionChangedEvent event) {
                updateButtonsState();
            }
        });

        final ScrolledForm form = managedForm.getForm();
        FormToolkit toolkit = managedForm.getToolkit();

        form.setText(getTitle());
        toolkit.decorateFormHeading(form.getForm());

        createErrorMessageHandler(managedForm);

        /*
         * launch the help system UI, displaying the documentation identified by
         * the href parameter.
         */
        final String href = getHelpResource();
        if (href != null) {
            IToolBarManager manager = form.getToolBarManager();
            Action helpAction = new Action(
                    MTJUIMessages.MidletsEditorPage_help_action) {
                @Override
                public void run() {
                    PlatformUI.getWorkbench().getHelpSystem()
                            .displayHelpResource(href);
                }
            };

            helpAction.setImageDescriptor(MTJUIPluginImages.DESC_LINKTOHELP);
            manager.add(helpAction);
        }
        form.updateToolBar();

        block.createContent(managedForm);
        
        add = block.getButtonBarBlock().getButton(
                ButtonBarBlock.BUTTON_ADD_INDEX);
        add.setEnabled(true);

        remove = block.getButtonBarBlock().getButton(
                ButtonBarBlock.BUTTON_REMOVE_INDEX);

        up = block.getButtonBarBlock()
                .getButton(ButtonBarBlock.BUTTON_UP_INDEX);

        down = block.getButtonBarBlock().getButton(
                ButtonBarBlock.BUTTON_DOWN_INDEX);

        createButtonsListeners();
        updateButtonsState();
    }

    /* (non-Javadoc)
     * @see org.eclipse.mtj.ui.editors.jad.AbstractJADEditorPage#getHelpResource()
     */
    @Override
    protected String getHelpResource() {
        return "/org.eclipse.mtj.doc.user/html/reference/editors/jad_editor/midlets.html"; //$NON-NLS-1$
    }

    /* (non-Javadoc)
     * @see org.eclipse.ui.part.EditorPart#setInput(org.eclipse.ui.IEditorInput)
     */
    @Override
    protected void setInput(IEditorInput input) {
        super.setInput(input);
        setDirty(false);
        updateMidletProperties();
    }
}
