/**
 * Copyright (c) 2020 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform

import com.google.inject.Inject
import org.eclipse.n4js.n4JS.Expression
import org.eclipse.n4js.n4JS.LiteralOrComputedPropertyName
import org.eclipse.n4js.n4JS.PropertyNameKind
import org.eclipse.n4js.n4JS.StringLiteral
import org.eclipse.n4js.transpiler.Transformation
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef
import org.eclipse.n4js.typesystem.N4JSTypeSystem
import org.eclipse.n4js.utils.N4JSLanguageUtils

import static org.eclipse.n4js.transpiler.TranspilerBuilderBlocks.*

import static extension org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions.*

/**
 * Performs various simplifications of the output code. This should run rather late to have
 * those simplifications be applied to the results of, i.e. code generated by, the main transformations.
 */
class SimplifyTransformation extends Transformation {

	@Inject
	private N4JSTypeSystem ts;

	override assertPreConditions() {
	}

	override assertPostConditions() {
	}

	override analyze() {
		// ignore
	}

	override transform() {
		collectNodes(state.im, LiteralOrComputedPropertyName, false).forEach[simplify(it)];
	}

	def private void simplify(LiteralOrComputedPropertyName name) {
		// simplify { ["name"+1]: ... } to { ["name1"]: ... }
		if (name.kind === PropertyNameKind.COMPUTED
			&& isOfTypeStringInAST(name.expression)) {

			replace(name.expression, _StringLiteral(name.computedName));
		}
		
		// simplify { ["name"]: ... } to { "name": ... }
		if (name.kind === PropertyNameKind.COMPUTED
			&& name.expression instanceof StringLiteral) {

			name.kind = PropertyNameKind.STRING;
			name.literalName = name.computedName;
			name.computedName = null;
			remove(name.expression);
		}

		// simplify { "name": ... } to { name: ... } (iff 'name' is a valid identifier)
		if (name.kind === PropertyNameKind.STRING
			&& name.literalName !== null
			&& N4JSLanguageUtils.isValidIdentifier(name.literalName)) {

			name.kind = PropertyNameKind.IDENTIFIER;
		}
	}

	/**
	 * Returns true iff the given expression has a corresponding expression in the original AST and
	 * that expression in the AST is of type 'string'. Returns false in all error cases.
	 * <p>
	 * Does not support expressions created by earlier transformations (i.e. without original AST node);
	 * returns false in those cases.
	 */
	def private boolean isOfTypeStringInAST(Expression expressionInIM) {
		val expressionInAST = state.tracer.getOriginalASTNodeOfSameType(expressionInIM, false);
		if (expressionInAST !== null) {
			val typeRef = ts.type(state.G, expressionInAST);
			if (!(typeRef instanceof UnknownTypeRef)) {
				return ts.subtypeSucceeded(state.G, typeRef, state.G.stringTypeRef);
			}
		}
		return false;
	}
}
