/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.compileTime;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.compileTime.CompileTimeEvaluationError;
import org.eclipse.n4js.compileTime.CompileTimeValue;
import org.eclipse.n4js.n4JS.AdditiveExpression;
import org.eclipse.n4js.n4JS.AdditiveOperator;
import org.eclipse.n4js.n4JS.BinaryLogicalExpression;
import org.eclipse.n4js.n4JS.BinaryLogicalOperator;
import org.eclipse.n4js.n4JS.BooleanLiteral;
import org.eclipse.n4js.n4JS.ConditionalExpression;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.MultiplicativeExpression;
import org.eclipse.n4js.n4JS.MultiplicativeOperator;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.NullLiteral;
import org.eclipse.n4js.n4JS.NumericLiteral;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.ParenExpression;
import org.eclipse.n4js.n4JS.StringLiteral;
import org.eclipse.n4js.n4JS.TemplateLiteral;
import org.eclipse.n4js.n4JS.TemplateSegment;
import org.eclipse.n4js.n4JS.UnaryExpression;
import org.eclipse.n4js.n4JS.UnaryOperator;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.postprocessing.ASTMetaInfoUtils;
import org.eclipse.n4js.postprocessing.ASTProcessor;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TAnnotableElement;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TConstableElement;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TEnumLiteral;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.TypesPackage;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.RecursionGuard;
import org.eclipse.n4js.validation.N4JSElementKeywordProvider;
import org.eclipse.n4js.validation.validators.N4JSExpressionValidator;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Helper class to evaluate compile-time expressions.
 * <p>
 * <u>IMPORTANT IMPLEMENTATION NOTES:</u>
 * <ul>
 * <li>It is a design decision<sup>1</sup> to handle compile-time evaluation and computed property names as a separate,
 * up-front phase during post-processing before the main AST traversal begins (for details about the phases of
 * post-processing, see method {@link ASTProcessor#processAST(RuleEnvironment, Script, ASTMetaInfoCache)}).
 * <li>To achieve this, we must <b>avoid using type information during compile-time evaluation</b>, because typing
 * is part of main AST traversal, so typing an AST node would inevitably start the main AST traversal.
 * <li>the only place where this limitation becomes tricky is the evaluation of property access expressions, see
 * {@link #eval(RuleEnvironment, ParameterizedPropertyAccessExpression, RecursionGuard)}.
 * </ul>
 * 
 * <sup>1</sup> main rationale for this decision was to keep the handling of computed property names from complicating
 * the scoping and main AST traversal. Without resolving computed property names beforehand, all the code in scoping,
 * AST traversal, type system, and helper classes such as {@link ContainerTypesHelper} would have to cope with
 * unresolved property names, i.e. {@code #getName()} on a property or member would return <code>null</code> or trigger
 * some potentially complex computation in the background that might confuse AST traversal.
 */
@SuppressWarnings("all")
public class CompileTimeEvaluator {
  /**
   * Special kind of {@link CompileTimeEvaluationError} used to denote a particular case in which the
   * {@link CompileTimeEvaluator} cannot come up with the correct error message and thus delegates finding a proper
   * message to the validation, i.e. to class {@link N4JSExpressionValidator}.
   */
  public static final class UnresolvedPropertyAccessError extends CompileTimeEvaluationError {
    public UnresolvedPropertyAccessError(final ParameterizedPropertyAccessExpression astNode) {
      super("*** UnresolvedPropertyAccessError ***", astNode, 
        N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Property());
    }
    
    public ParameterizedPropertyAccessExpression getAstNodeCasted() {
      return ((ParameterizedPropertyAccessExpression) this.astNode);
    }
  }
  
  @Inject
  private N4JSElementKeywordProvider keywordProvider;
  
  /**
   * <b>
   * IMPORTANT: CLIENT CODE SHOULD NOT CALL THIS METHOD!<br>
   * Instead, read compile-time values from the cache using method {@link ASTMetaInfoUtils#getCompileTimeValue(Expression)}.<br>
   * If the evaluation result of the expression you are interested in is not being cached, add your use case to method
   * {@link N4JSLanguageUtils#isProcessedAsCompileTimeExpression(Expression)}. Only expressions for which this method
   * returns <code>true</code> will be evaluated and cached during post-processing.
   * </b>
   * <p>
   * Computes and returns the value of the given expression as a {@link CompileTimeValue}. If the given expression is
   * not a valid compile-time expression, the returned value will be {@link CompileTimeValue#isValid() invalid}. Never
   * returns <code>null</code>.
   */
  public CompileTimeValue evaluateCompileTimeExpression(final RuleEnvironment G, final Expression expr) {
    RecursionGuard<EObject> _recursionGuard = new RecursionGuard<EObject>();
    return this.eval(G, expr, _recursionGuard);
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final Expression expr, final RecursionGuard<EObject> guard) {
    String _keywordWithIndefiniteArticle = this.keywordProvider.keywordWithIndefiniteArticle(expr);
    String _plus = (_keywordWithIndefiniteArticle + " is never a compile-time expression");
    return CompileTimeValue.error(_plus, expr);
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final ParenExpression expr, final RecursionGuard<EObject> guard) {
    Expression _expression = expr.getExpression();
    boolean _tripleEquals = (_expression == null);
    if (_tripleEquals) {
      return CompileTimeValue.error();
    }
    return this.eval(G, expr.getExpression(), guard);
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final NullLiteral expr, final RecursionGuard<EObject> guard) {
    return CompileTimeValue.NULL;
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final BooleanLiteral expr, final RecursionGuard<EObject> guard) {
    return CompileTimeValue.of(Boolean.valueOf(expr.isTrue()));
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final NumericLiteral expr, final RecursionGuard<EObject> guard) {
    return CompileTimeValue.of(expr.getValue());
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final StringLiteral expr, final RecursionGuard<EObject> guard) {
    return CompileTimeValue.of(expr.getValue());
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final TemplateSegment expr, final RecursionGuard<EObject> guard) {
    return CompileTimeValue.of(expr.getValue());
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final TemplateLiteral expr, final RecursionGuard<EObject> guard) {
    final StringBuilder buff = new StringBuilder();
    final ArrayList<CompileTimeValue> invalidValues = CollectionLiterals.<CompileTimeValue>newArrayList();
    EList<Expression> _segments = expr.getSegments();
    for (final Expression seg : _segments) {
      {
        final CompileTimeValue segValue = this.eval(G, seg, guard);
        boolean _isValid = segValue.isValid();
        if (_isValid) {
          buff.append(segValue.toString());
        } else {
          invalidValues.add(segValue);
        }
      }
    }
    boolean _isEmpty = invalidValues.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      return CompileTimeValue.combineErrors(((CompileTimeValue[])Conversions.unwrapArray(invalidValues, CompileTimeValue.class)));
    }
    return CompileTimeValue.of(buff.toString());
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final UnaryExpression expr, final RecursionGuard<EObject> guard) {
    CompileTimeValue _xifexpression = null;
    Expression _expression = expr.getExpression();
    boolean _tripleNotEquals = (_expression != null);
    if (_tripleNotEquals) {
      _xifexpression = this.eval(G, expr.getExpression(), guard);
    }
    final CompileTimeValue value = _xifexpression;
    CompileTimeValue _switchResult = null;
    UnaryOperator _op = expr.getOp();
    if (_op != null) {
      switch (_op) {
        case NOT:
          _switchResult = CompileTimeValue.invert(value, expr.getExpression());
          break;
        case POS:
          CompileTimeValue _elvis = null;
          CompileTimeValue.ValueInvalid _requireValueType = CompileTimeValue.requireValueType(value, CompileTimeValue.ValueNumber.class, "operand must be a number", expr.getExpression());
          if (_requireValueType != null) {
            _elvis = _requireValueType;
          } else {
            _elvis = value;
          }
          _switchResult = _elvis;
          break;
        case NEG:
          _switchResult = CompileTimeValue.negate(value, expr.getExpression());
          break;
        case VOID:
          _switchResult = CompileTimeValue.UNDEFINED;
          break;
        default:
          UnaryOperator _op_1 = expr.getOp();
          String _plus = ("invalid operator: " + _op_1);
          _switchResult = CompileTimeValue.error(_plus, expr);
          break;
      }
    } else {
      UnaryOperator _op_1 = expr.getOp();
      String _plus = ("invalid operator: " + _op_1);
      _switchResult = CompileTimeValue.error(_plus, expr);
    }
    return _switchResult;
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final AdditiveExpression expr, final RecursionGuard<EObject> guard) {
    final Expression lhs = expr.getLhs();
    final Expression rhs = expr.getRhs();
    CompileTimeValue _xifexpression = null;
    if ((lhs != null)) {
      _xifexpression = this.eval(G, lhs, guard);
    }
    final CompileTimeValue leftValue = _xifexpression;
    CompileTimeValue _xifexpression_1 = null;
    if ((rhs != null)) {
      _xifexpression_1 = this.eval(G, rhs, guard);
    }
    final CompileTimeValue rightValue = _xifexpression_1;
    CompileTimeValue _switchResult = null;
    AdditiveOperator _op = expr.getOp();
    if (_op != null) {
      switch (_op) {
        case ADD:
          _switchResult = CompileTimeValue.add(leftValue, rightValue, expr);
          break;
        case SUB:
          _switchResult = CompileTimeValue.subtract(leftValue, rightValue, lhs, rhs);
          break;
        default:
          AdditiveOperator _op_1 = expr.getOp();
          String _plus = ("invalid operator: " + _op_1);
          _switchResult = CompileTimeValue.error(_plus, expr);
          break;
      }
    } else {
      AdditiveOperator _op_1 = expr.getOp();
      String _plus = ("invalid operator: " + _op_1);
      _switchResult = CompileTimeValue.error(_plus, expr);
    }
    return _switchResult;
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final MultiplicativeExpression expr, final RecursionGuard<EObject> guard) {
    final Expression lhs = expr.getLhs();
    final Expression rhs = expr.getRhs();
    CompileTimeValue _xifexpression = null;
    if ((lhs != null)) {
      _xifexpression = this.eval(G, lhs, guard);
    }
    final CompileTimeValue leftValue = _xifexpression;
    CompileTimeValue _xifexpression_1 = null;
    if ((rhs != null)) {
      _xifexpression_1 = this.eval(G, rhs, guard);
    }
    final CompileTimeValue rightValue = _xifexpression_1;
    CompileTimeValue _switchResult = null;
    MultiplicativeOperator _op = expr.getOp();
    if (_op != null) {
      switch (_op) {
        case TIMES:
          _switchResult = CompileTimeValue.multiply(leftValue, rightValue, lhs, rhs);
          break;
        case DIV:
          _switchResult = CompileTimeValue.divide(leftValue, rightValue, lhs, rhs);
          break;
        case MOD:
          _switchResult = CompileTimeValue.remainder(leftValue, rightValue, lhs, rhs);
          break;
        default:
          MultiplicativeOperator _op_1 = expr.getOp();
          String _plus = ("invalid operator: " + _op_1);
          _switchResult = CompileTimeValue.error(_plus, expr);
          break;
      }
    } else {
      MultiplicativeOperator _op_1 = expr.getOp();
      String _plus = ("invalid operator: " + _op_1);
      _switchResult = CompileTimeValue.error(_plus, expr);
    }
    return _switchResult;
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final BinaryLogicalExpression expr, final RecursionGuard<EObject> guard) {
    final Expression lhs = expr.getLhs();
    final Expression rhs = expr.getRhs();
    CompileTimeValue _xifexpression = null;
    if ((lhs != null)) {
      _xifexpression = this.eval(G, lhs, guard);
    }
    final CompileTimeValue leftValue = _xifexpression;
    CompileTimeValue _xifexpression_1 = null;
    if ((rhs != null)) {
      _xifexpression_1 = this.eval(G, rhs, guard);
    }
    final CompileTimeValue rightValue = _xifexpression_1;
    CompileTimeValue _switchResult = null;
    BinaryLogicalOperator _op = expr.getOp();
    if (_op != null) {
      switch (_op) {
        case AND:
          _switchResult = CompileTimeValue.and(leftValue, rightValue, lhs, rhs);
          break;
        case OR:
          _switchResult = CompileTimeValue.or(leftValue, rightValue, lhs, rhs);
          break;
        default:
          BinaryLogicalOperator _op_1 = expr.getOp();
          String _plus = ("invalid operator: " + _op_1);
          _switchResult = CompileTimeValue.error(_plus, expr);
          break;
      }
    } else {
      BinaryLogicalOperator _op_1 = expr.getOp();
      String _plus = ("invalid operator: " + _op_1);
      _switchResult = CompileTimeValue.error(_plus, expr);
    }
    return _switchResult;
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final ConditionalExpression expr, final RecursionGuard<EObject> guard) {
    final Expression condition = expr.getExpression();
    final Expression trueExpr = expr.getTrueExpression();
    final Expression falseExpr = expr.getFalseExpression();
    CompileTimeValue _xifexpression = null;
    if ((condition != null)) {
      _xifexpression = this.eval(G, condition, guard);
    }
    final CompileTimeValue conditionValue = _xifexpression;
    CompileTimeValue _xifexpression_1 = null;
    if ((trueExpr != null)) {
      _xifexpression_1 = this.eval(G, trueExpr, guard);
    }
    final CompileTimeValue trueValue = _xifexpression_1;
    CompileTimeValue _xifexpression_2 = null;
    if ((falseExpr != null)) {
      _xifexpression_2 = this.eval(G, falseExpr, guard);
    }
    final CompileTimeValue falseValue = _xifexpression_2;
    final CompileTimeValue.ValueInvalid error = CompileTimeValue.combineErrors(
      CompileTimeValue.requireValueType(conditionValue, CompileTimeValue.ValueBoolean.class, "condition must be a boolean", 
        expr.getExpression()), trueValue, falseValue);
    if ((error != null)) {
      return error;
    }
    CompileTimeValue _xifexpression_3 = null;
    Boolean _value = ((CompileTimeValue.ValueBoolean) conditionValue).getValue();
    if ((_value).booleanValue()) {
      _xifexpression_3 = trueValue;
    } else {
      _xifexpression_3 = falseValue;
    }
    return _xifexpression_3;
  }
  
  private CompileTimeValue _eval(final RuleEnvironment G, final IdentifierRef expr, final RecursionGuard<EObject> guard) {
    boolean _isUndefinedLiteral = N4JSLanguageUtils.isUndefinedLiteral(G, expr);
    if (_isUndefinedLiteral) {
      return CompileTimeValue.UNDEFINED;
    }
    final IdentifiableElement id = expr.getId();
    if (((id != null) && (!id.eIsProxy()))) {
      return this.obtainValueIfConstFieldOrVariable(G, id, expr, guard);
    }
    return CompileTimeValue.error();
  }
  
  /**
   * Handles compile-time evaluation of property access expressions.
   * <p>
   * <u>IMPORTANT IMPLEMENTATION NOTES:</u>
   * <ul>
   * <li>We must not make use of type information during compile-time evaluation (see {@link CompileTimeEvaluator}
   * for details why this rule exists).
   * <li>Since scoping of property access requires type information, we cannot use this form of scoping.
   * <li>Since this scoping would be triggered when invoking {@code #getProperty()} on the given property access
   * expression, we cannot make use of that property in this method.
   * <li>APPROACH: avoid using (ordinary) scoping but instead implement custom member lookup for the very limited cases
   * supported by compile-time expressions.
   * </ul>
   * YES, this approach introduces an unfortunate duplication of logic, but greatly simplifies other parts of the
   * system, i.e. (ordinary) scoping, AST traversal, type system.
   */
  private CompileTimeValue _eval(final RuleEnvironment G, final ParameterizedPropertyAccessExpression expr, final RecursionGuard<EObject> guard) {
    final Expression targetExpr = expr.getTarget();
    IdentifiableElement _xifexpression = null;
    if ((targetExpr instanceof IdentifierRef)) {
      _xifexpression = ((IdentifierRef)targetExpr).getId();
    }
    final IdentifiableElement targetElem = _xifexpression;
    final String propName = expr.getPropertyAsText();
    final TObjectPrototype sym = RuleEnvironmentExtensions.symbolObjectType(G);
    if ((targetElem == sym)) {
      final TField memberInSym = N4JSLanguageUtils.getAccessedBuiltInSymbol(G, expr, false);
      if ((memberInSym != null)) {
        return CompileTimeValue.of(memberInSym);
      }
    } else {
      if ((targetElem instanceof TEnum)) {
        boolean _hasAnnotation = AnnotationDefinition.STRING_BASED.hasAnnotation(((TAnnotableElement)targetElem));
        if (_hasAnnotation) {
          final Function1<TEnumLiteral, Boolean> _function = (TEnumLiteral it) -> {
            String _name = it.getName();
            return Boolean.valueOf(Objects.equal(_name, propName));
          };
          final TEnumLiteral litInEnum = IterableExtensions.<TEnumLiteral>findFirst(((TEnum)targetElem).getLiterals(), _function);
          if ((litInEnum != null)) {
            return CompileTimeValue.of(litInEnum.getValueOrName());
          }
        }
      } else {
        if ((targetElem instanceof TClassifier)) {
          final Function1<TMember, Boolean> _function_1 = (TMember it) -> {
            return Boolean.valueOf(((Objects.equal(it.getName(), propName) && it.isReadable()) && it.isStatic()));
          };
          final TMember member = IterableExtensions.<TMember>findFirst(IterableExtensions.<TMember>filterNull(((TClassifier)targetElem).getOwnedMembers()), _function_1);
          if (((member instanceof TField) && (!((TField) member).isHasComputedName()))) {
            return this.obtainValueIfConstFieldOrVariable(G, member, expr, guard);
          } else {
            CompileTimeEvaluator.UnresolvedPropertyAccessError _unresolvedPropertyAccessError = new CompileTimeEvaluator.UnresolvedPropertyAccessError(expr);
            return CompileTimeValue.error(_unresolvedPropertyAccessError);
          }
        }
      }
    }
    if (((targetElem != sym) && (!((targetElem instanceof TClassifier) || (targetElem instanceof TEnum))))) {
      return CompileTimeValue.error(
        "target of a property access must be a direct reference to a class, interface, or enum", expr, 
        N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Target());
    }
    return CompileTimeValue.error("property access must point to const fields, literals of @StringBased enums, or built-in symbols", expr);
  }
  
  /**
   * Iff the given element is a const field or variable with a valid compile-time expression as initializer, then this
   * method returns its compile-time value; otherwise, an invalid compile-time value with an appropriate error message
   * is returned. Never returns <code>null</code>.
   * <p>
   * This method only handles infinite recursion; main logic in
   * {@link #obtainValueIfConstFieldOrVariableUnguarded(RuleEnvironment, IdentifiableElement, EObject, RecursionGuard)}.
   */
  private CompileTimeValue obtainValueIfConstFieldOrVariable(final RuleEnvironment G, final IdentifiableElement targetElem, final EObject astNodeForErrorMessage, final RecursionGuard<EObject> guard) {
    boolean _tryNext = guard.tryNext(targetElem);
    if (_tryNext) {
      try {
        return this.obtainValueIfConstFieldOrVariableUnguarded(G, targetElem, astNodeForErrorMessage, guard);
      } finally {
        guard.done(targetElem);
      }
    } else {
      return CompileTimeValue.error("cyclic definition of compile-time expression", astNodeForErrorMessage);
    }
  }
  
  private CompileTimeValue obtainValueIfConstFieldOrVariableUnguarded(final RuleEnvironment G, final IdentifiableElement targetElem, final EObject astNodeForErrorMessage, final RecursionGuard<EObject> guard) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (targetElem instanceof TConstableElement) {
      _matched=true;
      _switchResult = ((TConstableElement)targetElem).isConst();
    }
    if (!_matched) {
      if (targetElem instanceof N4FieldDeclaration) {
        _matched=true;
        _switchResult = ((N4FieldDeclaration)targetElem).isConst();
      }
    }
    if (!_matched) {
      if (targetElem instanceof VariableDeclaration) {
        _matched=true;
        _switchResult = ((VariableDeclaration)targetElem).isConst();
      }
    }
    final boolean targetElemIsConst = _switchResult;
    if ((!targetElemIsConst)) {
      String _keyword = this.keywordProvider.keyword(targetElem);
      String _plus = (_keyword + " ");
      String _name = targetElem.getName();
      String _plus_1 = (_plus + _name);
      String _plus_2 = (_plus_1 + " is not const");
      return CompileTimeValue.error(_plus_2, astNodeForErrorMessage);
    }
    final CompileTimeValue valueOfTargetElem = this.obtainCompileTimeValueOfTargetElement(G, astNodeForErrorMessage.eResource(), targetElem, guard);
    if ((valueOfTargetElem != null)) {
      if ((valueOfTargetElem instanceof CompileTimeValue.ValueInvalid)) {
        String _keyword_1 = this.keywordProvider.keyword(targetElem);
        String _plus_3 = (_keyword_1 + " ");
        String _name_1 = targetElem.getName();
        String _plus_4 = (_plus_3 + _name_1);
        final String baseMsg = (_plus_4 + 
          " is const but does not have a compile-time expression as initializer");
        final String msg = CompileTimeEvaluator.combineErrorMessageWithNestedErrors(baseMsg, ((CompileTimeEvaluationError[])Conversions.unwrapArray(((CompileTimeValue.ValueInvalid)valueOfTargetElem).getErrors(), CompileTimeEvaluationError.class)));
        EReference _xifexpression = null;
        if ((astNodeForErrorMessage instanceof ParameterizedPropertyAccessExpression)) {
          _xifexpression = N4JSPackage.eINSTANCE.getParameterizedPropertyAccessExpression_Property();
        }
        final EReference feature = _xifexpression;
        return CompileTimeValue.error(msg, astNodeForErrorMessage, feature);
      }
      return valueOfTargetElem;
    }
    return CompileTimeValue.error(
      "only references to const variables with a compile-time expression as initializer are allowed", astNodeForErrorMessage);
  }
  
  private CompileTimeValue obtainCompileTimeValueOfTargetElement(final RuleEnvironment G, final Resource currentResource, final IdentifiableElement targetElem, final RecursionGuard<EObject> guard) {
    if (((targetElem.eResource() == currentResource) || CompileTimeEvaluator.hasLoadedASTElement(targetElem))) {
      EObject _xifexpression = null;
      if ((targetElem instanceof SyntaxRelatedTElement)) {
        _xifexpression = ((SyntaxRelatedTElement)targetElem).getAstElement();
      } else {
        _xifexpression = targetElem;
      }
      final EObject astNodeOfTargetElem = _xifexpression;
      Expression _switchResult = null;
      boolean _matched = false;
      if (astNodeOfTargetElem instanceof N4FieldDeclaration) {
        _matched=true;
        _switchResult = ((N4FieldDeclaration)astNodeOfTargetElem).getExpression();
      }
      if (!_matched) {
        if (astNodeOfTargetElem instanceof VariableDeclaration) {
          _matched=true;
          _switchResult = ((VariableDeclaration)astNodeOfTargetElem).getExpression();
        }
      }
      final Expression expressionOfTargetElem = _switchResult;
      if ((expressionOfTargetElem != null)) {
        return this.eval(G, expressionOfTargetElem, guard);
      }
    } else {
      if ((targetElem instanceof TConstableElement)) {
        return CompileTimeValue.deserialize(((TConstableElement)targetElem).getCompileTimeValue());
      }
    }
    return null;
  }
  
  /**
   * Tells if given element has an AST element in an already loaded AST (i.e. it is safe to invoke method
   * {@code #getASTElement()} without triggering a demand-load of the AST).
   */
  private static boolean hasLoadedASTElement(final IdentifiableElement elem) {
    EObject _xifexpression = null;
    if ((elem instanceof SyntaxRelatedTElement)) {
      Object _eGet = elem.eGet(TypesPackage.eINSTANCE.getSyntaxRelatedTElement_AstElement(), false);
      _xifexpression = ((EObject) _eGet);
    }
    final EObject astElemNonResolved = _xifexpression;
    return ((astElemNonResolved != null) && (!astElemNonResolved.eIsProxy()));
  }
  
  private static String combineErrorMessageWithNestedErrors(final String mainMessage, final CompileTimeEvaluationError... nestedErrors) {
    int _length = nestedErrors.length;
    boolean _equals = (_length == 0);
    if (_equals) {
      return mainMessage;
    } else {
      int _length_1 = nestedErrors.length;
      boolean _equals_1 = (_length_1 == 1);
      if (_equals_1) {
        String _messageWithLocation = nestedErrors[0].getMessageWithLocation();
        return ((mainMessage + ": ") + _messageWithLocation);
      } else {
        final Function1<CompileTimeEvaluationError, String> _function = (CompileTimeEvaluationError it) -> {
          return it.getMessageWithLocation();
        };
        String _join = IterableExtensions.join(ListExtensions.<CompileTimeEvaluationError, String>map(((List<CompileTimeEvaluationError>)Conversions.doWrapArray(nestedErrors)), _function), "\n- ");
        return ((mainMessage + ":\n- ") + _join);
      }
    }
  }
  
  private CompileTimeValue eval(final RuleEnvironment G, final Expression expr, final RecursionGuard<EObject> guard) {
    if (expr instanceof BooleanLiteral) {
      return _eval(G, (BooleanLiteral)expr, guard);
    } else if (expr instanceof NullLiteral) {
      return _eval(G, (NullLiteral)expr, guard);
    } else if (expr instanceof NumericLiteral) {
      return _eval(G, (NumericLiteral)expr, guard);
    } else if (expr instanceof StringLiteral) {
      return _eval(G, (StringLiteral)expr, guard);
    } else if (expr instanceof TemplateSegment) {
      return _eval(G, (TemplateSegment)expr, guard);
    } else if (expr instanceof IdentifierRef) {
      return _eval(G, (IdentifierRef)expr, guard);
    } else if (expr instanceof ParenExpression) {
      return _eval(G, (ParenExpression)expr, guard);
    } else if (expr instanceof TemplateLiteral) {
      return _eval(G, (TemplateLiteral)expr, guard);
    } else if (expr instanceof AdditiveExpression) {
      return _eval(G, (AdditiveExpression)expr, guard);
    } else if (expr instanceof BinaryLogicalExpression) {
      return _eval(G, (BinaryLogicalExpression)expr, guard);
    } else if (expr instanceof ConditionalExpression) {
      return _eval(G, (ConditionalExpression)expr, guard);
    } else if (expr instanceof MultiplicativeExpression) {
      return _eval(G, (MultiplicativeExpression)expr, guard);
    } else if (expr instanceof ParameterizedPropertyAccessExpression) {
      return _eval(G, (ParameterizedPropertyAccessExpression)expr, guard);
    } else if (expr instanceof UnaryExpression) {
      return _eval(G, (UnaryExpression)expr, guard);
    } else if (expr != null) {
      return _eval(G, expr, guard);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(G, expr, guard).toString());
    }
  }
}
