/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4idl.migrations;

import com.google.common.base.Objects;
import java.util.Arrays;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4idl.migrations.AndSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.ArrayTypeSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.ConstantSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.OrSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.SwitchCondition;
import org.eclipse.n4js.n4idl.migrations.TypeSwitchCondition;
import org.eclipse.n4js.n4idl.migrations.TypeTypeCondition;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * The MigrationSwitchComputer can be used to compute a {@link SwitchCondition} which
 * represents a runtime condition that matches values of a given compile-time {@link TypeRef}
 * at runtime (within limits).
 */
@SuppressWarnings("all")
public class MigrationSwitchComputer {
  /**
   * This exception is thrown when a {@link TypeRef} is passed to a {@link MigrationSwitchComputer}
   * but the type of {@link TypeRef} is not handled by the current implementation of the computer.
   * 
   * For instance, this may happen for {@link ComposedTypeRef}s.
   */
  public static final class UnhandledTypeRefException extends Exception {
    public UnhandledTypeRefException(final TypeRef typeRef) {
      super(String.format("The (sub-)expression %s cannot be handled by the TypeSwitch computer", typeRef.getTypeRefAsString()));
    }
  }
  
  /**
   * Converter to convert a {@link SwitchCondition} back to an equivalent {@link TypeRef}.
   * 
   * Use dynamically dispatched method {@link #toTypeRef} to trigger a recursive transformation.
   */
  private static final class SwitchCondition2TypeRefConverter {
    public static TypeRef _toTypeRef(final RuleEnvironment env, final AndSwitchCondition condition) {
      final Function1<SwitchCondition, TypeRef> _function = (SwitchCondition o) -> {
        return MigrationSwitchComputer.SwitchCondition2TypeRefConverter.toTypeRef(env, o);
      };
      return TypeUtils.createNonSimplifiedIntersectionType(ListExtensions.<SwitchCondition, TypeRef>map(condition.operands, _function));
    }
    
    public static TypeRef _toTypeRef(final RuleEnvironment env, final OrSwitchCondition condition) {
      final Function1<SwitchCondition, TypeRef> _function = (SwitchCondition o) -> {
        return MigrationSwitchComputer.SwitchCondition2TypeRefConverter.toTypeRef(env, o);
      };
      return TypeUtils.createNonSimplifiedUnionType(ListExtensions.<SwitchCondition, TypeRef>map(condition.operands, _function));
    }
    
    public static TypeRef _toTypeRef(final RuleEnvironment env, final TypeSwitchCondition condition) {
      return TypeUtils.createTypeRef(condition.type, TypingStrategy.DEFAULT, true);
    }
    
    public static TypeRef _toTypeRef(final RuleEnvironment env, final TypeTypeCondition condition) {
      return TypeUtils.createTypeTypeRef(condition.type, false);
    }
    
    public static TypeRef _toTypeRef(final RuleEnvironment env, final ConstantSwitchCondition condition) {
      return TypeUtils.createTypeRef(RuleEnvironmentExtensions.anyType(env));
    }
    
    public static TypeRef _toTypeRef(final RuleEnvironment env, final ArrayTypeSwitchCondition condition) {
      return RuleEnvironmentExtensions.arrayTypeRef(env, MigrationSwitchComputer.SwitchCondition2TypeRefConverter.toTypeRef(env, condition.elementTypeCondition));
    }
    
    public static TypeRef toTypeRef(final RuleEnvironment env, final SwitchCondition condition) {
      if (condition instanceof AndSwitchCondition) {
        return _toTypeRef(env, (AndSwitchCondition)condition);
      } else if (condition instanceof ArrayTypeSwitchCondition) {
        return _toTypeRef(env, (ArrayTypeSwitchCondition)condition);
      } else if (condition instanceof ConstantSwitchCondition) {
        return _toTypeRef(env, (ConstantSwitchCondition)condition);
      } else if (condition instanceof OrSwitchCondition) {
        return _toTypeRef(env, (OrSwitchCondition)condition);
      } else if (condition instanceof TypeSwitchCondition) {
        return _toTypeRef(env, (TypeSwitchCondition)condition);
      } else if (condition instanceof TypeTypeCondition) {
        return _toTypeRef(env, (TypeTypeCondition)condition);
      } else {
        throw new IllegalArgumentException("Unhandled parameter types: " +
          Arrays.<Object>asList(env, condition).toString());
      }
    }
  }
  
  /**
   * Computes a {@link SwitchCondition which detects the given {@link TypeRef}
   * at runtime (within limits).
   * 
   *  Currently the generated switch conditions support the following {@link TypeRef} features:
   * - parameterized array types (such as [A#1] or Array<A#1>)
   * - plain non-parameterized types (such as A#1)
   * 
   * There is currently no support for composed type references (such as A#1|A#2).
   * 
   * Furthermore, the following {@link TypeRef}s are ignored and therefore always evaluate to true
   * in the generated switch condition:
   * 
   * - TypeTypeRef
   * 
   * All other possible {@link TypeRef}s will lead to an {@link IllegalArgumentException}.
   */
  public SwitchCondition compute(final TypeRef ref) throws MigrationSwitchComputer.UnhandledTypeRefException {
    boolean _matched = false;
    if (ref instanceof ParameterizedTypeRef) {
      boolean _isParameterizedArrayTypeRef = this.isParameterizedArrayTypeRef(((ParameterizedTypeRef)ref));
      if (_isParameterizedArrayTypeRef) {
        _matched=true;
        TypeArgument _get = ((ParameterizedTypeRef)ref).getTypeArgs().get(0);
        return SwitchCondition.arrayOf(this.compute(((TypeRef) _get)));
      }
    }
    if (!_matched) {
      if (ref instanceof ParameterizedTypeRef) {
        boolean _isUnhandledBuiltInType = this.isUnhandledBuiltInType(((ParameterizedTypeRef)ref).getDeclaredType());
        if (_isUnhandledBuiltInType) {
          _matched=true;
          throw new MigrationSwitchComputer.UnhandledTypeRefException(ref);
        }
      }
    }
    if (!_matched) {
      if (ref instanceof ParameterizedTypeRef) {
        _matched=true;
        return SwitchCondition.instanceOf(((ParameterizedTypeRef)ref).getDeclaredType());
      }
    }
    if (!_matched) {
      if (ref instanceof TypeTypeRef) {
        if (((((TypeTypeRef)ref).getTypeArg() instanceof TypeRef) && (((TypeRef) ((TypeTypeRef)ref).getTypeArg()).getDeclaredType() != null))) {
          _matched=true;
          TypeArgument _typeArg = ((TypeTypeRef)ref).getTypeArg();
          return SwitchCondition.type(((TypeRef) _typeArg).getDeclaredType());
        }
      }
    }
    throw new MigrationSwitchComputer.UnhandledTypeRefException(ref);
  }
  
  /**
   * Infers the generalized {@link TypeRef} of the given typeRef, which can be recognized by
   * a type switch.
   * 
   * In many cases this {@link TypeRef} will be more generic than the given typeRef, since at runtime
   * only limited type information is available (e.g. usually no type arguments). However, it always
   * holds true that the returned type reference is a subtype of the given type reference typeRef.
   */
  public TypeRef toSwitchRecognizableTypeRef(final RuleEnvironment ruleEnv, final TypeRef typeRef) throws MigrationSwitchComputer.UnhandledTypeRefException {
    final SwitchCondition condition = this.compute(typeRef);
    return this.toTypeRef(ruleEnv, condition);
  }
  
  /**
   * Converts a given {@link SwitchCondition} to the corresponding recognized {@link TypeRef}.
   */
  public TypeRef toTypeRef(final RuleEnvironment ruleEnv, final SwitchCondition condition) {
    return MigrationSwitchComputer.SwitchCondition2TypeRefConverter.toTypeRef(ruleEnv, condition);
  }
  
  /**
   * Returns the BuiltInTypeScope that is used for the given context object.
   */
  private BuiltInTypeScope getBuiltInTypes(final EObject context) {
    return BuiltInTypeScope.get(context.eResource().getResourceSet());
  }
  
  /**
   * Returns {@code true} iff the given {@link TypeRef} refers to a parameterized Array type.
   * 
   * This excludes Array type references with type variables or wildcards as type argument.
   */
  private boolean isParameterizedArrayTypeRef(final ParameterizedTypeRef typeRef) {
    return ((Objects.equal(typeRef.getDeclaredType(), this.getBuiltInTypes(typeRef).getArrayType()) && (typeRef.getTypeArgs().size() > 0)) && (typeRef.getTypeArgs().get(0) instanceof TypeRef));
  }
  
  /**
   * Returns {@code true} iff the given {@code type} is an unhandled built-in type.
   */
  private boolean isUnhandledBuiltInType(final Type type) {
    return ((((type != null) && 
      (!(type instanceof PrimitiveType))) && 
      (type instanceof TInterface)) && 
      type.isProvidedByRuntime());
  }
}
