/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.naming;

import com.google.common.base.Objects;
import com.google.common.base.Optional;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.List;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.N4JSGlobals;
import org.eclipse.n4js.projectModel.IN4JSCore;
import org.eclipse.n4js.projectModel.IN4JSSourceContainer;
import org.eclipse.n4js.utils.ResourceType;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.xbase.lib.Extension;

/**
 * Utility class to calculate the qualified name of the resource depending on the project configuration.
 * The project configuration is provided by a package.json file in the root folder of the project.
 * This file contains the definition which folders of the current project should be handled as src rsp.
 * src-test folders. So if a resource is placed under MyProject/src/my/pack/MyResource.n4js, the calculated
 * qualified name would be my.pack.MyResource.
 * 
 * Client code should usually use {@code TModule.getName()} or {@code Script.getModule().getName()} to access the
 * module's name.
 */
@Singleton
@SuppressWarnings("all")
public class ModuleNameComputer {
  @Inject
  @Extension
  private IN4JSCore core;
  
  /**
   * Returns the qualified module name which is implicitly defined by the given resource.
   * <p>
   * Please note there is also a special treatment for Xpect test files that may have a file extension
   * like {@code ".n4js.xt"}. The calculation will handle this as a hole file extension, so {@code ".n4js"} will be pruned, too.
   */
  public QualifiedName getQualifiedModuleName(final Resource resource) {
    return this.getQualifiedModuleName(resource.getURI());
  }
  
  /**
   * Returns the qualified module name which is implicitly defined by the given resource description.
   * <p>
   * Please note there is also a special treatment for Xpect test files that may have a file extension
   * like {@code ".n4js.xt"}. The calculation will handle this as a hole file extension, so {@code ".n4js"} will be pruned, too.
   */
  public QualifiedName getQualifiedModuleName(final IResourceDescription resourceDesc) {
    return this.getQualifiedModuleName(resourceDesc.getURI());
  }
  
  /**
   * Returns the qualified module name which is explicitly defined by the given uri.
   * <p>
   * Please note there is also a special treatment for Xpect test files that may have a file extension
   * like {@code ".n4js.xt"}. The calculation will handle this as a hole file extension, so {@code ".n4js"} will be pruned, too.
   */
  public QualifiedName getQualifiedModuleName(final URI uri) {
    final Optional<? extends IN4JSSourceContainer> maybeSourceContainer = this.core.findN4JSSourceContainer(uri);
    boolean _isPresent = maybeSourceContainer.isPresent();
    if (_isPresent) {
      final IN4JSSourceContainer sourceContainer = maybeSourceContainer.get();
      final URI location = sourceContainer.getLocation();
      boolean _uriStartsWith = this.uriStartsWith(uri, location);
      if (_uriStartsWith) {
        URI relativeURI = uri.deresolve(location.appendSegment(""));
        if ((ResourceType.xtHidesOtherExtension(uri) || Objects.equal(N4JSGlobals.XT_FILE_EXTENSION, uri.fileExtension().toLowerCase()))) {
          relativeURI = relativeURI.trimFileExtension().trimFileExtension();
        } else {
          relativeURI = relativeURI.trimFileExtension();
        }
        return QualifiedName.create(relativeURI.segments());
      }
    } else {
      if (((uri.segmentCount() == 1) && (uri.fileExtension() != null))) {
        if ((ResourceType.xtHidesOtherExtension(uri) || Objects.equal(N4JSGlobals.XT_FILE_EXTENSION, uri.fileExtension().toLowerCase()))) {
          return QualifiedName.create(uri.trimFileExtension().trimFileExtension().segments());
        }
      }
    }
    return this.createDefaultQualifiedName(uri);
  }
  
  /**
   * Called only for URIs without container, e.g. from tests, or built-ins. Hardcoded values should be fine for those cases.
   */
  private QualifiedName createDefaultQualifiedName(final URI uri) {
    List<String> segmentList = uri.trimFileExtension().segmentsList();
    final int srcFolder = Math.max(segmentList.indexOf("src"), segmentList.indexOf("src-test"));
    if ((srcFolder != (-1))) {
      segmentList = segmentList.subList((srcFolder + 1), segmentList.size());
    }
    return QualifiedName.create(segmentList);
  }
  
  private boolean uriStartsWith(final URI resourceLocation, final URI containerLocation) {
    final int maxSegments = containerLocation.segmentCount();
    int _segmentCount = resourceLocation.segmentCount();
    boolean _lessThan = (_segmentCount < maxSegments);
    if (_lessThan) {
      return false;
    }
    for (int i = 0; (i < maxSegments); i++) {
      String _segment = resourceLocation.segment(i);
      String _segment_1 = containerLocation.segment(i);
      boolean _notEquals = (!Objects.equal(_segment, _segment_1));
      if (_notEquals) {
        return false;
      }
    }
    return true;
  }
}
