/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.organize.imports;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TAnnotationArgument;
import org.eclipse.n4js.ts.types.TAnnotationTypeRefArgument;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.util.AllSuperTypesCollector;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Collection of utility methods for working with N4JS DI.
 */
@SuppressWarnings("all")
public class DIUtility {
  public static boolean isSingleton(final Type type) {
    final Function1<TAnnotation, Boolean> _function = (TAnnotation it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, AnnotationDefinition.SINGLETON.name));
    };
    return IterableExtensions.<TAnnotation>exists(type.getAnnotations(), _function);
  }
  
  public static boolean hasSuperType(final Type type) {
    boolean _xifexpression = false;
    if ((type instanceof TClass)) {
      ParameterizedTypeRef _superClassRef = ((TClass)type).getSuperClassRef();
      Type _declaredType = null;
      if (_superClassRef!=null) {
        _declaredType=_superClassRef.getDeclaredType();
      }
      _xifexpression = (_declaredType instanceof TClass);
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * Returns true if provided type is instanceof {@link TClass}
   * and at least owned member is annotated with {@link AnnotationDefinition#INJECT}.
   */
  public static boolean hasInjectedMembers(final Type type) {
    boolean _xifexpression = false;
    if ((type instanceof TClass)) {
      final Function1<TClassifier, Boolean> _function = (TClassifier it) -> {
        final Function1<TMember, Boolean> _function_1 = (TMember it_1) -> {
          return Boolean.valueOf(AnnotationDefinition.INJECT.hasAnnotation(it_1));
        };
        return Boolean.valueOf(IterableExtensions.<TMember>exists(it.getOwnedMembers(), _function_1));
      };
      _xifexpression = IterableExtensions.<TClassifier>exists(AllSuperTypesCollector.collect(((ContainerType<?>)type)), _function);
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * Generate DI meta info for classes that have injected members,
   * or are can be injected and have DI relevant information, e.g. scope annotation.
   * Also if type has a super type (injection of inherited members)
   */
  public static boolean isInjectedClass(final Type it) {
    return ((DIUtility.isSingleton(it) || DIUtility.hasInjectedMembers(it)) || DIUtility.hasSuperType(it));
  }
  
  public static boolean isBinder(final Type type) {
    return AnnotationDefinition.BINDER.hasAnnotation(type);
  }
  
  public static boolean isDIComponent(final Type type) {
    return AnnotationDefinition.GENERATE_INJECTOR.hasAnnotation(type);
  }
  
  /**
   * Checks if diComponent has parent component, that is one specified by the superclass
   * or by the inheritance.
   */
  public static boolean hasParentInjector(final Type type) {
    boolean _or = false;
    boolean _hasAnnotation = AnnotationDefinition.WITH_PARENT_INJECTOR.hasAnnotation(type);
    if (_hasAnnotation) {
      _or = true;
    } else {
      boolean _xifexpression = false;
      if ((type instanceof TClass)) {
        ParameterizedTypeRef _superClassRef = ((TClass)type).getSuperClassRef();
        _xifexpression = (_superClassRef != null);
      } else {
        _xifexpression = false;
      }
      _or = _xifexpression;
    }
    return _or;
  }
  
  /**
   * @returns {@link Type} of the parent DIComponent.
   * @throws {@link RuntimeException} if no parent on provided type.
   */
  public static Type findParentDIC(final Type type) {
    TypeRef _xifexpression = null;
    boolean _hasAnnotation = AnnotationDefinition.WITH_PARENT_INJECTOR.hasAnnotation(type);
    if (_hasAnnotation) {
      TAnnotation _ownedAnnotation = AnnotationDefinition.WITH_PARENT_INJECTOR.getOwnedAnnotation(type);
      EList<TAnnotationArgument> _args = null;
      if (_ownedAnnotation!=null) {
        _args=_ownedAnnotation.getArgs();
      }
      TAnnotationArgument _head = IterableExtensions.<TAnnotationArgument>head(_args);
      _xifexpression = ((TAnnotationTypeRefArgument) _head).getTypeRef();
    } else {
      ParameterizedTypeRef _xifexpression_1 = null;
      if ((type instanceof TClass)) {
        _xifexpression_1 = ((TClass)type).getSuperClassRef();
      } else {
        _xifexpression_1 = null;
      }
      _xifexpression = _xifexpression_1;
    }
    TypeRef parent = _xifexpression;
    if ((parent != null)) {
      return parent.getDeclaredType();
    }
    String _name = type.getName();
    String _plus = ("no parent on " + _name);
    throw new RuntimeException(_plus);
  }
  
  /**
   * returns list of types that are parameters of {@link AnnotationDefinition#USE_BINDER} annotations
   * attached to a given type or empty list
   */
  public static List<Type> resolveBinders(final Type type) {
    final Function1<TAnnotation, EList<TAnnotationArgument>> _function = (TAnnotation it) -> {
      return it.getArgs();
    };
    final Function1<TAnnotationArgument, Type> _function_1 = (TAnnotationArgument it) -> {
      return ((TAnnotationTypeRefArgument) it).getTypeRef().getDeclaredType();
    };
    return IterableExtensions.<Type>toList(IterableExtensions.<Type>filterNull(IterableExtensions.<TAnnotationArgument, Type>map(Iterables.<TAnnotationArgument>concat(IterableExtensions.<TAnnotation, EList<TAnnotationArgument>>map(AnnotationDefinition.USE_BINDER.getAllOwnedAnnotations(type), _function)), _function_1)));
  }
  
  /**
   * Returns with {@code true} if one or more members of the given type are annotated with {@code @Inject} annotation.
   */
  public static boolean requiresInjection(final Type type) {
    boolean _xifexpression = false;
    if ((type instanceof TClass)) {
      final Function1<TClassifier, Boolean> _function = (TClassifier it) -> {
        final Function1<TMember, Boolean> _function_1 = (TMember it_1) -> {
          return Boolean.valueOf(AnnotationDefinition.INJECT.hasAnnotation(it_1));
        };
        return Boolean.valueOf(IterableExtensions.<TMember>exists(it.getOwnedMembers(), _function_1));
      };
      _xifexpression = IterableExtensions.<TClassifier>exists(AllSuperTypesCollector.collect(((ContainerType<?>)type)), _function);
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * Returns with {@code true} if the type reference argument requires injection. Either the declared type requires injection,
   * or the type reference represents an N4 provider, and the dependency of the provider requires injection. Otherwise
   * returns with {@code false}.
   */
  public static boolean requiresInjection(final TypeRef it) {
    return (DIUtility.requiresInjection(it.getDeclaredType()) || (DIUtility.isProviderType(it) && DIUtility.requiresInjection(DIUtility.getProvidedType(it))));
  }
  
  /**
   * Returns with {@code true} if the type reference argument is an N4 provider. Otherwise returns with {@code false}.
   * Also returns with {@code false} if the type reference is sub interface of N4 provider or a class which implements
   * the N4 provider interface.
   */
  public static boolean isProviderType(final TypeRef it) {
    return ((null != it) && (it.getDeclaredType() == RuleEnvironmentExtensions.n4ProviderType(RuleEnvironmentExtensions.newRuleEnvironment(it))));
  }
  
  /**
   * Returns with the type most nested dependency if the type reference argument represents and N4 provider.
   * Otherwise returns with {@code null}.
   */
  public static Type getProvidedType(final TypeRef it) {
    boolean _isProviderType = DIUtility.isProviderType(it);
    boolean _not = (!_isProviderType);
    if (_not) {
      return null;
    }
    TypeRef nestedTypeRef = it;
    while ((DIUtility.isProviderType(nestedTypeRef) && (nestedTypeRef instanceof ParameterizedTypeRef))) {
      {
        final Iterable<TypeRef> typeArgs = Iterables.<TypeRef>filter(((ParameterizedTypeRef) nestedTypeRef).getTypeArgs(), TypeRef.class);
        TypeRef _xifexpression = null;
        boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(typeArgs);
        if (_isNullOrEmpty) {
          _xifexpression = null;
        } else {
          _xifexpression = IterableExtensions.<TypeRef>head(typeArgs);
        }
        nestedTypeRef = _xifexpression;
      }
    }
    Type _declaredType = null;
    if (nestedTypeRef!=null) {
      _declaredType=nestedTypeRef.getDeclaredType();
    }
    return _declaredType;
  }
}
