/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.inject.Inject;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.ArrowFunction;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyAssignmentAnnotationList;
import org.eclipse.n4js.n4JS.PropertyGetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertySetterDeclaration;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4idl.versioning.MigrationUtils;
import org.eclipse.n4js.postprocessing.ASTMetaInfoUtils;
import org.eclipse.n4js.postprocessing.AbstractProcessor;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.InferenceVariable;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.xsemantics.runtime.Result;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xsemantics.runtime.RuleFailedException;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Base for all poly processors. Contains some utility and convenience methods.
 */
@SuppressWarnings("all")
abstract class AbstractPolyProcessor extends AbstractProcessor {
  @Inject
  private N4JSTypeSystem ts;
  
  /**
   * Convenience method for {@link #isPoly(Expression)} and {@link #isPoly(PropertyAssignment)}, accepting any type of
   * EObject.
   */
  public boolean isPoly(final EObject obj) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (obj instanceof Expression) {
      _matched=true;
      _switchResult = this.isPoly(((Expression)obj));
    }
    if (!_matched) {
      if (obj instanceof PropertyAssignment) {
        _matched=true;
        _switchResult = this.isPoly(((PropertyAssignment)obj));
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  /**
   * Tells whether the given expression is a poly expression, i.e. requires constraint-based type inference.
   */
  public boolean isPoly(final Expression obj) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (obj instanceof ParameterizedCallExpression) {
      _matched=true;
      boolean _xblockexpression = false;
      {
        boolean _isMigrateCall = MigrationUtils.isMigrateCall(obj);
        if (_isMigrateCall) {
          return false;
        }
        final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(obj);
        final TypeRef targetTypeRef = this.ts.type(G, ((ParameterizedCallExpression)obj).getTarget()).getValue();
        boolean _xifexpression = false;
        if ((targetTypeRef instanceof FunctionTypeExprOrRef)) {
          _xifexpression = (((FunctionTypeExprOrRef)targetTypeRef).isGeneric() && (((ParameterizedCallExpression)obj).getTypeArgs().size() < ((FunctionTypeExprOrRef)targetTypeRef).getTypeVars().size()));
        } else {
          _xifexpression = false;
        }
        _xblockexpression = _xifexpression;
      }
      _switchResult = _xblockexpression;
    }
    if (!_matched) {
      if (obj instanceof FunctionExpression) {
        _matched=true;
        _switchResult = (IterableExtensions.<FormalParameter>exists(((FunctionExpression)obj).getFpars(), ((Function1<FormalParameter, Boolean>) (FormalParameter it) -> {
          TypeRef _declaredTypeRef = it.getDeclaredTypeRef();
          return Boolean.valueOf((_declaredTypeRef == null));
        })) || (((FunctionExpression)obj).getReturnTypeRef() == null));
      }
    }
    if (!_matched) {
      if (obj instanceof ArrayLiteral) {
        _matched=true;
        _switchResult = true;
      }
    }
    if (!_matched) {
      if (obj instanceof ObjectLiteral) {
        _matched=true;
        final Function1<PropertyAssignment, Boolean> _function = (PropertyAssignment it) -> {
          return Boolean.valueOf(this.isPoly(it));
        };
        _switchResult = IterableExtensions.<PropertyAssignment>exists(((ObjectLiteral)obj).getPropertyAssignments(), _function);
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    return _switchResult;
  }
  
  /**
   * Tells whether the given PropertyAssignment is a poly "expression", i.e. requires constraint-based type inference.
   */
  private boolean isPoly(final PropertyAssignment pa) {
    boolean _switchResult = false;
    boolean _matched = false;
    if (pa instanceof PropertyNameValuePair) {
      _matched=true;
      _switchResult = ((((PropertyNameValuePair)pa).getExpression() != null) && (((PropertyNameValuePair)pa).getDeclaredTypeRef() == null));
    }
    if (!_matched) {
      if (pa instanceof PropertyGetterDeclaration) {
        _matched=true;
        TypeRef _declaredTypeRef = ((PropertyGetterDeclaration)pa).getDeclaredTypeRef();
        _switchResult = (_declaredTypeRef == null);
      }
    }
    if (!_matched) {
      if (pa instanceof PropertySetterDeclaration) {
        _matched=true;
        TypeRef _declaredTypeRef = ((PropertySetterDeclaration)pa).getDeclaredTypeRef();
        _switchResult = (_declaredTypeRef == null);
      }
    }
    if (!_matched) {
      if (pa instanceof PropertyMethodDeclaration) {
        _matched=true;
        _switchResult = false;
      }
    }
    if (!_matched) {
      if (pa instanceof PropertyAssignmentAnnotationList) {
        _matched=true;
        _switchResult = false;
      }
    }
    if (!_matched) {
      String _name = pa.eClass().getName();
      String _plus = ("unsupported subclass of PropertyAssignment: " + _name);
      throw new IllegalArgumentException(_plus);
    }
    return _switchResult;
  }
  
  /**
   * Convenience method for {@link #isRootPoly(Expression)}, accepting any type of EObject.
   */
  public boolean isRootPoly(final EObject obj) {
    boolean _xifexpression = false;
    if ((obj instanceof Expression)) {
      _xifexpression = this.isRootPoly(((Expression)obj));
    } else {
      _xifexpression = false;
    }
    return _xifexpression;
  }
  
  /**
   * Tells whether the given expression is a root poly expression, i.e. it
   * <ol>
   * <li>is a {@link #isPoly(Expression) poly expression}, <em>and</em>
   * <li>represents the root of a tree of nested poly expressions which have to be inferred together within a single
   * constraint system (this tree may have depth 0, i.e. consist only of the given expression).
   * </ol>
   */
  public boolean isRootPoly(final Expression obj) {
    boolean _isPoly = this.isPoly(obj);
    if (_isPoly) {
      final EObject p = this.getParentPolyCandidate(obj);
      return ((p == null) || (!this.isPoly(p)));
    }
    return false;
  }
  
  /**
   * Given a poly expression, returns the parent expression that <em>might</em> be the parent poly expression.
   * If the given expression is not poly, the return value is undefined.
   */
  private EObject getParentPolyCandidate(final Expression poly) {
    EObject _eContainer = null;
    if (poly!=null) {
      _eContainer=poly.eContainer();
    }
    final EObject directParent = _eContainer;
    EObject _eContainer_1 = null;
    if (directParent!=null) {
      _eContainer_1=directParent.eContainer();
    }
    final EObject grandParent = _eContainer_1;
    EObject _switchResult = null;
    boolean _matched = false;
    if (directParent instanceof Argument) {
      if (((grandParent instanceof ParameterizedCallExpression) && 
        ListExtensions.<Argument, Expression>map(((ParameterizedCallExpression) grandParent).getArguments(), ((Function1<Argument, Expression>) (Argument it) -> {
          return it.getExpression();
        })).contains(poly))) {
        _matched=true;
        _switchResult = grandParent;
      }
    }
    if (!_matched) {
      if (directParent instanceof FunctionExpression) {
        _matched=true;
        _switchResult = null;
      }
    }
    if (!_matched) {
      if (directParent instanceof ArrayElement) {
        Expression _expression = ((ArrayElement)directParent).getExpression();
        boolean _tripleEquals = (_expression == poly);
        if (_tripleEquals) {
          _matched=true;
          EObject _eContainer_2 = ((ArrayElement)directParent).eContainer();
          _switchResult = ((ArrayLiteral) _eContainer_2);
        }
      }
    }
    if (!_matched) {
      if (directParent instanceof PropertyNameValuePair) {
        Expression _expression = ((PropertyNameValuePair)directParent).getExpression();
        boolean _tripleEquals = (_expression == poly);
        if (_tripleEquals) {
          _matched=true;
          _switchResult = directParent;
        }
      }
    }
    if (!_matched) {
      if (directParent instanceof PropertyGetterDeclaration) {
        _matched=true;
        _switchResult = null;
      }
    }
    if (!_matched) {
      if (directParent instanceof PropertySetterDeclaration) {
        _matched=true;
        _switchResult = null;
      }
    }
    return _switchResult;
  }
  
  /**
   * Returns the type of a nested poly expression. The final type is returned, i.e. not the one created when preparing
   * the constraint system that may contain inference variables.
   * <p>
   * Because final types are created and stored in the typing cache in the onSuccess/onFailure lambdas and those
   * lambdas of nested poly expressions are registered before those of outer expression, we can here simply read the
   * nested poly expression's type from the cache.
   */
  protected TypeRef getFinalResultTypeOfNestedPolyExpression(final Expression nestedPolyExpression) {
    Result<TypeRef> _typeFailSafe = ASTMetaInfoUtils.getTypeFailSafe(nestedPolyExpression);
    TypeRef _value = null;
    if (_typeFailSafe!=null) {
      _value=_typeFailSafe.getValue();
    }
    return _value;
  }
  
  protected TypeRef subst(final TypeRef typeRef, final RuleEnvironment G, final Map<TypeVariable, ? extends TypeVariable> substitutions) {
    return this.subst(typeRef, G, substitutions, false);
  }
  
  protected TypeRef subst(final TypeRef typeRef, final RuleEnvironment G, final Map<TypeVariable, ? extends TypeVariable> substitutions, final boolean reverse) {
    final RuleEnvironment Gx = RuleEnvironmentExtensions.wrap(G);
    final Consumer<Map.Entry<TypeVariable, ? extends TypeVariable>> _function = (Map.Entry<TypeVariable, ? extends TypeVariable> e) -> {
      if (reverse) {
        Gx.add(e.getValue(), TypeUtils.createTypeRef(e.getKey()));
      } else {
        Gx.add(e.getKey(), TypeUtils.createTypeRef(e.getValue()));
      }
    };
    substitutions.entrySet().forEach(_function);
    final Result<TypeArgument> result = this.ts.substTypeVariables(Gx, typeRef);
    boolean _failed = result.failed();
    if (_failed) {
      RuleFailedException _ruleFailedException = result.getRuleFailedException();
      throw new IllegalArgumentException("substitution failed", _ruleFailedException);
    }
    TypeArgument _value = result.getValue();
    return ((TypeRef) _value);
  }
  
  protected TypeRef applySolution(final TypeRef typeRef, final RuleEnvironment G, final Map<InferenceVariable, TypeRef> solution) {
    if ((((typeRef == null) || (solution == null)) || solution.isEmpty())) {
      return typeRef;
    }
    final RuleEnvironment Gx = RuleEnvironmentExtensions.wrap(G);
    final Consumer<Map.Entry<InferenceVariable, TypeRef>> _function = (Map.Entry<InferenceVariable, TypeRef> e) -> {
      Gx.add(e.getKey(), e.getValue());
    };
    solution.entrySet().forEach(_function);
    final Result<TypeArgument> result = this.ts.substTypeVariables(Gx, typeRef);
    boolean _failed = result.failed();
    if (_failed) {
      RuleFailedException _ruleFailedException = result.getRuleFailedException();
      throw new IllegalArgumentException("substitution failed", _ruleFailedException);
    }
    TypeArgument _value = result.getValue();
    return ((TypeRef) _value);
  }
  
  protected Map<InferenceVariable, TypeRef> createPseudoSolution(final InferenceContext infCtx, final TypeRef defaultTypeRef) {
    final HashMap<InferenceVariable, TypeRef> pseudoSolution = CollectionLiterals.<InferenceVariable, TypeRef>newHashMap();
    Set<InferenceVariable> _inferenceVariables = infCtx.getInferenceVariables();
    for (final InferenceVariable iv : _inferenceVariables) {
      pseudoSolution.put(iv, defaultTypeRef);
    }
    return pseudoSolution;
  }
  
  protected boolean isReturningValue(final FunctionDefinition fun) {
    boolean _or = false;
    if (((fun.getBody() != null) && IteratorExtensions.<ReturnStatement>exists(fun.getBody().getAllReturnStatements(), ((Function1<ReturnStatement, Boolean>) (ReturnStatement it) -> {
      Expression _expression = it.getExpression();
      return Boolean.valueOf((_expression != null));
    })))) {
      _or = true;
    } else {
      boolean _xifexpression = false;
      if ((fun instanceof ArrowFunction)) {
        _xifexpression = ((ArrowFunction)fun).isSingleExprImplicitReturn();
      } else {
        _xifexpression = false;
      }
      _or = _xifexpression;
    }
    return _or;
  }
  
  protected TypeRef getTypeOfMember(final TMember m) {
    TypeRef _switchResult = null;
    boolean _matched = false;
    if (m instanceof TField) {
      _matched=true;
      _switchResult = ((TField)m).getTypeRef();
    }
    if (!_matched) {
      if (m instanceof TGetter) {
        _matched=true;
        _switchResult = ((TGetter)m).getDeclaredTypeRef();
      }
    }
    if (!_matched) {
      if (m instanceof TSetter) {
        _matched=true;
        TFormalParameter _fpar = null;
        if (((TSetter)m)!=null) {
          _fpar=((TSetter)m).getFpar();
        }
        _switchResult = _fpar.getTypeRef();
      }
    }
    if (!_matched) {
      if (m instanceof TMethod) {
        _matched=true;
        throw new IllegalArgumentException("this method should not be used for TMethod");
      }
    }
    if (!_matched) {
      EClass _eClass = null;
      if (m!=null) {
        _eClass=m.eClass();
      }
      String _name = null;
      if (_eClass!=null) {
        _name=_eClass.getName();
      }
      String _plus = ("unknown subtype of TMember: " + _name);
      throw new IllegalArgumentException(_plus);
    }
    return _switchResult;
  }
  
  protected void setTypeOfMember(final TMember m, final TypeRef type) {
    boolean _matched = false;
    if (m instanceof TField) {
      _matched=true;
      ((TField)m).setTypeRef(type);
    }
    if (!_matched) {
      if (m instanceof TGetter) {
        _matched=true;
        ((TGetter)m).setDeclaredTypeRef(type);
      }
    }
    if (!_matched) {
      if (m instanceof TSetter) {
        _matched=true;
        TFormalParameter _fpar = ((TSetter)m).getFpar();
        boolean _tripleNotEquals = (_fpar != null);
        if (_tripleNotEquals) {
          TFormalParameter _fpar_1 = ((TSetter)m).getFpar();
          _fpar_1.setTypeRef(type);
        }
      }
    }
    if (!_matched) {
      if (m instanceof TMethod) {
        _matched=true;
        throw new IllegalArgumentException("this method should not be used for TMethod");
      }
    }
    if (!_matched) {
      EClass _eClass = null;
      if (m!=null) {
        _eClass=m.eClass();
      }
      String _name = null;
      if (_eClass!=null) {
        _name=_eClass.getName();
      }
      String _plus = ("unknown subtype of TMember: " + _name);
      throw new IllegalArgumentException(_plus);
    }
  }
}
