/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.compileTime.CompileTimeEvaluator;
import org.eclipse.n4js.compileTime.CompileTimeValue;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.ts.types.TConstableElement;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * Processing of compile-time expressions.
 * <p>
 * All expressions for which {@link N4JSLanguageUtils#isProcessedAsCompileTimeExpression(Expression)} returns
 * <code>true</code> are evaluated and the resulting {@link CompileTimeValue} is stored in the cache. In some cases,
 * the value is also stored in the TModule.
 */
@Singleton
@SuppressWarnings("all")
public class CompileTimeExpressionProcessor {
  @Inject
  private CompileTimeEvaluator compileTimeEvaluator;
  
  /**
   * If the given AST node is an expression that is directly processed as a compile-time expression (cf.
   * {@link N4JSLanguageUtils#isProcessedAsCompileTimeExpression(Expression)}, this method will evaluate the
   * expression and store the evaluation result in the given cache; otherwise, this method will do nothing.
   */
  public void evaluateCompileTimeExpression(final RuleEnvironment G, final Expression astNode, final ASTMetaInfoCache cache, final int indentLevel) {
    boolean _isProcessedAsCompileTimeExpression = N4JSLanguageUtils.isProcessedAsCompileTimeExpression(astNode);
    if (_isProcessedAsCompileTimeExpression) {
      final CompileTimeValue value = this.compileTimeEvaluator.evaluateCompileTimeExpression(G, astNode);
      cache.storeCompileTimeValue(astNode, value);
      final EObject parent = astNode.eContainer();
      if ((parent instanceof ExportedVariableDeclaration)) {
        this.storeValueInTModule(G, ((ExportedVariableDeclaration)parent).getDefinedVariable(), value);
      } else {
        if ((parent instanceof N4FieldDeclaration)) {
          this.storeValueInTModule(G, ((N4FieldDeclaration)parent).getDefinedField(), value);
        }
      }
    }
  }
  
  private void storeValueInTModule(final RuleEnvironment G, final TConstableElement elem, final CompileTimeValue value) {
    if (((elem != null) && elem.isConst())) {
      final String valueStr = CompileTimeValue.serialize(value);
      final Procedure0 _function = () -> {
        elem.setCompileTimeValue(valueStr);
      };
      EcoreUtilN4.doWithDeliver(false, _function, elem);
    }
  }
}
