/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.BindingElement;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.ASTProcessor;
import org.eclipse.n4js.postprocessing.AbstractProcessor;
import org.eclipse.n4js.postprocessing.PolyProcessor;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.xsemantics.runtime.Result;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Deals with destructuring patterns during post processing of an N4JS resource (only the destructuring pattern;
 * the value to be destructured is handled normally by the other processors).
 * <p>
 * TODO clean up handling of destructuring patterns during AST traversal, IDE-1714
 */
@Singleton
@SuppressWarnings("all")
class DestructureProcessor extends AbstractProcessor {
  @Inject
  private ASTProcessor astProcessor;
  
  @Inject
  private PolyProcessor polyProcessor;
  
  /**
   * Temporary handling of destructuring patterns while typing the AST.
   */
  public void typeDestructuringPattern(final RuleEnvironment G, final EObject node, final ASTMetaInfoCache cache, final int indentLevel) {
    UnknownTypeRef _createUnknownTypeRef = TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
    Result<TypeRef> _result = new Result<TypeRef>(_createUnknownTypeRef);
    cache.storeType(((TypableElement) node), _result);
    if ((node instanceof ObjectLiteral)) {
      final Function1<PropertyNameValuePair, Expression> _function = (PropertyNameValuePair it) -> {
        return it.getExpression();
      };
      final Function1<Expression, Boolean> _function_1 = (Expression it) -> {
        return Boolean.valueOf((this.polyProcessor.isResponsibleFor(it) && (!this.polyProcessor.isEntryPoint(it))));
      };
      final Consumer<Expression> _function_2 = (Expression it) -> {
        this.polyProcessor.inferType(G, it, cache);
      };
      IterableExtensions.<Expression>filter(IterableExtensions.<Expression>filterNull(IterableExtensions.<PropertyNameValuePair, Expression>map(Iterables.<PropertyNameValuePair>filter(((ObjectLiteral)node).getPropertyAssignments(), PropertyNameValuePair.class), _function)), _function_1).forEach(_function_2);
      final Procedure1<DeferredTypeRef> _function_3 = (DeferredTypeRef dtr) -> {
        final Procedure0 _function_4 = () -> {
          EcoreUtil.replace(dtr, TypeRefsFactory.eINSTANCE.createUnknownTypeRef());
        };
        EcoreUtilN4.doWithDeliver(false, _function_4, dtr.eContainer());
      };
      IteratorExtensions.<DeferredTypeRef>forEach(Iterators.<DeferredTypeRef>filter(((ObjectLiteral)node).getDefinedType().eAllContents(), DeferredTypeRef.class), _function_3);
      final Consumer<PropertyAssignment> _function_4 = (PropertyAssignment it) -> {
        UnknownTypeRef _createUnknownTypeRef_1 = TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
        Result<TypeRef> _result_1 = new Result<TypeRef>(_createUnknownTypeRef_1);
        cache.storeType(it, _result_1);
      };
      ((ObjectLiteral)node).getPropertyAssignments().forEach(_function_4);
    }
    final Function1<EObject, Boolean> _function_5 = (EObject it) -> {
      return Boolean.valueOf(((it instanceof ObjectLiteral) || (it instanceof ArrayLiteral)));
    };
    final Function1<EObject, Boolean> _function_6 = (EObject it) -> {
      Result<TypeRef> _typeFailSafe = cache.getTypeFailSafe(((TypableElement) it));
      return Boolean.valueOf((_typeFailSafe == null));
    };
    final Procedure1<EObject> _function_7 = (EObject it) -> {
      UnknownTypeRef _createUnknownTypeRef_1 = TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
      Result<TypeRef> _result_1 = new Result<TypeRef>(_createUnknownTypeRef_1);
      cache.storeType(((TypableElement) it), _result_1);
    };
    IteratorExtensions.<EObject>forEach(IteratorExtensions.<EObject>filter(IteratorExtensions.<EObject>filter(node.eAllContents(), _function_5), _function_6), _function_7);
  }
  
  /**
   * Temporary handling of forward references within destructuring patterns.
   */
  public Result<TypeRef> handleForwardReferenceWhileTypingDestructuringPattern(final RuleEnvironment G, final TypableElement node, final ASTMetaInfoCache cache) {
    final EObject parent = node.eContainer();
    final boolean isCyclicForwardReference = cache.astNodesCurrentlyBeingTyped.contains(node);
    if (isCyclicForwardReference) {
      if (((parent instanceof VariableBinding) && (((VariableBinding) parent).getExpression() == node))) {
        ParameterizedTypeRef _anyTypeRef = RuleEnvironmentExtensions.anyTypeRef(G);
        return new Result<TypeRef>(_anyTypeRef);
      } else {
        if (((parent instanceof ForStatement) && (((ForStatement) parent).getExpression() == node))) {
          ParameterizedTypeRef _anyTypeRef_1 = RuleEnvironmentExtensions.anyTypeRef(G);
          return new Result<TypeRef>(_anyTypeRef_1);
        }
      }
    }
    AbstractProcessor.log(0, "===START of other identifiable sub-tree");
    final RuleEnvironment G_fresh = RuleEnvironmentExtensions.wrap(G);
    this.astProcessor.processSubtree(G_fresh, node, cache, 0);
    cache.forwardProcessedSubTrees.add(node);
    AbstractProcessor.log(0, "===END of other identifiable sub-tree");
    return cache.getType(node);
  }
  
  public boolean isForwardReferenceWhileTypingDestructuringPattern(final EObject obj) {
    if ((obj instanceof Expression)) {
      final EObject parent = ((Expression)obj).eContainer();
      if ((parent instanceof ForStatement)) {
        return DestructureUtils.isTopOfDestructuringForStatement(parent);
      }
      if ((parent instanceof AssignmentExpression)) {
        return DestructureUtils.isTopOfDestructuringAssignment(parent);
      }
      return (((parent instanceof VariableBinding) || (parent instanceof BindingElement)) || ((parent instanceof VariableDeclaration) && (parent.eContainer() instanceof BindingElement)));
    }
    return false;
  }
}
