/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.base.Optional;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.AbstractPolyProcessor;
import org.eclipse.n4js.postprocessing.PolyProcessor;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.InferenceVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.n4js.utils.DestructureHelper;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * {@link PolyProcessor} delegates here for processing array literals.
 * 
 * @see PolyProcessor#inferType(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,ASTMetaInfoCache)
 * @see PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)
 */
@Singleton
@SuppressWarnings("all")
class PolyProcessor_ArrayLiteral extends AbstractPolyProcessor {
  @Inject
  private PolyProcessor polyProcessor;
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private DestructureHelper destructureHelper;
  
  /**
   * BEFORE CHANGING THIS METHOD, READ THIS:
   * {@link PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)}
   */
  TypeRef processArrayLiteral(final RuleEnvironment G, final ArrayLiteral arrLit, final TypeRef expectedTypeRef, final InferenceContext infCtx, final ASTMetaInfoCache cache) {
    final int numOfElems = arrLit.getElements().size();
    final List<TypeRef> expectedElemTypeRefs = this.getExpectedElemTypeRefs(G, expectedTypeRef);
    final boolean isValueToBeDestructured = DestructureUtils.isArrayOrObjectLiteralBeingDestructured(arrLit);
    if (isValueToBeDestructured) {
      while ((expectedElemTypeRefs.size() < numOfElems)) {
        expectedElemTypeRefs.add(RuleEnvironmentExtensions.anyTypeRef(G));
      }
    }
    boolean _isEmpty = expectedElemTypeRefs.isEmpty();
    final boolean haveUsableExpectedType = (!_isEmpty);
    if (((!haveUsableExpectedType) && (!TypeUtils.isInferenceVariable(expectedTypeRef)))) {
      final ArrayList<TypeRef> elemTypeRefs = CollectionLiterals.<TypeRef>newArrayList();
      final Function1<ArrayElement, Boolean> _function = (ArrayElement it) -> {
        Expression _expression = it.getExpression();
        return Boolean.valueOf((_expression != null));
      };
      final Iterable<ArrayElement> nonNullElems = IterableExtensions.<ArrayElement>filter(arrLit.getElements(), _function);
      for (final ArrayElement arrElem : nonNullElems) {
        TypeRef _processExpr = this.polyProcessor.processExpr(G, arrElem.getExpression(), null, infCtx, cache);
        elemTypeRefs.add(_processExpr);
      }
      final Consumer<Optional<Map<InferenceVariable, TypeRef>>> _function_1 = (Optional<Map<InferenceVariable, TypeRef>> solution) -> {
        this.handleOnSolvedPerformanceTweak(G, cache, arrLit, expectedElemTypeRefs);
      };
      infCtx.onSolved(_function_1);
      TypeRef _xifexpression = null;
      boolean _isEmpty_1 = elemTypeRefs.isEmpty();
      boolean _not = (!_isEmpty_1);
      if (_not) {
        _xifexpression = this.tsh.createUnionType(G, ((TypeRef[])Conversions.unwrapArray(elemTypeRefs, TypeRef.class)));
      } else {
        _xifexpression = RuleEnvironmentExtensions.anyTypeRef(G);
      }
      final TypeRef unionOfElemTypes = _xifexpression;
      return RuleEnvironmentExtensions.arrayTypeRef(G, unionOfElemTypes);
    }
    final int resultLen = this.getResultLength(arrLit, expectedElemTypeRefs);
    final TypeVariable[] resultInfVars = infCtx.newInferenceVariables(resultLen);
    this.processElements(G, cache, infCtx, arrLit, resultLen, resultInfVars);
    final TypeRef resultTypeRef = this.getResultTypeRef(G, resultLen, resultInfVars);
    final Consumer<Optional<Map<InferenceVariable, TypeRef>>> _function_2 = (Optional<Map<InferenceVariable, TypeRef>> solution) -> {
      this.handleOnSolved(G, cache, arrLit, expectedElemTypeRefs, resultTypeRef, solution);
    };
    infCtx.onSolved(_function_2);
    return resultTypeRef;
  }
  
  /**
   * The return value is as follows:
   * <ul>
   * <li>#[ T ] for an expectedTypeRef of the form Array<T> or Iterable<T>,</li>
   * <li>#[ T1, T2, ..., TN ] for an expectedTypeRef of the form IterableN<T1,T2,...,TN>,</li>
   * <li>#[] for any other kind of expectedTypeRef</li>
   * </ul>
   */
  private List<TypeRef> getExpectedElemTypeRefs(final RuleEnvironment G, final TypeRef expectedTypeRef) {
    if ((expectedTypeRef != null)) {
      final Iterable<TypeRef> extractedTypeRefs = this.destructureHelper.extractIterableElementTypesUBs(G, expectedTypeRef);
      return IterableExtensions.<TypeRef>toList(extractedTypeRefs);
    } else {
      return CollectionLiterals.<TypeRef>newArrayList();
    }
  }
  
  /**
   * Makes a best effort for building a type in case something went awry. It's only non-trivial in case we have an
   * expectation of IterableN.
   */
  private TypeRef buildFallbackTypeForArrayLiteral(final boolean isIterableN, final int resultLen, final List<TypeRef> elemTypeRefs, final List<TypeRef> expectedElemTypeRefs, final RuleEnvironment G) {
    if (isIterableN) {
      final TypeRef[] typeArgs = new TypeRef[resultLen];
      for (int i = 0; (i < resultLen); i++) {
        {
          final boolean isLastElem = (i == (resultLen - 1));
          TypeRef typeRef = null;
          if ((isLastElem && (elemTypeRefs.size() > resultLen))) {
            final ArrayList<TypeRef> allRemainingElementTypeRefs = CollectionLiterals.<TypeRef>newArrayList();
            final TypeRef currExpectedElemTypeRef = expectedElemTypeRefs.get(i);
            boolean allMatch = true;
            for (int j = i; (j < elemTypeRefs.size()); j++) {
              {
                final TypeRef currElementTypeRef = elemTypeRefs.get(j);
                allRemainingElementTypeRefs.add(currElementTypeRef);
                if (allMatch) {
                  final boolean actualIsSubtypeOfExpected = this.ts.subtypeSucceeded(G, currElementTypeRef, currExpectedElemTypeRef);
                  if ((!actualIsSubtypeOfExpected)) {
                    allMatch = false;
                  }
                }
              }
            }
            if (allMatch) {
              typeRef = currExpectedElemTypeRef;
            } else {
              typeRef = this.tsh.createUnionType(G, ((TypeRef[])Conversions.unwrapArray(allRemainingElementTypeRefs, TypeRef.class)));
            }
          } else {
            final TypeRef currElemTypeRef = elemTypeRefs.get(i);
            final TypeRef currExpectedElemTypeRef_1 = expectedElemTypeRefs.get(i);
            final boolean actualIsSubtypeOfExpected = this.ts.subtypeSucceeded(G, currElemTypeRef, currExpectedElemTypeRef_1);
            if (actualIsSubtypeOfExpected) {
              typeRef = currExpectedElemTypeRef_1;
            } else {
              typeRef = currElemTypeRef;
            }
          }
          typeArgs[i] = typeRef;
        }
      }
      int _size = elemTypeRefs.size();
      boolean _greaterThan = (_size > resultLen);
      if (_greaterThan) {
        final TypeRef[] remaining = Arrays.<TypeRef>copyOfRange(((TypeRef[])Conversions.unwrapArray(elemTypeRefs, TypeRef.class)), (resultLen - 1), elemTypeRefs.size());
        typeArgs[(resultLen - 1)] = this.tsh.createUnionType(G, remaining);
      }
      return RuleEnvironmentExtensions.iterableNTypeRef(G, resultLen, typeArgs);
    } else {
      TypeRef _xifexpression = null;
      boolean _isEmpty = elemTypeRefs.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        _xifexpression = this.tsh.createUnionType(G, ((TypeRef[])Conversions.unwrapArray(elemTypeRefs, TypeRef.class)));
      } else {
        _xifexpression = RuleEnvironmentExtensions.anyTypeRef(G);
      }
      final TypeRef unionOfElemTypes = _xifexpression;
      return RuleEnvironmentExtensions.arrayTypeRef(G, unionOfElemTypes);
    }
  }
  
  /**
   * choose correct number of type arguments in our to-be-created resultTypeRef
   * (always 1 for Array<T> or Iterable<T> but N for IterableN<..>, e.g. 3 for Iterable3<T1,T2,T3>)
   */
  private int getResultLength(final ArrayLiteral arrLit, final List<TypeRef> expectedElemTypeRefs) {
    final int numOfElems = arrLit.getElements().size();
    final int lenA = Math.min(
      expectedElemTypeRefs.size(), numOfElems);
    final int lenB = Math.min(lenA, 
      BuiltInTypeScope.ITERABLE_N__MAX_LEN);
    final int resultLen = Math.max(lenB, 
      1);
    return resultLen;
  }
  
  /**
   * Creates temporary type (i.e. may contain inference variables):
   * <ul>
   * <li>Array<T> (where T is a new inference variable) or</li>
   * <li>Iterable<T> (where T is a new inference variable) or</li>
   * <li>IterableN<T1,T2,...,TN> (where T1,...TN are new inference variables, N>=2)</li>
   * </ul>
   */
  private TypeRef getResultTypeRef(final RuleEnvironment G, final int resultLen, final TypeVariable[] resultInfVars) {
    final boolean isIterableN = (resultLen >= 2);
    Type _xifexpression = null;
    if (isIterableN) {
      _xifexpression = RuleEnvironmentExtensions.iterableNType(G, resultLen);
    } else {
      _xifexpression = RuleEnvironmentExtensions.arrayType(G);
    }
    final Type declaredType = ((Type)_xifexpression);
    final Function1<TypeVariable, ParameterizedTypeRef> _function = (TypeVariable it) -> {
      return TypeUtils.createTypeRef(it);
    };
    final List<ParameterizedTypeRef> typeArgs = ListExtensions.<TypeVariable, ParameterizedTypeRef>map(((List<TypeVariable>)Conversions.doWrapArray(resultInfVars)), _function);
    final TypeRef resultTypeRef = TypeUtils.createTypeRef(declaredType, ((TypeArgument[])Conversions.unwrapArray(typeArgs, TypeArgument.class)));
    return resultTypeRef;
  }
  
  /**
   * for each array element, add a constraint to ensure that its corresponding infVar in result type will be
   * a super type of the array element's expression
   */
  private void processElements(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final ArrayLiteral arrLit, final int resultLen, final TypeVariable[] resultInfVars) {
    final int numOfElems = arrLit.getElements().size();
    for (int idxElem = 0; (idxElem < numOfElems); idxElem++) {
      {
        final ArrayElement currElem = arrLit.getElements().get(idxElem);
        Expression _expression = null;
        if (currElem!=null) {
          _expression=currElem.getExpression();
        }
        boolean _tripleEquals = (_expression == null);
        if (_tripleEquals) {
        } else {
          final int idxResult = Math.min(idxElem, (resultLen - 1));
          final TypeVariable currResultInfVar = resultInfVars[idxResult];
          final TypeRef currElemTypeRef = this.polyProcessor.processExpr(G, currElem.getExpression(), TypeUtils.createTypeRef(currResultInfVar), infCtx, cache);
          infCtx.addConstraint(currElemTypeRef, TypeUtils.createTypeRef(currResultInfVar), Variance.CO);
        }
      }
    }
  }
  
  /**
   * Writes final types to cache.
   */
  private void handleOnSolvedPerformanceTweak(final RuleEnvironment G, final ASTMetaInfoCache cache, final ArrayLiteral arrLit, final List<TypeRef> expectedElemTypeRefs) {
    final Function1<ArrayElement, Boolean> _function = (ArrayElement it) -> {
      Expression _expression = it.getExpression();
      return Boolean.valueOf((_expression != null));
    };
    final Function1<ArrayElement, TypeRef> _function_1 = (ArrayElement it) -> {
      return this.getFinalResultTypeOfNestedPolyExpression(it.getExpression());
    };
    final List<TypeRef> betterElemTypeRefs = IterableExtensions.<TypeRef>toList(IterableExtensions.<ArrayElement, TypeRef>map(IterableExtensions.<ArrayElement>filter(arrLit.getElements(), _function), _function_1));
    final TypeRef fallbackTypeRef = this.buildFallbackTypeForArrayLiteral(false, 1, betterElemTypeRefs, expectedElemTypeRefs, G);
    cache.storeType(arrLit, fallbackTypeRef);
  }
  
  /**
   * Writes final types to cache.
   */
  private void handleOnSolved(final RuleEnvironment G, final ASTMetaInfoCache cache, final ArrayLiteral arrLit, final List<TypeRef> expectedElemTypeRefs, final TypeRef resultTypeRef, final Optional<Map<InferenceVariable, TypeRef>> solution) {
    final int resultLen = this.getResultLength(arrLit, expectedElemTypeRefs);
    final boolean isIterableN = (resultLen >= 2);
    boolean _isPresent = solution.isPresent();
    if (_isPresent) {
      final TypeRef typeRef = this.applySolution(resultTypeRef, G, solution.get());
      cache.storeType(arrLit, typeRef);
    } else {
      final Function1<ArrayElement, TypeRef> _function = (ArrayElement it) -> {
        TypeRef _xifexpression = null;
        Expression _expression = it.getExpression();
        boolean _tripleNotEquals = (_expression != null);
        if (_tripleNotEquals) {
          _xifexpression = this.getFinalResultTypeOfNestedPolyExpression(it.getExpression());
        } else {
          _xifexpression = RuleEnvironmentExtensions.anyTypeRef(G);
        }
        return _xifexpression;
      };
      final List<TypeRef> betterElemTypeRefs = ListExtensions.<ArrayElement, TypeRef>map(arrLit.getElements(), _function);
      final TypeRef typeRef_1 = this.buildFallbackTypeForArrayLiteral(isIterableN, resultLen, betterElemTypeRefs, expectedElemTypeRefs, G);
      cache.storeType(arrLit, typeRef_1);
    }
    EList<ArrayElement> _elements = arrLit.getElements();
    for (final ArrayElement arrElem : _elements) {
      {
        Expression _expression = null;
        if (arrElem!=null) {
          _expression=arrElem.getExpression();
        }
        final Expression expr = _expression;
        if ((expr != null)) {
          final TypeRef exprType = this.getFinalResultTypeOfNestedPolyExpression(expr);
          if ((exprType != null)) {
            cache.storeType(arrElem, exprType);
          }
        }
      }
    }
  }
}
