/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.base.Optional;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.AbstractPolyProcessor;
import org.eclipse.n4js.postprocessing.PolyProcessor;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.InferenceVariable;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * {@link PolyProcessor} delegates here for processing array literals.
 * 
 * @see PolyProcessor#inferType(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,ASTMetaInfoCache)
 * @see PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)
 */
@Singleton
@SuppressWarnings("all")
class PolyProcessor_CallExpression extends AbstractPolyProcessor {
  @Inject
  private PolyProcessor polyProcessor;
  
  @Inject
  private N4JSTypeSystem ts;
  
  /**
   * BEFORE CHANGING THIS METHOD, READ THIS:
   * {@link PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)}
   */
  TypeRef processCallExpression(final RuleEnvironment G, final ParameterizedCallExpression callExpr, final TypeRef expectedTypeRef, final InferenceContext infCtx, final ASTMetaInfoCache cache) {
    final Expression target = callExpr.getTarget();
    final TypeRef targetTypeRef = this.ts.type(G, target).getValue();
    if ((!(targetTypeRef instanceof FunctionTypeExprOrRef))) {
      return TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
    }
    final FunctionTypeExprOrRef fteor = ((FunctionTypeExprOrRef) targetTypeRef);
    final boolean isPoly = (fteor.isGeneric() && (callExpr.getTypeArgs().size() < fteor.getTypeVars().size()));
    if ((!isPoly)) {
      final TypeRef result = this.ts.type(G, callExpr).getValue();
      return result;
    }
    final Map<TypeVariable, InferenceVariable> typeParam2infVar = CollectionLiterals.<TypeVariable, InferenceVariable>newLinkedHashMap();
    EList<TypeVariable> _typeVars = fteor.getTypeVars();
    for (final TypeVariable typeParam : _typeVars) {
      typeParam2infVar.put(typeParam, infCtx.newInferenceVariable());
    }
    this.processParameters(G, cache, infCtx, callExpr, fteor, typeParam2infVar);
    final TypeRef resultTypeRefRaw = fteor.getReturnTypeRef();
    final TypeRef resultTypeRef = this.subst(resultTypeRefRaw, G, typeParam2infVar);
    final Consumer<Optional<Map<InferenceVariable, TypeRef>>> _function = (Optional<Map<InferenceVariable, TypeRef>> solution) -> {
      this.handleOnSolved(G, cache, callExpr, resultTypeRef, typeParam2infVar, solution);
    };
    infCtx.onSolved(_function);
    return resultTypeRef;
  }
  
  /**
   * Processes all parameters and derives constraints from their bounds and matching types.
   */
  private void processParameters(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final ParameterizedCallExpression callExpr, final FunctionTypeExprOrRef fteor, final Map<TypeVariable, InferenceVariable> typeParam2infVar) {
    final EList<TypeVariable> funcTypeVars = fteor.getTypeVars();
    for (final TypeVariable currTypeVar : funcTypeVars) {
      {
        TypeRef _elvis = null;
        TypeRef _typeVarUpperBound = fteor.getTypeVarUpperBound(currTypeVar);
        if (_typeVarUpperBound != null) {
          _elvis = _typeVarUpperBound;
        } else {
          TypeRef _typeVariableImplicitUpperBound = N4JSLanguageUtils.getTypeVariableImplicitUpperBound(G);
          _elvis = _typeVariableImplicitUpperBound;
        }
        final TypeRef currUB = _elvis;
        final ParameterizedTypeRef leftTypeRef = TypeUtils.createTypeRef(currTypeVar);
        final TypeRef leftTypeRefSubst = this.subst(leftTypeRef, G, typeParam2infVar);
        final TypeRef rightTypeRef = currUB;
        final TypeRef rightTypeRefSubst = this.subst(rightTypeRef, G, typeParam2infVar);
        infCtx.addConstraint(leftTypeRefSubst, rightTypeRefSubst, Variance.CO);
      }
    }
    final int argsSize = callExpr.getArguments().size();
    for (int i = 0; (i < argsSize); i++) {
      {
        Argument _get = callExpr.getArguments().get(i);
        Expression _expression = null;
        if (_get!=null) {
          _expression=_get.getExpression();
        }
        final Expression arg = _expression;
        final TFormalParameter curr_fpar = fteor.getFparForArgIdx(i);
        if (((arg != null) && (curr_fpar != null))) {
          final TypeRef fparTypeRef = curr_fpar.getTypeRef();
          final TypeRef fparTypeRefSubst = this.subst(fparTypeRef, G, typeParam2infVar);
          final TypeRef argType = this.polyProcessor.processExpr(G, arg, fparTypeRefSubst, infCtx, cache);
          if ((argType != null)) {
            infCtx.addConstraint(fparTypeRefSubst, argType, Variance.CONTRA);
          }
        } else {
          if ((arg != null)) {
            this.polyProcessor.processExpr(G, arg, null, infCtx, cache);
          }
        }
      }
    }
  }
  
  /**
   * Writes final types to cache.
   */
  private void handleOnSolved(final RuleEnvironment G, final ASTMetaInfoCache cache, final ParameterizedCallExpression callExpr, final TypeRef resultTypeRef, final Map<TypeVariable, InferenceVariable> typeParam2infVar, final Optional<Map<InferenceVariable, TypeRef>> solution) {
    boolean _isPresent = solution.isPresent();
    if (_isPresent) {
      cache.storeType(callExpr, this.applySolution(resultTypeRef, G, solution.get()));
      final Function1<InferenceVariable, TypeRef> _function = (InferenceVariable it) -> {
        return solution.get().get(it);
      };
      final List<TypeRef> inferredTypeArgs = IterableExtensions.<TypeRef>toList(IterableExtensions.<InferenceVariable, TypeRef>map(typeParam2infVar.values(), _function));
      cache.storeInferredTypeArgs(callExpr, inferredTypeArgs);
    } else {
      final HashMap<InferenceVariable, TypeRef> fakeSolution = CollectionLiterals.<InferenceVariable, TypeRef>newHashMap();
      Set<Map.Entry<TypeVariable, InferenceVariable>> _entrySet = typeParam2infVar.entrySet();
      for (final Map.Entry<TypeVariable, InferenceVariable> e : _entrySet) {
        fakeSolution.put(e.getValue(), TypeUtils.createTypeRef(e.getKey()));
      }
      cache.storeType(callExpr, this.applySolution(resultTypeRef, G, fakeSolution));
      cache.storeInferredTypeArgs(callExpr, Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList()));
    }
    EList<Argument> _arguments = callExpr.getArguments();
    for (final Argument arg : _arguments) {
      {
        Expression _expression = null;
        if (arg!=null) {
          _expression=arg.getExpression();
        }
        final Expression expr = _expression;
        if ((expr != null)) {
          final TypeRef exprType = this.getFinalResultTypeOfNestedPolyExpression(expr);
          if ((exprType != null)) {
            cache.storeType(arg, exprType);
          }
        }
      }
    }
  }
}
