/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.base.Optional;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FieldAccessor;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyGetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.PropertySetterDeclaration;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.AbstractPolyProcessor;
import org.eclipse.n4js.postprocessing.AbstractProcessor;
import org.eclipse.n4js.postprocessing.PolyProcessor;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.OptionalFieldStrategy;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.Versionable;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.InferenceVariable;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TStructGetter;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.TStructuralType;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * {@link PolyProcessor} delegates here for processing array literals.
 * 
 * @see PolyProcessor#inferType(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,ASTMetaInfoCache)
 * @see PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)
 */
@Singleton
@SuppressWarnings("all")
class PolyProcessor_ObjectLiteral extends AbstractPolyProcessor {
  @Inject
  private PolyProcessor polyProcessor;
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  /**
   * BEFORE CHANGING THIS METHOD, READ THIS:
   * {@link PolyProcessor#processExpr(RuleEnvironment,org.eclipse.n4js.n4JS.Expression,TypeRef,InferenceContext,ASTMetaInfoCache)}
   */
  TypeRef processObjectLiteral(final RuleEnvironment G, final ObjectLiteral objLit, final TypeRef expectedTypeRef, final InferenceContext infCtx, final ASTMetaInfoCache cache) {
    boolean _isPoly = this.isPoly(objLit);
    boolean _not = (!_isPoly);
    if (_not) {
      final TypeRef result = this.ts.type(G, objLit).getValue();
      return result;
    }
    final boolean haveUsableExpectedType = ((expectedTypeRef != null) && (expectedTypeRef.isUseSiteStructuralTyping() || expectedTypeRef.isDefSiteStructuralTyping()));
    final boolean quickMode = ((!haveUsableExpectedType) && (!TypeUtils.isInferenceVariable(expectedTypeRef)));
    final List<TStructMember> tMembers = CollectionLiterals.<TStructMember>newArrayList();
    final List<Pair<PropertyAssignment, ? extends Versionable>> props2InfVarOrFallbackType = CollectionLiterals.<Pair<PropertyAssignment, ? extends Versionable>>newArrayList();
    this.processProperties(G, cache, infCtx, objLit, tMembers, quickMode, props2InfVarOrFallbackType);
    this.linkGetterSetterPairs(infCtx, tMembers, quickMode);
    final ParameterizedTypeRefStructural resultTypeRef = TypeUtils.createParameterizedTypeRefStructural(RuleEnvironmentExtensions.objectType(G), TypingStrategy.STRUCTURAL, ((TStructMember[])Conversions.unwrapArray(tMembers, TStructMember.class)));
    resultTypeRef.setASTNodeOptionalFieldStrategy(OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL);
    final Consumer<Optional<Map<InferenceVariable, TypeRef>>> _function = (Optional<Map<InferenceVariable, TypeRef>> solution) -> {
      this.handleOnSolved(G, cache, infCtx, objLit, quickMode, props2InfVarOrFallbackType, solution);
    };
    infCtx.onSolved(_function);
    return resultTypeRef;
  }
  
  /**
   * Processes all properties of an object literal.
   * As of now, methods are not processed.
   */
  private void processProperties(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final ObjectLiteral objLit, final List<TStructMember> tMembers, final boolean quickMode, final List<Pair<PropertyAssignment, ? extends Versionable>> props2InfVarOrFallbackType) {
    EList<PropertyAssignment> _propertyAssignments = objLit.getPropertyAssignments();
    for (final PropertyAssignment pa : _propertyAssignments) {
      if ((pa != null)) {
        TStructMember tMember = null;
        TStructMember _definedMember = pa.getDefinedMember();
        boolean _tripleNotEquals = (_definedMember != null);
        if (_tripleNotEquals) {
          tMember = TypeUtils.<TStructMember>copy(pa.getDefinedMember());
          tMembers.add(tMember);
        } else {
        }
        boolean _isPoly = this.isPoly(pa);
        if (_isPoly) {
          if ((!(tMember instanceof TMethod))) {
            this.processNonMethodProperties(G, cache, infCtx, tMember, pa, quickMode, props2InfVarOrFallbackType);
          }
        }
      }
    }
  }
  
  /**
   * For each property in the object literal:
   *  a) introduce a new inference variable representing the property's type (except for methods)
   *  b) add a constraint: expressionTypeRef <: iv
   *  c) create a TStructMember to be used in the structural result type reference
   * 
   * @param tMember  may be null in case of invalid members.
   */
  private void processNonMethodProperties(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final TStructMember tMember, final PropertyAssignment propAssignm, final boolean quickMode, final List<Pair<PropertyAssignment, ? extends Versionable>> props2InfVarOrFallbackType) {
    if ((tMember != null)) {
      final TypeRef originalMemberType = this.getTypeOfMember(tMember);
      String _name = propAssignm.eClass().getName();
      String _plus = ("type of " + _name);
      String _plus_1 = (_plus + " in TModule should be a DeferredTypeRef");
      AbstractProcessor.assertTrueIfRigid(cache, _plus_1, 
        (originalMemberType instanceof DeferredTypeRef));
      if ((!(originalMemberType instanceof DeferredTypeRef))) {
        return;
      }
    }
    if ((!quickMode)) {
      final InferenceVariable iv = infCtx.newInferenceVariable();
      if ((tMember != null)) {
        this.setTypeOfMember(tMember, TypeUtils.createTypeRef(iv));
      }
      if ((propAssignm instanceof PropertyNameValuePair)) {
        Expression _expression = ((PropertyNameValuePair)propAssignm).getExpression();
        boolean _tripleNotEquals = (_expression != null);
        if (_tripleNotEquals) {
          final TypeRef exprTypeRef = this.polyProcessor.processExpr(G, ((PropertyNameValuePair)propAssignm).getExpression(), TypeUtils.createTypeRef(iv), infCtx, cache);
          infCtx.addConstraint(exprTypeRef, TypeUtils.createTypeRef(iv), Variance.CO);
        }
      }
      Pair<PropertyAssignment, InferenceVariable> _mappedTo = Pair.<PropertyAssignment, InferenceVariable>of(propAssignm, iv);
      props2InfVarOrFallbackType.add(_mappedTo);
    } else {
      TypeRef _switchResult = null;
      boolean _matched = false;
      if (propAssignm instanceof PropertyNameValuePair) {
        Expression _expression_1 = ((PropertyNameValuePair)propAssignm).getExpression();
        boolean _tripleNotEquals_1 = (_expression_1 != null);
        if (_tripleNotEquals_1) {
          _matched=true;
          _switchResult = this.polyProcessor.processExpr(G, ((PropertyNameValuePair)propAssignm).getExpression(), null, infCtx, cache);
        }
      }
      if (!_matched) {
        if (propAssignm instanceof PropertyGetterDeclaration) {
          _matched=true;
          TypeRef _elvis = null;
          TypeRef _declaredTypeOfOtherAccessorInPair = this.getDeclaredTypeOfOtherAccessorInPair(((FieldAccessor)propAssignm));
          if (_declaredTypeOfOtherAccessorInPair != null) {
            _elvis = _declaredTypeOfOtherAccessorInPair;
          } else {
            ParameterizedTypeRef _anyTypeRef = RuleEnvironmentExtensions.anyTypeRef(G);
            _elvis = _anyTypeRef;
          }
          _switchResult = _elvis;
        }
      }
      if (!_matched) {
        if (propAssignm instanceof PropertySetterDeclaration) {
          _matched=true;
          TypeRef _elvis = null;
          TypeRef _declaredTypeOfOtherAccessorInPair = this.getDeclaredTypeOfOtherAccessorInPair(((FieldAccessor)propAssignm));
          if (_declaredTypeOfOtherAccessorInPair != null) {
            _elvis = _declaredTypeOfOtherAccessorInPair;
          } else {
            ParameterizedTypeRef _anyTypeRef = RuleEnvironmentExtensions.anyTypeRef(G);
            _elvis = _anyTypeRef;
          }
          _switchResult = _elvis;
        }
      }
      if (!_matched) {
        _switchResult = RuleEnvironmentExtensions.anyTypeRef(G);
      }
      final TypeRef fallbackType = _switchResult;
      if ((tMember != null)) {
        this.setTypeOfMember(tMember, TypeUtils.<TypeRef>copy(fallbackType));
      }
      Pair<PropertyAssignment, TypeRef> _mappedTo_1 = Pair.<PropertyAssignment, TypeRef>of(propAssignm, fallbackType);
      props2InfVarOrFallbackType.add(_mappedTo_1);
    }
  }
  
  private TypeRef getDeclaredTypeOfOtherAccessorInPair(final FieldAccessor accAST) {
    final org.eclipse.n4js.ts.types.FieldAccessor otherPair = this.findOtherAccessorInPair(accAST.getDefinedAccessor());
    EObject _astElement = null;
    if (otherPair!=null) {
      _astElement=otherPair.getAstElement();
    }
    final FieldAccessor otherPairAST = ((FieldAccessor) _astElement);
    TypeRef _declaredTypeRef = null;
    if (otherPairAST!=null) {
      _declaredTypeRef=otherPairAST.getDeclaredTypeRef();
    }
    final TypeRef declTypeRef = _declaredTypeRef;
    return declTypeRef;
  }
  
  private org.eclipse.n4js.ts.types.FieldAccessor findOtherAccessorInPair(final org.eclipse.n4js.ts.types.FieldAccessor acc) {
    String _name = null;
    if (acc!=null) {
      _name=acc.getName();
    }
    boolean _tripleNotEquals = (_name != null);
    if (_tripleNotEquals) {
      final EObject type = acc.eContainer();
      if ((type instanceof ContainerType<?>)) {
        final boolean lookForWriteAccess = (acc instanceof TGetter);
        final TMember result = ((ContainerType<?>)type).findOwnedMember(acc.getName(), lookForWriteAccess, acc.isStatic());
        if ((result instanceof org.eclipse.n4js.ts.types.FieldAccessor)) {
          return ((org.eclipse.n4js.ts.types.FieldAccessor)result);
        }
      }
    }
    return null;
  }
  
  /**
   * Add a constraint for each getter/setter pair reflecting the relation between the getter's and setter's type.
   * Required to make the getter obtain its implicit type from the corresponding setter, and vice versa.
   */
  private void linkGetterSetterPairs(final InferenceContext infCtx, final List<TStructMember> tMembers, final boolean quickMode) {
    if ((!quickMode)) {
      for (final TStructMember tMember : tMembers) {
        if ((tMember instanceof TStructGetter)) {
          final org.eclipse.n4js.ts.types.FieldAccessor tOtherInPair = this.findOtherAccessorInPair(((org.eclipse.n4js.ts.types.FieldAccessor)tMember));
          if ((tOtherInPair != null)) {
            final TypeRef typeGetter = this.getTypeOfMember(tMember);
            final TypeRef typeSetter = this.getTypeOfMember(tOtherInPair);
            if ((TypeUtils.isInferenceVariable(typeGetter) || TypeUtils.isInferenceVariable(typeSetter))) {
              infCtx.addConstraint(typeGetter, typeSetter, Variance.CO);
            } else {
            }
          }
        }
      }
    }
  }
  
  /**
   * Writes final types to cache
   */
  private void handleOnSolved(final RuleEnvironment G, final ASTMetaInfoCache cache, final InferenceContext infCtx, final ObjectLiteral objLit, final boolean quickMode, final List<? extends Pair<PropertyAssignment, ? extends Versionable>> props2InfVarOrFallbackType, final Optional<Map<InferenceVariable, TypeRef>> solution) {
    for (final Pair<PropertyAssignment, ? extends Versionable> propPair : props2InfVarOrFallbackType) {
      {
        final PropertyAssignment propAssignm = propPair.getKey();
        final TStructMember memberInTModule = propAssignm.getDefinedMember();
        if ((memberInTModule != null)) {
          final TypeRef memberType = this.getMemberType(G, solution, quickMode, propPair);
          final TypeRef memberTypeSane = this.tsh.sanitizeTypeOfVariableFieldProperty(G, memberType);
          final Procedure0 _function = () -> {
            this.setTypeOfMember(memberInTModule, TypeUtils.<TypeRef>copy(memberTypeSane));
          };
          EcoreUtilN4.doWithDeliver(false, _function, memberInTModule);
        }
      }
    }
    Type _definedType = objLit.getDefinedType();
    final ParameterizedTypeRefStructural resultFinal = TypeUtils.createParameterizedTypeRefStructural(RuleEnvironmentExtensions.objectType(G), TypingStrategy.STRUCTURAL, 
      ((TStructuralType) _definedType));
    resultFinal.setASTNodeOptionalFieldStrategy(OptionalFieldStrategy.FIELDS_AND_ACCESSORS_OPTIONAL);
    cache.storeType(objLit, resultFinal);
    EList<PropertyAssignment> _propertyAssignments = objLit.getPropertyAssignments();
    for (final PropertyAssignment currAss : _propertyAssignments) {
      TStructMember _definedMember = currAss.getDefinedMember();
      boolean _tripleNotEquals = (_definedMember != null);
      if (_tripleNotEquals) {
        if ((currAss instanceof PropertyMethodDeclaration)) {
          cache.storeType(currAss, TypeUtils.createTypeRef(((PropertyMethodDeclaration)currAss).getDefinedMember()));
        } else {
          cache.storeType(currAss, TypeUtils.<TypeRef>copy(this.getTypeOfMember(currAss.getDefinedMember())));
        }
      } else {
        cache.storeType(currAss, TypeRefsFactory.eINSTANCE.createUnknownTypeRef());
      }
    }
  }
  
  private TypeRef getMemberType(final RuleEnvironment G, final Optional<Map<InferenceVariable, TypeRef>> solution, final boolean quickMode, final Pair<PropertyAssignment, ? extends Versionable> prop2InfVarOrFallbackType) {
    TypeRef memberType = null;
    final PropertyAssignment propAssignm = prop2InfVarOrFallbackType.getKey();
    boolean _isPresent = solution.isPresent();
    if (_isPresent) {
      if (quickMode) {
        Versionable _value = prop2InfVarOrFallbackType.getValue();
        final TypeRef fallbackType = ((TypeRef) _value);
        if ((propAssignm instanceof PropertyNameValuePair)) {
          memberType = this.getFinalResultTypeOfNestedPolyExpression(((PropertyNameValuePair)propAssignm).getExpression());
        } else {
          memberType = this.applySolution(TypeUtils.<TypeRef>copy(fallbackType), G, solution.get());
        }
      } else {
        Versionable _value_1 = prop2InfVarOrFallbackType.getValue();
        final InferenceVariable infVar = ((InferenceVariable) _value_1);
        final TypeRef fromSolution = solution.get().get(infVar);
        if ((propAssignm instanceof PropertyNameValuePair)) {
          TypeRef _xifexpression = null;
          Expression _expression = ((PropertyNameValuePair)propAssignm).getExpression();
          if ((_expression instanceof ObjectLiteral)) {
            _xifexpression = this.getFinalResultTypeOfNestedPolyExpression(((PropertyNameValuePair)propAssignm).getExpression());
          } else {
            _xifexpression = null;
          }
          final TypeRef fromCache = _xifexpression;
          if (((fromCache != null) && this.ts.equaltypeSucceeded(G, fromCache, fromSolution))) {
            memberType = fromCache;
          } else {
            memberType = fromSolution;
          }
        } else {
          memberType = fromSolution;
        }
      }
    } else {
      if ((propAssignm instanceof PropertyNameValuePair)) {
        memberType = this.getFinalResultTypeOfNestedPolyExpression(((PropertyNameValuePair)propAssignm).getExpression());
      } else {
        memberType = RuleEnvironmentExtensions.anyTypeRef(G);
      }
    }
    return memberType;
  }
}
