/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.PropertyMethodDeclaration;
import org.eclipse.n4js.n4JS.SetterDeclaration;
import org.eclipse.n4js.n4JS.TypedElement;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.AbstractProcessor;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TTypedElement;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure0;

/**
 * Processor for handling {@link DeferredTypeRef}s, <b>except</b> those related to poly expressions, which are handled
 * by the {@link PolyProcessor}s.
 */
@Singleton
@SuppressWarnings("all")
class TypeDeferredProcessor extends AbstractProcessor {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  public void handleDeferredTypeRefs_preChildren(final RuleEnvironment G, final EObject obj, final ASTMetaInfoCache cache) {
    boolean _matched = false;
    if (obj instanceof N4MethodDeclaration) {
      _matched=true;
      final TypeRef returnTypeRef = ((N4MethodDeclaration)obj).getReturnTypeRef();
      boolean _isConstructor = ((N4MethodDeclaration)obj).isConstructor();
      if (_isConstructor) {
        Type _definedType = ((N4MethodDeclaration)obj).getDefinedType();
        final TMethod tCtor = ((TMethod) _definedType);
        if ((null != tCtor)) {
          AbstractProcessor.assertTrueIfRigid(cache, "TMethod in TModule should be a constructor", tCtor.isConstructor());
          TypeRef _returnTypeRef = tCtor.getReturnTypeRef();
          AbstractProcessor.assertTrueIfRigid(cache, "return type of constructor in TModule should be a DeferredTypeRef", 
            (_returnTypeRef instanceof DeferredTypeRef));
          final ThisTypeRef implicitReturnTypeRef = TypeRefsFactory.eINSTANCE.createThisTypeRef();
          final TypeRef boundThisTypeRef = this.tsh.bindAndSubstituteThisTypeRef(G, obj, implicitReturnTypeRef);
          final Procedure0 _function = () -> {
            tCtor.setReturnValueMarkedOptional(true);
            tCtor.setReturnTypeRef(TypeUtils.<TypeRef>copy(boundThisTypeRef));
          };
          EcoreUtilN4.doWithDeliver(false, _function, tCtor);
        }
      } else {
        if ((returnTypeRef instanceof ThisTypeRef)) {
          Type _definedType_1 = ((N4MethodDeclaration)obj).getDefinedType();
          final TMethod tMethod = ((TMethod) _definedType_1);
          if ((null != tMethod)) {
            TypeRef _returnTypeRef_1 = tMethod.getReturnTypeRef();
            AbstractProcessor.assertTrueIfRigid(cache, "return type of TMethod in TModule should be a DeferredTypeRef", 
              (_returnTypeRef_1 instanceof DeferredTypeRef));
            final TypeRef boundThisTypeRef_1 = this.tsh.bindAndSubstituteThisTypeRef(G, returnTypeRef, returnTypeRef);
            final Procedure0 _function_1 = () -> {
              tMethod.setReturnTypeRef(TypeUtils.<TypeRef>copy(boundThisTypeRef_1));
            };
            EcoreUtilN4.doWithDeliver(false, _function_1, tMethod);
          }
        }
      }
    }
    if (!_matched) {
      if (obj instanceof N4GetterDeclaration) {
        _matched=true;
        final TypeRef returnTypeRef = ((N4GetterDeclaration)obj).getDeclaredTypeRef();
        if ((returnTypeRef instanceof ThisTypeRef)) {
          final TGetter tGetter = ((N4GetterDeclaration)obj).getDefinedGetter();
          TypeRef _declaredTypeRef = tGetter.getDeclaredTypeRef();
          AbstractProcessor.assertTrueIfRigid(cache, "return type of TGetter in TModule should be a DeferredTypeRef", 
            (_declaredTypeRef instanceof DeferredTypeRef));
          final TypeRef boundThisTypeRef = this.ts.thisTypeRef(G, returnTypeRef).getValue();
          final Procedure0 _function = () -> {
            tGetter.setDeclaredTypeRef(TypeUtils.<TypeRef>copy(boundThisTypeRef));
          };
          EcoreUtilN4.doWithDeliver(false, _function, tGetter);
        }
      }
    }
  }
  
  public void handleDeferredTypeRefs_postChildren(final RuleEnvironment G, final EObject obj, final ASTMetaInfoCache cache) {
    boolean _matched = false;
    if (obj instanceof ExportedVariableDeclaration) {
      _matched=true;
      final TVariable tVariable = ((ExportedVariableDeclaration)obj).getDefinedVariable();
      this.<ExportedVariableDeclaration>setTypeRef(((ExportedVariableDeclaration)obj), tVariable, false, G, cache);
    }
    if (!_matched) {
      if (obj instanceof N4FieldDeclaration) {
        _matched=true;
        final TField tField = ((N4FieldDeclaration)obj).getDefinedField();
        this.<N4FieldDeclaration>setTypeRef(((N4FieldDeclaration)obj), tField, true, G, cache);
      }
    }
    if (!_matched) {
      if (obj instanceof FormalParameter) {
        _matched=true;
        final EObject parent = ((FormalParameter)obj).eContainer();
        boolean _matched_1 = false;
        if (parent instanceof FunctionExpression) {
          _matched_1=true;
        }
        if (!_matched_1) {
          if (parent instanceof PropertyMethodDeclaration) {
            _matched_1=true;
          }
        }
        if (!_matched_1) {
          if (parent instanceof FunctionDefinition) {
            _matched_1=true;
            final TFormalParameter tFPar = ((FormalParameter)obj).getDefinedTypeElement();
            TypeRef _typeRef = null;
            if (tFPar!=null) {
              _typeRef=tFPar.getTypeRef();
            }
            if ((_typeRef instanceof DeferredTypeRef)) {
              this.<FormalParameter>setTypeRef(((FormalParameter)obj), tFPar, true, G, cache);
            }
          }
        }
        if (!_matched_1) {
          if (parent instanceof SetterDeclaration) {
            _matched_1=true;
          }
        }
        if (!_matched_1) {
          throw new IllegalArgumentException("Unsupported parent type of FormalParameter");
        }
      }
    }
  }
  
  private <T extends TypableElement & TypedElement> void setTypeRef(final T typedElem, final TTypedElement tte, final boolean useContext, final RuleEnvironment G, final ASTMetaInfoCache cache) {
    TypeRef _declaredTypeRef = typedElem.getDeclaredTypeRef();
    boolean _tripleEquals = (_declaredTypeRef == null);
    if (_tripleEquals) {
      if ((tte != null)) {
        String _name = typedElem.getClass().getName();
        String _plus = ("return type of " + _name);
        String _plus_1 = (_plus + " in TModule should be a DeferredTypeRef");
        TypeRef _typeRef = tte.getTypeRef();
        AbstractProcessor.assertTrueIfRigid(cache, _plus_1, 
          (_typeRef instanceof DeferredTypeRef));
        RuleEnvironment G2 = G;
        if (useContext) {
          ParameterizedTypeRef context = null;
          EObject _eContainer = tte.eContainer();
          if ((_eContainer instanceof ContainerType<?>)) {
            EObject _eContainer_1 = tte.eContainer();
            context = TypeUtils.createTypeRef(((ContainerType<?>) _eContainer_1));
          }
          G2 = this.ts.createRuleEnvironmentForContext(context, RuleEnvironmentExtensions.getContextResource(G));
        }
        TypeArgument fieldTypeRef = this.askXsemanticsForType(G2, null, typedElem).getValue();
        if (useContext) {
          fieldTypeRef = this.ts.substTypeVariables(G2, fieldTypeRef).getValue();
        }
        final TypeRef fieldTypeRefSane = this.tsh.sanitizeTypeOfVariableFieldProperty(G, fieldTypeRef);
        final Procedure0 _function = () -> {
          tte.setTypeRef(TypeUtils.<TypeRef>copy(fieldTypeRefSane));
        };
        EcoreUtilN4.doWithDeliver(false, _function, tte);
      }
    }
  }
}
