/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.postprocessing;

import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.function.BooleanSupplier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FieldAccessor;
import org.eclipse.n4js.n4JS.N4ClassExpression;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4JSASTUtils;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.YieldExpression;
import org.eclipse.n4js.postprocessing.ASTMetaInfoCache;
import org.eclipse.n4js.postprocessing.ASTProcessor;
import org.eclipse.n4js.postprocessing.AbstractProcessor;
import org.eclipse.n4js.postprocessing.DestructureProcessor;
import org.eclipse.n4js.postprocessing.PolyProcessor;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.OptionalFieldStrategy;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnknownTypeRef;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TypableElement;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.CustomInternalTypeSystem;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xsemantics.runtime.Result;
import org.eclipse.xsemantics.runtime.RuleApplicationTrace;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xsemantics.runtime.RuleFailedException;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;

/**
 * Processor for handling type inference during post-processing of an N4JS resource. Roughly corresponds to
 * 'type' judgment in Xsemantics, but handles also more complex cases, e.g. poly expressions.
 * <p>
 * Invoked from {@link ASTProcessor} and delegates to {@link PolyProcessor}s.
 */
@Singleton
@SuppressWarnings("all")
public class TypeProcessor extends AbstractProcessor {
  @Inject
  private ASTProcessor astProcessor;
  
  @Inject
  private PolyProcessor polyProcessor;
  
  @Inject
  private DestructureProcessor destructureProcessor;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private OperationCanceledManager operationCanceledManager;
  
  /**
   * If the given AST node is typable this method will infer its type and store the result in the given cache.
   * <p>
   * This method mainly checks if the given node is typable. Main processing is done in
   * {@link #typeNode2(RuleEnvironment, TypableElement, ASTMetaInfoCache, int) typeNode2()}.
   */
  public void typeNode(final RuleEnvironment G, final EObject node, final ASTMetaInfoCache cache, final int indentLevel) {
    boolean _isTypableNode = N4JSLanguageUtils.isTypableNode(node);
    if (_isTypableNode) {
      final TypableElement nodeCasted = ((TypableElement) node);
      if ((DestructureUtils.isArrayOrObjectLiteralUsedAsDestructuringPattern(node) && this.polyProcessor.isEntryPoint(nodeCasted))) {
        AbstractProcessor.log(indentLevel, "ignored (array or object literal being used as a destructuring pattern)");
        this.destructureProcessor.typeDestructuringPattern(G, node, cache, indentLevel);
      } else {
        this.typeNode2(G, nodeCasted, cache, indentLevel);
      }
    } else {
      EClass _eClass = null;
      if (node!=null) {
        _eClass=node.eClass();
      }
      String _name = null;
      if (_eClass!=null) {
        _name=_eClass.getName();
      }
      String _plus = ("ignored (not a typable node: " + _name);
      String _plus_1 = (_plus + ")");
      AbstractProcessor.log(indentLevel, _plus_1);
    }
  }
  
  /**
   * Infers type of given AST node and stores the result in the given cache.
   * <p>
   * More precisely:
   * <ol>
   * <li>if given node is part of a poly expression:
   *     <ol>
   *     <li>if given node is the root of a tree of nested poly expressions (including the case that node is a poly
   *         expression without any nested poly expressions):<br>
   *         --> inference of entire tree of nested poly expressions AND storage of all results in cache is delegated
   *             to class {@link PolyProcessor}.
   *     <li>otherwise:<br>
   *         --> ignore this node ({@code PolyProcessor} will deal with it when processing the parent poly expression)
   *     </ol>
   * <li>otherwise (standard case):<br>
   *     --> infer type of node by asking Xsemantics + store the result in the given cache.
   * </ol>
   */
  private void typeNode2(final RuleEnvironment G, final TypableElement node, final ASTMetaInfoCache cache, final int indentLevel) {
    try {
      boolean _isResponsibleFor = this.polyProcessor.isResponsibleFor(node);
      if (_isResponsibleFor) {
        boolean _isEntryPoint = this.polyProcessor.isEntryPoint(node);
        if (_isEntryPoint) {
          AbstractProcessor.log(indentLevel, "asking PolyComputer ...");
          this.polyProcessor.inferType(G, ((Expression) node), cache);
          final BooleanSupplier _function = () -> {
            final EObject typeModelElem = N4JSLanguageUtils.getDefinedTypeModelElement(node);
            return ((typeModelElem == null) || IteratorExtensions.isEmpty(Iterators.<DeferredTypeRef>filter(typeModelElem.eAllContents(), DeferredTypeRef.class)));
          };
          AbstractProcessor.assertTrueIfRigid(cache, "poly computer did not replace DeferredTypeRef", _function);
        } else {
          AbstractProcessor.log(indentLevel, 
            "deferred (nested in poly expression --> will be inferred during inference of outer poly expression)");
          return;
        }
      } else {
        AbstractProcessor.log(indentLevel, "asking Xsemantics ...");
        final Result<TypeRef> result = this.askXsemanticsForType(G, null, node);
        final Result<TypeRef> resultAdjusted = this.<TypeRef>adjustResultForLocationInAST(G, result, N4JSASTUtils.skipParenExpressionDownward(node));
        this.checkCanceled(G);
        cache.storeType(node, resultAdjusted);
      }
    } catch (final Throwable _t) {
      if (_t instanceof RuleFailedException) {
        final RuleFailedException e = (RuleFailedException)_t;
        Result<TypeRef> _result = new Result<TypeRef>(e);
        cache.storeType(node, _result);
      } else if (_t instanceof Throwable) {
        final Throwable th = (Throwable)_t;
        this.operationCanceledManager.propagateIfCancelException(th);
        th.printStackTrace();
        String _message = th.getMessage();
        String _plus = ("error while asking Xsemantics: " + _message);
        RuleFailedException _ruleFailedException = new RuleFailedException(_plus, "YYY", th);
        Result<TypeRef> _result_1 = new Result<TypeRef>(_ruleFailedException);
        cache.storeType(node, _result_1);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    AbstractProcessor.log(indentLevel, cache.getTypeFailSafe(node));
  }
  
  /**
   * Make sure that the value of the two location-dependent special properties <code>typeOfObjectLiteral</code> and
   * <code>typeOfNewExpressionOrFinalNominal</code> in {@link ParameterizedTypeRef} correctly reflect the current
   * location in the AST, i.e. the the location of the given <code>astNode</code>, no matter where the type reference
   * in the given <code>result</code> stems from.
   * <p>
   * For more details see {@link TypeRef#isTypeOfObjectLiteral()}.
   */
  private <T extends TypeRef> Result<T> adjustResultForLocationInAST(final RuleEnvironment G, final Result<T> result, final TypableElement astNode) {
    boolean _failed = result.failed();
    boolean _not = (!_failed);
    if (_not) {
      final T typeRef = result.getValue();
      if ((typeRef instanceof ParameterizedTypeRef)) {
        final OptionalFieldStrategy optionalFieldStrategy = N4JSLanguageUtils.calculateOptionalFieldStrategy(astNode, typeRef);
        OptionalFieldStrategy _aSTNodeOptionalFieldStrategy = typeRef.getASTNodeOptionalFieldStrategy();
        boolean _tripleNotEquals = (_aSTNodeOptionalFieldStrategy != optionalFieldStrategy);
        if (_tripleNotEquals) {
          final T typeRefCpy = ((T)TypeUtils.<T>copy(((T)typeRef)));
          ((ParameterizedTypeRef)typeRefCpy).setASTNodeOptionalFieldStrategy(optionalFieldStrategy);
          return new Result<T>(((T)typeRefCpy));
        }
      }
    }
    return result;
  }
  
  /**
   * This is the single, central method for obtaining the type of a typable element (AST node or TModule element).
   * <b>It should never be invoked directly by client code!</b> Instead, client code should always call
   * {@link N4JSTypeSystem#type(RuleEnvironment,TypableElement) N4JSTypeSystem#type()} or, when inside Xsemantics,
   * use the special syntax for invoking the 'type' judgment: <code>G |- someExpression : var TypeRef result</code>.
   * <p>
   * The behavior of this method depends on the state the containing {@link N4JSResource} is in:
   * <ul>
   * <li>before post-processing has started:<br>
   *     -> simply initiate post-processing; once it's finished, return type from AST meta-info cache.
   * <li>during post-processing:
   *     <ul>
   *     <li>in case of a backward reference:<br>
   *         -> simply return type from AST meta-info cache.
   *     <li>in case of a forward reference:<br>
   *         -> trigger forward-processing of the identifiable subtree below the given typable element, see
   *         {@link #getTypeOfForwardReference(RuleEnvironment,TypableElement,ASTMetaInfoCache) #getTypeOfForwardReference()},
   *         which delegates to {@link ASTProcessor#processSubtree_forwardReference(
   *         RuleEnvironment,TypableElement,ASTMetaInfoCache) ASTProcessor#processSubtree_forwardReference()}.
   *     </ul>
   * <li>after post-processing has completed:<br>
   *     -> simply return type from AST meta-info cache.
   * </ul>
   * This overview is simplified, check the code for precise rules!
   * <p>
   * Two methods delegate here (no one else should call this method):
   * <ol>
   * <li>{@link N4JSTypeSystem#type(RuleEnvironment,TypableElement)}
   * <li>{@link CustomInternalTypeSystem#typeInternal(RuleEnvironment,
   *     RuleApplicationTrace,TypableElement)}
   * </ol>
   */
  public Result<TypeRef> getType(final RuleEnvironment G, final RuleApplicationTrace trace, final TypableElement objRaw) {
    if ((objRaw == null)) {
      UnknownTypeRef _createUnknownTypeRef = TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
      return new Result<TypeRef>(_createUnknownTypeRef);
    }
    TypableElement _xifexpression = null;
    boolean _eIsProxy = objRaw.eIsProxy();
    if (_eIsProxy) {
      TypableElement _xblockexpression = null;
      {
        final ResourceSet resSet = RuleEnvironmentExtensions.getContextResource(G).getResourceSet();
        EObject _resolve = EcoreUtil.resolve(objRaw, resSet);
        _xblockexpression = ((TypableElement) _resolve);
      }
      _xifexpression = _xblockexpression;
    } else {
      _xifexpression = objRaw;
    }
    TypableElement obj = _xifexpression;
    final Resource res = obj.eResource();
    if ((res instanceof N4JSResource)) {
      if ((((N4JSResource)res).isFullyProcessed() && ((N4JSResource)res).getScript().eIsProxy())) {
        boolean _isTypeModelElement = N4JSLanguageUtils.isTypeModelElement(obj);
        boolean _not = (!_isTypeModelElement);
        if (_not) {
          throw new IllegalStateException(("not a type model element: " + obj));
        }
        return this.askXsemanticsForType(G, trace, obj);
      }
      ((N4JSResource)res).performPostProcessing(RuleEnvironmentExtensions.getCancelIndicator(G));
      if ((((N4JSResource)res).isPostProcessing() && N4JSLanguageUtils.isTypeModelElement(obj))) {
        EObject _xifexpression_1 = null;
        if ((obj instanceof SyntaxRelatedTElement)) {
          _xifexpression_1 = ((SyntaxRelatedTElement)obj).getAstElement();
        }
        final EObject astNodeToProcess = _xifexpression_1;
        if ((astNodeToProcess instanceof TypableElement)) {
          obj = ((TypableElement)astNodeToProcess);
        }
      }
      return this.getTypeInN4JSResource(G, trace, ((N4JSResource)res), obj);
    } else {
      return this.askXsemanticsForType(G, trace, obj);
    }
  }
  
  /**
   * See {@link TypeProcessor#getType(RuleEnvironment,RuleApplicationTrace,TypableElement)}.
   */
  private Result<TypeRef> getTypeInN4JSResource(final RuleEnvironment G, final RuleApplicationTrace trace, final N4JSResource res, final TypableElement obj) {
    boolean _isTypeModelElement = N4JSLanguageUtils.isTypeModelElement(obj);
    if (_isTypeModelElement) {
      return this.askXsemanticsForType(G, trace, obj);
    } else {
      if ((N4JSLanguageUtils.isASTNode(obj) && N4JSLanguageUtils.isTypableNode(obj))) {
        final ASTMetaInfoCache cache = res.getASTMetaInfoCacheVerifyContext();
        if (((!res.isPostProcessing()) && (!res.isFullyProcessed()))) {
          URI _uRI = res.getURI();
          String _plus = ("post-processing neither in progress nor completed after calling #performPostProcessing() in resource: " + _uRI);
          throw new IllegalStateException(_plus);
        } else {
          if (((!cache.isPostProcessing()) && (!cache.isFullyProcessed()))) {
            final IllegalStateException e = new IllegalStateException("post-processing flags out of sync between resource and cache (hint: this is often caused by an accidental cache clear!!)");
            e.printStackTrace();
            throw e;
          } else {
            boolean _isPostProcessing = cache.isPostProcessing();
            if (_isPostProcessing) {
              final Result<TypeRef> resultFromCache = cache.getTypeFailSafe(obj);
              if ((resultFromCache == null)) {
                AbstractProcessor.log(0, ("***** forward reference to: " + obj));
                return this.getTypeOfForwardReference(G, obj, cache);
              } else {
                return resultFromCache;
              }
            } else {
              boolean _isFullyProcessed = cache.isFullyProcessed();
              if (_isFullyProcessed) {
                return cache.getType(obj);
              }
            }
          }
        }
      } else {
        CustomInternalTypeSystem.RuleFailedExceptionWithoutStacktrace _ruleFailedExceptionWithoutStacktrace = new CustomInternalTypeSystem.RuleFailedExceptionWithoutStacktrace(("cannot type object: " + obj));
        return new Result<TypeRef>(_ruleFailedExceptionWithoutStacktrace);
      }
    }
    return null;
  }
  
  /**
   * See {@link TypeProcessor#getType(RuleEnvironment,RuleApplicationTrace,TypableElement)}.
   */
  private Result<TypeRef> getTypeOfForwardReference(final RuleEnvironment G, final TypableElement node, final ASTMetaInfoCache cache) {
    AbstractProcessor.assertTrueIfRigid(cache, "argument \'node\' must be an AST node", N4JSLanguageUtils.isASTNode(node));
    boolean _isForwardReferenceWhileTypingDestructuringPattern = this.destructureProcessor.isForwardReferenceWhileTypingDestructuringPattern(node);
    if (_isForwardReferenceWhileTypingDestructuringPattern) {
      return this.destructureProcessor.handleForwardReferenceWhileTypingDestructuringPattern(G, node, cache);
    }
    final boolean isLegal = this.astProcessor.processSubtree_forwardReference(G, node, cache);
    if (isLegal) {
      final boolean isCyclicForwardReference = cache.astNodesCurrentlyBeingTyped.contains(node);
      if (isCyclicForwardReference) {
        if ((((node instanceof VariableDeclaration) || (node instanceof N4FieldDeclaration)) || 
          (node instanceof PropertyNameValuePair))) {
          final Expression expr = TypeProcessor.getExpressionOfVFP(node);
          if ((expr instanceof N4ClassExpression)) {
            return this.askXsemanticsForType(G, null, expr);
          }
          if ((expr instanceof NewExpression)) {
            final Expression callee = ((NewExpression)expr).getCallee();
            if ((callee instanceof N4ClassExpression)) {
              final TypeRef calleeType = this.askXsemanticsForType(G, null, callee).getValue();
              final Type calleeTypeStaticType = this.tsh.getStaticType(G, ((TypeTypeRef) calleeType));
              ParameterizedTypeRef _createTypeRef = TypeUtils.createTypeRef(calleeTypeStaticType);
              return new Result<TypeRef>(_createTypeRef);
            }
          }
          final TypeRef declTypeRef = TypeProcessor.getDeclaredTypeRefOfVFP(node);
          Result<TypeRef> _xifexpression = null;
          if ((declTypeRef != null)) {
            _xifexpression = new Result<TypeRef>(declTypeRef);
          } else {
            ParameterizedTypeRef _anyTypeRef = RuleEnvironmentExtensions.anyTypeRef(G);
            _xifexpression = new Result<TypeRef>(_anyTypeRef);
          }
          return _xifexpression;
        } else {
          if ((node instanceof FieldAccessor)) {
            final TypeRef declTypeRef_1 = ((FieldAccessor)node).getDeclaredTypeRef();
            Result<TypeRef> _xifexpression_1 = null;
            if ((declTypeRef_1 != null)) {
              _xifexpression_1 = new Result<TypeRef>(declTypeRef_1);
            } else {
              ParameterizedTypeRef _anyTypeRef_1 = RuleEnvironmentExtensions.anyTypeRef(G);
              _xifexpression_1 = new Result<TypeRef>(_anyTypeRef_1);
            }
            return _xifexpression_1;
          } else {
            if ((node instanceof TypeDefiningElement)) {
              TypeRef _wrapTypeInTypeRef = RuleEnvironmentExtensions.wrapTypeInTypeRef(G, ((TypeDefiningElement)node).getDefinedType());
              return new Result<TypeRef>(_wrapTypeInTypeRef);
            } else {
              if (((node instanceof Expression) && (node.eContainer() instanceof YieldExpression))) {
                return this.askXsemanticsForType(G, null, node);
              } else {
                final IllegalStateException e = new IllegalStateException(
                  "handling of a legal case of cyclic forward references missing in TypeProcessor");
                e.printStackTrace();
                return new Result(e);
              }
            }
          }
        }
      } else {
        boolean _isSemiCyclicForwardReferenceInForLoop = this.astProcessor.isSemiCyclicForwardReferenceInForLoop(node, cache);
        if (_isSemiCyclicForwardReferenceInForLoop) {
          final TypeRef declTypeRef_2 = ((VariableDeclaration) node).getDeclaredTypeRef();
          Result<TypeRef> _xifexpression_2 = null;
          if ((declTypeRef_2 != null)) {
            _xifexpression_2 = new Result<TypeRef>(declTypeRef_2);
          } else {
            ParameterizedTypeRef _anyTypeRef_2 = RuleEnvironmentExtensions.anyTypeRef(G);
            _xifexpression_2 = new Result<TypeRef>(_anyTypeRef_2);
          }
          return _xifexpression_2;
        } else {
          return cache.getType(node);
        }
      }
    } else {
      Resource _eResource = node.eResource();
      URI _uRI = null;
      if (_eResource!=null) {
        _uRI=_eResource.getURI();
      }
      final String msg = ((("*#*#*#*#*#* ILLEGAL FORWARD REFERENCE to " + node) + " in ") + _uRI);
      AbstractProcessor.logErr(msg);
      RuleFailedException _ruleFailedException = new RuleFailedException(msg);
      return new Result<TypeRef>(_ruleFailedException);
    }
  }
  
  private static Expression getExpressionOfVFP(final EObject vfp) {
    Expression _switchResult = null;
    boolean _matched = false;
    if (vfp instanceof VariableDeclaration) {
      _matched=true;
      _switchResult = ((VariableDeclaration)vfp).getExpression();
    }
    if (!_matched) {
      if (vfp instanceof N4FieldDeclaration) {
        _matched=true;
        _switchResult = ((N4FieldDeclaration)vfp).getExpression();
      }
    }
    if (!_matched) {
      if (vfp instanceof PropertyNameValuePair) {
        _matched=true;
        _switchResult = ((PropertyNameValuePair)vfp).getExpression();
      }
    }
    return _switchResult;
  }
  
  private static TypeRef getDeclaredTypeRefOfVFP(final EObject vfp) {
    TypeRef _switchResult = null;
    boolean _matched = false;
    if (vfp instanceof VariableDeclaration) {
      _matched=true;
      _switchResult = ((VariableDeclaration)vfp).getDeclaredTypeRef();
    }
    if (!_matched) {
      if (vfp instanceof N4FieldDeclaration) {
        _matched=true;
        _switchResult = ((N4FieldDeclaration)vfp).getDeclaredTypeRef();
      }
    }
    if (!_matched) {
      if (vfp instanceof PropertyNameValuePair) {
        _matched=true;
        _switchResult = ((PropertyNameValuePair)vfp).getDeclaredTypeRef();
      }
    }
    return _switchResult;
  }
}
