/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.scoping.diagnosing;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Arrays;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.ParenExpression;
import org.eclipse.n4js.n4JS.RelationalExpression;
import org.eclipse.n4js.n4JS.RelationalOperator;
import org.eclipse.n4js.n4JS.SuperLiteral;
import org.eclipse.n4js.resource.ErrorAwareLinkingService;
import org.eclipse.n4js.scoping.diagnosing.N4JSScopingConsumableMethodsDiagnosis;
import org.eclipse.n4js.scoping.diagnosing.N4JSScopingInstanceOfPrimitivTypeDiagnosis;
import org.eclipse.xtext.diagnostics.DiagnosticMessage;
import org.eclipse.xtext.naming.IQualifiedNameConverter;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.nodemodel.INode;

/**
 * This class provides enhanced error reporting in the case that
 * a reference can't be resolved by the N4JS scoping mechanisms.
 * 
 * This class can be extended by more special cases
 * to provide the user with more informative error messages.
 */
@SuppressWarnings("all")
public class N4JSScopingDiagnostician {
  @Inject
  private N4JSScopingConsumableMethodsDiagnosis consumableMethodsDiagnosis;
  
  @Inject
  private N4JSScopingInstanceOfPrimitivTypeDiagnosis instanceOfPrimitiveTypeDiagnosis;
  
  @Inject
  private ErrorAwareLinkingService linkingService;
  
  @Inject
  private IQualifiedNameConverter qualifiedNameConverter;
  
  /**
   * Returns a custom {@link DiagnosticMessage} for the given unresolvable reference.
   * May return {@code null} if no supported special case is applicable.
   * 
   * Note that this methods already assumes, that the given reference actually isn't resolvable.
   */
  public DiagnosticMessage getMessageFor(final EObject context, final EReference reference, final INode node) {
    final String crossRefAsString = this.linkingService.getCrossRefNodeAsString(context, reference, node);
    if (((null != crossRefAsString) && (!crossRefAsString.equals("")))) {
      final QualifiedName qualifiedName = this.qualifiedNameConverter.toQualifiedName(crossRefAsString);
      return this.diagnose(qualifiedName, context, reference);
    }
    return null;
  }
  
  private DiagnosticMessage _diagnose(final QualifiedName name, final ParameterizedPropertyAccessExpression context, final EReference reference) {
    Expression _target = context.getTarget();
    if ((_target instanceof SuperLiteral)) {
      return this.consumableMethodsDiagnosis.diagnose(name, context);
    }
    return null;
  }
  
  private DiagnosticMessage _diagnose(final QualifiedName name, final IdentifierRef context, final EReference reference) {
    EObject container = context.eContainer();
    EStructuralFeature containingFeature = context.eContainingFeature();
    while ((container instanceof ParenExpression)) {
      {
        containingFeature = ((ParenExpression)container).eContainmentFeature();
        container = ((ParenExpression)container).eContainer();
      }
    }
    if ((container instanceof RelationalExpression)) {
      if ((Objects.equal(((RelationalExpression)container).getOp(), RelationalOperator.INSTANCEOF) && 
        Objects.equal(containingFeature, N4JSPackage.Literals.RELATIONAL_EXPRESSION__RHS))) {
        return this.instanceOfPrimitiveTypeDiagnosis.diagnose(name, ((RelationalExpression)container));
      }
    }
    return null;
  }
  
  private DiagnosticMessage _diagnose(final QualifiedName name, final EObject context, final EReference reference) {
    return null;
  }
  
  private DiagnosticMessage diagnose(final QualifiedName name, final EObject context, final EReference reference) {
    if (context instanceof IdentifierRef) {
      return _diagnose(name, (IdentifierRef)context, reference);
    } else if (context instanceof ParameterizedPropertyAccessExpression) {
      return _diagnose(name, (ParameterizedPropertyAccessExpression)context, reference);
    } else if (context != null) {
      return _diagnose(name, context, reference);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(name, context, reference).toString());
    }
  }
}
