/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.scoping.imports;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Provider;
import com.google.inject.Singleton;
import java.util.HashMap;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ImportSpecifier;
import org.eclipse.n4js.n4JS.NamedImportSpecifier;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.Script;
import org.eclipse.n4js.n4idl.versioning.VersionHelper;
import org.eclipse.n4js.resource.N4JSEObjectDescription;
import org.eclipse.n4js.scoping.N4JSScopeProvider;
import org.eclipse.n4js.scoping.accessModifiers.AbstractTypeVisibilityChecker;
import org.eclipse.n4js.scoping.accessModifiers.InvisibleTypeOrVariableDescription;
import org.eclipse.n4js.scoping.accessModifiers.TypeVisibilityChecker;
import org.eclipse.n4js.scoping.accessModifiers.VariableVisibilityChecker;
import org.eclipse.n4js.scoping.builtin.GlobalObjectScope;
import org.eclipse.n4js.scoping.builtin.NoPrimitiveTypesScope;
import org.eclipse.n4js.scoping.imports.AmbiguousImportDescription;
import org.eclipse.n4js.scoping.imports.IEODesc2ISpec;
import org.eclipse.n4js.scoping.imports.ImportedElementsMap;
import org.eclipse.n4js.scoping.imports.OriginAwareScope;
import org.eclipse.n4js.scoping.imports.PlainAccessOfAliasedImportDescription;
import org.eclipse.n4js.scoping.imports.PlainAccessOfNamespacedImportDescription;
import org.eclipse.n4js.scoping.members.MemberScope;
import org.eclipse.n4js.scoping.utils.LocallyKnownTypesScopingHelper;
import org.eclipse.n4js.scoping.utils.MergedScope;
import org.eclipse.n4js.scoping.utils.ScopesHelper;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.Versionable;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.ModuleNamespaceVirtualType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TExportableElement;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.versions.VersionableUtils;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.naming.IQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.impl.AliasedEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.impl.SimpleScope;
import org.eclipse.xtext.util.IResourceScopeCache;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Helper for {@link N4JSScopeProvider N4JSScopeProvider}
 * {@link N4JSScopeProvider#scope_IdentifierRef_id(org.eclipse.n4js.n4JS.VariableEnvironmentElement) .scope_IdentifierRef_id()},
 * also used by helper {@link LocallyKnownTypesScopingHelper LocallyKnownTypesScopingHelper}.
 */
@Singleton
@SuppressWarnings("all")
public class ImportedElementsScopingHelper {
  @Inject
  private IResourceScopeCache cache;
  
  @Inject
  private TypeVisibilityChecker typeVisibilityChecker;
  
  @Inject
  private IQualifiedNameProvider qualifiedNameProvider;
  
  @Inject
  private VariableVisibilityChecker variableVisibilityChecker;
  
  @Inject
  private ImportedElementsMap.Provider elementsMapProvider;
  
  @Inject
  private MemberScope.MemberScopeFactory memberScopeFactory;
  
  @Inject
  private ScopesHelper scopesHelper;
  
  @Inject
  private JavaScriptVariantHelper variantHelper;
  
  @Inject
  private VersionHelper versionHelper;
  
  public IScope getImportedIdentifiables(final IScope parentScope, final Script script) {
    Pair<Script, String> _mappedTo = Pair.<Script, String>of(script, "importedIdentifiables");
    final Provider<IScope> _function = () -> {
      BuiltInTypeScope _get = BuiltInTypeScope.get(script.eResource().getResourceSet());
      final NoPrimitiveTypesScope builtInTypes = new NoPrimitiveTypesScope(_get);
      MergedScope _mergedScope = new MergedScope(parentScope, builtInTypes);
      final IScope globalObjectScope = this.getGlobalObjectProperties(_mergedScope, script);
      final IScope result = this.findImportedElements(script, globalObjectScope, true);
      return result;
    };
    final IScope scriptScope = this.cache.<IScope>get(_mappedTo, script.eResource(), _function);
    return scriptScope;
  }
  
  public IScope getImportedTypes(final IScope parentScope, final Script script) {
    Pair<Script, String> _mappedTo = Pair.<Script, String>of(script, "importedTypes");
    final Provider<IScope> _function = () -> {
      final IScope result = this.findImportedElements(script, parentScope, false);
      return result;
    };
    final IScope scriptScope = this.cache.<IScope>get(_mappedTo, script.eResource(), _function);
    return scriptScope;
  }
  
  /**
   * Creates a new QualifiedNamed for the given named import specifier.
   * 
   * Determines the local name of the imported element based on the given import specifier.
   */
  private QualifiedName createQualifiedNameForAlias(final NamedImportSpecifier specifier, final TExportableElement importedElement) {
    String _xifexpression = null;
    boolean _isDefaultImport = specifier.isDefaultImport();
    if (_isDefaultImport) {
      _xifexpression = specifier.getImportedElementAsText();
    } else {
      String _elvis = null;
      String _elvis_1 = null;
      String _alias = specifier.getAlias();
      if (_alias != null) {
        _elvis_1 = _alias;
      } else {
        String _exportedName = importedElement.getExportedName();
        _elvis_1 = _exportedName;
      }
      if (_elvis_1 != null) {
        _elvis = _elvis_1;
      } else {
        String _name = importedElement.getName();
        _elvis = _name;
      }
      _xifexpression = _elvis;
    }
    final String importedName = _xifexpression;
    return QualifiedName.create(importedName);
  }
  
  private QualifiedName createImportedQualifiedTypeName(final Type type) {
    return QualifiedName.create(this.getImportedName(type));
  }
  
  private String getImportedName(final Type type) {
    String _elvis = null;
    String _exportedName = type.getExportedName();
    if (_exportedName != null) {
      _elvis = _exportedName;
    } else {
      String _name = type.getName();
      _elvis = _name;
    }
    return _elvis;
  }
  
  private QualifiedName createImportedQualifiedTypeName(final String namespace, final Type type) {
    return QualifiedName.create(namespace, this.getImportedName(type));
  }
  
  private IScope findImportedElements(final Script script, final IScope parentScope, final boolean includeVariables) {
    final Resource contextResource = script.eResource();
    final Iterable<ImportDeclaration> imports = Iterables.<ImportDeclaration>filter(script.getScriptElements(), ImportDeclaration.class);
    boolean _isEmpty = IterableExtensions.isEmpty(imports);
    if (_isEmpty) {
      return parentScope;
    }
    final ImportedElementsMap invalidImports = this.elementsMapProvider.get(script);
    final ImportedElementsMap validImports = this.elementsMapProvider.get(script);
    final IEODesc2ISpec originatorMap = new IEODesc2ISpec();
    for (final ImportDeclaration imp : imports) {
      TModule _module = null;
      if (imp!=null) {
        _module=imp.getModule();
      }
      boolean _tripleNotEquals = (_module != null);
      if (_tripleNotEquals) {
        EList<ImportSpecifier> _importSpecifiers = imp.getImportSpecifiers();
        for (final ImportSpecifier specifier : _importSpecifiers) {
          boolean _matched = false;
          if (specifier instanceof NamedImportSpecifier) {
            _matched=true;
            this.processNamedImportSpecifier(((NamedImportSpecifier)specifier), imp, contextResource, originatorMap, validImports, invalidImports, includeVariables);
          }
          if (!_matched) {
            if (specifier instanceof NamespaceImportSpecifier) {
              _matched=true;
              this.processNamespaceSpecifier(((NamespaceImportSpecifier)specifier), imp, script, contextResource, originatorMap, validImports, invalidImports, includeVariables);
            }
          }
        }
      }
    }
    Iterable<IEObjectDescription> _values = invalidImports.values();
    final SimpleScope invalidLocalScope = new SimpleScope(_values);
    final MergedScope localBaseScope = new MergedScope(invalidLocalScope, parentScope);
    final IScope localValidScope = this.scopesHelper.mapBasedScopeFor(script, localBaseScope, validImports.values());
    return new OriginAwareScope(localValidScope, originatorMap);
  }
  
  protected void processNamedImportSpecifier(final NamedImportSpecifier specifier, final ImportDeclaration imp, final Resource contextResource, final IEODesc2ISpec originatorMap, final ImportedElementsMap validImports, final ImportedElementsMap invalidImports, final boolean importVariables) {
    final TExportableElement element = specifier.getImportedElement();
    if (((element != null) && (!element.eIsProxy()))) {
      if (((!importVariables) && this.isVariableFrom(element, imp))) {
        return;
      }
      final QualifiedName importedQName = this.createQualifiedNameForAlias(specifier, element);
      final AbstractTypeVisibilityChecker.TypeVisibility typeVisibility = this.isVisible(contextResource, element);
      if (typeVisibility.visibility) {
        this.addNamedImports(specifier, element, importedQName, originatorMap, validImports);
        final QualifiedName originalName = QualifiedName.create(element.getName());
        if (((specifier.getAlias() != null) && (!invalidImports.containsElement(originalName)))) {
          this.handleAliasedAccess(element, originalName, importedQName.toString(), invalidImports, originatorMap, specifier);
        }
      } else {
        this.handleInvisible(element, invalidImports, importedQName, typeVisibility.accessModifierSuggestion, originatorMap, specifier);
      }
    }
  }
  
  private void addNamedImports(final NamedImportSpecifier specifier, final TExportableElement element, final QualifiedName importedName, final IEODesc2ISpec originatorMap, final ImportedElementsMap validImports) {
    if ((this.variantHelper.allowVersionedTypes(specifier) && VersionableUtils.isTVersionable(element))) {
      final Consumer<Type> _function = (Type type) -> {
        final IEObjectDescription description = this.putOrError(validImports, type, importedName, 
          IssueCodes.IMP_AMBIGUOUS);
        this.putWithOrigin(originatorMap, description, specifier);
      };
      this.versionHelper.<Type>findTypeVersions(((Type) element)).forEach(_function);
    } else {
      final IEObjectDescription ieod = this.putOrError(validImports, element, importedName, IssueCodes.IMP_AMBIGUOUS);
      this.putWithOrigin(originatorMap, ieod, specifier);
    }
  }
  
  private void processNamespaceSpecifier(final NamespaceImportSpecifier specifier, final ImportDeclaration imp, final Script script, final Resource contextResource, final IEODesc2ISpec originatorMap, final ImportedElementsMap validImports, final ImportedElementsMap invalidImports, final boolean importVariables) {
    String _alias = specifier.getAlias();
    boolean _tripleEquals = (_alias == null);
    if (_tripleEquals) {
      return;
    }
    final String namespaceName = specifier.getAlias();
    final QualifiedName namespaceQName = QualifiedName.create(namespaceName);
    EList<Type> _internalTypes = script.getModule().getInternalTypes();
    EList<Type> _exposedInternalTypes = script.getModule().getExposedInternalTypes();
    final Function1<Type, Boolean> _function = (Type interType) -> {
      return Boolean.valueOf(((interType instanceof ModuleNamespaceVirtualType) && 
        (((ModuleNamespaceVirtualType) interType).getModule() == imp.getModule())));
    };
    final Type namespaceType = IterableExtensions.<Type>findFirst(Iterables.<Type>concat(_internalTypes, _exposedInternalTypes), _function);
    final IEObjectDescription ieodx = this.putOrError(validImports, namespaceType, namespaceQName, IssueCodes.IMP_AMBIGUOUS);
    this.putWithOrigin(originatorMap, ieodx, specifier);
    if (importVariables) {
      EList<TVariable> _variables = imp.getModule().getVariables();
      for (final TVariable importedVar : _variables) {
        {
          final AbstractTypeVisibilityChecker.TypeVisibility varVisibility = this.variableVisibilityChecker.isVisible(contextResource, importedVar);
          final String varName = importedVar.getExportedName();
          final QualifiedName qn = QualifiedName.create(namespaceName, varName);
          if (varVisibility.visibility) {
            final QualifiedName originalName = QualifiedName.create(varName);
            boolean _containsElement = invalidImports.containsElement(originalName);
            boolean _not = (!_containsElement);
            if (_not) {
              this.handleNamespacedAccess(importedVar, originalName, qn, invalidImports, originatorMap, specifier);
            }
          }
        }
      }
    }
    EList<Type> _topLevelTypes = imp.getModule().getTopLevelTypes();
    for (final Type importedType : _topLevelTypes) {
      {
        final AbstractTypeVisibilityChecker.TypeVisibility typeVisibility = this.typeVisibilityChecker.isVisible(contextResource, importedType);
        final QualifiedName qn = this.createImportedQualifiedTypeName(namespaceName, importedType);
        if (typeVisibility.visibility) {
          final QualifiedName originalName = this.createImportedQualifiedTypeName(importedType);
          boolean _containsElement = invalidImports.containsElement(originalName);
          boolean _not = (!_containsElement);
          if (_not) {
            this.handleNamespacedAccess(importedType, originalName, qn, invalidImports, originatorMap, specifier);
          }
        }
      }
    }
  }
  
  private void handleAliasedAccess(final IdentifiableElement element, final QualifiedName originalName, final String importedName, final ImportedElementsMap invalidImports, final IEODesc2ISpec originatorMap, final ImportSpecifier specifier) {
    IEObjectDescription _create = N4JSEObjectDescription.create(originalName, element);
    final PlainAccessOfAliasedImportDescription invalidAccess = new PlainAccessOfAliasedImportDescription(_create, importedName);
    invalidImports.put(originalName, invalidAccess);
  }
  
  private void handleNamespacedAccess(final IdentifiableElement importedType, final QualifiedName originalName, final QualifiedName qn, final ImportedElementsMap invalidImports, final IEODesc2ISpec originatorMap, final ImportSpecifier specifier) {
    IEObjectDescription _create = N4JSEObjectDescription.create(originalName, importedType);
    String _string = qn.toString();
    final PlainAccessOfNamespacedImportDescription invalidAccess = new PlainAccessOfNamespacedImportDescription(_create, _string);
    invalidImports.put(originalName, invalidAccess);
  }
  
  private boolean handleInvisible(final IdentifiableElement importedElement, final ImportedElementsMap invalidImports, final QualifiedName qn, final String visibilitySuggestion, final IEODesc2ISpec originatorMap, final ImportSpecifier specifier) {
    boolean _xblockexpression = false;
    {
      final IEObjectDescription invalidAccess = this.putOrError(invalidImports, importedElement, qn, null);
      this.addAccessModifierSuggestion(invalidAccess, visibilitySuggestion);
      _xblockexpression = this.putWithOrigin(originatorMap, invalidAccess, specifier);
    }
    return _xblockexpression;
  }
  
  /**
   * Add the description to the orginatorMap and include trace to the specifier in special Error-Aware IEObjectDesciptoins
   * like AmbigousImportDescriptions.
   */
  private boolean putWithOrigin(final HashMap<IEObjectDescription, ImportSpecifier> originiatorMap, final IEObjectDescription description, final ImportSpecifier specifier) {
    boolean _xblockexpression = false;
    {
      originiatorMap.put(description, specifier);
      boolean _switchResult = false;
      boolean _matched = false;
      if (description instanceof AmbiguousImportDescription) {
        _matched=true;
        boolean _xblockexpression_1 = false;
        {
          ((AmbiguousImportDescription)description).getOriginatingImports().add(specifier);
          final ImportSpecifier firstPlaceSpec = originiatorMap.get(((AmbiguousImportDescription)description).delegate());
          boolean _xifexpression = false;
          if (((firstPlaceSpec != null) && (!((AmbiguousImportDescription)description).getOriginatingImports().contains(firstPlaceSpec)))) {
            _xifexpression = ((AmbiguousImportDescription)description).getOriginatingImports().add(firstPlaceSpec);
          }
          _xblockexpression_1 = _xifexpression;
        }
        _switchResult = _xblockexpression_1;
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  private boolean isVariableFrom(final IdentifiableElement element, final ImportDeclaration imp) {
    boolean res = false;
    boolean _and = false;
    TModule _module = null;
    if (imp!=null) {
      _module=imp.getModule();
    }
    boolean _tripleNotEquals = (_module != null);
    if (!_tripleNotEquals) {
      _and = false;
    } else {
      TModule _module_1 = null;
      if (imp!=null) {
        _module_1=imp.getModule();
      }
      boolean _contains = _module_1.getVariables().contains(element);
      _and = _contains;
    }
    if (_and) {
      res = true;
    }
    return res;
  }
  
  private AbstractTypeVisibilityChecker.TypeVisibility isVisible(final Resource contextResource, final IdentifiableElement element) {
    AbstractTypeVisibilityChecker.TypeVisibility _xifexpression = null;
    if ((element instanceof Type)) {
      _xifexpression = this.typeVisibilityChecker.isVisible(contextResource, ((Type)element));
    } else {
      AbstractTypeVisibilityChecker.TypeVisibility _xifexpression_1 = null;
      if ((element instanceof TVariable)) {
        _xifexpression_1 = this.variableVisibilityChecker.isVisible(contextResource, ((TVariable)element));
      } else {
        return new AbstractTypeVisibilityChecker.TypeVisibility(false);
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * Returns {@code true} if an import of the given {@link IEObjectDescription} should be
   * regarded as ambiguous with the given {@link IdentifiableElement}.
   */
  protected boolean isAmbiguous(final IEObjectDescription existing, final IdentifiableElement element) {
    if (((existing.getEObjectOrProxy() instanceof Versionable) && (element instanceof Versionable))) {
      EObject _eObjectOrProxy = existing.getEObjectOrProxy();
      int _version = ((Versionable) _eObjectOrProxy).getVersion();
      int _version_1 = ((Versionable) element).getVersion();
      return (_version == _version_1);
    } else {
      return true;
    }
  }
  
  private IEObjectDescription putOrError(final ImportedElementsMap result, final IdentifiableElement element, final QualifiedName importedName, final String issueCode) {
    IEObjectDescription ret = null;
    final Iterable<IEObjectDescription> existing = result.getElements(importedName);
    if (((!IterableExtensions.isEmpty(existing)) && (IterableExtensions.<IEObjectDescription>findFirst(existing, ((Function1<IEObjectDescription, Boolean>) (IEObjectDescription it) -> {
      return Boolean.valueOf(this.isAmbiguous(it, element));
    })) != null))) {
      if ((issueCode != null)) {
        boolean _matched = false;
        if (existing instanceof AmbiguousImportDescription) {
          _matched=true;
          List<IdentifiableElement> _elements = ((AmbiguousImportDescription)existing).getElements();
          _elements.add(element);
          ret = ((AmbiguousImportDescription)existing);
        }
        if (!_matched) {
          {
            IEObjectDescription _head = IterableExtensions.<IEObjectDescription>head(existing);
            final AmbiguousImportDescription error = new AmbiguousImportDescription(_head, issueCode, element);
            result.put(importedName, error);
            List<IdentifiableElement> _elements = error.getElements();
            _elements.add(element);
            ret = error;
          }
        }
      }
    } else {
      if ((issueCode == null)) {
        ret = this.createDescription(importedName, element);
        InvisibleTypeOrVariableDescription _invisibleTypeOrVariableDescription = new InvisibleTypeOrVariableDescription(ret);
        ret = _invisibleTypeOrVariableDescription;
        result.put(importedName, ret);
      } else {
        ret = this.createDescription(importedName, element);
        result.put(importedName, ret);
      }
    }
    return ret;
  }
  
  private IEObjectDescription createDescription(final QualifiedName name, final IdentifiableElement element) {
    String _lastSegment = name.getLastSegment();
    String _name = element.getName();
    boolean _notEquals = (!Objects.equal(_lastSegment, _name));
    if (_notEquals) {
      IEObjectDescription _create = N4JSEObjectDescription.create(this.qualifiedNameProvider.getFullyQualifiedName(element), element);
      return new AliasedEObjectDescription(name, _create);
    } else {
      return N4JSEObjectDescription.create(name, element);
    }
  }
  
  /**
   * global object scope indirectly cached, as this method is only called by getImportedIdentifiables (which is cached)
   */
  private IScope getGlobalObjectProperties(final IScope parent, final EObject context) {
    IScope _xblockexpression = null;
    {
      final TClass globalObject = GlobalObjectScope.get(context.eResource().getResourceSet()).getGlobalObject();
      _xblockexpression = this.memberScopeFactory.create(parent, globalObject, context, false, false);
    }
    return _xblockexpression;
  }
  
  private void addAccessModifierSuggestion(final IEObjectDescription description, final String suggestion) {
    if ((description instanceof InvisibleTypeOrVariableDescription)) {
      ((InvisibleTypeOrVariableDescription)description).setAccessModifierSuggestion(suggestion);
    }
  }
}
