/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.GetterDeclaration;
import org.eclipse.n4js.n4JS.YieldExpression;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.typesystem.TypeSystemHelperStrategy;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Contains helper methods used by the rules of the 'expectedTypeIn' judgment.
 * Main reason for factoring out this code is the same logic is used by several rules.
 */
@Singleton
@SuppressWarnings("all")
public class ExpectedTypeComputer extends TypeSystemHelperStrategy {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  /**
   * Returns the expected type of an expression which be used to compute the return value of its containing function.
   * In the standard case, the expression should be an expression in a return statement and the expected type will
   * be the return type of its containing function. However, in case of a single-expression arrow function, the
   * given expression need not be the child of a return statement.
   * <p>
   * This method will not check that the given expression is actually an expression that will, at runtime, provide
   * the return value of a function. This has to be ensured by the client code.
   * 
   * @return the expected type or <code>null</code> if there is no type expectation or some error occurred (e.g. broken AST).
   */
  public TypeRef getExpectedTypeOfReturnValueExpression(final RuleEnvironment G, final Expression returnValueExpr) {
    EObject _eContainer = null;
    if (returnValueExpr!=null) {
      _eContainer=returnValueExpr.eContainer();
    }
    final FunctionOrFieldAccessor fofa = EcoreUtil2.<FunctionOrFieldAccessor>getContainerOfType(_eContainer, FunctionOrFieldAccessor.class);
    final RuleEnvironment G2 = RuleEnvironmentExtensions.wrap(G);
    final TypeRef myThisTypeRef = this.ts.thisTypeRef(G, returnValueExpr).getValue();
    RuleEnvironmentExtensions.addThisType(G2, myThisTypeRef);
    return this.getExpectedTypeOfFunctionOrFieldAccessor(G2, fofa);
  }
  
  public TypeRef getExpectedTypeOfFunctionOrFieldAccessor(final RuleEnvironment G, final FunctionOrFieldAccessor fofa) {
    RuleEnvironment _xifexpression = null;
    if ((G == null)) {
      _xifexpression = RuleEnvironmentExtensions.newRuleEnvironment(fofa);
    } else {
      _xifexpression = G;
    }
    final RuleEnvironment G2 = _xifexpression;
    if ((fofa instanceof FunctionDefinition)) {
      boolean _isAsync = ((FunctionDefinition)fofa).isAsync();
      if (_isAsync) {
        return this.getExpectedTypeOfReturnValueExpressionForAsyncFunction(G2, ((FunctionDefinition)fofa));
      } else {
        boolean _isGenerator = ((FunctionDefinition)fofa).isGenerator();
        if (_isGenerator) {
          return this.getExpectedTypeOfReturnValueExpressionForGeneratorFunction(G2, ((FunctionDefinition)fofa));
        } else {
          final TypeRef fType = this.ts.type(G2, fofa).getValue();
          if ((fType instanceof FunctionTypeExprOrRef)) {
            return this.ts.substTypeVariablesInTypeRef(G2, ((FunctionTypeExprOrRef)fType).getReturnTypeRef());
          }
        }
      }
    } else {
      if ((fofa instanceof GetterDeclaration)) {
        TGetter _definedGetter = null;
        if (((GetterDeclaration)fofa)!=null) {
          _definedGetter=((GetterDeclaration)fofa).getDefinedGetter();
        }
        TypeRef _declaredTypeRef = null;
        if (_definedGetter!=null) {
          _declaredTypeRef=_definedGetter.getDeclaredTypeRef();
        }
        return _declaredTypeRef;
      }
    }
    return null;
  }
  
  private TypeRef getExpectedTypeOfReturnValueExpressionForAsyncFunction(final RuleEnvironment G, final FunctionDefinition funDef) {
    final Type tFun = funDef.getDefinedType();
    if ((tFun instanceof TFunction)) {
      final TypeRef actualReturnTypeRef = ((TFunction)tFun).getReturnTypeRef();
      boolean _isPromise = TypeUtils.isPromise(actualReturnTypeRef, RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope);
      if (_isPromise) {
        final TypeArgument firstTypeArg = IterableExtensions.<TypeArgument>head(actualReturnTypeRef.getTypeArgs());
        if ((firstTypeArg != null)) {
          return this.ts.upperBound(G, firstTypeArg).getValue();
        }
      }
    }
    TypeRef _returnTypeRef = funDef.getReturnTypeRef();
    boolean _tripleNotEquals = (_returnTypeRef != null);
    if (_tripleNotEquals) {
      return funDef.getReturnTypeRef();
    }
    return null;
  }
  
  private TypeRef getExpectedTypeOfReturnValueExpressionForGeneratorFunction(final RuleEnvironment G, final FunctionDefinition funDef) {
    final TFunction tFun = funDef.getDefinedFunction();
    if ((tFun != null)) {
      final TypeRef actualReturnTypeRef = tFun.getReturnTypeRef();
      boolean _isGenerator = TypeUtils.isGenerator(actualReturnTypeRef, RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope);
      if (_isGenerator) {
        return this.tsh.getGeneratorTReturn(G, actualReturnTypeRef);
      }
    }
    TypeRef _returnTypeRef = funDef.getReturnTypeRef();
    boolean _tripleNotEquals = (_returnTypeRef != null);
    if (_tripleNotEquals) {
      return funDef.getReturnTypeRef();
    }
    return null;
  }
  
  /**
   * Returns the expected type of the yield value. It is retrieved from the type TYield of the actual function return type
   * (with regard to {@code Generator<TYield,TReturn,TNext>}). In case the yield expression is recursive (features a star),
   * the expected type must conform to {@code Generator<? extends TYield,any,? super TNext>}.
   */
  public TypeRef getExpectedTypeOfYieldValueExpression(final RuleEnvironment G, final YieldExpression yieldExpr, final TypeRef exprTypeRef) {
    final Expression expression = yieldExpr.getExpression();
    EObject _eContainer = null;
    if (expression!=null) {
      _eContainer=expression.eContainer();
    }
    final FunctionDefinition funDef = EcoreUtil2.<FunctionDefinition>getContainerOfType(_eContainer, FunctionDefinition.class);
    final RuleEnvironment G2 = RuleEnvironmentExtensions.wrap(G);
    final TypeRef myThisTypeRef = this.ts.thisTypeRef(G, expression).getValue();
    RuleEnvironmentExtensions.addThisType(G2, myThisTypeRef);
    if (((funDef == null) || (!funDef.isGenerator()))) {
      return null;
    }
    final TFunction tFun = funDef.getDefinedFunction();
    if ((tFun != null)) {
      final TypeRef actualReturnTypeRef = tFun.getReturnTypeRef();
      final BuiltInTypeScope scope = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope;
      boolean _isGenerator = TypeUtils.isGenerator(actualReturnTypeRef, scope);
      if (_isGenerator) {
        final TypeRef yieldTypeRef = this.tsh.getGeneratorTYield(G, actualReturnTypeRef);
        final TypeRef yieldTypeRefCopy = TypeUtils.<TypeRef>copyWithProxies(yieldTypeRef);
        boolean _isMany = yieldExpr.isMany();
        if (_isMany) {
          final boolean isGenerator = TypeUtils.isGenerator(exprTypeRef, scope);
          if (isGenerator) {
            final TypeRef nextTypeRef = this.tsh.getGeneratorTNext(G, actualReturnTypeRef);
            final TypeRef nextTypeRefCopy = TypeUtils.<TypeRef>copyWithProxies(nextTypeRef);
            final Wildcard superNext = TypeUtils.createWildcardSuper(nextTypeRefCopy);
            final Wildcard extendsYield = TypeUtils.createWildcardExtends(yieldTypeRefCopy);
            final ParameterizedTypeRef tReturn = scope.getAnyTypeRef();
            final ParameterizedTypeRef recursiveGeneratorSuperType = TypeUtils.createGeneratorTypeRef(scope, extendsYield, tReturn, superNext);
            return recursiveGeneratorSuperType;
          } else {
            final ParameterizedTypeRef iterableTypeRef = RuleEnvironmentExtensions.iterableTypeRef(G, yieldTypeRefCopy);
            return iterableTypeRef;
          }
        } else {
          return yieldTypeRefCopy;
        }
      }
    }
    return null;
  }
}
