/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.postprocessing.ASTMetaInfoUtils;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.StructuralTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeVariableMapping;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.utils.TypeCompareHelper;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelperStrategy;
import org.eclipse.n4js.utils.RecursionGuard;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;

/**
 * Type System Helper Strategy for managing type variable mappings in RuleEnvironments of XSemantics.
 * Note that low-level method for adding type mappings to a rule environment are contained in
 * {@link RuleEnvironmentExtensions}.
 */
@SuppressWarnings("all")
public class GenericsComputer extends TypeSystemHelperStrategy {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  @Extension
  private TypeCompareHelper _typeCompareHelper;
  
  /**
   * Given a type reference to a generic type G where type variables are already bound, e.g.,
   * G&lt;A,B>, this method adds to the given rule environment the mappings for the type variables
   * of the declared type G using the type arguments of the passed type reference. Such mappings
   * will be used later to bind type variables.
   * <p>
   * For instance, given a type reference G&lt;A,B> to a class
   * <pre>
   * interface I&lt;V> {
   *     // ...
   * }
   * class G&lt;T,U> implements I&lt;C> {
   *     // ...
   * }
   * </pre>
   * this function will add the mappings T -> A, U -> B, V -> C.
   */
  public void addSubstitutions(final RuleEnvironment G, final TypeRef typeRef) {
    final Type declType = typeRef.getDeclaredType();
    if ((typeRef instanceof BoundThisTypeRef)) {
      this.addSubstitutions(G, ((BoundThisTypeRef)typeRef).getActualThisTypeRef());
    } else {
      if ((declType instanceof TypeVariable)) {
        final TypeRef currBound = ((TypeVariable)declType).getDeclaredUpperBound();
        if ((currBound != null)) {
          this.addSubstitutions(G, currBound);
        }
      } else {
        if ((declType instanceof TClassifier)) {
          ArrayList<TypeRef> _collectSuperTypeRefs = this.collectSuperTypeRefs(typeRef);
          for (final TypeRef currBaseType : _collectSuperTypeRefs) {
            this.primAddSubstitutions(G, currBaseType);
          }
        } else {
          this.primAddSubstitutions(G, typeRef);
        }
      }
    }
  }
  
  /**
   * Adds substitutions for an individual type reference (i.e. without taking
   * into account inheritance hierarchies).
   */
  private void primAddSubstitutions(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef instanceof ParameterizedTypeRef)) {
      boolean _isEmpty = ((ParameterizedTypeRef)typeRef).getTypeArgs().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        Type gen = ((ParameterizedTypeRef)typeRef).getDeclaredType();
        if ((gen instanceof ContainerType<?>)) {
          Iterator<TypeVariable> varIter = ((ContainerType<?>)gen).getTypeVars().iterator();
          EList<TypeArgument> _typeArgs = ((ParameterizedTypeRef)typeRef).getTypeArgs();
          for (final TypeArgument typeArg : _typeArgs) {
            boolean _hasNext = varIter.hasNext();
            if (_hasNext) {
              TypeVariable typeVar = varIter.next();
              this.addSubstitution(G, typeVar, typeArg);
            }
          }
        }
      }
      if ((typeRef instanceof StructuralTypeRef)) {
        this.restorePostponedSubstitutionsFrom(G, ((StructuralTypeRef)typeRef));
      }
    }
  }
  
  /**
   * Adds an individual substitution.
   * TODO proper handling of type variables with multiple solutions (union/intersection on TypeRef level, not on variable level! or: disallow entirely)
   */
  private boolean addSubstitution(final RuleEnvironment G, final TypeVariable typeVar, final TypeArgument typeArg) {
    boolean _xblockexpression = false;
    {
      Object actualTypeArg = typeArg;
      while (RuleEnvironmentExtensions.hasSubstitutionFor(G, actualTypeArg)) {
        {
          final TypeRef actualTypeArgCasted = ((TypeRef) actualTypeArg);
          final Object fromEnv = G.getEnvironment().get(actualTypeArgCasted.getDeclaredType());
          Object _xifexpression = null;
          if ((fromEnv instanceof TypeRef)) {
            _xifexpression = TypeUtils.mergeTypeModifiers(((TypeRef)fromEnv), actualTypeArgCasted);
          } else {
            _xifexpression = fromEnv;
          }
          actualTypeArg = _xifexpression;
        }
      }
      if ((actualTypeArg instanceof Wildcard)) {
        actualTypeArg = TypeUtils.captureWildcard(typeVar, ((TypeArgument)actualTypeArg));
      }
      Object currSubstitute = G.get(typeVar);
      if ((currSubstitute == typeVar)) {
        currSubstitute = null;
      }
      _xblockexpression = G.add(typeVar, this.mergeTypeArgs(currSubstitute, actualTypeArg));
    }
    return _xblockexpression;
  }
  
  private Object mergeTypeArgs(final Object... typeArgs) {
    Object _xblockexpression = null;
    {
      final ArrayList<Object> result = CollectionLiterals.<Object>newArrayList();
      for (final Object currTypeArg : typeArgs) {
        {
          Collection<?> _xifexpression = null;
          if ((currTypeArg instanceof Collection<?>)) {
            _xifexpression = ((Collection<?>)currTypeArg);
          } else {
            _xifexpression = Collections.<Object>unmodifiableList(CollectionLiterals.<Object>newArrayList(currTypeArg));
          }
          final Collection<?> l = _xifexpression;
          for (final Object currO : l) {
            if (((currO != null) && (!this.typeRefAwareContains(result, currO)))) {
              result.add(currO);
            }
          }
        }
      }
      Object _xifexpression = null;
      int _size = result.size();
      boolean _greaterEqualsThan = (_size >= 2);
      if (_greaterEqualsThan) {
        _xifexpression = result;
      } else {
        Object _xifexpression_1 = null;
        int _size_1 = result.size();
        boolean _tripleEquals = (_size_1 == 1);
        if (_tripleEquals) {
          _xifexpression_1 = result.get(0);
        } else {
          _xifexpression_1 = null;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private boolean typeRefAwareContains(final Collection<?> l, final Object o) {
    if ((o instanceof TypeRef)) {
      for (final Object currO : l) {
        if ((currO instanceof TypeRef)) {
          int _compare = this._typeCompareHelper.compare(((TypeArgument)currO), ((TypeArgument)o));
          boolean _tripleEquals = (_compare == 0);
          if (_tripleEquals) {
            return true;
          }
        }
      }
      return false;
    } else {
      return l.contains(o);
    }
  }
  
  private ArrayList<TypeRef> collectSuperTypeRefs(final TypeRef typeRef) {
    final ArrayList<TypeRef> result = CollectionLiterals.<TypeRef>newArrayList();
    RecursionGuard<TypeRef> _recursionGuard = new RecursionGuard<TypeRef>();
    this.primCollectSuperTypeRefs(typeRef, result, _recursionGuard);
    return result;
  }
  
  private void primCollectSuperTypeRefs(final TypeRef typeRef, final List<? super TypeRef> addHere, final RecursionGuard<TypeRef> guard) {
    boolean _tryNext = guard.tryNext(typeRef);
    if (_tryNext) {
      addHere.add(typeRef);
      Type _declaredType = null;
      if (typeRef!=null) {
        _declaredType=typeRef.getDeclaredType();
      }
      Iterable<ParameterizedTypeRef> _allSuperTypes = this.getAllSuperTypes(_declaredType);
      for (final ParameterizedTypeRef currSuperTypeRef : _allSuperTypes) {
        this.primCollectSuperTypeRefs(currSuperTypeRef, addHere, guard);
      }
      guard.done(typeRef);
    }
  }
  
  private Iterable<ParameterizedTypeRef> getAllSuperTypes(final Type type) {
    Iterable<ParameterizedTypeRef> _switchResult = null;
    boolean _matched = false;
    if (type instanceof TClass) {
      _matched=true;
      List<ParameterizedTypeRef> _xifexpression = null;
      ParameterizedTypeRef _superClassRef = ((TClass)type).getSuperClassRef();
      boolean _tripleNotEquals = (_superClassRef != null);
      if (_tripleNotEquals) {
        ParameterizedTypeRef _superClassRef_1 = ((TClass)type).getSuperClassRef();
        _xifexpression = Collections.<ParameterizedTypeRef>unmodifiableList(CollectionLiterals.<ParameterizedTypeRef>newArrayList(_superClassRef_1));
      } else {
        _xifexpression = Collections.<ParameterizedTypeRef>unmodifiableList(CollectionLiterals.<ParameterizedTypeRef>newArrayList());
      }
      EList<ParameterizedTypeRef> _implementedInterfaceRefs = ((TClass)type).getImplementedInterfaceRefs();
      _switchResult = Iterables.<ParameterizedTypeRef>concat(_xifexpression, _implementedInterfaceRefs);
    }
    if (!_matched) {
      if (type instanceof TInterface) {
        _matched=true;
        _switchResult = ((TInterface)type).getSuperInterfaceRefs();
      }
    }
    if (!_matched) {
      _switchResult = Collections.<ParameterizedTypeRef>unmodifiableList(CollectionLiterals.<ParameterizedTypeRef>newArrayList());
    }
    return _switchResult;
  }
  
  /**
   * Add type variable substitutions for the given property access expression. If the property expression is
   * parameterized, the explicitly provided type arguments will be used, otherwise this method will do nothing.
   * 
   * @param G
   * @param paExpr
   */
  public void addSubstitutions(final RuleEnvironment G, final ParameterizedPropertyAccessExpression paExpr) {
    boolean _isParameterized = paExpr.isParameterized();
    if (_isParameterized) {
      final IdentifiableElement prop = paExpr.getProperty();
      if ((prop instanceof Type)) {
        RuleEnvironmentExtensions.addTypeMappings(G, ((Type)prop).getTypeVars(), paExpr.getTypeArgs());
      }
    }
  }
  
  /**
   * Add type variable substitutions for the given call expression. If the call expression is parameterized,
   * the explicitly provided type arguments will be used, otherwise this method tries to infer type arguments
   * from types of function/method arguments and the expected return type.
   * 
   * @param G
   * @param callExpr
   * @param targetTypeRef  the inferred type of <code>callExpr.target</code> or <code>null</code> to let
   *                       this method do the inference; only purpose of this argument is to avoid an
   *                       unnecessary 2nd type inference if caller has already performed this.
   */
  public void addSubstitutions(final RuleEnvironment G, final ParameterizedCallExpression callExpr, final TypeRef targetTypeRef) {
    if (((G == null) || (callExpr == null))) {
      return;
    }
    TypeRef _xifexpression = null;
    if ((targetTypeRef != null)) {
      _xifexpression = targetTypeRef;
    } else {
      RuleEnvironment _ruleEnvironment = new RuleEnvironment(G);
      _xifexpression = this.ts.type(_ruleEnvironment, callExpr.getTarget()).getValue();
    }
    final TypeRef actualTargetTypeRef = _xifexpression;
    if ((!(actualTargetTypeRef instanceof FunctionTypeExprOrRef))) {
      return;
    }
    final FunctionTypeExprOrRef F = ((FunctionTypeExprOrRef) actualTargetTypeRef);
    TypeRef _returnTypeRef = F.getReturnTypeRef();
    if ((_returnTypeRef instanceof StructuralTypeRef)) {
      TypeRef _returnTypeRef_1 = F.getReturnTypeRef();
      this.restorePostponedSubstitutionsFrom(G, ((StructuralTypeRef) _returnTypeRef_1));
    }
    EList<TFormalParameter> _fpars = F.getFpars();
    for (final TFormalParameter currFpar : _fpars) {
      TypeRef _typeRef = currFpar.getTypeRef();
      if ((_typeRef instanceof StructuralTypeRef)) {
        TypeRef _typeRef_1 = currFpar.getTypeRef();
        this.restorePostponedSubstitutionsFrom(G, ((StructuralTypeRef) _typeRef_1));
      }
    }
    boolean _isGeneric = F.isGeneric();
    if (_isGeneric) {
      List<TypeRef> _xifexpression_1 = null;
      boolean _isParameterized = callExpr.isParameterized();
      if (_isParameterized) {
        _xifexpression_1 = callExpr.getTypeArgs();
      } else {
        List<TypeRef> _elvis = null;
        List<TypeRef> _inferredTypeArgs = ASTMetaInfoUtils.getInferredTypeArgs(callExpr);
        if (_inferredTypeArgs != null) {
          _elvis = _inferredTypeArgs;
        } else {
          _elvis = Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList());
        }
        _xifexpression_1 = _elvis;
      }
      final List<TypeRef> typeArgs = _xifexpression_1;
      RuleEnvironmentExtensions.addTypeMappings(G, F.getTypeVars(), typeArgs);
    }
  }
  
  /**
   * Helper method for Xsemantics substTypeVariables judgment. Will either directly substitute in a copy of 'typeRef'
   * or postpone substitution. The given 'typeRef' is never changed.
   */
  public TypeRef substTypeVariablesInStructuralMembers(final RuleEnvironment G, final StructuralTypeRef typeRef) {
    if ((typeRef.getStructuralMembers().isEmpty() && typeRef.getPostponedSubstitutions().isEmpty())) {
      return ((TypeRef) typeRef);
    }
    final StructuralTypeRef result = TypeUtils.<StructuralTypeRef>copy(typeRef);
    int _size = result.getGenStructuralMembers().size();
    int _size_1 = result.getStructuralMembers().size();
    boolean _tripleEquals = (_size == _size_1);
    if (_tripleEquals) {
      final Consumer<TStructMember> _function = (TStructMember member) -> {
        final ArrayList<TypeArgument> l = CollectionLiterals.<TypeArgument>newArrayList();
        final TreeIterator<EObject> iter = member.eAllContents();
        while (iter.hasNext()) {
          {
            final EObject obj = iter.next();
            if ((obj instanceof TypeArgument)) {
              l.add(((TypeArgument)obj));
              iter.prune();
            }
          }
        }
        final Consumer<TypeArgument> _function_1 = (TypeArgument ta) -> {
          final TypeArgument taSubst = this.ts.substTypeVariables(G, ta).getValue();
          if (((taSubst != null) && (taSubst != ta))) {
            EcoreUtil.replace(ta, taSubst);
          }
        };
        l.forEach(_function_1);
      };
      result.getGenStructuralMembers().forEach(_function);
    } else {
      this.storePostponedSubstitutionsIn(G, result);
    }
    return ((TypeRef) result);
  }
  
  /**
   * For all type variables used in the structural members of 'typeRef', this method stores the type
   * variable mappings defined in G to the given 'typeRef' (stored in property 'postponedSubstitutions').
   * This allows to defer actual substitution in the structural members until a later point in time.
   * <p>
   * For more details, see the API documentation for property 'postponedSubstitutions' of
   * {@link StructuralTypeRef}.
   * 
   * @see #restorePostponedSubstitutionsFrom(RuleEnvironment,StructuralTypeRef)
   */
  public void storePostponedSubstitutionsIn(final RuleEnvironment G, final StructuralTypeRef typeRef) {
    final Set<TypeVariable> typeVarsInMembers = TypeUtils.getTypeVarsInStructMembers(typeRef);
    final Predicate<TypeVariable> _function = (TypeVariable currVar) -> {
      return typeRef.hasPostponedSubstitutionFor(currVar);
    };
    typeVarsInMembers.removeIf(_function);
    final List<TypeVariableMapping> bindings = CollectionLiterals.<TypeVariableMapping>newArrayList();
    for (final TypeVariable currVar : typeVarsInMembers) {
      {
        final Object currArg = G.get(currVar);
        if ((currArg instanceof TypeArgument)) {
          final TypeArgument currArgCpy = TypeUtils.<TypeArgument>copy(((TypeArgument)currArg));
          if ((currArgCpy instanceof StructuralTypeRef)) {
            this.storePostponedSubstitutionsIn(G, ((StructuralTypeRef)currArgCpy));
          }
          TypeVariableMapping _createTypeVariableMapping = TypeUtils.createTypeVariableMapping(currVar, currArgCpy);
          bindings.add(_createTypeVariableMapping);
        }
      }
    }
    EList<TypeVariableMapping> _postponedSubstitutions = typeRef.getPostponedSubstitutions();
    Iterables.<TypeVariableMapping>addAll(_postponedSubstitutions, bindings);
    EList<TypeVariableMapping> _postponedSubstitutions_1 = typeRef.getPostponedSubstitutions();
    for (final TypeVariableMapping tvmapping : _postponedSubstitutions_1) {
      {
        final TypeArgument typeArgSubst = this.ts.substTypeVariables(G, tvmapping.getTypeArg()).getValue();
        if (((typeArgSubst != null) && (typeArgSubst != tvmapping.getTypeArg()))) {
          tvmapping.setTypeArg(typeArgSubst);
        }
      }
    }
  }
  
  /**
   * Restores type variable mappings from the given structural type reference 'typeRef' to rule
   * environment 'G'. Assumes that method {@link #storePostponedSubstitutionsIn(RuleEnvironment,StructuralTypeRef)}
   * has been invoked on the given 'typeRef' before.
   */
  public void restorePostponedSubstitutionsFrom(final RuleEnvironment G, final StructuralTypeRef typeRef) {
    final Consumer<TypeVariableMapping> _function = (TypeVariableMapping currMapping) -> {
      G.add(currMapping.getTypeVar(), TypeUtils.<TypeArgument>copy(currMapping.getTypeArg()));
    };
    typeRef.getPostponedSubstitutions().forEach(_function);
  }
  
  TypeRef bindTypeVariables(final RuleEnvironment G, final TypeRef typeRef) {
    ParameterizedTypeRef _switchResult = null;
    boolean _matched = false;
    if (typeRef instanceof FunctionTypeRef) {
      _matched=true;
      return typeRef;
    }
    if (!_matched) {
      if (typeRef instanceof ParameterizedTypeRef) {
        _matched=true;
        ParameterizedTypeRef _xifexpression = null;
        Type _declaredType = ((ParameterizedTypeRef)typeRef).getDeclaredType();
        if ((_declaredType instanceof TypeVariable)) {
          Object _get = G.get(((ParameterizedTypeRef)typeRef).getDeclaredType());
          final TypeRef boundType = ((TypeRef) _get);
          return boundType;
        } else {
          ParameterizedTypeRef _xifexpression_1 = null;
          boolean _isGeneric = ((ParameterizedTypeRef)typeRef).getDeclaredType().isGeneric();
          if (_isGeneric) {
            final ParameterizedTypeRef ptr = TypeUtils.<ParameterizedTypeRef>copy(((ParameterizedTypeRef)typeRef));
            int i = 0;
            while ((i < ((ParameterizedTypeRef)typeRef).getDeclaredType().getTypeVars().size())) {
              {
                final TypeVariable typeVar = ((ParameterizedTypeRef)typeRef).getDeclaredType().getTypeVars().get(i);
                Object _get_1 = G.get(typeVar);
                final TypeRef boundType_1 = ((TypeRef) _get_1);
                if ((boundType_1 != null)) {
                  ptr.getTypeArgs().set(i, TypeUtils.<TypeRef>copy(boundType_1));
                }
                i = (i + 1);
              }
            }
            return ptr;
          } else {
            _xifexpression_1 = ((ParameterizedTypeRef)typeRef);
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
    }
    if (!_matched) {
      return typeRef;
    }
    return _switchResult;
  }
}
