/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem;

import com.google.common.base.Objects;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.ListMultimap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.scoping.builtin.GlobalObjectScope;
import org.eclipse.n4js.scoping.builtin.VirtualBaseTypeScope;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.DeferredTypeRef;
import org.eclipse.n4js.ts.typeRefs.ExistentialTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeRef;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.Wildcard;
import org.eclipse.n4js.ts.types.AnyType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.NullType;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.UndefinedType;
import org.eclipse.n4js.ts.types.VirtualBaseType;
import org.eclipse.n4js.ts.types.VoidType;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.ITypeReplacementProvider;
import org.eclipse.n4js.typesystem.PredefinedTypes;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.utils.RecursionGuard;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Extensions of class RuleEnvironment for handling substitutions and
 * retrieving build in types.
 */
@SuppressWarnings("all")
public class RuleEnvironmentExtensions {
  /**
   * Key used for storing a cancel indicator in a rule environment. Client code should not use this constant
   * directly, but instead use methods
   * {@link RuleEnvironmentExtensions#addCancelIndicator(RuleEnvironment,CancelIndicator)} and
   * {@link RuleEnvironmentExtensions#getCancelIndicator(RuleEnvironment)}.
   */
  private final static String KEY__CANCEL_INDICATOR = "cancelIndicator";
  
  /**
   * Key used for storing a 'this' binding in a rule environment. Client code should not use this constant
   * directly, but instead use methods
   * {@link RuleEnvironmentExtensions#addThisType(RuleEnvironment,TypeRef) addThisType(RuleEnvironment G, TypeRef actualThisTypeRef)} and
   * {@link RuleEnvironmentExtensions#getThisType(RuleEnvironment) getThisType(RuleEnvironment G)}.
   */
  private final static String KEY__THIS_BINDING = "this";
  
  /**
   * Key used for storing inconsistent substitutions in a rule environment. Client code should not use this constant
   * directly, but instead use methods
   * {@link RuleEnvironmentExtensions#recordInconsistentSubstitutions(RuleEnvironment)},
   * {@link RuleEnvironmentExtensions#addInconsistentSubstitutions(RuleEnvironment,TypeVariable,Collection)}, and
   * {@link RuleEnvironmentExtensions#getInconsistentSubstitutions(RuleEnvironment,TypeVariable)}.
   */
  private final static String KEY__INCONSISTENT_SUBSTITUTIONS = "inconsistentSubstitutions";
  
  /**
   * Key for a List&lt;ExistentialTypeRef> with existential type references that should be re-opened during
   * type inference, i.e. they should be treated like the Wildcard they were created from.
   * For detailed semantics, see xsemantics rules subtypeRefExistentialTypeRefLeft/Right.
   */
  private final static String KEY__REOPEN_EXISTENTIAL_TYPES = "reopenExistentialTypes";
  
  /**
   * Key for storing an ITypeReplacementProvider defining a replacement of some types by other types within
   * the context of a rule environment. Used when dealing with replacing an API by its implementation project.
   */
  private final static String KEY__TYPE_REPLACEMENT = "typeReplacement";
  
  public final static String GUARD_VARIABLE_DECLARATION = "varDecl";
  
  public final static String GUARD_TYPE_CALL_EXPRESSION = "typeCallExpression";
  
  public final static String GUARD_TYPE_PROPERTY_ACCESS_EXPRESSION = "typePropertyAccessExpression";
  
  public final static String GUARD_SUBTYPE_PARAMETERIZED_TYPE_REF__STRUCT = "subtypeRefParameterizedTypeRef__struct";
  
  public final static String GUARD_SUBTYPE_PARAMETERIZED_TYPE_REF__ARGS = "subtypeRefParameterizedTypeRef__args";
  
  public final static String GUARD_SUBST_TYPE_VARS = "substTypeVariablesInParameterizedTypeRef";
  
  public final static String GUARD_STRUCTURAL_TYPING_COMPUTER = "StructuralTypingComputer";
  
  public final static String GUARD_REDUCER_IS_SUBTYPE_OF = "Reducer#isSubtypeOf";
  
  /**
   * Returns a new {@code RuleEnvironment}; we need this because of the
   * {@code BuiltInTypeScope} and we cannot simply create a new empty environment.
   * 
   * @param context must not be null!
   */
  public static RuleEnvironment newRuleEnvironment(final EObject context) {
    Resource res = context.eResource();
    if ((res == null)) {
      if ((context instanceof BoundThisTypeRef)) {
        res = ((BoundThisTypeRef)context).getActualThisTypeRef().getDeclaredType().eResource();
      }
    }
    RuleEnvironment G = new RuleEnvironment();
    RuleEnvironmentExtensions.setPredefinedTypesFromObjectsResourceSet(G, res.getResourceSet());
    G.add(Resource.class, res);
    return G;
  }
  
  /**
   * Returns a new {@code RuleEnvironment} with a given resource to provide context information and xtext index access.
   */
  public static RuleEnvironment newRuleEnvironment(final Resource resource) {
    RuleEnvironment G = new RuleEnvironment();
    RuleEnvironmentExtensions.setPredefinedTypesFromObjectsResourceSet(G, resource.getResourceSet());
    G.add(Resource.class, resource);
    return G;
  }
  
  /**
   * Returns a new {@code RuleEnvironment} for the same predefined types, resource, and cancel indicator as the given
   * rule environment.
   * <p>
   * IMPORTANT: other key/value pairs from G will not be available in the returned rule environment! Compare this with
   * method {@link #wrap(RuleEnvironment)}.
   */
  public static RuleEnvironment newRuleEnvironment(final RuleEnvironment G) {
    RuleEnvironment Gnew = new RuleEnvironment();
    RuleEnvironmentExtensions.setPredefinedTypes(Gnew, RuleEnvironmentExtensions.getPredefinedTypes(G));
    Gnew.add(Resource.class, G.get(Resource.class));
    RuleEnvironmentExtensions.addCancelIndicator(Gnew, RuleEnvironmentExtensions.getCancelIndicator(G));
    return Gnew;
  }
  
  /**
   * Return a new rule environment wrapping the given rule environment 'G', i.e. the all key/value pairs of 'G'
   * will be readable in the returned environment, but changes to the returned environment will not affect 'G'.
   */
  public static RuleEnvironment wrap(final RuleEnvironment G) {
    return new RuleEnvironment(G);
  }
  
  public static boolean setPredefinedTypesFromObjectsResourceSet(final RuleEnvironment G, final ResourceSet resourceSet) {
    boolean _xblockexpression = false;
    {
      if ((resourceSet == null)) {
        throw new IllegalArgumentException("Resource set used to load predefined types must not be null at org.eclipse.n4js.typesystem.RuleEnvironmentExtensions.setPredefinedTypesFromObjectsResourceSet(RuleEnvironment, ResourceSet)");
      }
      final BuiltInTypeScope builtInTypeScope = BuiltInTypeScope.get(resourceSet);
      final GlobalObjectScope globalObjectTypeScope = GlobalObjectScope.get(resourceSet);
      final VirtualBaseTypeScope virtualBaseTypeScope = VirtualBaseTypeScope.get(resourceSet);
      PredefinedTypes _predefinedTypes = new PredefinedTypes(builtInTypeScope, globalObjectTypeScope, virtualBaseTypeScope);
      _xblockexpression = G.add(PredefinedTypes.PREDEFINED_TYPES_KEY, _predefinedTypes);
    }
    return _xblockexpression;
  }
  
  public static boolean setPredefinedTypes(final RuleEnvironment G, final PredefinedTypes predefinedTypes) {
    return G.add(PredefinedTypes.PREDEFINED_TYPES_KEY, predefinedTypes);
  }
  
  public static PredefinedTypes getPredefinedTypes(final RuleEnvironment G) {
    Object _get = G.get(PredefinedTypes.PREDEFINED_TYPES_KEY);
    final PredefinedTypes predefinedTypes = ((PredefinedTypes) _get);
    if ((predefinedTypes == null)) {
      throw new IllegalStateException(
        "Predefined types not set, call type system with configured rule environment");
    }
    return predefinedTypes;
  }
  
  /**
   * Convenience method returning the {@link BuiltInTypeScope} via this rule environment's {@link PredefinedTypes}.
   */
  public static BuiltInTypeScope getBuiltInTypeScope(final RuleEnvironment G) {
    PredefinedTypes _predefinedTypes = null;
    if (G!=null) {
      _predefinedTypes=RuleEnvironmentExtensions.getPredefinedTypes(G);
    }
    BuiltInTypeScope _builtInTypeScope = null;
    if (_predefinedTypes!=null) {
      _builtInTypeScope=_predefinedTypes.builtInTypeScope;
    }
    return _builtInTypeScope;
  }
  
  /**
   * Convenience method returning the {@link GlobalObjectScope} via this rule environment's {@link PredefinedTypes}.
   */
  public static GlobalObjectScope getGlobalObjectScope(final RuleEnvironment G) {
    PredefinedTypes _predefinedTypes = null;
    if (G!=null) {
      _predefinedTypes=RuleEnvironmentExtensions.getPredefinedTypes(G);
    }
    GlobalObjectScope _globalObjectScope = null;
    if (_predefinedTypes!=null) {
      _globalObjectScope=_predefinedTypes.globalObjectScope;
    }
    return _globalObjectScope;
  }
  
  /**
   * Returns the resource used to load built-in types and to resolve proxies. This is the resource of the object
   */
  public static Resource getContextResource(final RuleEnvironment G) {
    Object _get = G.get(Resource.class);
    return ((Resource) _get);
  }
  
  /**
   * Add a cancel indicator to the given rule environment.
   */
  public static void addCancelIndicator(final RuleEnvironment G, final CancelIndicator cancelIndicator) {
    G.add(RuleEnvironmentExtensions.KEY__CANCEL_INDICATOR, cancelIndicator);
  }
  
  /**
   * Returns the cancel indicator of this rule environment or <code>null</code> if none has been added, yet.
   */
  public static CancelIndicator getCancelIndicator(final RuleEnvironment G) {
    Object _get = G.get(RuleEnvironmentExtensions.KEY__CANCEL_INDICATOR);
    return ((CancelIndicator) _get);
  }
  
  /**
   * <b>IMPORTANT:</b><br>
   * use this only for rare special cases (e.g. logging); ordinary cancellation handling should be done by invoking
   * {@link OperationCanceledManager#checkCanceled(CancelIndicator)} with the cancel indicator returned by
   * {@link #getCancelIndicator(RuleEnvironment)}!
   * <p>
   * Tells if the given rule environment has a cancel indicator AND that indicator is canceled.
   */
  public static boolean isCanceled(final RuleEnvironment G) {
    final CancelIndicator cancelIndicator = RuleEnvironmentExtensions.getCancelIndicator(G);
    return ((cancelIndicator != null) && cancelIndicator.isCanceled());
  }
  
  /**
   * Adds the actual this type to the rule environment if the actual this type is either
   * a  ParameterizedTypeRef or a BoundThisTypeRef. The latter case happens if the receiver
   * of a function call is a function call itself, returning a this type.
   */
  public static void addThisType(final RuleEnvironment G, final TypeRef actualThisTypeRef) {
    boolean _matched = false;
    if (actualThisTypeRef instanceof TypeTypeRef) {
      _matched=true;
      TypeArgument _typeArg = ((TypeTypeRef)actualThisTypeRef).getTypeArg();
      if ((_typeArg instanceof TypeRef)) {
        TypeArgument _typeArg_1 = ((TypeTypeRef)actualThisTypeRef).getTypeArg();
        RuleEnvironmentExtensions.addThisType(G, ((TypeRef) _typeArg_1));
      }
    }
    if (!_matched) {
      if (actualThisTypeRef instanceof ParameterizedTypeRef) {
        _matched=true;
        G.add(RuleEnvironmentExtensions.KEY__THIS_BINDING, TypeUtils.createBoundThisTypeRef(((ParameterizedTypeRef)actualThisTypeRef)));
      }
    }
    if (!_matched) {
      if (actualThisTypeRef instanceof BoundThisTypeRef) {
        _matched=true;
        G.add(RuleEnvironmentExtensions.KEY__THIS_BINDING, actualThisTypeRef);
      }
    }
  }
  
  /**
   * Returns the current this type, this must have been added before via
   * {@link #addThisType(RuleEnvironment, TypeRef)}
   */
  public static TypeRef getThisType(final RuleEnvironment G) {
    Object _get = G.get(RuleEnvironmentExtensions.KEY__THIS_BINDING);
    return ((TypeRef) _get);
  }
  
  /**
   * Turn on recording of inconsistent substitutions in Xsemantics judgment {@code substTypeVariables}.
   * This is used by a validation which will then produce a corresponding error.
   */
  public static void recordInconsistentSubstitutions(final RuleEnvironment G) {
    G.add(RuleEnvironmentExtensions.KEY__INCONSISTENT_SUBSTITUTIONS, ArrayListMultimap.<TypeVariable, TypeRef>create());
  }
  
  /**
   * Iff recording of inconsistent substitutions has been turned on for the given rule environment (see method
   * {@link #recordInconsistentSubstitutions(RuleEnvironment)}), then this method will store such substitutions
   * in the given rule environment.
   */
  public static void addInconsistentSubstitutions(final RuleEnvironment G, final TypeVariable typeVar, final Collection<? extends TypeRef> substitutions) {
    Object _get = G.get(RuleEnvironmentExtensions.KEY__INCONSISTENT_SUBSTITUTIONS);
    final ListMultimap<TypeVariable, TypeRef> storage = ((ListMultimap<TypeVariable, TypeRef>) _get);
    if ((storage != null)) {
      storage.putAll(typeVar, substitutions);
    }
  }
  
  /**
   * Iff recording of inconsistent substitutions has been turned on for the given rule environment (see method
   * {@link #recordInconsistentSubstitutions(RuleEnvironment)}), then this method will return those substitutions.
   * Otherwise, an empty list is returned.
   */
  public static List<TypeRef> getInconsistentSubstitutions(final RuleEnvironment G, final TypeVariable typeVar) {
    Object _get = G.get(RuleEnvironmentExtensions.KEY__INCONSISTENT_SUBSTITUTIONS);
    final ListMultimap<TypeVariable, TypeRef> storage = ((ListMultimap<TypeVariable, TypeRef>) _get);
    List<TypeRef> _xifexpression = null;
    if ((storage != null)) {
      _xifexpression = storage.get(typeVar);
    } else {
      _xifexpression = Collections.<TypeRef>emptyList();
    }
    return _xifexpression;
  }
  
  /**
   * For semantics, see xsemantics rules subtypeRefExistentialTypeRefLeft/Right.
   */
  public static void addExistentialTypeToBeReopened(final RuleEnvironment G, final ExistentialTypeRef existentialTypeRef) {
    Wildcard _wildcard = existentialTypeRef.getWildcard();
    boolean _tripleNotEquals = (_wildcard != null);
    if (_tripleNotEquals) {
      Wildcard _wildcard_1 = existentialTypeRef.getWildcard();
      Pair<String, Wildcard> _mappedTo = Pair.<String, Wildcard>of(RuleEnvironmentExtensions.KEY__REOPEN_EXISTENTIAL_TYPES, _wildcard_1);
      G.add(_mappedTo, Boolean.TRUE, true);
    }
  }
  
  /**
   * For semantics, see xsemantics rules subtypeRefExistentialTypeRefLeft/Right.
   */
  public static boolean isExistentialTypeToBeReopened(final RuleEnvironment G, final ExistentialTypeRef existentialTypeRef) {
    return ((existentialTypeRef.getWildcard() != null) && (G.get(Pair.<String, Wildcard>of(RuleEnvironmentExtensions.KEY__REOPEN_EXISTENTIAL_TYPES, existentialTypeRef.getWildcard())) != null));
  }
  
  public static boolean isExistentialTypeToBeReopened(final RuleEnvironment G, final EObject obj, final boolean searchContents) {
    if ((obj instanceof ExistentialTypeRef)) {
      boolean _isExistentialTypeToBeReopened = RuleEnvironmentExtensions.isExistentialTypeToBeReopened(G, ((ExistentialTypeRef)obj));
      if (_isExistentialTypeToBeReopened) {
        return true;
      }
    }
    if ((searchContents && (obj != null))) {
      final Function1<ExistentialTypeRef, Boolean> _function = (ExistentialTypeRef it) -> {
        return Boolean.valueOf(RuleEnvironmentExtensions.isExistentialTypeToBeReopened(G, it));
      };
      return IteratorExtensions.<ExistentialTypeRef>exists(Iterators.<ExistentialTypeRef>filter(obj.eAllContents(), ExistentialTypeRef.class), _function);
    }
    return false;
  }
  
  public static ExistentialTypeRef createExistentialTypeRef(final TypeVariable typeVar, final Wildcard wildcard) {
    ExistentialTypeRef etr = TypeRefsFactory.eINSTANCE.createExistentialTypeRef();
    etr.setWildcard(wildcard);
    etr.setBoundTypeVariable(typeVar);
    return etr;
  }
  
  /**
   * For the moment we won't use that; it should also take intersection
   * types into consideration.
   */
  public static ExistentialTypeRef createExistentialTypeRef(final TypeVariable typeVar) {
    Wildcard _createWildcard = TypeRefsFactory.eINSTANCE.createWildcard();
    final Procedure1<Wildcard> _function = (Wildcard it) -> {
      final TypeRef declUB = typeVar.getDeclaredUpperBound();
      it.setDeclaredUpperBound(TypeUtils.<TypeRef>copyIfContained(declUB));
    };
    Wildcard _doubleArrow = ObjectExtensions.<Wildcard>operator_doubleArrow(_createWildcard, _function);
    return RuleEnvironmentExtensions.createExistentialTypeRef(typeVar, _doubleArrow);
  }
  
  public static void setTypeReplacement(final RuleEnvironment G, final ITypeReplacementProvider replacementProvider) {
    G.add(RuleEnvironmentExtensions.KEY__TYPE_REPLACEMENT, replacementProvider);
  }
  
  public static TypeRef getReplacement(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef instanceof ParameterizedTypeRef)) {
      if ((!(typeRef instanceof FunctionTypeRef))) {
        final Type type = ((ParameterizedTypeRef)typeRef).getDeclaredType();
        final Type replacement = RuleEnvironmentExtensions.<Type>getReplacement(G, type);
        if ((replacement != type)) {
          final ParameterizedTypeRef cpy = TypeUtils.<ParameterizedTypeRef>copyWithProxies(((ParameterizedTypeRef)typeRef));
          cpy.setDeclaredType(replacement);
          return cpy;
        }
      }
    } else {
    }
    return typeRef;
  }
  
  public static <T extends Type> T getReplacement(final RuleEnvironment G, final T type) {
    Object _get = G.get(RuleEnvironmentExtensions.KEY__TYPE_REPLACEMENT);
    final ITypeReplacementProvider replacementProvider = ((ITypeReplacementProvider) _get);
    T _replacement = null;
    if (replacementProvider!=null) {
      _replacement=replacementProvider.<T>getReplacement(type);
    }
    final T replacement = _replacement;
    T _xifexpression = null;
    if ((replacement != null)) {
      _xifexpression = replacement;
    } else {
      _xifexpression = type;
    }
    return _xifexpression;
  }
  
  public static TypeRef createTypeRefFromUpperBound(final TypeVariable typeVar) {
    return TypeUtils.<TypeRef>copyIfContained(typeVar.getDeclaredUpperBound());
  }
  
  /**
   * Returns the top type (which is currently 'any' but may change in the future).
   */
  public static AnyType topType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.anyType(G);
  }
  
  /**
   * Returns newly created reference to the top type (which is currently 'any' but may change in the future).
   */
  public static ParameterizedTypeRef topTypeRef(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.anyTypeRef(G);
  }
  
  /**
   * Returns the bottom type (which is currently 'undefined' but may change in the future).
   */
  public static UndefinedType bottomType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.undefinedType(G);
  }
  
  /**
   * Returns newly created reference to the bottom type (which is currently 'undefined' but may change in the future).
   */
  public static ParameterizedTypeRef bottomTypeRef(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.undefinedTypeRef(G);
  }
  
  /**
   * Returns built-in type {@code boolean}
   */
  public static PrimitiveType booleanType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getBooleanType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code boolean}
   */
  public static ParameterizedTypeRef booleanTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.booleanType(G));
  }
  
  /**
   * Returns built-in type {@code string}
   */
  public static PrimitiveType stringType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getStringType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code string}
   */
  public static ParameterizedTypeRef stringTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.stringType(G));
  }
  
  /**
   * Returns built-in object type {@code String}
   */
  public static TObjectPrototype stringObjectType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getStringObjectType();
  }
  
  /**
   * Returns newly created reference to built-in object type {@code String}
   */
  public static ParameterizedTypeRef stringObjectTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.stringObjectType(G));
  }
  
  /**
   * Returns built-in type {@code number}
   */
  public static PrimitiveType numberType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getNumberType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code number}
   */
  public static ParameterizedTypeRef numberTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.numberType(G));
  }
  
  /**
   * Returns built-in type {@code int}
   */
  public static PrimitiveType intType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getIntType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code int}
   */
  public static ParameterizedTypeRef intTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.intType(G));
  }
  
  /**
   * Returns built-in type {@code symbol}
   */
  public static PrimitiveType symbolType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getSymbolType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code symbol}
   */
  public static ParameterizedTypeRef symbolTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.symbolType(G));
  }
  
  /**
   * Returns built-in object type {@code Symbol}
   */
  public static TObjectPrototype symbolObjectType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getSymbolObjectType();
  }
  
  /**
   * Returns newly created reference to built-in object type {@code Symbol}
   */
  public static ParameterizedTypeRef symbolObjectTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.symbolObjectType(G));
  }
  
  /**
   * Returns built-in type {@code any}
   */
  public static AnyType anyType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getAnyType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code any}
   */
  public static ParameterizedTypeRef anyTypeRef(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getAnyTypeRef();
  }
  
  /**
   * Returns newly created dynamic reference to built-in type {@code any}, that is {@code any+}.
   * This is the default type used in JavaScript modes.
   */
  public static ParameterizedTypeRef anyTypeRefDynamic(final RuleEnvironment G) {
    final ParameterizedTypeRef result = TypeUtils.createTypeRef(RuleEnvironmentExtensions.anyType(G));
    result.setDynamic(true);
    return result;
  }
  
  /**
   * Returns newly created reference to built-in type {@code null}
   */
  public static ParameterizedTypeRef nullTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.nullType(G));
  }
  
  /**
   * Returns built-in type {@code undefined}
   */
  public static UndefinedType undefinedType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getUndefinedType();
  }
  
  /**
   * Returns built-in type {@code null}
   */
  public static NullType nullType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getNullType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code undefined}
   */
  public static ParameterizedTypeRef undefinedTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.undefinedType(G));
  }
  
  /**
   * Returns built-in type {@code void}
   */
  public static VoidType voidType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getVoidType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code void}
   */
  public static ParameterizedTypeRef voidTypeRef(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getVoidTypeRef();
  }
  
  /**
   * Returns built-in type {@code RegExp}
   */
  public static TObjectPrototype regexpType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getRegexpType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code RegExp}
   */
  public static ParameterizedTypeRef regexpTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.regexpType(G));
  }
  
  /**
   * Returns built-in type {@code Array<T>}
   */
  public static TObjectPrototype arrayType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getArrayType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code Array<T>}
   */
  public static ParameterizedTypeRef arrayTypeRef(final RuleEnvironment G, final TypeArgument... typeArgs) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.arrayType(G), typeArgs);
  }
  
  /**
   * Returns built-in type {@code Object}
   */
  public static TClassifier objectType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getObjectType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code Object}
   */
  public static ParameterizedTypeRef objectTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.objectType(G));
  }
  
  /**
   * Returns newly created reference to built-in global object type
   */
  public static TClass globalObjectType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).globalObjectScope.getGlobalObject();
  }
  
  /**
   * Returns newly created reference to built-in global object type
   */
  public static ParameterizedTypeRef globalObjectTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.globalObjectType(G));
  }
  
  /**
   * Returns built-in type {@code Function}
   */
  public static TObjectPrototype functionType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getFunctionType();
  }
  
  /**
   * Returns newly created structural reference to built-in type {@code Function}
   */
  public static ParameterizedTypeRef structuralFunctionTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.functionType(G), TypingStrategy.STRUCTURAL);
  }
  
  /**
   * Returns newly created reference to built-in type {@code Function}
   */
  public static ParameterizedTypeRef functionTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.functionType(G));
  }
  
  /**
   * Returns built-in type {@code N4Object}
   */
  public static TClass n4ObjectType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getN4ObjectType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code N4Object}
   */
  public static ParameterizedTypeRef n4ObjectTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.n4ObjectType(G));
  }
  
  /**
   * Returns built-in type {@code N4Enum}
   */
  public static TObjectPrototype n4EnumType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getN4EnumType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code N4Enum}
   */
  public static ParameterizedTypeRef n4EnumTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.n4EnumType(G));
  }
  
  /**
   * Returns built-in type {@code N4StringBasedEnum}
   */
  public static TObjectPrototype n4StringBasedEnumType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getN4StringBasedEnumType();
  }
  
  /**
   * Returns a newly created reference to the  built-in type {@code N4StringBasedEnum}
   */
  public static ParameterizedTypeRef n4StringBasedEnumTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.n4StringBasedEnumType(G));
  }
  
  /**
   * Returns built-in type {@code i18nKey}
   */
  public static PrimitiveType i18nKeyType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getI18nKeyType();
  }
  
  /**
   * Returns built-in type {@code pathSelector}
   */
  public static PrimitiveType pathSelectorType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getPathSelectorType();
  }
  
  /**
   * Returns built-in type {@code typeName}
   */
  public static PrimitiveType typeNameType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getTypeNameType();
  }
  
  /**
   * Returns built-in type {@code N4Provider}
   */
  public static TInterface n4ProviderType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getN4ProviderType();
  }
  
  /**
   * Returns built-in type {@code Error}.
   */
  public static TObjectPrototype errorType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getErrorType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code Error}.
   */
  public static ParameterizedTypeRef errorTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.errorType(G));
  }
  
  /**
   * Returns built-in type {@code ArgumentsType}
   */
  public static VirtualBaseType argumentsType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).virtualBaseTypeScope.getArgumentsType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code ArgumentsType}
   */
  public static ParameterizedTypeRef argumentsTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.argumentsType(G));
  }
  
  /**
   * Returns built-in type {@code MigrationContext}
   */
  public static TInterface migrationContextType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getMigrationContextType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code MigrationContext}
   */
  public static ParameterizedTypeRef migrationContextTypeRef(final RuleEnvironment G) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.migrationContextType(G));
  }
  
  /**
   * Returns built-in type {@code Iterable<T>}
   */
  public static TInterface iterableType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getIterableType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code Iterable<T>}
   */
  public static ParameterizedTypeRef iterableTypeRef(final RuleEnvironment G, final TypeArgument... typeArgs) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.iterableType(G), typeArgs);
  }
  
  /**
   * Returns built-in type {@code IterableN<T1...TN>}
   */
  public static TInterface iterableNType(final RuleEnvironment G, final int n) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getIterableNType(n);
  }
  
  /**
   * Returns newly created reference to built-in type {@code IterableN<T1...TN>}
   */
  public static ParameterizedTypeRef iterableNTypeRef(final RuleEnvironment G, final int n, final TypeArgument... typeArgs) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.iterableNType(G, n), typeArgs);
  }
  
  /**
   * Returns built-in type {@code IterableN<T1...TN>}
   */
  public static List<TInterface> iterableNTypes(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getIterableNTypes();
  }
  
  /**
   * Returns true iff <code>obj</code> is a {@link Type} or {@link TypeRef} and is or points to
   * one of the <code>IterableN&lt;...></code> built-in types. Does <b>not</b> check for the
   * built-in type <code>Iterable&lt;T></code>.
   */
  public static boolean isIterableN(final RuleEnvironment G, final EObject obj) {
    Type _switchResult = null;
    boolean _matched = false;
    if (obj instanceof Type) {
      _matched=true;
      _switchResult = ((Type)obj);
    }
    if (!_matched) {
      if (obj instanceof TypeRef) {
        _matched=true;
        _switchResult = ((TypeRef)obj).getDeclaredType();
      }
    }
    final Type type = _switchResult;
    return ((type != null) && RuleEnvironmentExtensions.iterableNTypes(G).contains(type));
  }
  
  /**
   * Returns built-in type {@code Promise<S,F>}
   */
  public static TClass promiseType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getPromiseType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code Promise<S,F>}
   */
  public static ParameterizedTypeRef promiseTypeRef(final RuleEnvironment G, final TypeArgument... typeArgs) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.promiseType(G), typeArgs);
  }
  
  /**
   * Returns built-in type {@code Generator<Y,R,N>}
   */
  public static TInterface generatorType(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getGeneratorType();
  }
  
  /**
   * Returns newly created reference to built-in type {@code Generator<Y,R,N>}
   */
  public static ParameterizedTypeRef generatorTypeRef(final RuleEnvironment G, final TypeArgument... typeArgs) {
    return TypeUtils.createTypeRef(RuleEnvironmentExtensions.generatorType(G), typeArgs);
  }
  
  /**
   * Returns true if the given type is one of the {@link BuiltInTypeScope#isNumeric(Type) numeric} primitive
   * built-in types.
   */
  public static boolean isNumeric(final RuleEnvironment G, final Type type) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.isNumeric(type);
  }
  
  /**
   * Returns true if the given type is any.
   */
  public static boolean isAny(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef == null)) {
      return false;
    }
    Type _declaredType = typeRef.getDeclaredType();
    AnyType _anyType = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getAnyType();
    return Objects.equal(_declaredType, _anyType);
  }
  
  /**
   * Returns true if the given type is symbol.
   */
  public static boolean isSymbol(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef == null)) {
      return false;
    }
    Type _declaredType = typeRef.getDeclaredType();
    PrimitiveType _symbolType = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getSymbolType();
    return Objects.equal(_declaredType, _symbolType);
  }
  
  /**
   * Returns true if the given type reference points to one of the {@link BuiltInTypeScope#isNumeric(Type) numeric}
   * primitive built-in types.
   */
  public static boolean isNumeric(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef == null)) {
      return false;
    }
    boolean _isNumeric = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.isNumeric(typeRef.getDeclaredType());
    if (_isNumeric) {
      return true;
    }
    if ((typeRef instanceof UnionTypeExpression)) {
      final Function1<TypeRef, Boolean> _function = (TypeRef e) -> {
        return Boolean.valueOf(RuleEnvironmentExtensions.isNumeric(G, e));
      };
      return IterableExtensions.<TypeRef>forall(((UnionTypeExpression)typeRef).getTypeRefs(), _function);
    }
    if ((typeRef instanceof IntersectionTypeExpression)) {
      final Function1<TypeRef, Boolean> _function_1 = (TypeRef e) -> {
        return Boolean.valueOf(RuleEnvironmentExtensions.isNumeric(G, e));
      };
      return IterableExtensions.<TypeRef>exists(((IntersectionTypeExpression)typeRef).getTypeRefs(), _function_1);
    }
    return false;
  }
  
  /**
   * Returns true iff typeRef is a union type and one if its elements
   * is numeric, boolean, null or undefined or contains one of these types.
   * Note that this method returns false for number types -- the
   * typeref needs to be a union type!
   */
  public static boolean containsNumericOperand(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef instanceof UnionTypeExpression)) {
      final Function1<TypeRef, Boolean> _function = (TypeRef e) -> {
        return Boolean.valueOf((RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.isNumericOperand(e.getDeclaredType()) || RuleEnvironmentExtensions.containsNumericOperand(G, e)));
      };
      return IterableExtensions.<TypeRef>exists(((UnionTypeExpression)typeRef).getTypeRefs(), _function);
    }
    return false;
  }
  
  /**
   * Returns true if the given type reference can be used in a numeric
   * operation as operand leading to a numeric result. This is true for
   * number, int, boolean, null, or even undefined, for unions of these types,
   * and for intersections containing any of these types.
   */
  public static boolean isNumericOperand(final RuleEnvironment G, final TypeRef typeRef) {
    if ((typeRef == null)) {
      return false;
    }
    boolean _isNumericOperand = RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.isNumericOperand(typeRef.getDeclaredType());
    if (_isNumericOperand) {
      return true;
    }
    if ((typeRef instanceof UnionTypeExpression)) {
      final Function1<TypeRef, Boolean> _function = (TypeRef e) -> {
        return Boolean.valueOf(RuleEnvironmentExtensions.isNumericOperand(G, e));
      };
      return IterableExtensions.<TypeRef>forall(((UnionTypeExpression)typeRef).getTypeRefs(), _function);
    }
    if ((typeRef instanceof IntersectionTypeExpression)) {
      final Function1<TypeRef, Boolean> _function_1 = (TypeRef e) -> {
        return Boolean.valueOf(RuleEnvironmentExtensions.isNumericOperand(G, e));
      };
      return IterableExtensions.<TypeRef>exists(((IntersectionTypeExpression)typeRef).getTypeRefs(), _function_1);
    }
    return false;
  }
  
  /**
   * Same as {@link TypeUtils#wrapInTypeRef(BuiltInTypeScope,Type,TypeArgument...)}, but will obtain
   * the required {@code BuiltInTypeScope} from the given rule environment.
   */
  public static TypeRef wrapTypeInTypeRef(final RuleEnvironment G, final Type type, final TypeArgument... typeArgs) {
    return TypeUtils.wrapTypeInTypeRef(RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope, type, typeArgs);
  }
  
  /**
   * Returns all type variables for which a type mapping is defined in the given rule environment.
   */
  public static Set<TypeVariable> getTypeMappingKeys(final RuleEnvironment G) {
    final LinkedHashSet<TypeVariable> result = CollectionLiterals.<TypeVariable>newLinkedHashSet();
    RuleEnvironment env = G;
    while ((env != null)) {
      {
        Iterable<TypeVariable> _filter = Iterables.<TypeVariable>filter(env.getEnvironment().keySet(), TypeVariable.class);
        Iterables.<TypeVariable>addAll(result, _filter);
        env = env.getNext();
      }
    }
    return result;
  }
  
  /**
   * Convenience method. Same as {@link #addTypeMapping(RuleEnvironment,TypeVariable,TypeArgument)},
   * but for adding several mappings.
   */
  public static void addTypeMappings(final RuleEnvironment G, final List<? extends TypeVariable> keys, final List<? extends TypeArgument> values) {
    if (((keys == null) || (values == null))) {
      return;
    }
    final int size = Math.min(keys.size(), values.size());
    for (int idx = 0; (idx < size); idx++) {
      RuleEnvironmentExtensions.addTypeMapping(G, keys.get(idx), values.get(idx));
    }
  }
  
  /**
   * Low-level method for adding a type variable -> type argument mapping to a rule environment.
   * Use this method only if you know the exact mapping and you do not need support for handling
   * existing mappings.
   * <p>
   * An existing mapping for type variable 'key' will be overwritten. If the given mapping is
   * invalid, i.e. {@link #isValidMapping(RulenEnvironment,TypeVariable,TypeArgument isValidMapping()}
   * returns false, then this method will do nothing.
   */
  public static void addTypeMapping(final RuleEnvironment G, final TypeVariable key, final TypeArgument value) {
    boolean _isValidTypeMapping = RuleEnvironmentExtensions.isValidTypeMapping(G, key, value);
    boolean _not = (!_isValidTypeMapping);
    if (_not) {
      return;
    }
    final TypeRef actualValue = TypeUtils.captureWildcard(key, value);
    G.add(key, actualValue);
  }
  
  /**
   * Checks if rule environment G defines an actual, i.e. non-reflexive, type variable
   * substitution for <code>typeVariable</code>. Argument <code>typeVariable</code> may
   * either be a {@link TypeVariable} itself or a {@link TypeRef} with a type variable
   * as its declared type.<p>
   * For convenience, this methods takes arguments of any type but will always return
   * <code>false</code> if the argument is neither an instance of {@link TypeVariable}
   * nor an instance of {@link TypeRef} with a declared type that is an instance of
   * {@link TypeVariable}.
   */
  public static boolean hasSubstitutionFor(final RuleEnvironment G, final Object typeVariable) {
    Object _xifexpression = null;
    if ((typeVariable instanceof TypeRef)) {
      _xifexpression = ((TypeRef)typeVariable).getDeclaredType();
    } else {
      _xifexpression = typeVariable;
    }
    final Object key = _xifexpression;
    if ((key instanceof TypeVariable)) {
      final Object value = G.get(key);
      return ((value != null) && (!((value instanceof TypeRef) && (((TypeRef) value).getDeclaredType() == key))));
    }
    return false;
  }
  
  public static boolean isValidTypeMapping(final RuleEnvironment G, final TypeVariable key, final TypeArgument value) {
    boolean _isOrContainsRefToTypeVar = TypeUtils.isOrContainsRefToTypeVar(value, key);
    if (_isOrContainsRefToTypeVar) {
      return false;
    }
    if ((value instanceof DeferredTypeRef)) {
      return false;
    }
    if ((value instanceof ParameterizedTypeRef)) {
      Type _declaredType = ((ParameterizedTypeRef)value).getDeclaredType();
      if ((_declaredType instanceof VoidType)) {
        return false;
      }
    }
    if ((value instanceof ParameterizedTypeRef)) {
      Type _declaredType_1 = ((ParameterizedTypeRef)value).getDeclaredType();
      if ((_declaredType_1 instanceof NullType)) {
        return false;
      }
    }
    return true;
  }
  
  /**
   * Returns the declared or implicit super type of a class. This might be a TClass or, in case
   * of implicit super types and external classes, a TObjectPrototype (i.e. "Object").
   */
  public static TClassifier getDeclaredOrImplicitSuperType(final RuleEnvironment G, final TClass tClass) {
    if (((tClass.getSuperClassRef() != null) && (tClass.getSuperClassRef().getDeclaredType() instanceof TClassifier))) {
      Type _declaredType = tClass.getSuperClassRef().getDeclaredType();
      return ((TClassifier) _declaredType);
    } else {
      boolean _isExternal = tClass.isExternal();
      if (_isExternal) {
        return RuleEnvironmentExtensions.objectType(G);
      } else {
        return RuleEnvironmentExtensions.n4ObjectType(G);
      }
    }
  }
  
  /**
   * Returns transitive, non-reflexive closure of implicit super types. All implicit super types are non-generic, so
   * type arguments can be ignored here savely.
   */
  public static List<ParameterizedTypeRef> collectAllImplicitSuperTypesOfType(final RuleEnvironment G, final Type declaredType) {
    RecursionGuard<Type> _recursionGuard = new RecursionGuard<Type>();
    return RuleEnvironmentExtensions.collectAllImplicitSuperTypesOfType(G, declaredType, _recursionGuard);
  }
  
  private static List<ParameterizedTypeRef> collectAllImplicitSuperTypesOfType(final RuleEnvironment G, final Type declaredType, final RecursionGuard<Type> guard) {
    List<ParameterizedTypeRef> _xblockexpression = null;
    {
      if ((null == declaredType)) {
        return CollectionLiterals.<ParameterizedTypeRef>emptyList();
      }
      boolean _tryNext = guard.tryNext(declaredType);
      boolean _not = (!_tryNext);
      if (_not) {
        if ((declaredType instanceof TClass)) {
          if (((Objects.equal(declaredType, RuleEnvironmentExtensions.n4ObjectType(G)) || (((TClass)declaredType).isExternal() && (!((TClass)declaredType).isDeclaredN4JS()))) || 
            Objects.equal(((TClass)declaredType).getTypingStrategy(), TypingStrategy.STRUCTURAL))) {
            return RuleEnvironmentExtensions.getObjectPrototypesAllImplicitSuperTypeRefs(G);
          } else {
            return RuleEnvironmentExtensions.getN4ClassifiersAllImplicitSuperTypeRefs(G);
          }
        } else {
          return CollectionLiterals.<ParameterizedTypeRef>emptyList();
        }
      }
      List<ParameterizedTypeRef> _switchResult = null;
      boolean _matched = false;
      if (declaredType instanceof TClass) {
        _matched=true;
        List<ParameterizedTypeRef> _xifexpression = null;
        if (((Objects.equal(declaredType, RuleEnvironmentExtensions.n4ObjectType(G)) || (((TClass)declaredType).isExternal() && (!((TClass)declaredType).isDeclaredN4JS()))) || 
          Objects.equal(((TClass)declaredType).getTypingStrategy(), TypingStrategy.STRUCTURAL))) {
          _xifexpression = RuleEnvironmentExtensions.getObjectPrototypesAllImplicitSuperTypeRefs(G);
        } else {
          List<ParameterizedTypeRef> _xifexpression_1 = null;
          ParameterizedTypeRef _superClassRef = ((TClass)declaredType).getSuperClassRef();
          boolean _tripleEquals = (_superClassRef == null);
          if (_tripleEquals) {
            _xifexpression_1 = RuleEnvironmentExtensions.getN4ClassifiersAllImplicitSuperTypeRefs(G);
          } else {
            _xifexpression_1 = RuleEnvironmentExtensions.collectAllImplicitSuperTypes(G, ((TClass)declaredType).getSuperClassRef(), guard);
          }
          _xifexpression = _xifexpression_1;
        }
        _switchResult = _xifexpression;
      }
      if (!_matched) {
        if (declaredType instanceof TN4Classifier) {
          _matched=true;
          _switchResult = RuleEnvironmentExtensions.getN4ClassifiersAllImplicitSuperTypeRefs(G);
        }
      }
      if (!_matched) {
        if (declaredType instanceof TObjectPrototype) {
          _matched=true;
          List<ParameterizedTypeRef> _xifexpression = null;
          TClassifier _objectType = RuleEnvironmentExtensions.objectType(G);
          boolean _equals = Objects.equal(declaredType, _objectType);
          if (_equals) {
            _xifexpression = CollectionLiterals.<ParameterizedTypeRef>emptyList();
          } else {
            _xifexpression = RuleEnvironmentExtensions.getObjectPrototypesAllImplicitSuperTypeRefs(G);
          }
          _switchResult = _xifexpression;
        }
      }
      if (!_matched) {
        if (declaredType instanceof TEnum) {
          _matched=true;
          List<ParameterizedTypeRef> _xifexpression = null;
          boolean _isStringBasedEnumeration = TypeSystemHelper.isStringBasedEnumeration(((TEnum)declaredType));
          if (_isStringBasedEnumeration) {
            ParameterizedTypeRef _n4StringBasedEnumTypeRef = RuleEnvironmentExtensions.n4StringBasedEnumTypeRef(G);
            _xifexpression = Collections.<ParameterizedTypeRef>unmodifiableList(CollectionLiterals.<ParameterizedTypeRef>newArrayList(_n4StringBasedEnumTypeRef));
          } else {
            ParameterizedTypeRef _objectTypeRef = RuleEnvironmentExtensions.objectTypeRef(G);
            _xifexpression = Collections.<ParameterizedTypeRef>unmodifiableList(CollectionLiterals.<ParameterizedTypeRef>newArrayList(_objectTypeRef));
          }
          _switchResult = _xifexpression;
        }
      }
      if (!_matched) {
        _switchResult = CollectionLiterals.<ParameterizedTypeRef>emptyList();
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns transitive, non-reflexive closure of implicit super types, delegates to
   * {@link #collectAllImplicitSuperTypesOfType((RuleEnvironment , Type )}.
   */
  public static List<ParameterizedTypeRef> _collectAllImplicitSuperTypes(final RuleEnvironment G, final TypeRef typeRef) {
    RecursionGuard<Type> _recursionGuard = new RecursionGuard<Type>();
    return RuleEnvironmentExtensions.collectAllImplicitSuperTypes(G, typeRef, _recursionGuard);
  }
  
  public static List<ParameterizedTypeRef> _collectAllImplicitSuperTypes(final RuleEnvironment G, final IntersectionTypeExpression typeRef) {
    RecursionGuard<Type> _recursionGuard = new RecursionGuard<Type>();
    return RuleEnvironmentExtensions.collectAllImplicitSuperTypes(G, typeRef, _recursionGuard);
  }
  
  public static List<ParameterizedTypeRef> _collectAllImplicitSuperTypes(final RuleEnvironment G, final FunctionTypeExprOrRef typeRef) {
    RecursionGuard<Type> _recursionGuard = new RecursionGuard<Type>();
    return RuleEnvironmentExtensions.collectAllImplicitSuperTypes(G, typeRef, _recursionGuard);
  }
  
  private static List<ParameterizedTypeRef> _collectAllImplicitSuperTypes(final RuleEnvironment G, final TypeRef typeRef, final RecursionGuard<Type> guard) {
    Type _declaredType = null;
    if (typeRef!=null) {
      _declaredType=typeRef.getDeclaredType();
    }
    return RuleEnvironmentExtensions.collectAllImplicitSuperTypesOfType(G, _declaredType, guard);
  }
  
  private static List<ParameterizedTypeRef> _collectAllImplicitSuperTypes(final RuleEnvironment G, final IntersectionTypeExpression typeRef, final RecursionGuard<Type> guard) {
    final Function1<TypeRef, List<ParameterizedTypeRef>> _function = (TypeRef it) -> {
      return RuleEnvironmentExtensions.collectAllImplicitSuperTypes(G, it, guard);
    };
    return IterableExtensions.<ParameterizedTypeRef>toList(Iterables.<ParameterizedTypeRef>concat(ListExtensions.<TypeRef, List<ParameterizedTypeRef>>map(typeRef.getTypeRefs(), _function)));
  }
  
  private static List<ParameterizedTypeRef> _collectAllImplicitSuperTypes(final RuleEnvironment G, final FunctionTypeExprOrRef typeRef, final RecursionGuard<Type> guard) {
    return RuleEnvironmentExtensions.getFunctionTypesAllImplicitSuperTypeRefs(G);
  }
  
  /**
   * returns an iterable of the assignment-compatible types, up to now only primitives have this concept.
   */
  public static Iterable<TypeRef> assignmentCompatibleTypes(final RuleEnvironment G, final TypeRef typeRef) {
    Iterable<TypeRef> _xblockexpression = null;
    {
      final Type declaredType = typeRef.getDeclaredType();
      Iterable<TypeRef> _switchResult = null;
      boolean _matched = false;
      if (declaredType instanceof PrimitiveType) {
        _matched=true;
        _switchResult = RuleEnvironmentExtensions.assignmentCompatibleTypes(G, ((PrimitiveType)declaredType));
      }
      if (!_matched) {
        _switchResult = CollectionLiterals.<TypeRef>emptyList();
      }
      _xblockexpression = _switchResult;
    }
    return _xblockexpression;
  }
  
  public static Iterable<TypeRef> assignmentCompatibleTypes(final RuleEnvironment G, final PrimitiveType pt) {
    List<TypeRef> _xifexpression = null;
    PrimitiveType _assignmentCompatible = pt.getAssignmentCompatible();
    boolean _tripleNotEquals = (_assignmentCompatible != null);
    if (_tripleNotEquals) {
      _xifexpression = ImmutableList.<TypeRef>of(TypeUtils.createTypeRef(pt.getAssignmentCompatible()));
    } else {
      _xifexpression = CollectionLiterals.<TypeRef>emptyList();
    }
    return _xifexpression;
  }
  
  /**
   * Returns unmodifiable list of type references to all function types (expressions and concrete functions):
   * {@code Function} and {@code Object}.
   */
  public static List<ParameterizedTypeRef> getFunctionTypesAllImplicitSuperTypeRefs(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getFunctionTypesAllImplicitSuperTypeRefs();
  }
  
  /**
   * Returns unmodifiable list of type references to all implicit super types of all built-in JavaScript object types,
   * object literals and via constructor created elements: {@code Object}.
   */
  public static List<ParameterizedTypeRef> getObjectPrototypesAllImplicitSuperTypeRefs(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getObjectPrototypesAllImplicitSuperTypeRefs();
  }
  
  /**
   * Returns unmodifiable list of type references to all implicit super types of all N4 classes, roles, and interfaces,
   * that is to {@code N4Object} and {@code Object}.
   */
  public static List<ParameterizedTypeRef> getN4ClassifiersAllImplicitSuperTypeRefs(final RuleEnvironment G) {
    return RuleEnvironmentExtensions.getPredefinedTypes(G).builtInTypeScope.getN4ClassifiersAllImplicitSuperTypeRefs();
  }
  
  public static String ruleEnvAsString(final RuleEnvironment G) {
    final String INDENT = "    ";
    final StringBuffer result = new StringBuffer();
    result.append("RuleEnvironment@");
    result.append(Integer.toHexString(System.identityHashCode(G)));
    result.append(" {\n");
    result.append(RuleEnvironmentExtensions.typeVariableSubstitutionsAsString(G.getEnvironment(), INDENT));
    RuleEnvironment _next = G.getNext();
    boolean _tripleNotEquals = (_next != null);
    if (_tripleNotEquals) {
      String _replaceAll = RuleEnvironmentExtensions.ruleEnvAsString(G.getNext()).replaceAll("\\n", ("\n" + INDENT));
      String _plus = (INDENT + _replaceAll);
      result.append(_plus);
    }
    result.append("}");
    return result.toString();
  }
  
  protected static String typeVariableSubstitutionsAsString(final Map<?, ?> substitutions, final String indent) {
    final ArrayList<String> pairs = CollectionLiterals.<String>newArrayList();
    Set<?> _keySet = substitutions.keySet();
    for (final Object currKey : _keySet) {
      String _typeRefOrVariableAsString = RuleEnvironmentExtensions.typeRefOrVariableAsString(currKey);
      String _plus = (indent + _typeRefOrVariableAsString);
      String _plus_1 = (_plus + " -> ");
      String _typeRefOrVariableAsString_1 = RuleEnvironmentExtensions.typeRefOrVariableAsString(substitutions.get(currKey));
      String _plus_2 = (_plus_1 + _typeRefOrVariableAsString_1);
      String _plus_3 = (_plus_2 + "\n");
      pairs.add(_plus_3);
    }
    Collections.<String>sort(pairs);
    return IterableExtensions.join(pairs);
  }
  
  protected static String typeRefOrVariableAsString(final Object obj) {
    String _xifexpression = null;
    if ((obj instanceof Collection<?>)) {
      final Function1<Object, String> _function = (Object it) -> {
        return RuleEnvironmentExtensions.typeRefOrVariableAsString(it);
      };
      String _join = IterableExtensions.join(IterableExtensions.map(((Iterable<?>)obj), _function), ", ");
      String _plus = ("[ " + _join);
      _xifexpression = (_plus + " ]");
    } else {
      String _xifexpression_1 = null;
      if ((obj instanceof TypeVariable)) {
        String _xblockexpression = null;
        {
          final EObject parent = ((TypeVariable)obj).eContainer();
          String _xifexpression_2 = null;
          if ((parent instanceof IdentifiableElement)) {
            String _name = ((IdentifiableElement)parent).getName();
            String _plus_1 = (_name + "#");
            String _name_1 = ((TypeVariable)obj).getName();
            _xifexpression_2 = (_plus_1 + _name_1);
          } else {
            String _xifexpression_3 = null;
            if (((parent != null) && (parent.eClass() != null))) {
              String _name_2 = parent.eClass().getName();
              String _plus_2 = (_name_2 + "#");
              String _name_3 = ((TypeVariable)obj).getName();
              _xifexpression_3 = (_plus_2 + _name_3);
            } else {
              String _name_4 = ((TypeVariable)obj).getName();
              _xifexpression_3 = ("#" + _name_4);
            }
            _xifexpression_2 = _xifexpression_3;
          }
          _xblockexpression = _xifexpression_2;
        }
        _xifexpression_1 = _xblockexpression;
      } else {
        String _xifexpression_2 = null;
        if (((obj instanceof TypeRef) && (((TypeRef) obj).getDeclaredType() instanceof TypeVariable))) {
          _xifexpression_2 = RuleEnvironmentExtensions.typeRefOrVariableAsString(((TypeRef) obj).getDeclaredType());
        } else {
          String _xifexpression_3 = null;
          if ((obj instanceof TypeRef)) {
            _xifexpression_3 = ((TypeRef)obj).getTypeRefAsString();
          } else {
            _xifexpression_3 = obj.toString();
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }
  
  /**
   * Check if {@code locationToCheck} is contained in the return part of {@code container}.
   */
  public static boolean isInReturnDeclaration_Of_StaticMethod(final EObject locationToCheck, final N4MethodDeclaration container) {
    boolean _isStatic = container.isStatic();
    boolean _not = (!_isStatic);
    if (_not) {
      return false;
    }
    final boolean isInReturn = EcoreUtil2.isAncestor(container.getReturnTypeRef(), locationToCheck);
    return isInReturn;
  }
  
  /**
   * Check if {@code locationToCheck} is contained in the body part of {@code container}.
   */
  public static boolean isInBody_Of_StaticMethod(final EObject locationToCheck, final N4MethodDeclaration container) {
    boolean _isStatic = container.isStatic();
    boolean _not = (!_isStatic);
    if (_not) {
      return false;
    }
    final boolean isInBody = EcoreUtil2.isAncestor(container.getBody(), locationToCheck);
    return isInBody;
  }
  
  public static List<ParameterizedTypeRef> collectAllImplicitSuperTypes(final RuleEnvironment G, final TypeRef typeRef) {
    if (typeRef instanceof IntersectionTypeExpression) {
      return _collectAllImplicitSuperTypes(G, (IntersectionTypeExpression)typeRef);
    } else if (typeRef instanceof FunctionTypeExprOrRef) {
      return _collectAllImplicitSuperTypes(G, (FunctionTypeExprOrRef)typeRef);
    } else if (typeRef != null) {
      return _collectAllImplicitSuperTypes(G, typeRef);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(G, typeRef).toString());
    }
  }
  
  private static List<ParameterizedTypeRef> collectAllImplicitSuperTypes(final RuleEnvironment G, final TypeRef typeRef, final RecursionGuard<Type> guard) {
    if (typeRef instanceof IntersectionTypeExpression) {
      return _collectAllImplicitSuperTypes(G, (IntersectionTypeExpression)typeRef, guard);
    } else if (typeRef instanceof FunctionTypeExprOrRef) {
      return _collectAllImplicitSuperTypes(G, (FunctionTypeExprOrRef)typeRef, guard);
    } else if (typeRef != null) {
      return _collectAllImplicitSuperTypes(G, typeRef, guard);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(G, typeRef, guard).toString());
    }
  }
}
