/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.utils.TypeCompareHelper;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelperStrategy;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Type System Helper Strategy for creating simplified composed types, i.e. union
 * and intersection types.
 */
@SuppressWarnings("all")
public class SimplifyComputer extends TypeSystemHelperStrategy {
  @Inject
  @Extension
  private TypeCompareHelper _typeCompareHelper;
  
  /**
   * Creates a simplified union type containing the given types; since it is simplified, the result is not necessarily a union type.
   * The result may be contained in another container, so clients may have to use Ecore2.cloneIfNecessary(EObject).
   */
  public TypeRef createUnionType(final RuleEnvironment G, final TypeRef... elements) {
    return this.<UnionTypeExpression>simplify(G, TypeUtils.createNonSimplifiedUnionType(elements));
  }
  
  /**
   * Creates a simplified intersection type containing the given types; since it is simplified, the result is not necessarily an intersection type.
   * The result may be contained in another container, so clients may have to use Ecore2.cloneIfNecessary(EObject).
   */
  public TypeRef createIntersectionType(final RuleEnvironment G, final TypeRef... elements) {
    return this.<IntersectionTypeExpression>simplify(G, TypeUtils.createNonSimplifiedIntersectionType(elements));
  }
  
  /**
   * Returns a simplified copy of a given composed type, i.e. union or intersection type.
   * The returned type may be one of the elements, without cloning it.
   * So clients need to clone the result if necessary.
   * @see [N4JS Spec], 4.13 Intersection Type
   */
  public <T extends ComposedTypeRef> TypeRef simplify(final RuleEnvironment G, final T composedType) {
    final List<TypeRef> tRs = this.<T>getSimplifiedTypeRefs(G, composedType, true);
    final EClass eClass = composedType.eClass();
    EObject _create = EcoreUtil.create(eClass);
    final ComposedTypeRef simplified = ((ComposedTypeRef) _create);
    final EList<TypeRef> typeRefs = simplified.getTypeRefs();
    typeRefs.addAll(tRs);
    int _size = typeRefs.size();
    switch (_size) {
      case 0:
        return RuleEnvironmentExtensions.undefinedTypeRef(G);
      case 1:
        return IterableExtensions.<TypeRef>head(typeRefs);
      default:
        return simplified;
    }
  }
  
  /**
   * Returns a simplified list of all TypeRefs of a composed type.
   */
  public <T extends ComposedTypeRef> List<TypeRef> getSimplifiedTypeRefs(final RuleEnvironment G, final T composedType) {
    return this.<T>getSimplifiedTypeRefs(G, composedType, false);
  }
  
  private <T extends ComposedTypeRef> List<TypeRef> getSimplifiedTypeRefs(final RuleEnvironment G, final T composedType, final boolean copyIfContained) {
    if ((composedType == null)) {
      return null;
    }
    final EClass eClass = composedType.eClass();
    Comparator<TypeRef> _typeRefComparator = this._typeCompareHelper.getTypeRefComparator();
    final Set<TypeRef> set = new TreeSet<TypeRef>(_typeRefComparator);
    final Function1<TypeRef, Boolean> _function = (TypeRef it) -> {
      return Boolean.valueOf(eClass.isInstance(it));
    };
    boolean _exists = IterableExtensions.<TypeRef>exists(composedType.getTypeRefs(), _function);
    if (_exists) {
      Iterables.<TypeRef>addAll(set, this.flattenComposedTypes(eClass, composedType));
    } else {
      set.addAll(composedType.getTypeRefs());
    }
    final List<TypeRef> typeRefs = new LinkedList<TypeRef>();
    final ParameterizedTypeRef undefinedTypeRef = RuleEnvironmentExtensions.undefinedTypeRef(G);
    final ParameterizedTypeRef nullTypeRef = RuleEnvironmentExtensions.nullTypeRef(G);
    for (final TypeRef e : set) {
      if (((this._typeCompareHelper.compare(e, undefinedTypeRef) != 0) && 
        (this._typeCompareHelper.compare(e, nullTypeRef) != 0))) {
        TypeRef cpy = e;
        if (copyIfContained) {
          cpy = TypeUtils.<TypeRef>copyIfContained(e);
        }
        typeRefs.add(cpy);
      }
    }
    return typeRefs;
  }
  
  private Iterable<TypeRef> flattenComposedTypes(final EClass eClass, final TypeRef typeRef) {
    Iterable<TypeRef> _xifexpression = null;
    boolean _isInstance = eClass.isInstance(typeRef);
    if (_isInstance) {
      final Function1<TypeRef, Iterable<TypeRef>> _function = (TypeRef it) -> {
        return this.flattenComposedTypes(eClass, it);
      };
      _xifexpression = Iterables.<TypeRef>concat(ListExtensions.<TypeRef, Iterable<TypeRef>>map(((ComposedTypeRef) typeRef).getTypeRefs(), _function));
    } else {
      _xifexpression = Collections.<TypeRef>singleton(typeRef);
    }
    return _xifexpression;
  }
}
