/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.typesystem;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.InferenceVariable;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.VoidType;
import org.eclipse.n4js.ts.types.util.Variance;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.typesystem.TypeSystemHelperStrategy;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.service.OperationCanceledManager;
import org.eclipse.xtext.util.CancelIndicator;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;

/**
 * Contains some helper methods to compute if type A is a subtype of type B.
 * Note that the main logic for subtype computation is contained in file
 * n4js.xsemantics. For structural typing there is a separate helper class
 * called {@link StructuralTypingComputer}.
 */
@Singleton
@SuppressWarnings("all")
public class SubtypeComputer extends TypeSystemHelperStrategy {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private OperationCanceledManager operationCanceledManager;
  
  /**
   * Returns true iff function/method 'left' is a subtype of function/method 'right'.
   */
  public boolean isSubtypeFunction(final RuleEnvironment G, final FunctionTypeExprOrRef left, final FunctionTypeExprOrRef right) {
    final EList<TypeVariable> leftTypeVars = left.getTypeVars();
    final EList<TypeVariable> rightTypeVars = right.getTypeVars();
    if ((leftTypeVars.isEmpty() && rightTypeVars.isEmpty())) {
      return this.primIsSubtypeFunction(G, left, right);
    } else {
      if (((!leftTypeVars.isEmpty()) && rightTypeVars.isEmpty())) {
        CancelIndicator _cancelIndicator = RuleEnvironmentExtensions.getCancelIndicator(G);
        final InferenceContext infCtx = new InferenceContext(this.ts, this.tsh, this.operationCanceledManager, _cancelIndicator, G);
        final FunctionTypeExprOrRef left_withInfVars = infCtx.newInferenceVariablesFor(left);
        infCtx.addConstraint(left_withInfVars, right, Variance.CO);
        final Map<InferenceVariable, TypeRef> solution = infCtx.solve();
        if ((solution != null)) {
          final RuleEnvironment G_solution = RuleEnvironmentExtensions.newRuleEnvironment(G);
          final Consumer<Map.Entry<InferenceVariable, TypeRef>> _function = (Map.Entry<InferenceVariable, TypeRef> it) -> {
            RuleEnvironmentExtensions.addTypeMapping(G_solution, it.getKey(), it.getValue());
          };
          solution.entrySet().forEach(_function);
          final TypeRef leftSubst = this.ts.substTypeVariablesInTypeRef(G_solution, left_withInfVars);
          if ((leftSubst instanceof FunctionTypeExprOrRef)) {
            return this.primIsSubtypeFunction(G, ((FunctionTypeExprOrRef)leftSubst), right);
          }
        }
        return false;
      } else {
        int _size = leftTypeVars.size();
        int _size_1 = rightTypeVars.size();
        boolean _tripleNotEquals = (_size != _size_1);
        if (_tripleNotEquals) {
          return false;
        }
        final RuleEnvironment G2 = RuleEnvironmentExtensions.wrap(G);
        int _size_2 = leftTypeVars.size();
        ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _size_2, true);
        for (final Integer i : _doubleDotLessThan) {
          RuleEnvironmentExtensions.addTypeMapping(G2, rightTypeVars.get((i).intValue()), TypeUtils.createTypeRef(leftTypeVars.get((i).intValue())));
        }
        final TypeRef rightSubst = this.ts.substTypeVariablesInTypeRef(G2, right);
        boolean _not = (!((rightSubst instanceof FunctionTypeExprOrRef) && 
          this.primIsSubtypeFunction(G, left, ((FunctionTypeExprOrRef) rightSubst))));
        if (_not) {
          return false;
        }
        Type _declaredType = left.getDeclaredType();
        EObject _eContainer = null;
        if (_declaredType!=null) {
          _eContainer=_declaredType.eContainer();
        }
        if ((_eContainer instanceof TClassifier)) {
          EObject _eContainer_1 = left.getDeclaredType().eContainer();
          this._typeSystemHelper.addSubstitutions(G2, TypeUtils.createTypeRef(((TClassifier) _eContainer_1)));
        }
        return this.isMatchingTypeVariableBounds(G2, leftTypeVars, rightTypeVars);
      }
    }
  }
  
  /**
   * Contains the core logic for subtype relation of functions/methods but <em>without</em>
   * taking into account type variables of generic functions/methods. Generic functions are handled
   * in method {@link #isSubtypeFunction(RuleEnvironment,FunctionTypeExprOrRef,FunctionTypeExprOrRef)}.
   */
  private boolean primIsSubtypeFunction(final RuleEnvironment G, final FunctionTypeExprOrRef left, final FunctionTypeExprOrRef right) {
    final TypeRef leftReturnTypeRef = left.getReturnTypeRef();
    final TypeRef rightReturnTypeRef = right.getReturnTypeRef();
    if ((rightReturnTypeRef != null)) {
      Type _declaredType = rightReturnTypeRef.getDeclaredType();
      VoidType _voidType = RuleEnvironmentExtensions.voidType(G);
      boolean _tripleNotEquals = (_declaredType != _voidType);
      if (_tripleNotEquals) {
        final Type rightFunType = right.getDeclaredType();
        boolean _xifexpression = false;
        if ((rightFunType instanceof TFunction)) {
          _xifexpression = ((TFunction)rightFunType).isReturnValueOptional();
        } else {
          _xifexpression = right.isReturnValueOptional();
        }
        final boolean isRightReturnOptional = _xifexpression;
        Type _declaredType_1 = leftReturnTypeRef.getDeclaredType();
        VoidType _voidType_1 = RuleEnvironmentExtensions.voidType(G);
        boolean _tripleNotEquals_1 = (_declaredType_1 != _voidType_1);
        if (_tripleNotEquals_1) {
          if ((left.isReturnValueOptional() && (!isRightReturnOptional))) {
            return false;
          } else {
            boolean _isSubtype = this.isSubtype(G, leftReturnTypeRef, rightReturnTypeRef);
            boolean _not = (!_isSubtype);
            if (_not) {
              return false;
            }
          }
        } else {
          if (((!isRightReturnOptional) && (!this.ts.equaltypeSucceeded(G, rightReturnTypeRef, RuleEnvironmentExtensions.undefinedTypeRef(G))))) {
            return false;
          }
        }
      }
    }
    final int k = left.getFpars().size();
    final int n = right.getFpars().size();
    if ((k <= n)) {
      if ((k > 0)) {
        int i = 0;
        while ((i < k)) {
          {
            final TFormalParameter R = right.getFpars().get(i);
            final TFormalParameter L = left.getFpars().get(i);
            if (((R.isVariadic() || R.isOptional()) && (!(L.isOptional() || L.isVariadic())))) {
              return false;
            }
            boolean _isSubtype_1 = this.isSubtype(G, R.getTypeRef(), L.getTypeRef());
            boolean _not_1 = (!_isSubtype_1);
            if (_not_1) {
              return false;
            }
            i = (i + 1);
          }
        }
        final TFormalParameter L = left.getFpars().get((k - 1));
        boolean _isVariadic = L.isVariadic();
        if (_isVariadic) {
          while ((i < n)) {
            {
              final TFormalParameter R = right.getFpars().get(i);
              boolean _isSubtype_1 = this.isSubtype(G, R.getTypeRef(), L.getTypeRef());
              boolean _not_1 = (!_isSubtype_1);
              if (_not_1) {
                return false;
              }
              i = (i + 1);
            }
          }
        }
      }
    } else {
      int i_1 = 0;
      while ((i_1 < n)) {
        {
          final TFormalParameter R = right.getFpars().get(i_1);
          final TFormalParameter L_1 = left.getFpars().get(i_1);
          if (((R.isVariadic() || R.isOptional()) && (!(L_1.isOptional() || L_1.isVariadic())))) {
            return false;
          }
          boolean _isSubtype_1 = this.isSubtype(G, R.getTypeRef(), L_1.getTypeRef());
          boolean _not_1 = (!_isSubtype_1);
          if (_not_1) {
            return false;
          }
          i_1 = (i_1 + 1);
        }
      }
      TFormalParameter _xifexpression_1 = null;
      if ((n > 0)) {
        _xifexpression_1 = right.getFpars().get((n - 1));
      } else {
        _xifexpression_1 = null;
      }
      final TFormalParameter R = _xifexpression_1;
      while ((i_1 < k)) {
        {
          final TFormalParameter L_1 = left.getFpars().get(i_1);
          boolean _not_1 = (!(L_1.isOptional() || L_1.isVariadic()));
          if (_not_1) {
            return false;
          }
          if (((R != null) && R.isVariadic())) {
            boolean _isSubtype_1 = this.isSubtype(G, R.getTypeRef(), L_1.getTypeRef());
            boolean _not_2 = (!_isSubtype_1);
            if (_not_2) {
              return false;
            }
          }
          i_1 = (i_1 + 1);
        }
      }
    }
    final TypeRef rThis = right.getDeclaredThisType();
    final TypeRef lThis = left.getDeclaredThisType();
    if ((rThis != null)) {
      return ((lThis == null) || this.isSubtype(G, rThis, lThis));
    } else {
      if ((lThis != null)) {
        return false;
      }
    }
    return true;
  }
  
  /**
   * Checks bounds of type variables in left and right.
   * Upper bound on left side must be a super type of upper bound on right side.
   */
  private boolean isMatchingTypeVariableBounds(final RuleEnvironment G, final List<TypeVariable> left, final List<TypeVariable> right) {
    for (int i = 0; (i < right.size()); i++) {
      {
        final TypeVariable leftTypeVar = left.get(i);
        final TypeVariable rightTypeVar = right.get(i);
        final TypeRef leftDeclUB = leftTypeVar.getDeclaredUpperBound();
        final TypeRef rightDeclUB = rightTypeVar.getDeclaredUpperBound();
        TypeRef _xifexpression = null;
        if ((leftDeclUB == null)) {
          _xifexpression = N4JSLanguageUtils.getTypeVariableImplicitUpperBound(G);
        } else {
          _xifexpression = leftDeclUB;
        }
        final TypeRef leftUpperBound = _xifexpression;
        TypeRef _xifexpression_1 = null;
        if ((rightDeclUB == null)) {
          _xifexpression_1 = N4JSLanguageUtils.getTypeVariableImplicitUpperBound(G);
        } else {
          _xifexpression_1 = rightDeclUB;
        }
        final TypeRef rightUpperBound = _xifexpression_1;
        final TypeRef rightUpperBoundSubst = this.ts.substTypeVariablesInTypeRef(G, rightUpperBound);
        boolean _isSubtype = this.isSubtype(G, rightUpperBoundSubst, leftUpperBound);
        boolean _not = (!_isSubtype);
        if (_not) {
          return false;
        }
      }
    }
    return true;
  }
  
  private boolean isSubtype(final RuleEnvironment G, final TypeArgument left, final TypeArgument right) {
    Boolean _elvis = null;
    Boolean _value = this.ts.subtype(G, left, right).getValue();
    if (_value != null) {
      _elvis = _value;
    } else {
      _elvis = Boolean.valueOf(false);
    }
    return (boolean) _elvis;
  }
}
