/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.DestructNode;
import org.eclipse.n4js.n4JS.DestructureUtils;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ForStatement;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.VariableBinding;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.scoping.accessModifiers.VisibilityAwareMemberScope;
import org.eclipse.n4js.scoping.members.MemberScopingHelper;
import org.eclipse.n4js.scoping.utils.AbstractDescriptionWithError;
import org.eclipse.n4js.ts.conversions.ComputedPropertyNameValueConverter;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.typeRefs.UnionTypeExpression;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.InferenceVariable;
import org.eclipse.n4js.ts.types.PrimitiveType;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TStructField;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypesFactory;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.util.AllSuperTypeRefsCollector;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.typesystem.constraints.InferenceContext;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Helper for dealing with destructuring patterns. For more details on destructuring patterns,
 * see documentation of class {@link DestructNode}.
 */
@Singleton
@SuppressWarnings("all")
public class DestructureHelper {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private MemberScopingHelper memberScopingHelper;
  
  /**
   * Infers the type of a variable declaration within a destructuring pattern from the value to be
   * destructured and/or the default value given in the pattern.
   * <p>
   * Returns <code>null</code> if <code>vdecl</code> has an explicitly declared type or in case of error.
   */
  public TypeRef getTypeOfVariableDeclarationInDestructuringPattern(final RuleEnvironment G, final VariableDeclaration vdecl) {
    TypeRef _declaredTypeRef = vdecl.getDeclaredTypeRef();
    boolean _tripleNotEquals = (_declaredTypeRef != null);
    if (_tripleNotEquals) {
      return null;
    }
    final EObject root = DestructureUtils.getRoot(vdecl.eContainer());
    if ((root == null)) {
      return null;
    }
    final EObject rootParent = root.eContainer();
    if ((rootParent instanceof VariableBinding)) {
      final EObject rootParent2 = ((VariableBinding)rootParent).eContainer();
      final boolean isLocatedUnderForInOf = ((rootParent2 instanceof ForStatement) && DestructureUtils.isTopOfDestructuringForStatement(rootParent2));
      DestructNode _xifexpression = null;
      if (isLocatedUnderForInOf) {
        _xifexpression = DestructNode.unify(((ForStatement) rootParent2));
      } else {
        _xifexpression = DestructNode.unify(((VariableBinding)rootParent));
      }
      final DestructNode rootNode = _xifexpression;
      if ((rootNode != null)) {
        final HashMap<DestructNode, TypeRef> valueTypePerNode = CollectionLiterals.<DestructNode, TypeRef>newHashMap();
        this.buildValueTypesMap(G, rootNode, valueTypePerNode, ((VariableBinding)rootParent).getPattern());
        final Predicate<DestructNode> _function = (DestructNode it) -> {
          VariableDeclaration _variableDeclaration = it.getVariableDeclaration();
          return (_variableDeclaration == vdecl);
        };
        final DestructNode node = rootNode.stream().filter(_function).findFirst().orElse(null);
        return valueTypePerNode.get(node);
      }
    }
    return null;
  }
  
  /**
   * For a unified destructuring pattern created with one of the unify() methods in {@link DestructNode}, this method will
   * return a mapping of the pattern's nodes to their value type, i.e. to the type of their value to be destructured.
   * The returned map might not contain key/value pairs for all nodes (in case of error) but won't contain <code>null</code>
   * values.
   * 
   * @param G  rule environment (used for type inference).
   * @param rootNode  root of the destructuring pattern.
   * @param addHere  types for each node will be added here.
   * @param contextObj  context object (used for member scoping and obtaining the containing resource or resource set).
   */
  public void buildValueTypesMap(final RuleEnvironment G, final DestructNode rootNode, final Map<DestructNode, TypeRef> addHere, final EObject contextObj) {
    Expression _defaultExpr = null;
    if (rootNode!=null) {
      _defaultExpr=rootNode.getDefaultExpr();
    }
    boolean _tripleEquals = (_defaultExpr == null);
    if (_tripleEquals) {
      return;
    }
    TypeRef valueTypeRef = this.ts.type(G, rootNode.getDefaultExpr()).getValue();
    valueTypeRef = this.ts.resolveType(G, valueTypeRef);
    EObject _eContainer = rootNode.getDefaultExpr().eContainer();
    if ((_eContainer instanceof ForStatement)) {
      valueTypeRef = this.extractIterableElementTypeUB(G, valueTypeRef);
    }
    if ((valueTypeRef != null)) {
      addHere.put(rootNode, valueTypeRef);
      final DestructNode[] nestedNodes = rootNode.getNestedNodes();
      if (((nestedNodes != null) && (!((List<DestructNode>)Conversions.doWrapArray(nestedNodes)).isEmpty()))) {
        this.buildValueTypesMap(G, nestedNodes, valueTypeRef, addHere, contextObj);
      }
    }
  }
  
  private void buildValueTypesMap(final RuleEnvironment G, final DestructNode[] nodes, final TypeRef valueTypeRef, final Map<DestructNode, TypeRef> addHere, final EObject contextObj) {
    boolean _arePositional = DestructNode.arePositional(nodes);
    if (_arePositional) {
      final List<TypeRef> elementTypeRefs = IterableExtensions.<TypeRef>toList(this.extractIterableElementTypesUBs(G, valueTypeRef));
      boolean _isEmpty = elementTypeRefs.isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        int _size = elementTypeRefs.size();
        final int maxIdx = (_size - 1);
        for (int idx = 0; (idx < ((List<DestructNode>)Conversions.doWrapArray(nodes)).size()); idx++) {
          {
            final DestructNode currNode = nodes[idx];
            final TypeRef currValueTypeRef = elementTypeRefs.get(Math.min(idx, maxIdx));
            this.addTypeAndContinueWithChildren(G, currNode, currValueTypeRef, addHere, contextObj);
          }
        }
      }
    } else {
      final IScope memberScope = this.createMemberScopeForPropertyAccess(valueTypeRef, contextObj, false);
      for (final DestructNode currNode : nodes) {
        {
          final TypeRef currValueTypeRef = this.getPropertyTypeForNode(G, valueTypeRef, memberScope, currNode.getPropName(), null);
          if ((currValueTypeRef != null)) {
            this.addTypeAndContinueWithChildren(G, currNode, currValueTypeRef, addHere, contextObj);
          }
        }
      }
    }
  }
  
  /**
   * Create a new member scope for use with method {@link #getPropertyTypeForNode(RuleEnvironment, TypeRef, IScope,
   * String, StringBuffer) #getPropertyTypeForNode()}. Do not use the scope returned by this method for any other
   * purpose (use methods in {@link MemberScopingHelper} instead)!
   * <p>
   * This is only provided as a separate method to avoid creating the same member scope over and over in case of
   * multiple invocations of {@code #getPropertyTypeForNode()}.
   * 
   * @param valueTypeRef
   *               type of the value to be destructured.
   * @param contextObj
   *               context object used for (a) obtaining context resource and (b) visibility checking.
   * @param checkVisibility
   *               if true, the member scope will be wrapped in a {@link VisibilityAwareMemberScope}; if
   *               false, method {@link getPropertyTypeForNode(IScope,String)} will <b>never</b> return
   *               {@link #INVISIBLE_MEMBER}.
   */
  public IScope createMemberScopeForPropertyAccess(final TypeRef receiverTypeRef, final EObject contextObj, final boolean checkVisibility) {
    TypingStrategy _typingStrategy = receiverTypeRef.getTypingStrategy();
    final boolean structFieldInitMode = (_typingStrategy == TypingStrategy.STRUCTURAL_FIELD_INITIALIZER);
    return this.memberScopingHelper.createMemberScopeAllowingNonContainedMembers(receiverTypeRef, contextObj, checkVisibility, false, structFieldInitMode);
  }
  
  /**
   * Returns type of a property within an object destructuring pattern or <code>null</code> if property does not exist.
   * In case the property exists but is not available (e.g. not visible), an error message is appended to given
   * StringBuffer 'errorMessage' (optional).
   * 
   * @param parentValueTypeRef
   *              value type of the parent node.
   * @param parentMemberScope
   *              a member scope as returned by method {@link #createMemberScopeForPropertyAccess(TypeRef,EObject,boolean)}.
   * @param propName
   *              name of property to look up.
   * @param errorMessage
   *              a string buffer where the error message will be stored in case the property exists but is not readable
   *              or <code>null</code> if the caller is not interested in receiving error messages.
   */
  public TypeRef getPropertyTypeForNode(final RuleEnvironment G, final TypeRef parentValueTypeRef, final IScope parentMemberScope, final String propName, final StringBuffer errorMessage) {
    if (((parentValueTypeRef == null) || (parentMemberScope == null))) {
      return null;
    }
    final IEObjectDescription mDesc = parentMemberScope.getSingleElement(QualifiedName.create(propName));
    if ((mDesc instanceof AbstractDescriptionWithError)) {
      if ((errorMessage != null)) {
        errorMessage.append(((AbstractDescriptionWithError)mDesc).getMessage());
      }
    }
    EObject _eObjectOrProxy = null;
    if (mDesc!=null) {
      _eObjectOrProxy=mDesc.getEObjectOrProxy();
    }
    final EObject m = _eObjectOrProxy;
    if (((m != null) && (!m.eIsProxy()))) {
      TypeRef _switchResult = null;
      boolean _matched = false;
      if (m instanceof TField) {
        _matched=true;
        _switchResult = ((TField)m).getTypeRef();
      }
      if (!_matched) {
        if (m instanceof TGetter) {
          _matched=true;
          _switchResult = ((TGetter)m).getDeclaredTypeRef();
        }
      }
      final TypeRef result = _switchResult;
      if ((result != null)) {
        final RuleEnvironment G2 = RuleEnvironmentExtensions.wrap(G);
        this.tsh.addSubstitutions(G2, parentValueTypeRef);
        final TypeArgument resultSubst = this.ts.substTypeVariables(G2, result).getValue();
        TypeRef _xifexpression = null;
        if ((resultSubst != null)) {
          _xifexpression = this.ts.upperBound(G2, resultSubst).getValue();
        }
        final TypeRef resultSubstUB = _xifexpression;
        return resultSubstUB;
      }
    }
    return null;
  }
  
  /**
   * Following code factored out from #buildTypesMap(), because it is common to the positional
   * and non-positional case. IMPORTANT: this method must also be called if valueTypeRef===null,
   * because there might be a default expression in 'currNode' that provides a type.
   */
  private void addTypeAndContinueWithChildren(final RuleEnvironment G, final DestructNode currNode, final TypeRef valueTypeRef, final Map<DestructNode, TypeRef> addHere, final EObject contextObj) {
    TypeRef _xifexpression = null;
    boolean _isRest = currNode.isRest();
    if (_isRest) {
      _xifexpression = RuleEnvironmentExtensions.arrayTypeRef(G, valueTypeRef);
    } else {
      _xifexpression = valueTypeRef;
    }
    final TypeRef actualValueTypeRef = _xifexpression;
    final TypeRef currTypeRef = this.mergeWithTypeOfDefaultExpression(G, actualValueTypeRef, currNode);
    if ((currTypeRef != null)) {
      addHere.put(currNode, currTypeRef);
      if (((currNode.getNestedNodes() != null) && (!((List<DestructNode>)Conversions.doWrapArray(currNode.getNestedNodes())).isEmpty()))) {
        this.buildValueTypesMap(G, currNode.getNestedNodes(), currTypeRef, addHere, contextObj);
      }
    }
  }
  
  /**
   * Infers type of the default expression of 'currNode' and merges it with the given valueTypeRef.
   * Both the given value type and inferred expression type may be null and then this returns null.
   */
  private TypeRef mergeWithTypeOfDefaultExpression(final RuleEnvironment G, final TypeRef valueTypeRef, final DestructNode node) {
    TypeRef _xifexpression = null;
    Expression _defaultExpr = node.getDefaultExpr();
    boolean _tripleNotEquals = (_defaultExpr != null);
    if (_tripleNotEquals) {
      _xifexpression = this.ts.type(G, node.getDefaultExpr()).getValue();
    }
    final TypeRef exprTypeRefRaw = _xifexpression;
    boolean _xifexpression_1 = false;
    if ((exprTypeRefRaw != null)) {
      _xifexpression_1 = (this.ts.subtypeSucceeded(G, exprTypeRefRaw, RuleEnvironmentExtensions.undefinedTypeRef(G)) || this.ts.subtypeSucceeded(G, exprTypeRefRaw, RuleEnvironmentExtensions.nullTypeRef(G)));
    }
    final boolean isNullOrUndef = _xifexpression_1;
    TypeRef _xifexpression_2 = null;
    if (((exprTypeRefRaw != null) && (!isNullOrUndef))) {
      _xifexpression_2 = this.ts.upperBound(G, exprTypeRefRaw).getValue();
    }
    final TypeRef exprTypeRef = _xifexpression_2;
    if (((valueTypeRef != null) && (exprTypeRef != null))) {
      TypeRef _xifexpression_3 = null;
      boolean _subtypeSucceeded = this.ts.subtypeSucceeded(G, valueTypeRef, exprTypeRef);
      if (_subtypeSucceeded) {
        _xifexpression_3 = exprTypeRef;
      } else {
        TypeRef _xifexpression_4 = null;
        boolean _subtypeSucceeded_1 = this.ts.subtypeSucceeded(G, exprTypeRef, valueTypeRef);
        if (_subtypeSucceeded_1) {
          _xifexpression_4 = valueTypeRef;
        } else {
          _xifexpression_4 = this.tsh.createUnionType(G, valueTypeRef, exprTypeRef);
        }
        _xifexpression_3 = _xifexpression_4;
      }
      return _xifexpression_3;
    } else {
      if ((valueTypeRef != null)) {
        return valueTypeRef;
      } else {
        if ((exprTypeRef != null)) {
          return exprTypeRef;
        }
      }
    }
    return null;
  }
  
  /**
   * Same as {@link #extractIterableElementTypes(RuleEnvironment,TypeRef)}, but returns the upper bounds.
   */
  public Iterable<TypeRef> extractIterableElementTypesUBs(final RuleEnvironment G, final TypeRef typeRef) {
    final Function1<TypeArgument, TypeRef> _function = (TypeArgument it) -> {
      return this.ts.upperBound(G, it).getValue();
    };
    return IterableExtensions.map(this.extractIterableElementTypes(G, typeRef), _function);
  }
  
  /**
   * Given a type that is or (directly or indirectly) implements one of the Iterable or IterableN built-in types,
   * this method will return the type of the first N elements returned by the Iterable's iterator. The last returned
   * type will be the type of all remaining elements (if any).
   * 
   * Never returns <code>null</code> but may return an empty result if 'typeRef' does not implement any of the
   * Iterable or IterableN interfaces. Usually never returns a result longer than {@link BuiltInTypeScope#ITERABLE_N__MAX_LEN},
   * but if there are invalid type references with too many arguments, this might happen.
   */
  public Iterable<? extends TypeArgument> extractIterableElementTypes(final RuleEnvironment G, final TypeRef typeRef) {
    return this.extractIterableElementTypes(G, typeRef, true);
  }
  
  /**
   * Same as {@link #extractIterableElementType(RuleEnvironment,TypeRef)}, but returns the upper bound.
   */
  public TypeRef extractIterableElementTypeUB(final RuleEnvironment G, final TypeRef typeRef) {
    final Function1<TypeRef, TypeRef> _function = (TypeRef it) -> {
      return this.ts.upperBound(G, it).getValue();
    };
    return IterableExtensions.<TypeRef>head(IterableExtensions.map(this.extractIterableElementTypes(G, typeRef, false), _function));
  }
  
  /**
   * Given a type that is or (directly or indirectly) implements the Iterable built-in types, this method will
   * return the type of the elements returned by the Iterable's iterator.
   * 
   * Returns <code>null</code> if 'typeRef' does not implement Iterable.
   */
  public TypeArgument extractIterableElementType(final RuleEnvironment G, final TypeRef typeRef) {
    return IterableExtensions.head(this.extractIterableElementTypes(G, typeRef, false));
  }
  
  /**
   * Return the expected type of a poly expression if it is used in a destructure pattern and null otherwise.
   */
  public TypeRef calculateExpectedType(final Expression rootPoly, final RuleEnvironment G, final InferenceContext infCtx) {
    DestructNode _xifexpression = null;
    EObject _eContainer = rootPoly.eContainer();
    if ((_eContainer instanceof VariableBinding)) {
      EObject _eContainer_1 = rootPoly.eContainer();
      _xifexpression = DestructNode.unify(((VariableBinding) _eContainer_1));
    } else {
      DestructNode _xifexpression_1 = null;
      EObject _eContainer_2 = rootPoly.eContainer();
      if ((_eContainer_2 instanceof AssignmentExpression)) {
        EObject _eContainer_3 = rootPoly.eContainer();
        _xifexpression_1 = DestructNode.unify(((AssignmentExpression) _eContainer_3));
      } else {
        DestructNode _xifexpression_2 = null;
        EObject _eContainer_4 = rootPoly.eContainer();
        if ((_eContainer_4 instanceof ForStatement)) {
          EObject _eContainer_5 = rootPoly.eContainer();
          _xifexpression_2 = DestructNode.unify(((ForStatement) _eContainer_5));
        } else {
          _xifexpression_2 = null;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    final DestructNode rootDestructNode = _xifexpression;
    if ((rootDestructNode == null)) {
      return null;
    }
    return this.calculateExpectedType(rootDestructNode, G, infCtx);
  }
  
  /**
   * Calculate expected type of a destructure pattern based on its structure.
   */
  private TypeRef calculateExpectedType(final DestructNode destructNode, final RuleEnvironment G, final InferenceContext infCtx) {
    final ArrayList<TypeArgument> elementTypes = new ArrayList<TypeArgument>();
    final ArrayList<TStructMember> elementMembers = new ArrayList<TStructMember>();
    final int elemCount = ((List<DestructNode>)Conversions.doWrapArray(destructNode.getNestedNodes())).size();
    DestructNode[] _nestedNodes = destructNode.getNestedNodes();
    for (final DestructNode nestedNode : _nestedNodes) {
      {
        TypeRef _xifexpression = null;
        if (((nestedNode.getNestedNodes() != null) && (((List<DestructNode>)Conversions.doWrapArray(nestedNode.getNestedNodes())).size() > 0))) {
          _xifexpression = this.calculateExpectedType(nestedNode, G, infCtx);
        } else {
          _xifexpression = this.createTypeFromLeafDestructNode(nestedNode, G);
        }
        final TypeRef elemExpectedType = _xifexpression;
        String _propName = nestedNode.getPropName();
        boolean _tripleNotEquals = (_propName != null);
        if (_tripleNotEquals) {
          final TStructField field = TypesFactory.eINSTANCE.createTStructField();
          field.setName(nestedNode.getPropName());
          TypeRef _xifexpression_1 = null;
          if ((elemExpectedType != null)) {
            _xifexpression_1 = elemExpectedType;
          } else {
            ParameterizedTypeRef _xblockexpression = null;
            {
              final InferenceVariable iv = infCtx.newInferenceVariable();
              _xblockexpression = TypeUtils.createTypeRef(iv);
            }
            _xifexpression_1 = _xblockexpression;
          }
          field.setTypeRef(_xifexpression_1);
          elementMembers.add(field);
        } else {
          if ((elemExpectedType != null)) {
            elementTypes.add(elemExpectedType);
          } else {
            elementTypes.add(TypeRefsFactory.eINSTANCE.createWildcard());
          }
        }
      }
    }
    ParameterizedTypeRef _xifexpression = null;
    int _size = elementMembers.size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      _xifexpression = TypeUtils.createParameterizedTypeRefStructural(RuleEnvironmentExtensions.objectType(G), TypingStrategy.STRUCTURAL, ((TStructMember[])Conversions.unwrapArray(elementMembers, TStructMember.class)));
    } else {
      ParameterizedTypeRef _xifexpression_1 = null;
      int _size_1 = elementTypes.size();
      boolean _greaterThan_1 = (_size_1 > 0);
      if (_greaterThan_1) {
        ParameterizedTypeRef _xifexpression_2 = null;
        if ((elemCount == 1)) {
          _xifexpression_2 = RuleEnvironmentExtensions.arrayTypeRef(G, elementTypes.get(0));
        } else {
          ParameterizedTypeRef _xifexpression_3 = null;
          if ((elemCount > 1)) {
            _xifexpression_3 = RuleEnvironmentExtensions.iterableNTypeRef(G, elemCount, ((TypeArgument[])Conversions.unwrapArray(elementTypes, TypeArgument.class)));
          } else {
            _xifexpression_3 = null;
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      } else {
        throw new IllegalStateException("elementTypes and elementMembers can not both contain elements at the same time.");
      }
      _xifexpression = _xifexpression_1;
    }
    ParameterizedTypeRef retTypeRef = _xifexpression;
    if (((retTypeRef != null) && (destructNode.getAstElement().eContainer() instanceof ForStatement))) {
      retTypeRef = RuleEnvironmentExtensions.iterableTypeRef(G, retTypeRef);
    }
    return retTypeRef;
  }
  
  /**
   * Create expected type for a leaf DestructNode
   */
  private TypeRef createTypeFromLeafDestructNode(final DestructNode leafNode, final RuleEnvironment G) {
    final VariableDeclaration varDecl = leafNode.getVarDecl();
    final IdentifierRef varRef = leafNode.getVarRef();
    if ((varDecl != null)) {
      TypeRef declaredTypeRef = varDecl.getDeclaredTypeRef();
      if ((declaredTypeRef != null)) {
        return declaredTypeRef;
      }
    } else {
      if ((varRef != null)) {
        if (((varRef.getId() instanceof VariableDeclaration) && (((VariableDeclaration) varRef.getId()).getDeclaredTypeRef() != null))) {
          IdentifiableElement _id = varRef.getId();
          return ((VariableDeclaration) _id).getDeclaredTypeRef();
        }
      }
    }
    return null;
  }
  
  private Iterable<? extends TypeRef> extractIterableElementTypes(final RuleEnvironment G, final TypeRef typeRef, final boolean includeIterableN) {
    Iterable<? extends TypeRef> result = null;
    Type _declaredType = null;
    if (typeRef!=null) {
      _declaredType=typeRef.getDeclaredType();
    }
    final Type declType = _declaredType;
    if (((declType == RuleEnvironmentExtensions.iterableType(G)) || (includeIterableN && RuleEnvironmentExtensions.isIterableN(G, declType)))) {
      result = this.toUpperBounds(typeRef.getTypeArgs(), G);
    } else {
      if ((declType instanceof PrimitiveType)) {
        final TypeRef elementType = ((PrimitiveType)declType).getElementType();
        if ((elementType != null)) {
          TypeRef _copy = TypeUtils.<TypeRef>copy(elementType);
          result = Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList(_copy));
        }
      } else {
        if ((typeRef instanceof ComposedTypeRef)) {
          final ArrayList<Iterable<? extends TypeRef>> results = CollectionLiterals.<Iterable<? extends TypeRef>>newArrayList();
          EList<TypeRef> _typeRefs = ((ComposedTypeRef)typeRef).getTypeRefs();
          for (final TypeRef containedTypeRef : _typeRefs) {
            {
              final Iterable<? extends TypeRef> currResult = this.extractIterableElementTypes(G, containedTypeRef, includeIterableN);
              boolean _isEmpty = IterableExtensions.isEmpty(currResult);
              if (_isEmpty) {
                return Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList());
              }
              results.add(currResult);
            }
          }
          boolean _isEmpty = results.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            result = this.mergeListsOfTypeRefs(G, ((ComposedTypeRef)typeRef).getClass(), ((Iterable<? extends TypeRef>[])Conversions.unwrapArray(results, Iterable.class)));
          }
        } else {
          if ((declType instanceof ContainerType<?>)) {
            final ArrayList<Iterable<? extends TypeRef>> results_1 = CollectionLiterals.<Iterable<? extends TypeRef>>newArrayList();
            List<ParameterizedTypeRef> _collect = AllSuperTypeRefsCollector.collect(((ContainerType<?>)declType));
            for (final ParameterizedTypeRef superTypeRef : _collect) {
              boolean _or = false;
              Type _declaredType_1 = null;
              if (superTypeRef!=null) {
                _declaredType_1=superTypeRef.getDeclaredType();
              }
              TInterface _iterableType = RuleEnvironmentExtensions.iterableType(G);
              boolean _tripleEquals = (_declaredType_1 == _iterableType);
              if (_tripleEquals) {
                _or = true;
              } else {
                _or = (includeIterableN && RuleEnvironmentExtensions.isIterableN(G, superTypeRef));
              }
              if (_or) {
                final boolean isContainedInIterable = RuleEnvironmentExtensions.isIterableN(G, superTypeRef.eContainer());
                if ((!(includeIterableN && isContainedInIterable))) {
                  results_1.add(this.toUpperBounds(superTypeRef.getTypeArgs(), G));
                }
              }
            }
            boolean _isEmpty_1 = results_1.isEmpty();
            boolean _not_1 = (!_isEmpty_1);
            if (_not_1) {
              result = this.mergeListsOfTypeRefs(G, IntersectionTypeExpression.class, ((Iterable<? extends TypeRef>[])Conversions.unwrapArray(results_1, Iterable.class)));
            }
            if ((result == null)) {
              final Object res = G.get(Resource.class);
              if ((res instanceof Resource)) {
                final TMember m = this.containerTypesHelper.fromContext(((Resource)res)).findMember(((ContainerType<?>)declType), ComputedPropertyNameValueConverter.SYMBOL_ITERATOR_MANGLED, false, false);
                if ((m instanceof TMethod)) {
                  TypeRef _returnTypeRef = ((TMethod)m).getReturnTypeRef();
                  EList<TypeArgument> _typeArgs = null;
                  if (_returnTypeRef!=null) {
                    _typeArgs=_returnTypeRef.getTypeArgs();
                  }
                  result = this.toUpperBounds(_typeArgs, G);
                } else {
                  if ((m instanceof TGetter)) {
                    TypeRef _declaredTypeRef = ((TGetter)m).getDeclaredTypeRef();
                    EList<TypeArgument> _typeArgs_1 = null;
                    if (_declaredTypeRef!=null) {
                      _typeArgs_1=_declaredTypeRef.getTypeArgs();
                    }
                    result = this.toUpperBounds(_typeArgs_1, G);
                  }
                }
              } else {
                throw new IllegalArgumentException("no or invalid Resource defined in rule environment G");
              }
            }
          }
        }
      }
    }
    if (((result == null) || IterableExtensions.isEmpty(result))) {
      return Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList());
    }
    final RuleEnvironment G2 = RuleEnvironmentExtensions.wrap(G);
    this.tsh.addSubstitutions(G2, typeRef);
    final Function1<TypeRef, TypeArgument> _function = (TypeRef it) -> {
      return this.ts.substTypeVariables(G2, it).getValue();
    };
    final Iterable<TypeRef> resultSubst = Iterables.<TypeRef>filter(IterableExtensions.map(result, _function), TypeRef.class);
    return resultSubst;
  }
  
  private Iterable<TypeRef> toUpperBounds(final Iterable<TypeArgument> typeArgs, final RuleEnvironment G) {
    final Function1<TypeArgument, TypeRef> _function = (TypeArgument it) -> {
      return this.ts.upperBound(G, it).getValue();
    };
    return IterableExtensions.<TypeArgument, TypeRef>map(typeArgs, _function);
  }
  
  private Iterable<? extends TypeRef> mergeListsOfTypeRefs(final RuleEnvironment G, final Class<? extends ComposedTypeRef> type, final Iterable<? extends TypeRef>... iterablesToMerge) {
    final int rs = ((List<Iterable<? extends TypeRef>>)Conversions.doWrapArray(iterablesToMerge)).size();
    if ((rs == 0)) {
      return CollectionLiterals.<TypeRef>emptyList();
    } else {
      if ((rs == 1)) {
        return IterableExtensions.<Iterable<? extends TypeRef>>head(((Iterable<Iterable<? extends TypeRef>>)Conversions.doWrapArray(iterablesToMerge)));
      } else {
        final Function1<Iterable<? extends TypeRef>, List<? extends TypeRef>> _function = (Iterable<? extends TypeRef> it) -> {
          return IterableExtensions.toList(it);
        };
        final List<List<? extends TypeRef>> listsToMerge = IterableExtensions.<List<? extends TypeRef>>toList(ListExtensions.<Iterable<? extends TypeRef>, List<? extends TypeRef>>map(((List<Iterable<? extends TypeRef>>)Conversions.doWrapArray(iterablesToMerge)), _function));
        final Function1<List<? extends TypeRef>, Integer> _function_1 = (List<? extends TypeRef> it) -> {
          return Integer.valueOf(it.size());
        };
        final Function2<Integer, Integer, Integer> _function_2 = (Integer a, Integer b) -> {
          return Integer.valueOf(Math.max((a).intValue(), (b).intValue()));
        };
        final int maxNumOfElems = IterableExtensions.<Integer>reduce(ListExtensions.<List<? extends TypeRef>, Integer>map(listsToMerge, _function_1), _function_2).intValue();
        final ArrayList<TypeRef> result = CollectionLiterals.<TypeRef>newArrayList();
        final TypeRef[] types_of_element_i_across_results = new TypeRef[rs];
        for (int i = 0; (i < maxNumOfElems); i++) {
          {
            for (int j = 0; (j < rs); j++) {
              {
                final List<? extends TypeRef> result_j = listsToMerge.get(j);
                if (((result_j == null) || result_j.isEmpty())) {
                  throw new IllegalArgumentException("iterablesToMerge may not contain null values or empty iterables");
                }
                int _size = result_j.size();
                int _minus = (_size - 1);
                final int idxSafe = Math.min(i, _minus);
                final TypeRef type_of_element_i_in_result_j = result_j.get(idxSafe);
                types_of_element_i_across_results[j] = type_of_element_i_in_result_j;
              }
            }
            TypeRef _xifexpression = null;
            boolean _isAssignableFrom = UnionTypeExpression.class.isAssignableFrom(type);
            if (_isAssignableFrom) {
              _xifexpression = this.tsh.createUnionType(G, types_of_element_i_across_results);
            } else {
              TypeRef _xifexpression_1 = null;
              boolean _isAssignableFrom_1 = IntersectionTypeExpression.class.isAssignableFrom(type);
              if (_isAssignableFrom_1) {
                _xifexpression_1 = this.tsh.createIntersectionType(G, types_of_element_i_across_results);
              } else {
                String _name = null;
                if (type!=null) {
                  _name=type.getName();
                }
                String _plus = ("unknown subtype of ComposedTypeRef: " + _name);
                throw new IllegalArgumentException(_plus);
              }
              _xifexpression = _xifexpression_1;
            }
            final TypeRef type_of_element_i = _xifexpression;
            result.add(type_of_element_i);
          }
        }
        return result;
      }
    }
  }
}
