/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.AwaitExpression;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PromisifyExpression;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeArgument;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsFactory;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Core logic for handling annotation <code>@Promisifiable</code> and operator <code>@Promisify</code>, i.e.
 * {@link PromisifyExpression}s.
 */
@SuppressWarnings("all")
public class PromisifyHelper {
  public enum CheckResult {
    OK,
    
    MISSING_CALLBACK,
    
    BAD_CALLBACK__MORE_THAN_ONE_ERROR,
    
    BAD_CALLBACK__ERROR_NOT_FIRST_ARG;
  }
  
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  /**
   * Checks if the given function or method may be annotated with <code>@Promisifiable</code>.
   */
  public PromisifyHelper.CheckResult checkPromisifiablePreconditions(final FunctionDefinition funDef) {
    Type _definedType = funDef.getDefinedType();
    EList<TFormalParameter> _fpars = null;
    if (((TFunction) _definedType)!=null) {
      _fpars=((TFunction) _definedType).getFpars();
    }
    TFormalParameter _last = null;
    if (_fpars!=null) {
      _last=IterableExtensions.<TFormalParameter>last(_fpars);
    }
    final TFormalParameter lastFpar = _last;
    TypeRef _typeRef = null;
    if (lastFpar!=null) {
      _typeRef=lastFpar.getTypeRef();
    }
    final TypeRef lastFparTypeRef = _typeRef;
    if ((!(lastFparTypeRef instanceof FunctionTypeExprOrRef))) {
      return PromisifyHelper.CheckResult.MISSING_CALLBACK;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(funDef);
    final FunctionTypeExprOrRef callbackTypeRef = ((FunctionTypeExprOrRef) lastFparTypeRef);
    final EList<TFormalParameter> callbackFpars = callbackTypeRef.getFpars();
    final Function1<TFormalParameter, Boolean> _function = (TFormalParameter it) -> {
      return Boolean.valueOf(this.isErrorOrSubtype(G, it.getTypeRef()));
    };
    final List<TFormalParameter> errorFpars = IterableExtensions.<TFormalParameter>toList(IterableExtensions.<TFormalParameter>filter(callbackFpars, _function));
    int _size = errorFpars.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      return PromisifyHelper.CheckResult.BAD_CALLBACK__MORE_THAN_ONE_ERROR;
    }
    final TFormalParameter errorFpar = IterableExtensions.<TFormalParameter>head(errorFpars);
    if (((errorFpar != null) && (errorFpar != callbackFpars.get(0)))) {
      return PromisifyHelper.CheckResult.BAD_CALLBACK__ERROR_NOT_FIRST_ARG;
    }
    return PromisifyHelper.CheckResult.OK;
  }
  
  /**
   * Checks if the given await expression is a case of "auto-promisify", i.e. if it is a short-hand form for
   * <pre>await @Promisify foo()</pre>.
   */
  public boolean isAutoPromisify(final AwaitExpression awaitExpr) {
    return (((awaitExpr != null) && (!(awaitExpr.getExpression() instanceof PromisifyExpression))) && this.isPromisifiableExpression(awaitExpr.getExpression()));
  }
  
  /**
   * Tells if the given expression is promisifiable, i.e. if it is an expression that may appear after a
   * <code>@Promisify</code> operator and thus is a valid expression within a {@link PromisifyExpression}.
   * However, this method will *not* check if the expression is actually contained in a {@code PromisifyExpression}.
   */
  public boolean isPromisifiableExpression(final Expression expr) {
    if ((expr instanceof ParameterizedCallExpression)) {
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(expr);
      final TypeRef targetTypeRef = this.ts.type(G, ((ParameterizedCallExpression)expr).getTarget()).getValue();
      if ((targetTypeRef instanceof FunctionTypeExprOrRef)) {
        final TFunction fun = ((FunctionTypeExprOrRef)targetTypeRef).getFunctionType();
        return ((fun != null) && AnnotationDefinition.PROMISIFIABLE.hasAnnotation(fun));
      }
    }
    return false;
  }
  
  /**
   * If the given expression is a promisifiable expression (see {@link #isPromisifiableExpression(Expression)}, this
   * method will return the return type of the invoked function or method after promisification.
   */
  public TypeRef extractPromisifiedReturnType(final Expression expr) {
    boolean _isPromisifiableExpression = this.isPromisifiableExpression(expr);
    if (_isPromisifiableExpression) {
      final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(expr);
      TypeRef _value = this.ts.type(G, ((ParameterizedCallExpression) expr).getTarget()).getValue();
      final FunctionTypeExprOrRef targetTypeRef = ((FunctionTypeExprOrRef) _value);
      final TypeRef promisifiedReturnTypeRef = this.extractPromisifiedReturnType(G, targetTypeRef);
      if ((promisifiedReturnTypeRef != null)) {
        return promisifiedReturnTypeRef;
      }
    }
    return TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
  }
  
  /**
   * Given the type of a promisifiable function (see {@link #checkPromisifiablePreconditions(FunctionDefinition)},
   * this method will return the function's return type after promisification. If the function is not promisifiable,
   * this method returns <code>null</code>.
   */
  public TypeRef extractPromisifiedReturnType(final RuleEnvironment G, final FunctionTypeExprOrRef targetTypeRef) {
    final TFormalParameter callbackFpar = IterableExtensions.<TFormalParameter>last(targetTypeRef.getFpars());
    if ((callbackFpar != null)) {
      final TypeRef callbackTypeRef = callbackFpar.getTypeRef();
      if ((callbackTypeRef instanceof FunctionTypeExprOrRef)) {
        TFormalParameter _head = IterableExtensions.<TFormalParameter>head(((FunctionTypeExprOrRef)callbackTypeRef).getFpars());
        TypeRef _typeRef = null;
        if (_head!=null) {
          _typeRef=_head.getTypeRef();
        }
        final TypeRef callback1stFparTypeRef = _typeRef;
        final boolean hasErrorFpar = this.isErrorOrSubtype(G, callback1stFparTypeRef);
        TypeRef _xifexpression = null;
        if (hasErrorFpar) {
          _xifexpression = callback1stFparTypeRef;
        } else {
          _xifexpression = RuleEnvironmentExtensions.undefinedTypeRef(G);
        }
        final TypeRef errorTypeRef = _xifexpression;
        Iterable<TFormalParameter> _xifexpression_1 = null;
        if (hasErrorFpar) {
          _xifexpression_1 = IterableExtensions.<TFormalParameter>drop(((FunctionTypeExprOrRef)callbackTypeRef).getFpars(), 1);
        } else {
          _xifexpression_1 = ((FunctionTypeExprOrRef)callbackTypeRef).getFpars();
        }
        final Iterable<TFormalParameter> successFpars = _xifexpression_1;
        final Function1<TFormalParameter, TypeRef> _function = (TFormalParameter it) -> {
          return it.getTypeRef();
        };
        final Function1<TypeRef, TypeRef> _function_1 = (TypeRef it) -> {
          TypeRef _xifexpression_2 = null;
          if ((it != null)) {
            _xifexpression_2 = it;
          } else {
            _xifexpression_2 = TypeRefsFactory.eINSTANCE.createUnknownTypeRef();
          }
          return _xifexpression_2;
        };
        final List<TypeRef> successFparTypeRefs = IterableExtensions.<TypeRef>toList(IterableExtensions.<TypeRef, TypeRef>map(IterableExtensions.<TFormalParameter, TypeRef>map(successFpars, _function), _function_1));
        final int len = successFparTypeRefs.size();
        TypeRef _xifexpression_2 = null;
        if ((len == 0)) {
          _xifexpression_2 = RuleEnvironmentExtensions.undefinedTypeRef(G);
        } else {
          TypeRef _xifexpression_3 = null;
          if ((len == 1)) {
            _xifexpression_3 = successFparTypeRefs.get(0);
          } else {
            ParameterizedTypeRef _xifexpression_4 = null;
            if ((len <= BuiltInTypeScope.ITERABLE_N__MAX_LEN)) {
              _xifexpression_4 = RuleEnvironmentExtensions.iterableNTypeRef(G, successFparTypeRefs.size(), ((TypeArgument[])Conversions.unwrapArray(successFparTypeRefs, TypeArgument.class)));
            } else {
              ParameterizedTypeRef _xblockexpression = null;
              {
                final TypeRef remaining = this.tsh.createUnionType(G, ((TypeRef[])Conversions.unwrapArray(IterableExtensions.<TypeRef>drop(successFparTypeRefs, (BuiltInTypeScope.ITERABLE_N__MAX_LEN - 1)), TypeRef.class)));
                Iterable<TypeRef> _take = IterableExtensions.<TypeRef>take(successFparTypeRefs, (BuiltInTypeScope.ITERABLE_N__MAX_LEN - 1));
                Iterable<TypeRef> _plus = Iterables.<TypeRef>concat(_take, Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList(remaining)));
                _xblockexpression = RuleEnvironmentExtensions.iterableNTypeRef(G, 
                  BuiltInTypeScope.ITERABLE_N__MAX_LEN, ((TypeArgument[])Conversions.unwrapArray(_plus, TypeArgument.class)));
              }
              _xifexpression_4 = _xblockexpression;
            }
            _xifexpression_3 = _xifexpression_4;
          }
          _xifexpression_2 = _xifexpression_3;
        }
        final TypeRef successTypeRef = _xifexpression_2;
        return TypeUtils.createPromiseTypeRef(RuleEnvironmentExtensions.getBuiltInTypeScope(G), successTypeRef, errorTypeRef);
      }
    }
    return null;
  }
  
  private boolean isErrorOrSubtype(final RuleEnvironment G, final TypeRef typeRef) {
    return ((typeRef != null) && this.ts.subtypeSucceeded(G, typeRef, RuleEnvironmentExtensions.errorTypeRef(G)));
  }
}
