/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Arrays;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.IntersectionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.utils.AndFunction1;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.StructuralMembersPredicates;
import org.eclipse.n4js.utils.StructuralMembersTripleIterator;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Helper methods for dealing with structural types.
 */
@SuppressWarnings("all")
public class StructuralTypesHelper {
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private TypeSystemHelper tsh;
  
  /**
   * Collect the members of the two given structural types and create an iterator for iterating over
   * the triple of corresponding members.
   * <p>
   * 
   * If the {@code restrictInitTyping} argument is {@code true}, then the members will be filtered into read- and write-only
   * subsets of members depending whether the type reference is on left or right side. If this flag is {@code false}, then all field structural
   * members will be included in the iterator.
   * 
   * <TODO IDEBUG-723 remove restrictInitTyping argument. Currently the getters are returned from the member filter if a type has a getter/setter accessor
   * pair. The getter is the proper one when creating the type constraints. But when having ~i~ type, setters should return if both getter and
   * setters are available, which breaks the type checking. For instance union{string,int} is returned when string should be returned.
   * See: Constraints_55_9_Structural_Fields_Can_Be_Assigned_Via_Public_Accessors.n4js.xt for more details.>
   */
  public StructuralMembersTripleIterator getMembersTripleIterator(final RuleEnvironment G, final TypeRef left, final TypeRef right, final boolean restrictInitTyping) {
    TypingStrategy rightStrategy = right.getTypingStrategy();
    TypingStrategy leftStrategy = left.getTypingStrategy();
    if ((TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == rightStrategy)) {
      TypingStrategy _xifexpression = null;
      if (restrictInitTyping) {
        _xifexpression = TypingStrategy.STRUCTURAL_READ_ONLY_FIELDS;
      } else {
        _xifexpression = TypingStrategy.STRUCTURAL_FIELDS;
      }
      leftStrategy = _xifexpression;
      TypingStrategy _xifexpression_1 = null;
      if (restrictInitTyping) {
        _xifexpression_1 = TypingStrategy.STRUCTURAL_WRITE_ONLY_FIELDS;
      } else {
        _xifexpression_1 = TypingStrategy.STRUCTURAL_FIELDS;
      }
      rightStrategy = _xifexpression_1;
    }
    if ((TypingStrategy.STRUCTURAL_FIELD_INITIALIZER == leftStrategy)) {
      TypingStrategy _xifexpression_2 = null;
      if (restrictInitTyping) {
        _xifexpression_2 = TypingStrategy.STRUCTURAL_READ_ONLY_FIELDS;
      } else {
        _xifexpression_2 = TypingStrategy.STRUCTURAL_FIELDS;
      }
      leftStrategy = _xifexpression_2;
    }
    final Iterable<TMember> membersLeft = this.collectStructuralMembers(G, left, leftStrategy);
    final Iterable<TMember> membersRight = this.collectStructuralMembers(G, right, rightStrategy);
    return StructuralMembersTripleIterator.ofUnprepared(membersLeft, membersRight, 
      left.getTypingStrategy(), 
      right.getTypingStrategy());
  }
  
  /**
   * Collect the members of a structural type.
   */
  public Iterable<TMember> collectStructuralMembers(final RuleEnvironment G, final TypeRef typeRef, final TypingStrategy strategy) {
    AndFunction1<TMember> _createBaseStructuralMembersPredicate = StructuralMembersPredicates.createBaseStructuralMembersPredicate(G);
    AndFunction1<TMember> _switchResult = null;
    if (strategy != null) {
      switch (strategy) {
        case STRUCTURAL_WRITE_ONLY_FIELDS:
          _switchResult = StructuralMembersPredicates.WRITABLE_FIELDS_PREDICATE;
          break;
        case STRUCTURAL_READ_ONLY_FIELDS:
          _switchResult = StructuralMembersPredicates.READABLE_FIELDS_PREDICATE;
          break;
        case STRUCTURAL_FIELDS:
          _switchResult = StructuralMembersPredicates.FIELDS_PREDICATE;
          break;
        case STRUCTURAL_FIELD_INITIALIZER:
          throw new IllegalStateException("Expected read-only and write-only variants instead.");
        default:
          _switchResult = StructuralMembersPredicates.MEMBERS_PREDICATE;
          break;
      }
    } else {
      _switchResult = StructuralMembersPredicates.MEMBERS_PREDICATE;
    }
    final AndFunction1<TMember> predicate = _createBaseStructuralMembersPredicate.and(_switchResult);
    return this.doCollectMembers(G, typeRef, predicate);
  }
  
  private Iterable<TMember> _doCollectMembers(final RuleEnvironment G, final TypeRef ref, final Function1<TMember, Boolean> predicate) {
    return CollectionLiterals.<TMember>emptyList();
  }
  
  private Iterable<TMember> _doCollectMembers(final RuleEnvironment G, final BoundThisTypeRef ref, final Function1<TMember, Boolean> predicate) {
    final Iterable<TMember> structMembersOfThis = this.doCollectMembers(G, ref.getActualThisTypeRef(), predicate);
    boolean _isEmpty = ref.getStructuralMembers().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      return Iterables.<TMember>concat(structMembersOfThis, ref.getStructuralMembers());
    }
    return structMembersOfThis;
  }
  
  private Iterable<TMember> _doCollectMembers(final RuleEnvironment G, final ParameterizedTypeRef ref, final Function1<TMember, Boolean> predicate) {
    final Iterable<TMember> nominalMembers = this.doCollectMembersOfType(G, ref.getDeclaredType(), predicate);
    boolean _isEmpty = ref.getStructuralMembers().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      return Iterables.<TMember>concat(nominalMembers, ref.getStructuralMembers());
    }
    return nominalMembers;
  }
  
  private Iterable<TMember> _doCollectMembersOfType(final RuleEnvironment G, final Type type, final Function1<TMember, Boolean> predicate) {
    return CollectionLiterals.<TMember>emptyList();
  }
  
  private Iterable<TMember> _doCollectMembersOfType(final RuleEnvironment G, final ContainerType<?> type, final Function1<TMember, Boolean> predicate) {
    return IterableExtensions.<TMember>filter(this.containerTypesHelper.fromContext(RuleEnvironmentExtensions.getContextResource(G)).members(type), predicate);
  }
  
  private Iterable<TMember> _doCollectMembersOfType(final RuleEnvironment G, final TypeVariable type, final Function1<TMember, Boolean> predicate) {
    final TypeRef declUB = type.getDeclaredUpperBound();
    TypeRef _xifexpression = null;
    if ((declUB instanceof IntersectionTypeExpression)) {
      _xifexpression = this.tsh.join(G, ((IntersectionTypeExpression)declUB).getTypeRefs());
    } else {
      _xifexpression = declUB;
    }
    final TypeRef joinOfUpperBounds = _xifexpression;
    if ((joinOfUpperBounds != null)) {
      return this.doCollectMembers(G, joinOfUpperBounds, predicate);
    }
    return CollectionLiterals.<TMember>emptyList();
  }
  
  private Iterable<TMember> doCollectMembers(final RuleEnvironment G, final TypeRef ref, final Function1<TMember, Boolean> predicate) {
    if (ref instanceof BoundThisTypeRef) {
      return _doCollectMembers(G, (BoundThisTypeRef)ref, predicate);
    } else if (ref instanceof ParameterizedTypeRef) {
      return _doCollectMembers(G, (ParameterizedTypeRef)ref, predicate);
    } else if (ref != null) {
      return _doCollectMembers(G, ref, predicate);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(G, ref, predicate).toString());
    }
  }
  
  private Iterable<TMember> doCollectMembersOfType(final RuleEnvironment G, final Type type, final Function1<TMember, Boolean> predicate) {
    if (type instanceof ContainerType) {
      return _doCollectMembersOfType(G, (ContainerType<?>)type, predicate);
    } else if (type instanceof TypeVariable) {
      return _doCollectMembersOfType(G, (TypeVariable)type, predicate);
    } else if (type != null) {
      return _doCollectMembersOfType(G, type, predicate);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(G, type, predicate).toString());
    }
  }
}
