/**
 * Copyright (c) 2018 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import java.util.stream.Stream;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4idl.migrations.MigrationSwitchComputer;
import org.eclipse.n4js.n4idl.migrations.SwitchCondition;
import org.eclipse.n4js.n4idl.versioning.MigrationUtils;
import org.eclipse.n4js.ts.scoping.builtin.BuiltInTypeScope;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.VersionedParameterizedTypeRef;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMigratable;
import org.eclipse.n4js.ts.types.TMigration;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.VoidType;
import org.eclipse.n4js.ts.versions.MigratableUtils;
import org.eclipse.n4js.ts.versions.VersionableUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.utils.collections.Collections2;
import org.eclipse.n4js.utils.collections.Iterables2;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xsemantics.runtime.Result;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.Functions.Function2;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.MapExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;

/**
 * Validates N4IDL migration declarations.
 */
@SuppressWarnings("all")
public class N4IDLMigrationValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private JavaScriptVariantHelper variantHelper;
  
  @Inject
  private N4JSTypeSystem typeSystem;
  
  @Inject
  private MigrationSwitchComputer switchComputer;
  
  /**
   * NEEDED
   * 
   * When removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator.
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * Validates migration declarations (function declarations annotated as {@code @Migration}.
   */
  @Check
  public void checkMigration(final FunctionDeclaration functionDeclaration) {
    boolean _allowVersionedTypes = this.variantHelper.allowVersionedTypes(functionDeclaration);
    boolean _not = (!_allowVersionedTypes);
    if (_not) {
      return;
    }
    if (((functionDeclaration == null) || (functionDeclaration.getName() == null))) {
      return;
    }
    Block _body = functionDeclaration.getBody();
    boolean _tripleEquals = (null == _body);
    if (_tripleEquals) {
      return;
    }
    boolean _isMigrationDefinition = MigrationUtils.isMigrationDefinition(functionDeclaration);
    boolean _not_1 = (!_isMigrationDefinition);
    if (_not_1) {
      return;
    }
    TFunction _definedFunction = functionDeclaration.getDefinedFunction();
    boolean _not_2 = (!(_definedFunction instanceof TMigration));
    if (_not_2) {
      String _name = functionDeclaration.getName();
      String _plus = ("FunctionDeclaration " + _name);
      String _plus_1 = (_plus + " does not refer to a valid TMigration type model instance");
      throw new IllegalStateException(_plus_1);
    }
    TFunction _definedFunction_1 = functionDeclaration.getDefinedFunction();
    final TMigration migration = ((TMigration) _definedFunction_1);
    if (((this.holdsExplicitlyDeclaresReturnType(functionDeclaration, migration) && this.holdsMigrationHasSourceAndTargetTypes(migration)) && this.holdsNoComposedSourceAndTargetTypes(migration))) {
      boolean _isEmpty = migration.getTypeVars().isEmpty();
      if (_isEmpty) {
        boolean _and = false;
        if (!((this.holdsHasPrincipalArgumentType(migration) && this.holdsMigrationHasValidSourceAndTargetVersion(migration)) && this.holdsMigrationHasVersionExclusiveSourceAndTargetVersion(migration))) {
          _and = false;
        } else {
          boolean _holdsMigrationHasDifferentSourceAndTargetVersions = this.holdsMigrationHasDifferentSourceAndTargetVersions(functionDeclaration, migration);
          _and = _holdsMigrationHasDifferentSourceAndTargetVersions;
        }
      }
      return;
    }
  }
  
  /**
   * Checks that the #targetTypeRefs of TMigration aren't inferred but explicitly declared.
   */
  private boolean holdsExplicitlyDeclaresReturnType(final FunctionDeclaration migrationDeclaration, final TMigration tMigration) {
    TypeRef _returnTypeRef = migrationDeclaration.getReturnTypeRef();
    boolean _tripleEquals = (null == _returnTypeRef);
    if (_tripleEquals) {
      this.addIssue(IssueCodes.getMessageForIDL_MIGRATION_MUST_EXPLICITLY_DECLARE_RETURN_TYPE(), migrationDeclaration, 
        N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, IssueCodes.IDL_MIGRATION_MUST_EXPLICITLY_DECLARE_RETURN_TYPE);
      return false;
    }
    return true;
  }
  
  /**
   * Checks that the given migration has at least one source type and one (non-void) target type.
   */
  private boolean holdsMigrationHasSourceAndTargetTypes(final TMigration migration) {
    if (((migration.getSourceTypeRefs().isEmpty() || migration.getTargetTypeRefs().isEmpty()) || ((migration.getTargetTypeRefs().size() == 1) && this.isVoidTypeRef(IterableExtensions.<TypeRef>head(migration.getTargetTypeRefs()))))) {
      this.addIssue(IssueCodes.getMessageForIDL_MIGRATION_MUST_DECLARE_IN_AND_OUTPUT(), migration.getAstElement(), 
        N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, IssueCodes.IDL_MIGRATION_MUST_DECLARE_IN_AND_OUTPUT);
      return false;
    }
    final Function1<TFormalParameter, Boolean> _function = (TFormalParameter p) -> {
      return Boolean.valueOf(this.isMigrationContextTypeRef(p.getTypeRef()));
    };
    final Consumer<TFormalParameter> _function_1 = (TFormalParameter p) -> {
      this.addIssueToMigrationTypeRef(IssueCodes.getMessageForIDL_MIGRATION_NO_EXPLICIT_CONTEXT_PARAMETER(), p.getTypeRef(), 
        IssueCodes.IDL_MIGRATION_NO_EXPLICIT_CONTEXT_PARAMETER);
    };
    IterableExtensions.<TFormalParameter>filter(migration.getFpars(), _function).forEach(_function_1);
    return true;
  }
  
  /**
   * Checks that the source and target types of the given migration do not contain any {@link ComposedTypeRef}.
   */
  private boolean holdsNoComposedSourceAndTargetTypes(final TMigration migration) {
    final Function<TypeRef, Boolean> _function = (TypeRef ref) -> {
      return Boolean.valueOf(this.holdsIsNotComposedTypeRef(migration, ref));
    };
    final boolean sourceTypesHoldNoComposedTypes = this.<TypeRef>allHold(migration.getSourceTypeRefs(), _function);
    final Function<TypeRef, Boolean> _function_1 = (TypeRef ref) -> {
      return Boolean.valueOf(this.holdsIsNotComposedTypeRef(migration, ref));
    };
    final boolean targetTypesHoldNoComposedTypes = this.<TypeRef>allHold(migration.getTargetTypeRefs(), _function_1);
    return (sourceTypesHoldNoComposedTypes && targetTypesHoldNoComposedTypes);
  }
  
  /**
   * Returns {@code true} iff the {@code constraintChecker} function returns true for all elements in {@code elements}.
   * Always executes {@code constraintChecker} for all {@code elements}.
   */
  private <T extends Object> boolean allHold(final Iterable<T> elements, final Function<T, Boolean> constraintChecker) {
    final Function2<Boolean, T, Boolean> _function = (Boolean previousResult, T element) -> {
      return Boolean.valueOf(((constraintChecker.apply(element)).booleanValue() && (previousResult).booleanValue()));
    };
    return (boolean) IterableExtensions.<T, Boolean>fold(elements, Boolean.valueOf(true), _function);
  }
  
  /**
   * Checks that the given migration has a valid principal argument type (non-null).
   */
  private boolean holdsHasPrincipalArgumentType(final TMigration migration) {
    TMigratable _principalArgumentType = migration.getPrincipalArgumentType();
    boolean _tripleEquals = (null == _principalArgumentType);
    if (_tripleEquals) {
      this.addIssueToMultiValueFeature(IssueCodes.getMessageForIDL_MIGRATION_HAS_PRINCIPAL_ARGUMENT(), migration.getAstElement(), 
        N4JSPackage.Literals.FUNCTION_DEFINITION__FPARS, IssueCodes.IDL_MIGRATION_HAS_PRINCIPAL_ARGUMENT);
      return false;
    }
    return true;
  }
  
  /**
   * Checks that the given {@link TypeRef} is not an {@link ComposedTypeRef}.
   */
  private boolean holdsIsNotComposedTypeRef(final TMigration migration, final TypeRef typeRef) {
    if ((typeRef instanceof ComposedTypeRef)) {
      this.addIssueToMigrationTypeRef(IssueCodes.getMessageForIDL_MIGRATION_SIGNATURE_NO_COMPOSED_TYPES(), typeRef, 
        IssueCodes.IDL_MIGRATION_SIGNATURE_NO_COMPOSED_TYPES);
      return false;
    }
    return true;
  }
  
  /**
   * Checks that the given migration source and target types allow to infer a source and target version.
   */
  private boolean holdsMigrationHasValidSourceAndTargetVersion(final TMigration migration) {
    int _sourceVersion = migration.getSourceVersion();
    boolean _equals = (_sourceVersion == 0);
    if (_equals) {
      final String message = IssueCodes.getMessageForIDL_MIGRATION_VERSION_CANNOT_BE_INFERRED("source", migration.getName());
      this.addIssueToMultiValueFeature(message, migration.getAstElement(), N4JSPackage.Literals.FUNCTION_DEFINITION__FPARS, 
        IssueCodes.IDL_MIGRATION_VERSION_CANNOT_BE_INFERRED);
      return false;
    }
    int _targetVersion = migration.getTargetVersion();
    boolean _equals_1 = (_targetVersion == 0);
    if (_equals_1) {
      final String message_1 = IssueCodes.getMessageForIDL_MIGRATION_VERSION_CANNOT_BE_INFERRED("target", migration.getName());
      this.addIssue(message_1, migration.getAstElement(), N4JSPackage.Literals.FUNCTION_DEFINITION__RETURN_TYPE_REF, 
        IssueCodes.IDL_MIGRATION_VERSION_CANNOT_BE_INFERRED);
      return false;
    }
    return true;
  }
  
  /**
   * Checks that the source and target types all are exclusively of one version (one source version, one target version).
   * 
   * Does nothing for migrations with {@TMigration#hasDeclaredSourceAndTargetVersion}.
   */
  private boolean holdsMigrationHasVersionExclusiveSourceAndTargetVersion(final TMigration migration) {
    boolean _isHasDeclaredSourceAndTargetVersion = migration.isHasDeclaredSourceAndTargetVersion();
    if (_isHasDeclaredSourceAndTargetVersion) {
      return true;
    }
    boolean _isVersionExclusive = this.isVersionExclusive(migration.getSourceTypeRefs());
    boolean _not = (!_isVersionExclusive);
    if (_not) {
      final String message = IssueCodes.getMessageForIDL_MIGRATION_AMBIGUOUS_VERSION("source", migration.getName());
      this.addIssueToMultiValueFeature(message, migration.getAstElement(), N4JSPackage.Literals.FUNCTION_DEFINITION__FPARS, 
        IssueCodes.IDL_MIGRATION_VERSION_CANNOT_BE_INFERRED);
      return false;
    }
    boolean _isVersionExclusive_1 = this.isVersionExclusive(migration.getTargetTypeRefs());
    boolean _not_1 = (!_isVersionExclusive_1);
    if (_not_1) {
      final String message_1 = IssueCodes.getMessageForIDL_MIGRATION_AMBIGUOUS_VERSION("target", migration.getName());
      this.addIssue(message_1, migration.getAstElement(), N4JSPackage.Literals.FUNCTION_DEFINITION__RETURN_TYPE_REF, 
        IssueCodes.IDL_MIGRATION_VERSION_CANNOT_BE_INFERRED);
      return false;
    }
    return true;
  }
  
  private boolean holdsMigrationHasDifferentSourceAndTargetVersions(final FunctionDeclaration declaration, final TMigration migration) {
    int _sourceVersion = migration.getSourceVersion();
    int _targetVersion = migration.getTargetVersion();
    boolean _equals = (_sourceVersion == _targetVersion);
    if (_equals) {
      final String msg = IssueCodes.getMessageForIDL_MIGRATION_SAME_SOURCE_AND_TARGET_VERSION(migration.getName(), Integer.valueOf(migration.getSourceVersion()));
      boolean _isHasDeclaredSourceAndTargetVersion = migration.isHasDeclaredSourceAndTargetVersion();
      if (_isHasDeclaredSourceAndTargetVersion) {
        final Annotation migrationAnno = AnnotationDefinition.MIGRATION.getAnnotation(declaration);
        this.addIssueToMultiValueFeature(msg, migrationAnno, 
          N4JSPackage.Literals.ANNOTATION__ARGS, 
          IssueCodes.IDL_MIGRATION_SAME_SOURCE_AND_TARGET_VERSION);
        return false;
      } else {
        this.addIssue(msg, declaration, 
          N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, 
          IssueCodes.IDL_MIGRATION_SAME_SOURCE_AND_TARGET_VERSION);
        return false;
      }
    }
    return true;
  }
  
  /**
   * Returns {@code true} iff the given type reference refers to the built-in {@code void} type.
   */
  private boolean isVoidTypeRef(final TypeRef typeRef) {
    final BuiltInTypeScope builtInTypes = BuiltInTypeScope.get(typeRef.eResource().getResourceSet());
    Type _declaredType = typeRef.getDeclaredType();
    VoidType _voidType = builtInTypes.getVoidType();
    return Objects.equal(_declaredType, _voidType);
  }
  
  /**
   * Returns {@code true} iff the given type reference refers to the built-in {@code MigrationContext} type.
   */
  private boolean isMigrationContextTypeRef(final TypeRef typeRef) {
    final BuiltInTypeScope builtInTypes = BuiltInTypeScope.get(typeRef.eResource().getResourceSet());
    Type _declaredType = typeRef.getDeclaredType();
    TInterface _migrationContextType = builtInTypes.getMigrationContextType();
    return Objects.equal(_declaredType, _migrationContextType);
  }
  
  @Check
  public void checkMigratableTypeDeclaration(final N4TypeDeclaration typeDeclaration) {
    final Type declaredType = typeDeclaration.getDefinedType();
    if ((null == declaredType)) {
      return;
    }
    if ((!(declaredType instanceof TMigratable))) {
      return;
    }
    final TMigratable migratable = ((TMigratable) declaredType);
    final Function1<TMigration, Pair<Integer, Integer>> _function = (TMigration it) -> {
      int _targetVersion = it.getTargetVersion();
      int _size = it.getSourceTypeRefs().size();
      return Pair.<Integer, Integer>of(Integer.valueOf(_targetVersion), Integer.valueOf(_size));
    };
    final Function2<Pair<Integer, Integer>, List<TMigration>, Boolean> _function_1 = (Pair<Integer, Integer> groupKey, List<TMigration> migrations) -> {
      return Boolean.valueOf(((((groupKey.getKey()).intValue() > 0) && ((groupKey.getValue()).intValue() > 0)) && (migrations.size() > 0)));
    };
    final Map<Pair<Integer, Integer>, List<TMigration>> migrationGroups = MapExtensions.<Pair<Integer, Integer>, List<TMigration>>filter(IterableExtensions.<Pair<Integer, Integer>, TMigration>groupBy(migratable.getMigrations(), _function), _function_1);
    final BiConsumer<Pair<Integer, Integer>, List<TMigration>> _function_2 = (Pair<Integer, Integer> groupKey, List<TMigration> migrations) -> {
      this.holdsTypeSwitchDistinguishable(migrations);
    };
    migrationGroups.forEach(_function_2);
  }
  
  private RuleEnvironment ruleEnvironment(final TypeRef ref) {
    return this.typeSystem.createRuleEnvironmentForContext(ref, ref.eResource());
  }
  
  /**
   * Convenience method to enable the use of {@link Procedure2} when iterating over {@link Pair} streams.
   */
  private static <T1 extends Object, T2 extends Object> void forPair(final Stream<Pair<T1, T2>> stream, final Procedure2<T1, T2> procedure) {
    final Consumer<Pair<T1, T2>> _function = (Pair<T1, T2> pair) -> {
      procedure.apply(pair.getKey(), pair.getValue());
    };
    stream.forEach(_function);
  }
  
  /**
   * Returns {@code true} iff all type references in left are equaltype (cf. {@link N4JSTypeSystem#equalType}) of
   * the corresponding type references in right.
   * 
   * Always returns {@code false} if left holds a different number of type reference than right.
   */
  private boolean areEqualTypes(final RuleEnvironment ruleEnv, final Iterable<TypeRef> left, final Iterable<TypeRef> right) {
    int _size = IterableExtensions.size(left);
    int _size_1 = IterableExtensions.size(right);
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      return false;
    }
    final Function1<Pair<TypeRef, TypeRef>, Boolean> _function = (Pair<TypeRef, TypeRef> pair) -> {
      final TypeRef l = pair.getKey();
      final TypeRef r = pair.getValue();
      final Result<Boolean> subtypingResult = this.typeSystem.equaltype(ruleEnv, l, r);
      return Boolean.valueOf((subtypingResult.failed() || ((subtypingResult.getValue()).booleanValue() == false)));
    };
    final Pair<TypeRef, TypeRef> firstNonSubtype = IterableExtensions.<Pair<TypeRef, TypeRef>>findFirst(Iterables2.<TypeRef, TypeRef>align(left, right), _function);
    return (firstNonSubtype == null);
  }
  
  /**
   * Returns {@code true} iff the given list of migrations is distinguishable by type switches.
   * 
   * This method assumes that all of the given migrations already have the same number of source types.
   */
  private boolean holdsTypeSwitchDistinguishable(final Iterable<TMigration> migrations) {
    final Function1<TMigration, Pair<TMigration, List<TypeRef>>> _function = (TMigration migration) -> {
      List<TypeRef> _switchRecognizableSourceTypeRefs = this.getSwitchRecognizableSourceTypeRefs(migration);
      return Pair.<TMigration, List<TypeRef>>of(migration, _switchRecognizableSourceTypeRefs);
    };
    final List<Pair<TMigration, List<TypeRef>>> migrationAndSwitchTypes = IterableExtensions.<Pair<TMigration, List<TypeRef>>>toList(IterableExtensions.<TMigration, Pair<TMigration, List<TypeRef>>>map(migrations, _function));
    final HashMap<TMigration, Set<TMigration>> conflictGroups = new HashMap<TMigration, Set<TMigration>>();
    final Procedure2<Pair<TMigration, List<TypeRef>>, Pair<TMigration, List<TypeRef>>> _function_1 = (Pair<TMigration, List<TypeRef>> m1, Pair<TMigration, List<TypeRef>> m2) -> {
      final TMigration tMigration1 = m1.getKey();
      final List<TypeRef> switchTypes1 = m1.getValue();
      final TMigration tMigration2 = m2.getKey();
      final List<TypeRef> switchTypes2 = m2.getValue();
      if (((switchTypes1.size() == 0) || (switchTypes2.size() == 0))) {
        return;
      }
      final RuleEnvironment ruleEnv = this.ruleEnvironment(IterableExtensions.<TypeRef>head(tMigration1.getSourceTypeRefs()));
      boolean _areEqualTypes = this.areEqualTypes(ruleEnv, switchTypes1, switchTypes2);
      if (_areEqualTypes) {
        this.addMigrationConflict(conflictGroups, tMigration1, tMigration2);
      }
    };
    N4IDLMigrationValidator.<Pair<TMigration, List<TypeRef>>, Pair<TMigration, List<TypeRef>>>forPair(Collections2.<Pair<TMigration, List<TypeRef>>>pairs(migrationAndSwitchTypes), _function_1);
    final Consumer<Set<TMigration>> _function_2 = (Set<TMigration> conflictGroup) -> {
      this.addIssueMigrationConflict(conflictGroup);
    };
    IterableExtensions.<Set<TMigration>>toSet(conflictGroups.values()).forEach(_function_2);
    return conflictGroups.isEmpty();
  }
  
  /**
   * Returns a list of switch-recognizable {@link TypeRef}s based on the source type refs of the given {@code migration}.
   * 
   * Returns an empty list if any of the given {@link TypeRef} cannot be handled by a {@link SwitchCondition}
   * (cf. {@link MigrationSwitchComputer#UnhandledTypeRefException}).
   * 
   * Adds issues for unsupported type refs using {@link #addUnhandledParameterTypeRefIssue}.
   */
  private List<TypeRef> getSwitchRecognizableSourceTypeRefs(final TMigration migration) {
    final Function1<TypeRef, TypeRef> _function = (TypeRef s) -> {
      TypeRef _xtrycatchfinallyexpression = null;
      try {
        _xtrycatchfinallyexpression = this.switchComputer.toSwitchRecognizableTypeRef(this.ruleEnvironment(s), s);
      } catch (final Throwable _t) {
        if (_t instanceof MigrationSwitchComputer.UnhandledTypeRefException) {
          final MigrationSwitchComputer.UnhandledTypeRefException e = (MigrationSwitchComputer.UnhandledTypeRefException)_t;
          final Function1<TFormalParameter, Boolean> _function_1 = (TFormalParameter par) -> {
            TypeRef _typeRef = par.getTypeRef();
            return Boolean.valueOf(Objects.equal(_typeRef, s));
          };
          this.addUnsupportedParameterTypeRefIssue(IterableExtensions.<TFormalParameter>findFirst(migration.getFpars(), _function_1));
          return null;
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      return _xtrycatchfinallyexpression;
    };
    final List<TypeRef> refs = IterableExtensions.<TypeRef>toList(IterableExtensions.<TypeRef>filterNull(ListExtensions.<TypeRef, TypeRef>map(migration.getSourceTypeRefs(), _function)));
    int _size = refs.size();
    int _size_1 = migration.getSourceTypeRefs().size();
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      return Collections.<TypeRef>unmodifiableList(CollectionLiterals.<TypeRef>newArrayList());
    }
    return refs;
  }
  
  /**
   * Adds an IDL_MIGRATION_UNSUPPORTED_PARAMETER_TYPE issue to the given {@link TFormalParameter}'s TypeRef.
   */
  private void addUnsupportedParameterTypeRefIssue(final TFormalParameter tParam) {
    EObject _astElement = tParam.getAstElement();
    final TypeRef astTypeRef = ((FormalParameter) _astElement).getDeclaredTypeRef();
    if ((astTypeRef instanceof ComposedTypeRef)) {
      return;
    }
    this.addIssue(IssueCodes.getMessageForIDL_MIGRATION_UNSUPPORTED_PARAMETER_TYPE(astTypeRef.getTypeRefAsString()), astTypeRef, IssueCodes.IDL_MIGRATION_UNSUPPORTED_PARAMETER_TYPE);
  }
  
  /**
   * Adds a migration conflict to given map of conflictGroups.
   * 
   * That is, if m1 and m2 were not yet known to conflict, there corresponding set associated via conflictGroups
   * are merged and re-assigned to their entries in the map.
   * 
   * Otherwise, this method does nothing.
   */
  private void addMigrationConflict(final Map<TMigration, Set<TMigration>> conflictGroups, final TMigration m1, final TMigration m2) {
    HashSet<TMigration> _hashSet = new HashSet<TMigration>(Collections.<TMigration>unmodifiableList(CollectionLiterals.<TMigration>newArrayList(m1)));
    final Set<TMigration> mergedGroup = conflictGroups.getOrDefault(m1, _hashSet);
    boolean _contains = mergedGroup.contains(m2);
    if (_contains) {
      return;
    }
    HashSet<TMigration> _hashSet_1 = new HashSet<TMigration>(Collections.<TMigration>unmodifiableList(CollectionLiterals.<TMigration>newArrayList(m2)));
    mergedGroup.addAll(conflictGroups.getOrDefault(m2, _hashSet_1));
    conflictGroups.put(m1, mergedGroup);
    conflictGroups.put(m2, mergedGroup);
  }
  
  /**
   * Adds an {@link IssueCodes#IDL_MIGRATION_CONFLICT_WITH} issue for all given migrations,
   * if the list of migration exceeds a length of 1.
   */
  private void addIssueMigrationConflict(final Iterable<TMigration> migrations) {
    int _size = IterableExtensions.size(migrations);
    boolean _lessEqualsThan = (_size <= 1);
    if (_lessEqualsThan) {
      return;
    }
    final Consumer<TMigration> _function = (TMigration m) -> {
      final String argumentDescription = MigratableUtils.getMigrationArgumentsDescription(m.getSourceTypeRefs());
      final Function1<TMigration, Boolean> _function_1 = (TMigration other) -> {
        return Boolean.valueOf((!Objects.equal(other, m)));
      };
      final String conflictingMigrationsDescription = this.listOrSingleMigrationDescription(IterableExtensions.<TMigration>filter(migrations, _function_1));
      final String msg = IssueCodes.getMessageForIDL_MIGRATION_CONFLICT_WITH(m.getName(), argumentDescription, Integer.valueOf(m.getTargetVersion()), conflictingMigrationsDescription);
      this.addIssue(msg, m.getAstElement(), N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, IssueCodes.IDL_MIGRATION_CONFLICT_WITH);
    };
    migrations.forEach(_function);
  }
  
  /**
   * Returns either a bullet-list description of multiple migrations,
   * or in the case of a single migration just the description of that migration.
   */
  private String listOrSingleMigrationDescription(final Iterable<TMigration> migrations) {
    int _size = IterableExtensions.size(migrations);
    boolean _equals = (_size == 1);
    if (_equals) {
      return IterableExtensions.<TMigration>head(migrations).getMigrationAsString();
    } else {
      final Function1<TMigration, String> _function = (TMigration m) -> {
        String _migrationAsString = m.getMigrationAsString();
        return ("\n\t - " + _migrationAsString);
      };
      return IterableExtensions.join(IterableExtensions.<TMigration, String>map(migrations, _function), ", ");
    }
  }
  
  /**
   * Returns {@code true} if the given list of {@link TypeRef}s do not mix mulitple
   * model versions (Multiple {@link VersionedParameterizedTypeRef} with different versions).
   */
  private boolean isVersionExclusive(final Collection<TypeRef> typeRefs) {
    final Function<TypeRef, Stream<VersionedParameterizedTypeRef>> _function = (TypeRef ref) -> {
      return VersionableUtils.streamVersionedSubReferences(ref);
    };
    final ToIntFunction<VersionedParameterizedTypeRef> _function_1 = (VersionedParameterizedTypeRef versionedRef) -> {
      return Integer.valueOf(versionedRef.getVersion()).intValue();
    };
    final int[] versions = typeRefs.stream().<VersionedParameterizedTypeRef>flatMap(_function).mapToInt(_function_1).distinct().limit(2).toArray();
    int _size = ((List<Integer>)Conversions.doWrapArray(versions)).size();
    return (_size == 1);
  }
  
  /**
   * Adds an issue to a migration source or target type ref.
   * 
   * Makes sure to only mark the relevant offset in the code (e.g. only the parameter declaration at the correct index).
   */
  private void addIssueToMigrationTypeRef(final String message, final TypeRef typeRef, final String issueCode) {
    final TMigration migration = EcoreUtil2.<TMigration>getContainerOfType(typeRef, TMigration.class);
    if ((null == migration)) {
      this.addIssue(message, typeRef, issueCode);
    }
    EObject _eContainer = typeRef.eContainer();
    if ((_eContainer instanceof TFormalParameter)) {
      final int parameterIndex = migration.getFpars().indexOf(typeRef.eContainer());
      this.addIssue(message, migration.getAstElement(), N4JSPackage.Literals.FUNCTION_DEFINITION__FPARS, parameterIndex, issueCode);
    } else {
      EObject _eContainer_1 = typeRef.eContainer();
      if ((_eContainer_1 instanceof TMigration)) {
        this.addIssue(message, migration.getAstElement(), N4JSPackage.Literals.FUNCTION_DEFINITION__RETURN_TYPE_REF, issueCode);
      } else {
        this.addIssue(message, typeRef, issueCode);
      }
    }
  }
}
