/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.ExportableElement;
import org.eclipse.n4js.n4JS.ExportedVariableDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.NamespaceImportSpecifier;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedPropertyAccessExpression;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.ThisLiteral;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.ts.typeRefs.BoundThisTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.SyntaxRelatedTElement;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TVariable;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeAccessModifier;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.StaticPolyfillHelper;
import org.eclipse.n4js.utils.StructuralTypesHelper;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

@SuppressWarnings("all")
public class N4JSAccessModifierValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  protected N4JSTypeSystem ts;
  
  @Inject
  protected ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private StructuralTypesHelper structuralTypesHelper;
  
  @Inject
  private StaticPolyfillHelper staticPolyfillHelper;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkExportedWhenVisibilityHigherThanPrivate(final TypeDefiningElement typeDefiningElement) {
    boolean _requireCheckExportedWhenVisibilityHigherThanPrivate = this.jsVariantHelper.requireCheckExportedWhenVisibilityHigherThanPrivate(typeDefiningElement);
    boolean _not = (!_requireCheckExportedWhenVisibilityHigherThanPrivate);
    if (_not) {
      return;
    }
    if ((typeDefiningElement instanceof ObjectLiteral)) {
      return;
    }
    if ((typeDefiningElement instanceof NamespaceImportSpecifier)) {
      return;
    }
    final Type type = typeDefiningElement.getDefinedType();
    if ((((type != null) && (!type.isExported())) && (type.getTypeAccessModifier().ordinal() > TypeAccessModifier.PRIVATE.ordinal()))) {
      if ((type instanceof SyntaxRelatedTElement)) {
        EObject _astElement = ((SyntaxRelatedTElement)type).getAstElement();
        boolean _tripleNotEquals = (_astElement != null);
        if (_tripleNotEquals) {
          final String message = IssueCodes.getMessageForCLF_NOT_EXPORTED_NOT_PRIVATE(this.keywordProvider.keyword(type), 
            this.keywordProvider.keyword(type.getTypeAccessModifier()));
          final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(((SyntaxRelatedTElement)type).getAstElement());
          this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), 
            IssueCodes.CLF_NOT_EXPORTED_NOT_PRIVATE);
        }
      }
    }
  }
  
  @Check
  public void checkInternalUsedOnlyWithPublicAndProtected(final ExportableElement exportableElement) {
    final EObject parent = exportableElement.eContainer();
    Annotation _xifexpression = null;
    if ((exportableElement instanceof AnnotableElement)) {
      _xifexpression = AnnotationDefinition.INTERNAL.getAnnotation(((AnnotableElement)exportableElement));
    }
    Annotation annotation = _xifexpression;
    if ((annotation == null)) {
      Annotation _xifexpression_1 = null;
      if ((parent instanceof ExportDeclaration)) {
        _xifexpression_1 = AnnotationDefinition.INTERNAL.getAnnotation(((AnnotableElement)parent));
      }
      annotation = _xifexpression_1;
    }
    if ((annotation != null)) {
      TypeAccessModifier _switchResult = null;
      final ExportableElement it = exportableElement;
      boolean _matched = false;
      if (it instanceof VariableStatement) {
        _matched=true;
        ExportedVariableDeclaration _head = IterableExtensions.<ExportedVariableDeclaration>head(Iterables.<ExportedVariableDeclaration>filter(((VariableStatement)it).getVarDecl(), ExportedVariableDeclaration.class));
        TVariable _definedVariable = null;
        if (_head!=null) {
          _definedVariable=_head.getDefinedVariable();
        }
        TypeAccessModifier _typeAccessModifier = null;
        if (_definedVariable!=null) {
          _typeAccessModifier=_definedVariable.getTypeAccessModifier();
        }
        _switchResult = _typeAccessModifier;
      }
      if (!_matched) {
        if (it instanceof FunctionDeclaration) {
          _matched=true;
          Type _definedType = ((FunctionDeclaration)it).getDefinedType();
          TypeAccessModifier _typeAccessModifier = null;
          if (_definedType!=null) {
            _typeAccessModifier=_definedType.getTypeAccessModifier();
          }
          _switchResult = _typeAccessModifier;
        }
      }
      if (!_matched) {
        if (it instanceof TypeDefiningElement) {
          _matched=true;
          Type _definedType = ((TypeDefiningElement)it).getDefinedType();
          TypeAccessModifier _typeAccessModifier = null;
          if (_definedType!=null) {
            _typeAccessModifier=_definedType.getTypeAccessModifier();
          }
          _switchResult = _typeAccessModifier;
        }
      }
      if (!_matched) {
        _switchResult = null;
      }
      final TypeAccessModifier typeAccessModifier = _switchResult;
      if (((typeAccessModifier != null) && (typeAccessModifier.ordinal() <= TypeAccessModifier.PROJECT.ordinal()))) {
        final String message = IssueCodes.getMessageForCLF_LOW_ACCESSOR_WITH_INTERNAL(this.keywordProvider.keyword(exportableElement), 
          this.keywordProvider.keyword(typeAccessModifier));
        final Pair<? extends EObject, ? extends EStructuralFeature> eObjectToNameFeature = this.findNameFeature(exportableElement);
        this.addIssue(message, eObjectToNameFeature.getKey(), eObjectToNameFeature.getValue(), IssueCodes.CLF_LOW_ACCESSOR_WITH_INTERNAL);
      }
    }
  }
  
  @Check
  public void checkTypeRefOptionalFlag(final TypeRef typeRef) {
    boolean _isFollowedByQuestionMark = typeRef.isFollowedByQuestionMark();
    if (_isFollowedByQuestionMark) {
      final EObject parent = typeRef.eContainer();
      final boolean isLegalUseOfOptional = this.isReturnTypeButNotOfAGetter(typeRef, parent);
      if ((!isLegalUseOfOptional)) {
        if ((parent instanceof TFormalParameter)) {
          return;
        } else {
          if (((parent instanceof N4FieldDeclaration) || (parent instanceof TField))) {
            final String message = IssueCodes.getMessageForCLF_FIELD_OPTIONAL_OLD_SYNTAX();
            final ICompositeNode node = NodeModelUtils.findActualNodeFor(typeRef);
            if ((node != null)) {
              this.addIssue(message, typeRef, node.getOffset(), node.getLength(), IssueCodes.CLF_FIELD_OPTIONAL_OLD_SYNTAX);
            }
          } else {
            final String message_1 = IssueCodes.getMessageForEXP_OPTIONAL_INVALID_PLACE();
            final ICompositeNode node_1 = NodeModelUtils.findActualNodeFor(typeRef);
            if ((node_1 != null)) {
              this.addIssue(message_1, typeRef, node_1.getOffset(), node_1.getLength(), IssueCodes.EXP_OPTIONAL_INVALID_PLACE);
            }
          }
        }
      }
    }
  }
  
  private boolean isReturnTypeButNotOfAGetter(final TypeRef typeRef, final EObject parent) {
    return (((((parent instanceof FunctionDefinition) && (((FunctionDefinition) parent).getReturnTypeRef() == typeRef)) && 
      (!(parent instanceof N4GetterDeclaration))) || (((parent instanceof TFunction) && (((TFunction) parent).getReturnTypeRef() == typeRef)) && 
      (!(parent instanceof TGetter)))) || ((parent instanceof FunctionTypeExpression) && (((FunctionTypeExpression) parent).getReturnTypeRef() == typeRef)));
  }
  
  @Check
  public void checkFieldConstFinalValidCombinations(final N4FieldDeclaration n4field) {
    if ((n4field.isConst() && n4field.isDeclaredStatic())) {
      final String msg = IssueCodes.getMessageForCLF_FIELD_CONST_STATIC();
      this.addIssue(msg, n4field, N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName(), IssueCodes.CLF_FIELD_CONST_STATIC);
    } else {
      if ((n4field.isConst() && n4field.isFinal())) {
        final String msg_1 = IssueCodes.getMessageForCLF_FIELD_CONST_FINAL();
        this.addIssue(msg_1, n4field, N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName(), IssueCodes.CLF_FIELD_CONST_FINAL);
      } else {
        if ((n4field.isFinal() && n4field.isDeclaredStatic())) {
          final String msg_2 = IssueCodes.getMessageForCLF_FIELD_FINAL_STATIC();
          this.addIssue(msg_2, n4field, N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName(), IssueCodes.CLF_FIELD_FINAL_STATIC);
        }
      }
    }
  }
  
  @Check
  public void checkFieldConstInitialization(final N4FieldDeclaration n4field) {
    boolean _constantHasInitializer = this.jsVariantHelper.constantHasInitializer(n4field);
    boolean _not = (!_constantHasInitializer);
    if (_not) {
      return;
    }
    if ((n4field.isConst() && (n4field.getExpression() == null))) {
      final String msg = IssueCodes.getMessageForCLF_FIELD_CONST_MISSING_INIT(n4field.getName());
      this.addIssue(msg, n4field, N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName(), IssueCodes.CLF_FIELD_CONST_MISSING_INIT);
    }
  }
  
  /**
   * 5.2.3.1, Constraint 58.3
   */
  @Check
  public void checkFieldFinalInitialization(final N4ClassifierDefinition n4classifier) {
    boolean _requireCheckFinalFieldIsInitialized = this.jsVariantHelper.requireCheckFinalFieldIsInitialized(n4classifier);
    boolean _not = (!_requireCheckFinalFieldIsInitialized);
    if (_not) {
      return;
    }
    final Function1<N4FieldDeclaration, Boolean> _function = (N4FieldDeclaration it) -> {
      return Boolean.valueOf((it.isFinal() && (it.getExpression() == null)));
    };
    Iterable<N4FieldDeclaration> finalFieldsWithoutInit = IterableExtensions.<N4FieldDeclaration>filter(IterableExtensions.<N4FieldDeclaration>filterNull(n4classifier.getOwnedFields()), _function);
    boolean _isEmpty = IterableExtensions.isEmpty(finalFieldsWithoutInit);
    if (_isEmpty) {
      return;
    }
    finalFieldsWithoutInit = this.filterFieldsInitializedViaSpecConstructor(n4classifier, finalFieldsWithoutInit);
    boolean _isEmpty_1 = IterableExtensions.isEmpty(finalFieldsWithoutInit);
    if (_isEmpty_1) {
      return;
    }
    finalFieldsWithoutInit = this.filterFieldsInitializedExplicitlyInConstructor(n4classifier, finalFieldsWithoutInit);
    boolean _isEmpty_2 = IterableExtensions.isEmpty(finalFieldsWithoutInit);
    if (_isEmpty_2) {
      return;
    }
    N4MethodDeclaration _ownedCtor = n4classifier.getOwnedCtor();
    N4MethodDeclaration _polyfilledOrOwnCtor = this.polyfilledOrOwnCtor(n4classifier);
    final boolean replacedByPolyfill = (_ownedCtor != _polyfilledOrOwnCtor);
    if (replacedByPolyfill) {
      final Consumer<N4FieldDeclaration> _function_1 = (N4FieldDeclaration it) -> {
        final String msg = IssueCodes.getMessageForCLF_FIELD_FINAL_MISSING_INIT_IN_STATIC_POLYFILL(it.getName());
        this.addIssue(msg, it, N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName(), 
          IssueCodes.CLF_FIELD_FINAL_MISSING_INIT_IN_STATIC_POLYFILL);
      };
      finalFieldsWithoutInit.forEach(_function_1);
    } else {
      final Consumer<N4FieldDeclaration> _function_2 = (N4FieldDeclaration it) -> {
        final String msg = IssueCodes.getMessageForCLF_FIELD_FINAL_MISSING_INIT(it.getName());
        this.addIssue(msg, it, N4JSPackage.eINSTANCE.getPropertyNameOwner_DeclaredName(), IssueCodes.CLF_FIELD_FINAL_MISSING_INIT);
      };
      finalFieldsWithoutInit.forEach(_function_2);
    }
  }
  
  private Iterable<N4FieldDeclaration> filterFieldsInitializedViaSpecConstructor(final N4ClassifierDefinition n4classifier, final Iterable<N4FieldDeclaration> finalFieldsWithoutInit) {
    Type _definedType = n4classifier.getDefinedType();
    boolean _tripleEquals = (null == _definedType);
    if (_tripleEquals) {
      return CollectionLiterals.<N4FieldDeclaration>emptyList();
    }
    final N4MethodDeclaration ctor = this.polyfilledOrOwnCtor(n4classifier);
    TMethod _xifexpression = null;
    if ((ctor == null)) {
      Type _definedType_1 = n4classifier.getDefinedType();
      _xifexpression = this.containerTypesHelper.fromContext(n4classifier).findConstructor(
        ((ContainerType<?>) _definedType_1));
    } else {
      TMember _definedTypeElement = ctor.getDefinedTypeElement();
      _xifexpression = ((TMethod) _definedTypeElement);
    }
    final TMethod tctor = _xifexpression;
    EList<TFormalParameter> _fpars = null;
    if (tctor!=null) {
      _fpars=tctor.getFpars();
    }
    TFormalParameter _findFirst = null;
    if (_fpars!=null) {
      final Function1<TFormalParameter, Boolean> _function = (TFormalParameter it) -> {
        return Boolean.valueOf(AnnotationDefinition.SPEC.hasAnnotation(it));
      };
      _findFirst=IterableExtensions.<TFormalParameter>findFirst(_fpars, _function);
    }
    final TFormalParameter specPar = _findFirst;
    TypeRef _typeRef = null;
    if (specPar!=null) {
      _typeRef=specPar.getTypeRef();
    }
    final TypeRef typeRef = _typeRef;
    if ((typeRef instanceof ThisTypeRefStructural)) {
      final BoundThisTypeRef boundThisTypeRef = TypeUtils.createBoundThisTypeRefStructural(
        TypeUtils.createTypeRef(n4classifier.getDefinedType()), ((ThisTypeRefStructural)typeRef));
      final Function1<TMember, String> _function_1 = (TMember it) -> {
        return it.getName();
      };
      final Set<String> specMemberFieldName = IterableExtensions.<String>toSet(IterableExtensions.<TMember, String>map(this.structuralTypesHelper.collectStructuralMembers(RuleEnvironmentExtensions.newRuleEnvironment(n4classifier), boundThisTypeRef, TypingStrategy.STRUCTURAL_FIELDS), _function_1));
      final Function1<N4FieldDeclaration, Boolean> _function_2 = (N4FieldDeclaration it) -> {
        boolean _contains = specMemberFieldName.contains(it.getName());
        return Boolean.valueOf((!_contains));
      };
      return IterableExtensions.<N4FieldDeclaration>filter(finalFieldsWithoutInit, _function_2);
    }
    return finalFieldsWithoutInit;
  }
  
  private Iterable<N4FieldDeclaration> filterFieldsInitializedExplicitlyInConstructor(final N4ClassifierDefinition n4classifier, final Iterable<N4FieldDeclaration> finalFieldsWithoutInit) {
    final N4MethodDeclaration ctor = this.polyfilledOrOwnCtor(n4classifier);
    Collection<TField> _xifexpression = null;
    Block _body = null;
    if (ctor!=null) {
      _body=ctor.getBody();
    }
    boolean _tripleEquals = (_body == null);
    if (_tripleEquals) {
      _xifexpression = Collections.<TField>unmodifiableList(CollectionLiterals.<TField>newArrayList());
    } else {
      final Function1<Statement, Iterable<EObject>> _function = (Statement it) -> {
        return IteratorExtensions.<EObject>toIterable(it.eAllContents());
      };
      final Function1<EObject, TField> _function_1 = (EObject it) -> {
        return this.isAssignmentToFinalFieldInThis(it);
      };
      _xifexpression = IterableExtensions.<TField>toSet(IterableExtensions.<TField>filterNull(IterableExtensions.<EObject, TField>map(Iterables.<EObject>concat(IteratorExtensions.<Iterable<EObject>>toIterable(IteratorExtensions.<Statement, Iterable<EObject>>map(ctor.getBody().getAllStatements(), _function))), _function_1)));
    }
    final Collection<TField> finalFieldsAssignedInCtor = _xifexpression;
    final Function1<N4FieldDeclaration, Boolean> _function_2 = (N4FieldDeclaration it) -> {
      boolean _contains = finalFieldsAssignedInCtor.contains(it.getDefinedField());
      return Boolean.valueOf((!_contains));
    };
    return IterableExtensions.<N4FieldDeclaration>filter(finalFieldsWithoutInit, _function_2);
  }
  
  private TField isAssignmentToFinalFieldInThis(final EObject astNode) {
    if ((astNode instanceof AssignmentExpression)) {
      final Expression lhs = ((AssignmentExpression)astNode).getLhs();
      if ((lhs instanceof ParameterizedPropertyAccessExpression)) {
        Expression _target = ((ParameterizedPropertyAccessExpression)lhs).getTarget();
        if ((_target instanceof ThisLiteral)) {
          final IdentifiableElement prop = ((ParameterizedPropertyAccessExpression)lhs).getProperty();
          if ((prop instanceof TField)) {
            boolean _isFinal = ((TField)prop).isFinal();
            if (_isFinal) {
              return ((TField)prop);
            }
          }
        }
      }
    }
    return null;
  }
  
  /**
   * fetch the statically polyfilled ctor, or the own one. returns null if both are not defined.
   */
  private N4MethodDeclaration polyfilledOrOwnCtor(final N4ClassifierDefinition n4classifier) {
    final boolean polyAware = n4classifier.getDefinedType().getContainingModule().isStaticPolyfillAware();
    N4ClassDeclaration _xifexpression = null;
    if (polyAware) {
      _xifexpression = this.staticPolyfillHelper.getStaticPolyfill(n4classifier.getDefinedType());
    } else {
      _xifexpression = null;
    }
    final N4ClassDeclaration polyfill = _xifexpression;
    N4MethodDeclaration _ownedCtor = null;
    if (polyfill!=null) {
      _ownedCtor=polyfill.getOwnedCtor();
    }
    final N4MethodDeclaration polyfillCtor = _ownedCtor;
    N4MethodDeclaration _elvis = null;
    if (polyfillCtor != null) {
      _elvis = polyfillCtor;
    } else {
      N4MethodDeclaration _ownedCtor_1 = n4classifier.getOwnedCtor();
      _elvis = _ownedCtor_1;
    }
    final N4MethodDeclaration ctor = _elvis;
    return ctor;
  }
}
