/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.common.base.Predicate;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.function.Function;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.ExportDeclaration;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.FieldAccessor;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDeclaration;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.GetterDeclaration;
import org.eclipse.n4js.n4JS.IdentifierRef;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.ReturnStatement;
import org.eclipse.n4js.n4JS.SetterDeclaration;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.ts.typeRefs.ComposedTypeRef;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExpression;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.Versionable;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TStructSetter;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.VoidType;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.TypeSystemHelper;
import org.eclipse.n4js.utils.EcoreUtilN4;
import org.eclipse.n4js.utils.N4JSLanguageHelper;
import org.eclipse.n4js.utils.nodemodel.HiddenLeafAccess;
import org.eclipse.n4js.utils.nodemodel.HiddenLeafs;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.validation.helper.FunctionValidationHelper;
import org.eclipse.n4js.validation.validators.StaticPolyfillValidatorExtension;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure3;

@SuppressWarnings("all")
public class N4JSFunctionValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private HiddenLeafAccess hla;
  
  @Inject
  private N4JSLanguageHelper languageHelper;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  /**
   * "Also, an ExpressionStatement cannot start with the function keyword because that might make it ambiguous with a FunctionDeclaration."
   * [ECMAScript] 12.4 Expression Statement, [N4JS] 8.2. Function Definition
   * 
   * see also ECMAScript Test Suite Sbp_12.5_A9_T3.js
   */
  @Check
  public void checkFunctionExpressionInExpressionStatement(final FunctionDeclaration functionDeclaration) {
    final EObject container = functionDeclaration.eContainer();
    if (((container instanceof Block) && this.jsVariantHelper.requireCheckFunctionExpressionInExpressionStatement(functionDeclaration))) {
      final String msg = IssueCodes.getMessageForFUN_BLOCK();
      String _name = functionDeclaration.getName();
      boolean _tripleNotEquals = (_name != null);
      if (_tripleNotEquals) {
        this.addIssue(msg, functionDeclaration, N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, IssueCodes.FUN_BLOCK);
      } else {
        this.addIssue(msg, container, functionDeclaration.eContainmentFeature(), IssueCodes.FUN_BLOCK);
      }
    }
  }
  
  /**
   * TODO once ISSUE-666 is resolved this method could be dropped when
   * the check is carried out with #checkFunctionReturn(FunctionOrFieldAccessor)
   * 
   * Return-Type checking.
   * 
   * [N4JSSpec] 7.1.4 Return Statement
   * 
   * Constraint 111
   * 
   * @see #checkFunctionReturn(FunctionOrFieldAccessor)
   */
  @Check
  public boolean checkSetter(final SetterDeclaration setterDeclaration) {
    return this.holdsFunctionReturnsVoid(setterDeclaration, true);
  }
  
  /**
   * Given a function/method with return type void, checking the lack of returns or presence of empty returns
   * 
   * @param functionDefinition definition with void-return-type
   * @param _void precomputed built-in void type
   * @param isSetter true for setter and therefore ensuring no return at all, false in case of ordinary function/methods
   * 			where TS already does the job
   */
  private boolean holdsFunctionReturnsVoid(final FunctionOrFieldAccessor functionOrFieldAccessor, final boolean isSetter) {
    final Iterable<ReturnStatement> retstatements = this.allReturnstatementsAsList(functionOrFieldAccessor);
    final VoidType _void = RuleEnvironmentExtensions.voidType(RuleEnvironmentExtensions.newRuleEnvironment(functionOrFieldAccessor));
    for (final ReturnStatement rst : retstatements) {
      Expression _expression = rst.getExpression();
      boolean _tripleNotEquals = (_expression != null);
      if (_tripleNotEquals) {
        final TypeRef expressionType = this.ts.tau(rst.getExpression());
        Versionable _xifexpression = null;
        if ((expressionType instanceof ParameterizedTypeRef)) {
          _xifexpression = ((ParameterizedTypeRef)expressionType).getDeclaredType();
        } else {
          _xifexpression = expressionType;
        }
        final Versionable actualType = _xifexpression;
        if (((actualType != null) && (!Objects.equal(actualType, _void)))) {
          if (isSetter) {
            final String msg = IssueCodes.getMessageForFUN_RETURNTYPE_VOID_FOR_SETTER_VIOLATED();
            this.addIssue(msg, rst, IssueCodes.FUN_RETURNTYPE_VOID_FOR_SETTER_VIOLATED);
            return false;
          } else {
          }
        }
      }
    }
    return true;
  }
  
  /**
   * Method for checking whether the name of a function definition such as:
   * <p>
   * <ul>
   * <li>{@link FunctionDeclaration <em>Function declaration</em>},</li>
   * <li>{@link N4MethodDeclaration <em>Method declaration</em>} or</li>
   * <li>{@link FunctionExpression <em>Function expression</em>}</li>
   * </ul>
   * conflicts with a reserved keyword or a future reserved keyword.
   * In case of conflicts this method creates a validation if the name violates the constraint.
   * </p>
   * IDEBUG-287
   * 
   * @param definition the function definition to validate in respect if its name.
   */
  @Check
  public boolean checkFunctionName(final FunctionDefinition definition) {
    final String name = Strings.nullToEmpty(definition.getName());
    final String desc = this.validatorMessageHelper.description(definition);
    String errorMessage = "";
    boolean _requireCheckFunctionName = this.jsVariantHelper.requireCheckFunctionName(definition);
    if (_requireCheckFunctionName) {
      if ((Boolean.valueOf((definition instanceof N4MethodDeclaration)) == Boolean.valueOf(false))) {
        boolean _contains = N4JSLanguageConstants.FUTURE_RESERVED_WORDS.contains(name);
        if (_contains) {
          errorMessage = IssueCodes.getMessageForFUN_NAME_RESERVED(desc, "future reserved word");
        }
        if (((!Objects.equal(N4JSLanguageConstants.YIELD_KEYWORD, name)) && this.languageHelper.getECMAKeywords().contains(name))) {
          errorMessage = IssueCodes.getMessageForFUN_NAME_RESERVED(desc, "keyword");
        }
      }
    }
    boolean _isNullOrEmpty = Strings.isNullOrEmpty(errorMessage);
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      EStructuralFeature _switchResult = null;
      boolean _matched = false;
      if (definition instanceof FunctionDeclaration) {
        _matched=true;
        _switchResult = N4JSPackage.Literals.FUNCTION_DECLARATION__NAME;
      }
      if (!_matched) {
        if (definition instanceof N4MethodDeclaration) {
          _matched=true;
          _switchResult = N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME;
        }
      }
      if (!_matched) {
        if (definition instanceof FunctionExpression) {
          _matched=true;
          _switchResult = N4JSPackage.Literals.FUNCTION_EXPRESSION__NAME;
        }
      }
      if (!_matched) {
        _switchResult = null;
      }
      EStructuralFeature feature = _switchResult;
      this.addIssue(org.eclipse.xtext.util.Strings.toFirstUpper(errorMessage), definition, feature, IssueCodes.FUN_NAME_RESERVED);
    }
    return Strings.isNullOrEmpty(errorMessage);
  }
  
  /**
   * Grab all returns in body that apply to this function/get/set. (Leave out returns of nested definitions)
   * @param functionOrFieldAccessor body to inspect
   * @return all return statements in a functionOrFieldAccessor
   * 				(=function-definition or field accessor) (nested) but not in included functionExpressions, function definitions ....
   */
  private Iterable<ReturnStatement> allReturnstatementsAsList(final FunctionOrFieldAccessor functionOrFieldAccessor) {
    List<ReturnStatement> _xifexpression = null;
    Block _body = functionOrFieldAccessor.getBody();
    boolean _tripleEquals = (_body == null);
    if (_tripleEquals) {
      _xifexpression = Collections.<ReturnStatement>unmodifiableList(CollectionLiterals.<ReturnStatement>newArrayList());
    } else {
      final Predicate<EObject> _function = (EObject it) -> {
        return (!((it instanceof Expression) || (it instanceof FunctionOrFieldAccessor)));
      };
      _xifexpression = IteratorExtensions.<ReturnStatement>toList(Iterators.<ReturnStatement>filter(EcoreUtilN4.getAllContentsFiltered(functionOrFieldAccessor.getBody(), _function), ReturnStatement.class));
    }
    final List<ReturnStatement> retsByEcore = _xifexpression;
    return retsByEcore;
  }
  
  /**
   * Assuring all return statements do have an expression
   * @param returnTypeRef only used for error message
   */
  @Check
  public boolean checkReturnExpression(final ReturnStatement retStmt) {
    final FunctionOrFieldAccessor fofa = EcoreUtil2.<FunctionOrFieldAccessor>getContainerOfType(retStmt, FunctionOrFieldAccessor.class);
    boolean _requireCheckFunctionReturn = this.jsVariantHelper.requireCheckFunctionReturn(fofa);
    boolean _not = (!_requireCheckFunctionReturn);
    if (_not) {
      return false;
    }
    if ((fofa instanceof SetterDeclaration)) {
      return false;
    }
    if ((fofa == null)) {
      return false;
    }
    boolean _isReturnValueOptional = fofa.isReturnValueOptional();
    if (_isReturnValueOptional) {
      return false;
    }
    final VoidType _void = RuleEnvironmentExtensions.voidType(RuleEnvironmentExtensions.newRuleEnvironment(fofa));
    final TypeRef returnTypeRef = this.tsh.getExpectedTypeOfFunctionOrFieldAccessor(null, fofa);
    boolean _xifexpression = false;
    if ((fofa instanceof FieldAccessor)) {
      _xifexpression = TypeUtils.isOrContainsType(((FieldAccessor)fofa).getDeclaredTypeRef(), _void);
    } else {
      boolean _xifexpression_1 = false;
      if ((fofa instanceof FunctionDefinition)) {
        _xifexpression_1 = TypeUtils.isOrContainsType(returnTypeRef, _void);
      } else {
        _xifexpression_1 = false;
      }
      _xifexpression = _xifexpression_1;
    }
    final boolean isDeclaredVoid = _xifexpression;
    final boolean isUndefined = TypeUtils.isUndefined(returnTypeRef);
    final boolean isVoid = TypeUtils.isOrContainsType(returnTypeRef, _void);
    final boolean isComposed = ((returnTypeRef instanceof ComposedTypeRef) && (((ComposedTypeRef) returnTypeRef).getTypeRefs().size() > 1));
    final boolean isGetter = (fofa instanceof GetterDeclaration);
    if (((!isGetter) && (((isDeclaredVoid || isVoid) || isUndefined) || isComposed))) {
      return false;
    }
    if (((returnTypeRef != null) && (retStmt.getExpression() == null))) {
      final String msg = IssueCodes.getMessageForFUN_MISSING_RETURN_EXPRESSION(returnTypeRef.getTypeRefAsString());
      this.addIssue(msg, retStmt, IssueCodes.FUN_MISSING_RETURN_EXPRESSION);
      return true;
    }
    return false;
  }
  
  /**
   * additional check on top of {@link #checkFunctionName()}
   */
  @Check
  public void checkFunctionDeclarationName(final FunctionDeclaration functionDeclaration) {
    String _name = functionDeclaration.getName();
    boolean _tripleEquals = (_name == null);
    if (_tripleEquals) {
      final EObject container = functionDeclaration.eContainer();
      if ((container instanceof ExportDeclaration)) {
        boolean _isDefaultExport = ((ExportDeclaration)container).isDefaultExport();
        if (_isDefaultExport) {
          return;
        }
      }
      Block _body = functionDeclaration.getBody();
      boolean _tripleNotEquals = (_body != null);
      if (_tripleNotEquals) {
        final ICompositeNode firstNode = NodeModelUtils.findActualNodeFor(functionDeclaration);
        final ICompositeNode lastNode = NodeModelUtils.findActualNodeFor(functionDeclaration.getBody());
        final HiddenLeafs hLeafs = this.hla.getHiddenLeafsBefore(lastNode);
        final int off = firstNode.getOffset();
        int _offset = hLeafs.getOffset();
        int _offset_1 = firstNode.getOffset();
        final int len = (_offset - _offset_1);
        this.addIssue(IssueCodes.getMessageForFUN_NAME_MISSING(), functionDeclaration, off, len, IssueCodes.FUN_NAME_MISSING);
      } else {
        this.addIssue(IssueCodes.getMessageForFUN_NAME_MISSING(), functionDeclaration, IssueCodes.FUN_NAME_MISSING);
      }
    }
  }
  
  @Check
  public void checkFunctionDeclarationBody(final FunctionDeclaration functionDeclaration) {
    if ((((functionDeclaration.getBody() == null) && (functionDeclaration.getDefinedType() instanceof TFunction)) && 
      (!((TFunction) functionDeclaration.getDefinedType()).isExternal()))) {
      this.addIssue(IssueCodes.getMessageForFUN_BODY(), functionDeclaration, N4JSPackage.Literals.FUNCTION_DECLARATION__NAME, 
        IssueCodes.FUN_BODY);
    }
  }
  
  @Check
  public void checkParameters(final SetterDeclaration fun) {
    boolean _xifexpression = false;
    FormalParameter _fpar = null;
    if (fun!=null) {
      _fpar=fun.getFpar();
    }
    boolean _tripleNotEquals = (_fpar != null);
    if (_tripleNotEquals) {
      _xifexpression = fun.getFpar().isVariadic();
    } else {
      _xifexpression = false;
    }
    final boolean isVariadic = _xifexpression;
    boolean _xifexpression_1 = false;
    FormalParameter _fpar_1 = null;
    if (fun!=null) {
      _fpar_1=fun.getFpar();
    }
    boolean _tripleNotEquals_1 = (_fpar_1 != null);
    if (_tripleNotEquals_1) {
      _xifexpression_1 = fun.getFpar().isHasInitializerAssignment();
    } else {
      _xifexpression_1 = false;
    }
    final boolean hasInitializerAssignment = _xifexpression_1;
    this.<FormalParameter>internalCheckSetterParameters(fun.getFpar(), isVariadic, hasInitializerAssignment);
  }
  
  @Check
  public void checkParameters(final TStructSetter fun) {
    boolean _xifexpression = false;
    TFormalParameter _fpar = null;
    if (fun!=null) {
      _fpar=fun.getFpar();
    }
    boolean _tripleNotEquals = (_fpar != null);
    if (_tripleNotEquals) {
      _xifexpression = fun.getFpar().isVariadic();
    } else {
      _xifexpression = false;
    }
    final boolean isVariadic = _xifexpression;
    boolean _xifexpression_1 = false;
    TFormalParameter _fpar_1 = null;
    if (fun!=null) {
      _fpar_1=fun.getFpar();
    }
    boolean _tripleNotEquals_1 = (_fpar_1 != null);
    if (_tripleNotEquals_1) {
      _xifexpression_1 = fun.getFpar().isHasInitializerAssignment();
    } else {
      _xifexpression_1 = false;
    }
    final boolean hasInitializerAssignment = _xifexpression_1;
    this.<TFormalParameter>internalCheckSetterParameters(fun.getFpar(), isVariadic, hasInitializerAssignment);
  }
  
  private <T extends EObject> void internalCheckSetterParameters(final T fpar, final boolean isVariadic, final boolean hasInitializerAssignment) {
    if (isVariadic) {
      final String msg = IssueCodes.getMessageForFUN_SETTER_CANT_BE_VARIADIC();
      this.addIssue(msg, fpar, IssueCodes.FUN_SETTER_CANT_BE_VARIADIC);
    }
    if (hasInitializerAssignment) {
      final String msg_1 = IssueCodes.getMessageForFUN_SETTER_CANT_BE_DEFAULT();
      this.addIssue(msg_1, fpar, IssueCodes.FUN_SETTER_CANT_BE_DEFAULT);
    }
  }
  
  @Check
  public void checkOptionalModifier(final FormalParameter fpar) {
    if (((fpar.getDeclaredTypeRef() != null) && fpar.getDeclaredTypeRef().isFollowedByQuestionMark())) {
      final String msg = IssueCodes.getMessageForFUN_PARAM_OPTIONAL_WRONG_SYNTAX(fpar.getName());
      this.addIssue(msg, fpar, IssueCodes.FUN_PARAM_OPTIONAL_WRONG_SYNTAX);
    }
  }
  
  @Check
  public void checkOptionalModifierT(final TFormalParameter fpar) {
    if (((fpar.getTypeRef() != null) && fpar.getTypeRef().isFollowedByQuestionMark())) {
      TypeRef _typeRef = fpar.getTypeRef();
      Type _declaredType = null;
      if (_typeRef!=null) {
        _declaredType=_typeRef.getDeclaredType();
      }
      String _name = null;
      if (_declaredType!=null) {
        _name=_declaredType.getName();
      }
      final String msg = IssueCodes.getMessageForFUN_PARAM_OPTIONAL_WRONG_SYNTAX(_name);
      this.addIssue(msg, fpar, IssueCodes.FUN_PARAM_OPTIONAL_WRONG_SYNTAX);
    }
  }
  
  @Check
  public void checkFormalParametersIn(final FunctionTypeExpression fun) {
    this.internalCheckOptionalsHaveType(((TFormalParameter[])Conversions.unwrapArray(fun.getFpars(), TFormalParameter.class)));
    final Procedure3<String, String, EObject> _function = (String msg, String id, EObject eObj) -> {
      this.addIssue(msg, eObj, id);
    };
    final Procedure3<String, String, EObject> issueConsumer = _function;
    final java.util.function.Predicate<TFormalParameter> _function_1 = (TFormalParameter it) -> {
      return it.isVariadic();
    };
    final java.util.function.Predicate<TFormalParameter> _function_2 = (TFormalParameter it) -> {
      return it.isHasInitializerAssignment();
    };
    final Function<TFormalParameter, String> _function_3 = (TFormalParameter it) -> {
      TypeRef _typeRef = it.getTypeRef();
      Type _declaredType = null;
      if (_typeRef!=null) {
        _declaredType=_typeRef.getDeclaredType();
      }
      String _name = null;
      if (_declaredType!=null) {
        _name=_declaredType.getName();
      }
      return _name;
    };
    FunctionValidationHelper.<TFormalParameter>internalCheckFormalParameters(((TFormalParameter[])Conversions.unwrapArray(fun.getFpars(), TFormalParameter.class)), _function_1, _function_2, _function_3, new FunctionValidationHelper.TripleConsumer<String, String, EObject>() {
        public void accept(String arg0, String arg1, EObject arg2) {
          issueConsumer.apply(arg0, arg1, arg2);
        }
    });
  }
  
  @Check
  public void checkFormalParametersIn(final TFunction fun) {
    this.internalCheckOptionalsHaveType(((TFormalParameter[])Conversions.unwrapArray(fun.getFpars(), TFormalParameter.class)));
    final Procedure3<String, String, EObject> _function = (String msg, String id, EObject eObj) -> {
      this.addIssue(msg, eObj, id);
    };
    final Procedure3<String, String, EObject> issueConsumer = _function;
    final java.util.function.Predicate<TFormalParameter> _function_1 = (TFormalParameter it) -> {
      return it.isVariadic();
    };
    final java.util.function.Predicate<TFormalParameter> _function_2 = (TFormalParameter it) -> {
      return it.isHasInitializerAssignment();
    };
    final Function<TFormalParameter, String> _function_3 = (TFormalParameter it) -> {
      TypeRef _typeRef = it.getTypeRef();
      Type _declaredType = null;
      if (_typeRef!=null) {
        _declaredType=_typeRef.getDeclaredType();
      }
      String _name = null;
      if (_declaredType!=null) {
        _name=_declaredType.getName();
      }
      return _name;
    };
    FunctionValidationHelper.<TFormalParameter>internalCheckFormalParameters(((TFormalParameter[])Conversions.unwrapArray(fun.getFpars(), TFormalParameter.class)), _function_1, _function_2, _function_3, new FunctionValidationHelper.TripleConsumer<String, String, EObject>() {
        public void accept(String arg0, String arg1, EObject arg2) {
          issueConsumer.apply(arg0, arg1, arg2);
        }
    });
  }
  
  private void internalCheckOptionalsHaveType(final TFormalParameter[] fpars) {
    for (final TFormalParameter fp : fpars) {
      if ((fp.hasASTInitializer() && (!"undefined".equals(fp.getAstInitializer())))) {
        this.addIssue(IssueCodes.getMessageForFUN_PARAM_INITIALIZER_ONLY_UNDEFINED_ALLOWED(), fp, IssueCodes.FUN_PARAM_INITIALIZER_ONLY_UNDEFINED_ALLOWED);
      }
    }
  }
  
  @Check
  public void checkFormalParametersIn(final FunctionDefinition fun) {
    this.internalCheckInitializerBindings(fun);
    final Procedure3<String, String, EObject> _function = (String msg, String id, EObject eObj) -> {
      this.addIssue(msg, eObj, id);
    };
    final Procedure3<String, String, EObject> issueConsumer = _function;
    final java.util.function.Predicate<FormalParameter> _function_1 = (FormalParameter it) -> {
      return it.isVariadic();
    };
    final java.util.function.Predicate<FormalParameter> _function_2 = (FormalParameter it) -> {
      return it.isHasInitializerAssignment();
    };
    final Function<FormalParameter, String> _function_3 = (FormalParameter it) -> {
      return it.getName();
    };
    FunctionValidationHelper.<FormalParameter>internalCheckFormalParameters(((FormalParameter[])Conversions.unwrapArray(fun.getFpars(), FormalParameter.class)), _function_1, _function_2, _function_3, new FunctionValidationHelper.TripleConsumer<String, String, EObject>() {
        public void accept(String arg0, String arg1, EObject arg2) {
          issueConsumer.apply(arg0, arg1, arg2);
        }
    });
  }
  
  private void internalCheckInitializerBindings(final FunctionDefinition fun) {
    Block _body = fun.getBody();
    boolean _tripleEquals = (_body == null);
    if (_tripleEquals) {
      return;
    }
    final Function1<FormalParameter, List<IdentifierRef>> _function = (FormalParameter it) -> {
      return EcoreUtil2.<IdentifierRef>eAllOfType(it, IdentifierRef.class);
    };
    final Iterator<IdentifierRef> idRefs = Iterables.<IdentifierRef>concat(ListExtensions.<FormalParameter, List<IdentifierRef>>map(fun.getFpars(), _function)).iterator();
    final Function1<VariableDeclaration, String> _function_1 = (VariableDeclaration it) -> {
      return it.getName();
    };
    final List<String> varDeclNamesInBody = ListExtensions.<VariableDeclaration, String>map(EcoreUtil2.<VariableDeclaration>eAllOfType(fun.getBody(), VariableDeclaration.class), _function_1);
    while (idRefs.hasNext()) {
      {
        final IdentifierRef idRef = idRefs.next();
        boolean _contains = varDeclNamesInBody.contains(idRef.getId().getName());
        if (_contains) {
          final FormalParameter fpar = EcoreUtil2.<FormalParameter>getContainerOfType(idRef, FormalParameter.class);
          final String msg = IssueCodes.getMessageForFUN_PARAM_INITIALIZER_ILLEGAL_REFERENCE_TO_BODY_VARIABLE(fpar.getName(), idRef.getId().getName());
          this.addIssue(msg, idRef, IssueCodes.FUN_PARAM_INITIALIZER_ILLEGAL_REFERENCE_TO_BODY_VARIABLE);
        }
      }
    }
  }
  
  /**
   * IDEBUG-211 invalid combination of undefined, variadic & omitting type
   */
  public void holdsModifierOfParamsHaveType(final EList<FormalParameter> list) {
    for (final FormalParameter fp : list) {
      boolean _isHasInitializerAssignment = fp.getDefinedTypeElement().isHasInitializerAssignment();
      if (_isHasInitializerAssignment) {
        boolean _isVariadic = fp.isVariadic();
        if (_isVariadic) {
          this.addIssue(IssueCodes.getMessageForFUN_PARAM_VARIADIC_WITH_INITIALIZER(), fp, IssueCodes.FUN_PARAM_VARIADIC_WITH_INITIALIZER);
        }
      }
    }
  }
  
  /**
   * IDEBUG-211 invalid combination of undefined, variadic & omitting type
   */
  public void holdsModifierOfParamsHaveTType(final List<TFormalParameter> list) {
    for (final TFormalParameter fp : list) {
      boolean _isHasInitializerAssignment = fp.isHasInitializerAssignment();
      if (_isHasInitializerAssignment) {
        boolean _isVariadic = fp.isVariadic();
        if (_isVariadic) {
          this.addIssue(IssueCodes.getMessageForFUN_PARAM_VARIADIC_WITH_INITIALIZER(), fp, IssueCodes.FUN_PARAM_VARIADIC_WITH_INITIALIZER);
        }
      }
    }
  }
  
  /**
   * IDE-1534 Only Promise allowed as inferred return type of an async {@link FunctionDefinition}
   */
  @Check
  public void checkNonVoidAsyncMethod(final FunctionDefinition funDef) {
    if ((funDef.isAsync() && (null != funDef.getDefinedType()))) {
      Type _definedType = funDef.getDefinedType();
      final TypeRef tfunctionRetType = ((TFunction) _definedType).getReturnTypeRef();
      boolean _isVoid = TypeUtils.isVoid(tfunctionRetType);
      if (_isVoid) {
        final String message = IssueCodes.getMessageForTYS_NON_VOID_ASYNC();
        this.addIssue(message, funDef, IssueCodes.TYS_NON_VOID_ASYNC);
      }
    }
  }
  
  /**
   * IDE-1534 The return type of an async {@link FunctionDefinition} is not allowed to refer to the this-type.
   */
  @Check
  public void checkNoThisAsyncMethod(final FunctionDefinition funDef) {
    boolean _isAsync = funDef.isAsync();
    if (_isAsync) {
      boolean _isOrContainsThisType = TypeUtils.isOrContainsThisType(funDef.getReturnTypeRef());
      if (_isOrContainsThisType) {
        final String message = IssueCodes.getMessageForTYS_NON_THIS_ASYNC();
        this.addIssue(message, funDef, IssueCodes.TYS_NON_THIS_ASYNC);
      }
    }
  }
  
  @Override
  public void addIssue(final String message, final EObject source, final EStructuralFeature feature, final String issueCode, final String... issueData) {
    super.addIssue(message, source, feature, issueCode, issueData);
  }
  
  /**
   * IDE-1735 restrict content in filling modules.
   */
  @Check
  public void checkFunctionDeclarationInStaticPolyfillModule(final FunctionDeclaration functionDeclaration) {
    StaticPolyfillValidatorExtension.internalCheckNotInStaticPolyfillModule(functionDeclaration, this);
  }
  
  /**
   * GHOLD-234 add warning for unused type variables in function declarations.
   */
  @Check
  public void checkNoUnusedTypeParameters(final FunctionDeclaration functionDeclaration) {
    this.<FunctionDeclaration>internalCheckNoUnusedTypeParameters(functionDeclaration);
  }
  
  /**
   * GHOLD-234 add warning for unused type variables in function expressions.
   */
  @Check
  public void checkNoUnusedTypeParameters(final FunctionExpression functionExpression) {
    this.<FunctionExpression>internalCheckNoUnusedTypeParameters(functionExpression);
  }
  
  /**
   * GHOLD-234 add warning for unused type variables in function type expressions.
   */
  @Check
  public void checkNoUnusedTypeParameters(final FunctionTypeExpression functionTypeExp) {
    this.internalCheckNoUnusedTypeParameters(functionTypeExp);
  }
}
