/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.GenericDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDefinition;
import org.eclipse.n4js.n4JS.N4ClassifierDefinition;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4JSPackage;
import org.eclipse.n4js.n4JS.N4MemberAnnotationList;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.PropertyNameOwner;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.ThisTypeRefStructural;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRefsPackage;
import org.eclipse.n4js.ts.types.ContainerType;
import org.eclipse.n4js.ts.types.FieldAccessor;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.MemberAccessModifier;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TSetter;
import org.eclipse.n4js.ts.types.TStructMember;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.VoidType;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.ContainerTypesHelper;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.validation.AbstractN4JSDeclarativeValidator;
import org.eclipse.n4js.validation.IssueCodes;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.n4js.validation.validators.LazyOverrideAwareMemberCollector;
import org.eclipse.xsemantics.runtime.RuleEnvironment;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.util.Tuples;
import org.eclipse.xtext.validation.Check;
import org.eclipse.xtext.validation.EValidatorRegistrar;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Validation of rules that apply to individual members of a classifier.<p>
 * 
 * Validation of rules about members:
 * <ul>
 * <li>if the rules require to take into account the other owned or inherited members of the
 *     containing classifier, then the validation is contained in {@link N4JSClassifierValidator},
 * <li>if they can be checked by only looking at each member individually, then the validation
 *     is contained here.
 * </ul>
 */
@SuppressWarnings("all")
public class N4JSMemberValidator extends AbstractN4JSDeclarativeValidator {
  @Inject
  private ContainerTypesHelper containerTypesHelper;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * NEEDED
   * 
   * when removed check methods will be called twice once by N4JSValidator, and once by
   * AbstractDeclarativeN4JSValidator
   */
  @Override
  public void register(final EValidatorRegistrar registrar) {
  }
  
  @Check
  public void checkN4MemberDeclaration(final N4MemberDeclaration n4Member) {
    if ((n4Member instanceof N4MemberAnnotationList)) {
      return;
    }
    final TMember it = n4Member.getDefinedTypeElement();
    if ((it == null)) {
      return;
    }
    this.internalCheckNameStartsWithDollar(it);
    this.internalCheckAbstractAndFinal(it);
    this.internalCheckPrivateOrProjectWithInternalAnnotation(n4Member, it);
  }
  
  /**
   * This check was previously covered by ASTStructureValidator#validateName, but since computed property names are
   * only set during post processing as of IDE-2468, we now have to perform validations for computed property names
   * here.
   */
  @Check
  public void checkComputedPropertyName(final PropertyNameOwner owner) {
    boolean _hasComputedPropertyName = owner.hasComputedPropertyName();
    if (_hasComputedPropertyName) {
      final String name = owner.getName();
      if ((name != null)) {
        boolean _isValidName = owner.isValidName();
        boolean _not = (!_isValidName);
        if (_not) {
          final String message = IssueCodes.getMessageForAST_RESERVED_IDENTIFIER(name);
          this.addIssue(message, owner, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.AST_RESERVED_IDENTIFIER);
        }
      }
    }
  }
  
  @Check
  public void checkN4FieldDeclaration(final N4FieldDeclaration n4Field) {
    if ((n4Field instanceof N4MemberAnnotationList)) {
      return;
    }
    final TMember member = n4Field.getDefinedTypeElement();
    if ((member == null)) {
      return;
    }
    this.holdsMinimalMemberAccessModifier(member);
  }
  
  @Check
  public void checkN4MethodDeclaration(final N4MethodDeclaration n4Method) {
    if ((n4Method instanceof N4MemberAnnotationList)) {
      return;
    }
    this.holdsCallableConstructorConstraints(n4Method);
    TMember _definedTypeElement = n4Method.getDefinedTypeElement();
    boolean _tripleEquals = (_definedTypeElement == null);
    if (_tripleEquals) {
      return;
    }
    TMember _definedTypeElement_1 = n4Method.getDefinedTypeElement();
    final TMethod tmethod = ((TMethod) _definedTypeElement_1);
    this.holdsAbstractAndBodyPropertiesOfMethod(tmethod);
    this.holdsConstructorConstraints(tmethod);
  }
  
  @Check
  public void checkN4GetterDeclaration(final N4GetterDeclaration n4Getter) {
    if ((n4Getter instanceof N4MemberAnnotationList)) {
      return;
    }
    TGetter _definedGetter = n4Getter.getDefinedGetter();
    boolean _tripleEquals = (_definedGetter == null);
    if (_tripleEquals) {
      return;
    }
    final TGetter it = n4Getter.getDefinedGetter();
    this.holdsAbstractAndBodyPropertiesOfMethod(it);
    this.internalCheckGetterType(n4Getter);
  }
  
  @Check
  public void checkN4SetterDeclaration(final N4SetterDeclaration n4Setter) {
    if ((n4Setter instanceof N4MemberAnnotationList)) {
      return;
    }
    TSetter _definedSetter = n4Setter.getDefinedSetter();
    boolean _tripleEquals = (_definedSetter == null);
    if (_tripleEquals) {
      return;
    }
    final TSetter it = n4Setter.getDefinedSetter();
    this.holdsAbstractAndBodyPropertiesOfMethod(it);
  }
  
  @Check
  public void checkN4StructuralWithOnTypeVariables(final ParameterizedTypeRefStructural ptrs) {
    Type _declaredType = ptrs.getDeclaredType();
    boolean _not = (!(_declaredType instanceof TypeVariable));
    if (_not) {
      return;
    }
    EList<TStructMember> _astStructuralMembers = ptrs.getAstStructuralMembers();
    for (final TStructMember sm : _astStructuralMembers) {
      {
        final String message = IssueCodes.getMessageForTYS_ADDITIONAL_STRUCTURAL_MEMBERS_ON_TYPE_VARS();
        this.addIssue(message, sm, IssueCodes.TYS_ADDITIONAL_STRUCTURAL_MEMBERS_ON_TYPE_VARS);
      }
    }
  }
  
  public boolean holdsAbstractAndBodyPropertiesOfMethod(final TMember accessorOrMethod) {
    return ((((this.holdsAbstractOrHasBody(accessorOrMethod) && this.holdsAbstractMethodMustHaveNoBody(accessorOrMethod)) && this.holdsAbstractMethodMustNotBeStatic(accessorOrMethod)) && this.holdsAbstractMemberContainedInAbstractClassifier(accessorOrMethod)) && this.holdsMinimalMemberAccessModifier(accessorOrMethod));
  }
  
  private void internalCheckNameStartsWithDollar(final TMember member) {
    boolean _requireCheckNameStartsWithDollar = this.jsVariantHelper.requireCheckNameStartsWithDollar(member);
    boolean _not = (!_requireCheckNameStartsWithDollar);
    if (_not) {
      return;
    }
    String _name = null;
    if (member!=null) {
      _name=member.getName();
    }
    final String name = _name;
    if (((name != null) && name.startsWith("$"))) {
      final String message = IssueCodes.getMessageForCLF_NAME_DOLLAR();
      this.addIssue(message, member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_NAME_DOLLAR);
    }
  }
  
  private void internalCheckAbstractAndFinal(final TMember member) {
    boolean _isFinal = member.isFinal();
    if (_isFinal) {
      boolean _isAbstract = member.isAbstract();
      if (_isAbstract) {
        final String message = IssueCodes.getMessageForCLF_ABSTRACT_FINAL(this.keywordProvider.keyword(member));
        this.addIssue(message, member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_ABSTRACT_FINAL);
      } else {
        if (((member.getContainingType() instanceof TInterface) && (!(member instanceof TMethod)))) {
          final String message_1 = IssueCodes.getMessageForCLF_NO_FINAL_INTERFACE_MEMBER();
          this.addIssue(message_1, member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
            IssueCodes.CLF_NO_FINAL_INTERFACE_MEMBER);
        }
      }
    }
  }
  
  private boolean holdsConstructorConstraints(final TMethod method) {
    boolean _isConstructor = method.isConstructor();
    if (_isConstructor) {
      boolean _holdsConstructorInInterfaceDoesNotHaveBody = this.holdsConstructorInInterfaceDoesNotHaveBody(method);
      boolean _not = (!_holdsConstructorInInterfaceDoesNotHaveBody);
      if (_not) {
        return false;
      }
      boolean _holdsConstructorInInterfaceRequiresCovarianceAnnotation = this.holdsConstructorInInterfaceRequiresCovarianceAnnotation(method);
      boolean _not_1 = (!_holdsConstructorInInterfaceRequiresCovarianceAnnotation);
      if (_not_1) {
        return false;
      }
      boolean _holdsConstructorNoReturnType = this.holdsConstructorNoReturnType(method);
      boolean _not_2 = (!_holdsConstructorNoReturnType);
      if (_not_2) {
        return false;
      }
      boolean _holdsConstructorNoTypeParameters = this.holdsConstructorNoTypeParameters(method);
      boolean _not_3 = (!_holdsConstructorNoTypeParameters);
      if (_not_3) {
        return false;
      }
      boolean result = this.holdsConstructorModifiers(method);
      return (this.holdsRequiredExplicitSuperCallIsFound(method) && result);
    }
    return true;
  }
  
  /**
   * N4JS spec constraints 51.1
   */
  private boolean holdsConstructorModifiers(final TMethod constructor) {
    if (((((constructor.isAbstract() && (!(constructor.getContainingType() instanceof TInterface))) || constructor.isStatic()) || constructor.isFinal()) || this.getHasIllegalOverride(constructor))) {
      final String message = IssueCodes.getMessageForCLF_CTOR_ILLEGAL_MODIFIER();
      this.addIssue(message, constructor.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_CTOR_ILLEGAL_MODIFIER);
      return false;
    }
    return true;
  }
  
  /**
   * @return true if not a static polyfill but has a {@code @Override} annotation.
   */
  private boolean getHasIllegalOverride(final TMethod constructor) {
    return ((!constructor.getContainingType().isStaticPolyfill()) && 
      AnnotationDefinition.OVERRIDE.hasAnnotation(((N4MethodDeclaration) constructor.getAstElement())));
  }
  
  /**
   * Requirement 56 (Defining and Calling Constructors), #5.a
   */
  private boolean holdsConstructorInInterfaceDoesNotHaveBody(final TMethod constructor) {
    if (((constructor.getContainingType() instanceof TInterface) && (!constructor.isHasNoBody()))) {
      this.addIssue(IssueCodes.getMessageForITF_CONSTRUCTOR_BODY(), constructor.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
        IssueCodes.ITF_CONSTRUCTOR_BODY);
      return false;
    }
    return true;
  }
  
  /**
   * Requirement 56 (Defining and Calling Constructors), #5.b
   */
  private boolean holdsConstructorInInterfaceRequiresCovarianceAnnotation(final TMethod constructor) {
    final ContainerType<?> container = constructor.getContainingType();
    if (((container instanceof TInterface) && (!N4JSLanguageUtils.hasCovariantConstructor(((TInterface) container))))) {
      this.addIssue(IssueCodes.getMessageForITF_CONSTRUCTOR_COVARIANCE(), constructor.getAstElement(), 
        N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.ITF_CONSTRUCTOR_COVARIANCE);
      return false;
    }
    return true;
  }
  
  /**
   * Requirement 56 (Defining and Calling Constructors), #6
   */
  private boolean holdsConstructorNoReturnType(final TMethod constructor) {
    EObject _astElement = constructor.getAstElement();
    final N4MethodDeclaration constructorDecl = ((N4MethodDeclaration) _astElement);
    TypeRef _returnTypeRef = constructorDecl.getReturnTypeRef();
    boolean _tripleNotEquals = (_returnTypeRef != null);
    if (_tripleNotEquals) {
      this.addIssue(IssueCodes.getMessageForCLF_CTOR_RETURN_TYPE(), constructorDecl, N4JSPackage.Literals.FUNCTION_DEFINITION__RETURN_TYPE_REF, 
        IssueCodes.CLF_CTOR_RETURN_TYPE);
      return false;
    }
    return true;
  }
  
  /**
   * Requirement 56 (Defining and Calling Constructors), #8
   */
  private boolean holdsConstructorNoTypeParameters(final TMethod method) {
    boolean _isEmpty = method.getTypeVars().isEmpty();
    if (_isEmpty) {
      return true;
    }
    EObject _astElement = method.getAstElement();
    final N4MethodDeclaration constructorDecl = ((N4MethodDeclaration) _astElement);
    final Pair<Integer, Integer> offsetLength = this.findTypeVariablesOffset(constructorDecl);
    this.addIssue(IssueCodes.getMessageForCLF_CTOR_NO_TYPE_PARAMETERS(), constructorDecl, (offsetLength.getKey()).intValue(), 
      (offsetLength.getValue()).intValue(), IssueCodes.CLF_CTOR_NO_TYPE_PARAMETERS);
    return false;
  }
  
  /**
   * Determines the offset and length of the full list of type variable of the given {@link GenericDeclaration}.
   * 
   * This does not include the delimiting characters '<' and '>'.
   * 
   * @throws IllegalArgumentException if the GenericDeclaration does not have any type variables.
   */
  private Pair<Integer, Integer> findTypeVariablesOffset(final GenericDeclaration genericDeclaration) {
    boolean _isEmpty = genericDeclaration.getTypeVars().isEmpty();
    if (_isEmpty) {
      throw new IllegalArgumentException(
        "Cannot determine offset of type variables for a GenericDeclaration without any type variables.");
    }
    final List<INode> typeVariableNodes = NodeModelUtils.findNodesForFeature(genericDeclaration, N4JSPackage.Literals.GENERIC_DECLARATION__TYPE_VARS);
    final INode firstTypeVariable = typeVariableNodes.get(0);
    final INode lastTypeVariable = IterableExtensions.<INode>last(typeVariableNodes);
    int _offset = firstTypeVariable.getOffset();
    int _offset_1 = lastTypeVariable.getOffset();
    int _length = lastTypeVariable.getLength();
    int _plus = (_offset_1 + _length);
    int _offset_2 = firstTypeVariable.getOffset();
    int _minus = (_plus - _offset_2);
    return Pair.<Integer, Integer>of(Integer.valueOf(_offset), Integer.valueOf(_minus));
  }
  
  /**
   * N4JS spec constraints 44.3
   */
  private boolean holdsRequiredExplicitSuperCallIsFound(final TMethod constructor) {
    EObject _astElement = constructor.getAstElement();
    Block _body = this.getBody(((N4MemberDeclaration) _astElement));
    boolean _tripleNotEquals = (_body != null);
    if (_tripleNotEquals) {
      final EObject type = constructor.eContainer();
      if ((type instanceof TClass)) {
        final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(constructor);
        final TClassifier superClass = RuleEnvironmentExtensions.getDeclaredOrImplicitSuperType(G, ((TClass)type));
        final TMethod ctor = this.containerTypesHelper.fromContext(constructor).findConstructor(superClass);
        if (((ctor != null) && (ctor != RuleEnvironmentExtensions.objectType(G).getOwnedCtor()))) {
          int _size = ctor.getFpars().size();
          boolean _greaterThan = (_size > 0);
          if (_greaterThan) {
            EObject _astElement_1 = constructor.getAstElement();
            final boolean existsSuperCall = ((N4MethodDeclaration) _astElement_1).existsExplicitSuperCall();
            if ((!existsSuperCall)) {
              EObject _eContainer = ctor.eContainer();
              final String className = ((IdentifiableElement) _eContainer).getName();
              this.addIssue(
                IssueCodes.getMessageForKEY_SUP_REQUIRE_EXPLICIT_SUPERCTOR_CALL(className), 
                constructor.getAstElement(), 
                N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
                IssueCodes.KEY_SUP_REQUIRE_EXPLICIT_SUPERCTOR_CALL);
              return false;
            }
          }
        }
      }
    }
    return true;
  }
  
  private boolean holdsCallableConstructorConstraints(final N4MethodDeclaration method) {
    boolean _isCallableConstructor = method.isCallableConstructor();
    if (_isCallableConstructor) {
      EObject _eContainer = method.eContainer();
      boolean _not = (!(_eContainer instanceof N4ClassDefinition));
      if (_not) {
        this.addIssue(IssueCodes.getMessageForCLF_CTOR_CALLABLE_ONLY_IN_CLASS(), method, IssueCodes.CLF_CTOR_CALLABLE_ONLY_IN_CLASS);
        return false;
      }
      boolean _isExternalMode = this.jsVariantHelper.isExternalMode(method);
      boolean _not_1 = (!_isExternalMode);
      if (_not_1) {
        this.addIssue(IssueCodes.getMessageForCLF_CTOR_CALLABLE_ONLY_IN_N4JSD(), method, IssueCodes.CLF_CTOR_CALLABLE_ONLY_IN_N4JSD);
        return false;
      }
      EObject _eContainer_1 = method.eContainer();
      final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
        return Boolean.valueOf(it.isCallableConstructor());
      };
      int _size = IterableExtensions.size(IterableExtensions.<N4MemberDeclaration>filter(((N4ClassifierDefinition) _eContainer_1).getOwnedMembersRaw(), _function));
      boolean _greaterEqualsThan = (_size >= 2);
      if (_greaterEqualsThan) {
        this.addIssue(IssueCodes.getMessageForCLF_CTOR_CALLABLE_DUPLICATE(), method, IssueCodes.CLF_CTOR_CALLABLE_DUPLICATE);
      }
    }
    return true;
  }
  
  private boolean holdsAbstractOrHasBody(final TMember member) {
    final boolean requireCheckForMissingBody = this.jsVariantHelper.requireCheckForMissingBody(member);
    boolean _switchResult = false;
    boolean _matched = false;
    if (member instanceof TMethod) {
      _matched=true;
      _switchResult = ((TMethod)member).isAbstract();
    }
    if (!_matched) {
      if (member instanceof FieldAccessor) {
        _matched=true;
        _switchResult = ((FieldAccessor)member).isAbstract();
      }
    }
    if (!_matched) {
      _switchResult = false;
    }
    final boolean memberIsAbstract = _switchResult;
    if (((requireCheckForMissingBody && (!memberIsAbstract)) && 
      (this.getBody(((N4MemberDeclaration) member.getAstElement())) == null))) {
      boolean _isConstructor = member.isConstructor();
      if (_isConstructor) {
        this.addIssue(IssueCodes.getMessageForCLF_MISSING_CTOR_BODY(), member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
          IssueCodes.CLF_MISSING_CTOR_BODY);
      } else {
        final String message = IssueCodes.getMessageForCLF_MISSING_BODY(this.keywordProvider.keyword(member), member.getName());
        this.addIssue(message, member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_MISSING_BODY);
      }
      return false;
    }
    return true;
  }
  
  /**
   * Constraints 49: abstract methods/getters/setters must not be static and vice versa.
   */
  private boolean holdsAbstractMethodMustNotBeStatic(final TMember member) {
    if ((member.isAbstract() && member.isStatic())) {
      this.addIssue(IssueCodes.getMessageForCLF_STATIC_ABSTRACT(this.keywordProvider.keyword(member), member.getName()), member.getAstElement(), 
        N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_STATIC_ABSTRACT);
      return false;
    }
    return true;
  }
  
  private boolean holdsAbstractMethodMustHaveNoBody(final TMember member) {
    if ((member.isAbstract() && (this.getBody(((N4MemberDeclaration) member.getAstElement())) != null))) {
      final String message = IssueCodes.getMessageForCLF_ABSTRACT_BODY();
      this.addIssue(message, member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_ABSTRACT_BODY);
      return false;
    }
    return true;
  }
  
  private boolean holdsAbstractMemberContainedInAbstractClassifier(final TMember member) {
    boolean _isAbstract = member.isAbstract();
    if (_isAbstract) {
      final TClassifier classifier = EcoreUtil2.<TClassifier>getContainerOfType(member, TClassifier.class);
      if (((classifier != null) && (!classifier.isAbstract()))) {
        final String message = IssueCodes.getMessageForCLF_ABSTRACT_MISSING(this.keywordProvider.keyword(member), member.getName(), classifier.getName());
        this.addIssue(message, member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
          IssueCodes.CLF_ABSTRACT_MISSING);
        return false;
      }
    }
    return true;
  }
  
  /**
   * Internally, internal project and internal private do not exist. (IDEBUG-658)
   */
  private void internalCheckPrivateOrProjectWithInternalAnnotation(final N4MemberDeclaration n4Member, final TMember tmember) {
    boolean _hasAnnotation = AnnotationDefinition.INTERNAL.hasAnnotation(n4Member);
    if (_hasAnnotation) {
      final MemberAccessModifier memberAccessModifier = tmember.getMemberAccessModifier();
      final boolean hasPrivateModifier = (memberAccessModifier == MemberAccessModifier.PRIVATE);
      final boolean hasProjectModifier = (memberAccessModifier == MemberAccessModifier.PROJECT);
      if ((hasPrivateModifier || hasProjectModifier)) {
        final String message = IssueCodes.getMessageForCLF_INTERNAL_BAD_WITH_PRIVATE_OR_PROJECT();
        this.addIssue(message, tmember.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
          IssueCodes.CLF_INTERNAL_BAD_WITH_PRIVATE_OR_PROJECT);
      }
    }
  }
  
  /**
   * Constraint 44.3: Members of an interface must not be declared private or project
   */
  private boolean holdsMinimalMemberAccessModifier(final TMember member) {
    ContainerType<?> _containingType = member.getContainingType();
    if ((_containingType instanceof TInterface)) {
      final MemberAccessModifier memberAccessModifier = member.getMemberAccessModifier();
      if ((memberAccessModifier == MemberAccessModifier.PRIVATE)) {
        final String message = IssueCodes.getMessageForCLF_MINIMAL_ACCESSIBILITY_IN_INTERFACES();
        this.addIssue(message, member.getAstElement(), N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, 
          IssueCodes.CLF_MINIMAL_ACCESSIBILITY_IN_INTERFACES);
        return false;
      }
    }
    return true;
  }
  
  private void internalCheckGetterType(final N4GetterDeclaration n4GetterDeclaration) {
    TypeRef _declaredTypeRef = n4GetterDeclaration.getDeclaredTypeRef();
    Type _declaredType = null;
    if (_declaredTypeRef!=null) {
      _declaredType=_declaredTypeRef.getDeclaredType();
    }
    final Type getterType = _declaredType;
    if (((getterType != null) && (getterType instanceof VoidType))) {
      final String message = IssueCodes.getMessageForCLF_VOID_ACCESSOR();
      this.addIssue(message, n4GetterDeclaration, N4JSPackage.Literals.PROPERTY_NAME_OWNER__DECLARED_NAME, IssueCodes.CLF_VOID_ACCESSOR);
    }
  }
  
  @Check
  public void checkDuplicateFieldsIn(final ThisTypeRefStructural thisTypeRefStructural) {
    final N4ClassifierDefinition n4ClassifierDefinition = EcoreUtil2.<N4ClassifierDefinition>getContainerOfType(thisTypeRefStructural, N4ClassifierDefinition.class);
    if ((n4ClassifierDefinition != null)) {
      final Type tClass = n4ClassifierDefinition.getDefinedType();
      if ((tClass instanceof TClass)) {
        this.internalCheckDuplicateFieldsIn(((TClass)tClass), thisTypeRefStructural);
      }
    }
  }
  
  private void internalCheckDuplicateFieldsIn(final TClass tclass, final ThisTypeRefStructural thisTypeRefStructural) {
    TypingStrategy _typingStrategy = thisTypeRefStructural.getTypingStrategy();
    final boolean structFieldInitMode = Objects.equal(_typingStrategy, TypingStrategy.STRUCTURAL_FIELD_INITIALIZER);
    final List<TMember> members = LazyOverrideAwareMemberCollector.collectAllMembers(tclass);
    final Function1<TMember, org.eclipse.xtext.util.Pair<String, Boolean>> _function = (TMember it) -> {
      return Tuples.<String, Boolean>pair(it.getName(), Boolean.valueOf(it.isStatic()));
    };
    final Map<org.eclipse.xtext.util.Pair<String, Boolean>, List<TMember>> membersByNameAndStatic = IterableExtensions.<org.eclipse.xtext.util.Pair<String, Boolean>, TMember>groupBy(members, _function);
    final Function1<TStructMember, org.eclipse.xtext.util.Pair<String, Boolean>> _function_1 = (TStructMember it) -> {
      return Tuples.<String, Boolean>pair(it.getName(), Boolean.valueOf(it.isStatic()));
    };
    final Map<org.eclipse.xtext.util.Pair<String, Boolean>, List<TStructMember>> structuralMembersByNameAndStatic = IterableExtensions.<org.eclipse.xtext.util.Pair<String, Boolean>, TStructMember>groupBy(thisTypeRefStructural.getStructuralMembers(), _function_1);
    final Consumer<org.eclipse.xtext.util.Pair<String, Boolean>> _function_2 = (org.eclipse.xtext.util.Pair<String, Boolean> it) -> {
      boolean _containsKey = membersByNameAndStatic.containsKey(it);
      if (_containsKey) {
        final TStructMember structuralFieldDuplicate = IterableExtensions.<TStructMember>head(structuralMembersByNameAndStatic.get(it));
        final TMember existingClassifierMember = this.headExceptNonExistantGetters(membersByNameAndStatic.get(it), structFieldInitMode);
        MemberAccessModifier _memberAccessModifier = null;
        if (existingClassifierMember!=null) {
          _memberAccessModifier=existingClassifierMember.getMemberAccessModifier();
        }
        boolean _equals = Objects.equal(_memberAccessModifier, MemberAccessModifier.PUBLIC);
        if (_equals) {
          final String message = IssueCodes.getMessageForCLF_DUP_MEMBER(this.validatorMessageHelper.descriptionWithLine(structuralFieldDuplicate), 
            this.validatorMessageHelper.descriptionWithLine(existingClassifierMember));
          final int index = thisTypeRefStructural.getStructuralMembers().indexOf(structuralFieldDuplicate);
          this.addIssue(message, thisTypeRefStructural, 
            TypeRefsPackage.Literals.STRUCTURAL_TYPE_REF__AST_STRUCTURAL_MEMBERS, index, IssueCodes.CLF_DUP_MEMBER);
        }
      }
    };
    structuralMembersByNameAndStatic.keySet().forEach(_function_2);
  }
  
  private TMember headExceptNonExistantGetters(final Iterable<? extends TMember> members, final boolean structFieldInitMode) {
    if (structFieldInitMode) {
      final Function1<TMember, Boolean> _function = (TMember it) -> {
        return Boolean.valueOf((!(it instanceof TGetter)));
      };
      return IterableExtensions.head(IterableExtensions.filter(members, _function));
    }
    return IterableExtensions.head(members);
  }
  
  /**
   * A list of annotations that requires {@code @Test} annotation as well.
   * Currently used by {@link #checkFixmeUsedWithTestAnnotation(N4MethodDeclaration)} validation method.
   */
  private final static List<AnnotationDefinition> ANNOTATIONS_REQUIRE_TEST = Collections.<AnnotationDefinition>unmodifiableList(CollectionLiterals.<AnnotationDefinition>newArrayList(AnnotationDefinition.TEST_FIXME, AnnotationDefinition.TEST_IGNORE));
  
  /**
   * Raises an issue if a method has any arbitrary annotation that requires {@Test} annotation as well but it does not have it.
   * Currently:
   * <ul>
   * <li>{@code @Ignore} and</li>
   * <li>{@code @Fixme}</li>
   * </ul>
   * requires {@code @Test} annotation.
   */
  @Check
  public void checkFixmeUsedWithTestAnnotation(final N4MethodDeclaration methodDecl) {
    final Consumer<AnnotationDefinition> _function = (AnnotationDefinition annotation) -> {
      if ((annotation.hasAnnotation(methodDecl) && (!AnnotationDefinition.TEST_METHOD.hasAnnotation(methodDecl)))) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("@");
        _builder.append(annotation.name);
        final Function1<Annotation, Boolean> _function_1 = (Annotation it) -> {
          String _name = it.getName();
          return Boolean.valueOf(Objects.equal(_name, annotation.name));
        };
        this.addIssue(IssueCodes.getMessageForANN_REQUIRES_TEST(_builder), IterableExtensions.<Annotation>findFirst(methodDecl.getAnnotations(), _function_1), N4JSPackage.eINSTANCE.getAnnotation_Name(), IssueCodes.ANN_REQUIRES_TEST);
      }
    };
    N4JSMemberValidator.ANNOTATIONS_REQUIRE_TEST.forEach(_function);
  }
}
