/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.validation.validators.packagejson;

import com.google.common.collect.Iterables;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Pattern;
import org.eclipse.n4js.utils.io.FileMatcher;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Utility methods for the resolution of module filter wildcards as used in N4JS {@code package.json} files.
 */
@SuppressWarnings("all")
public class WildcardPathFilterUtils {
  private final static String MATCHING_FILE_EXT = ".*js*";
  
  private final static List<String> supportedFileExtension = Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList(".js", ".n4js", ".n4jsd", ".n4js.xt", ".n4jsd.xt", ".js.xt"));
  
  public static List<String> collectPathsByWildcardPath(final String absoluteProjectPath, final String projectRelativeWildcardPath) {
    List<String> _xblockexpression = null;
    {
      final Pair<List<String>, List<String>> absoluteFolderAndFilePaths = WildcardPathFilterUtils.collectAllFoldersAndFilesByWildcardPath(absoluteProjectPath, projectRelativeWildcardPath, true);
      List<String> _xifexpression = null;
      boolean _isEmpty = absoluteFolderAndFilePaths.getValue().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        _xifexpression = absoluteFolderAndFilePaths.getValue();
      } else {
        _xifexpression = absoluteFolderAndFilePaths.getKey();
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  public static List<String> collectAllFilesByWildcardPath(final String absoluteProjectPath, final String projectRelativeWildcardPath) {
    List<String> _xblockexpression = null;
    {
      final Pair<List<String>, List<String>> absoluteFolderAndFilePaths = WildcardPathFilterUtils.collectAllFoldersAndFilesByWildcardPath(absoluteProjectPath, projectRelativeWildcardPath, true);
      final Function1<String, List<String>> _function = (String path) -> {
        final Function1<String, String> _function_1 = (String it) -> {
          return ((path + File.separator) + it);
        };
        return IterableExtensions.<String>toList(IterableExtensions.<String, String>map(WildcardPathFilterUtils.matchFiles(path, "*.*"), _function_1));
      };
      final List<String> files = IterableExtensions.<String>toList(Iterables.<String>concat(ListExtensions.<String, List<String>>map(absoluteFolderAndFilePaths.getKey(), _function)));
      List<String> _value = absoluteFolderAndFilePaths.getValue();
      Iterables.<String>addAll(files, _value);
      _xblockexpression = files;
    }
    return _xblockexpression;
  }
  
  private static List<String> filterByFileExtensions(final List<String> paths) {
    final Function1<String, Boolean> _function = (String it) -> {
      final Function1<String, Boolean> _function_1 = (String fe) -> {
        return Boolean.valueOf(it.endsWith(fe));
      };
      return Boolean.valueOf(IterableExtensions.<String>exists(WildcardPathFilterUtils.supportedFileExtension, _function_1));
    };
    return IterableExtensions.<String>toList(IterableExtensions.<String>filter(paths, _function));
  }
  
  public static List<String> collectAllFoldersByWildcardPath(final String absoluteProjectPath, final String projectRelativeWildcardPath) {
    List<String> _xblockexpression = null;
    {
      final Pair<List<String>, List<String>> absoluteFolderAndFilePaths = WildcardPathFilterUtils.collectAllFoldersAndFilesByWildcardPath(absoluteProjectPath, projectRelativeWildcardPath, false);
      _xblockexpression = absoluteFolderAndFilePaths.getKey();
    }
    return _xblockexpression;
  }
  
  private static Pair<List<String>, List<String>> collectAllFoldersAndFilesByWildcardPath(final String absoluteProjectPath, final String matchString, final boolean addFileExtensionPattern) {
    boolean _exists = new File(absoluteProjectPath).exists();
    boolean _not = (!_exists);
    if (_not) {
      return Pair.<List<String>, List<String>>of(Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList()), Collections.<String>unmodifiableList(CollectionLiterals.<String>newArrayList()));
    }
    final String newMatchString = WildcardPathFilterUtils.adaptWildcard(matchString.replace("\\", File.separator).replace("/", File.separator), addFileExtensionPattern);
    final String separator = (File.separator + "..");
    final Iterator<String> splittedMatchStringParts = ((List<String>)Conversions.doWrapArray(newMatchString.split(Pattern.quote(separator)))).iterator();
    final ArrayList<String> absoluteFolderPaths = CollectionLiterals.<String>newArrayList(absoluteProjectPath);
    final ArrayList<String> absoluteFilePaths = CollectionLiterals.<String>newArrayList();
    while (splittedMatchStringParts.hasNext()) {
      {
        final String splittedMatchStringPart = splittedMatchStringParts.next();
        final Function1<String, Pair<String, List<String>>> _function = (String it) -> {
          List<String> _list = IterableExtensions.<String>toList(WildcardPathFilterUtils.matchFolders(it, splittedMatchStringPart));
          return Pair.<String, List<String>>of(it, _list);
        };
        final List<Pair<String, List<String>>> folders = ListExtensions.<String, Pair<String, List<String>>>map(absoluteFolderPaths, _function);
        final ArrayList<String> newAbsoluteFolderPaths = CollectionLiterals.<String>newArrayList();
        for (final Pair<String, List<String>> folderEntry : folders) {
          final Function1<String, String> _function_1 = (String it) -> {
            String _xtrycatchfinallyexpression = null;
            try {
              String _key = folderEntry.getKey();
              String _plus = (_key + File.separator);
              String _plus_1 = (_plus + it);
              String _xifexpression = null;
              if (((!splittedMatchStringParts.hasNext()) && newMatchString.endsWith(separator))) {
                _xifexpression = separator;
              } else {
                _xifexpression = "";
              }
              String _plus_2 = (_plus_1 + _xifexpression);
              _xtrycatchfinallyexpression = new File(_plus_2).getCanonicalPath();
            } catch (final Throwable _t) {
              if (_t instanceof IOException) {
                final IOException exc = (IOException)_t;
                throw new RuntimeException("Canonical path cannot be calculated", exc);
              } else {
                throw Exceptions.sneakyThrow(_t);
              }
            }
            return _xtrycatchfinallyexpression;
          };
          List<String> _map = ListExtensions.<String, String>map(folderEntry.getValue(), _function_1);
          Iterables.<String>addAll(newAbsoluteFolderPaths, _map);
        }
        if (((!splittedMatchStringParts.hasNext()) && (!newMatchString.endsWith(separator)))) {
          final Function1<String, Pair<String, List<String>>> _function_2 = (String it) -> {
            List<String> _list = IterableExtensions.<String>toList(WildcardPathFilterUtils.matchFiles(it, splittedMatchStringPart));
            return Pair.<String, List<String>>of(it, _list);
          };
          final List<Pair<String, List<String>>> files = ListExtensions.<String, Pair<String, List<String>>>map(absoluteFolderPaths, _function_2);
          for (final Pair<String, List<String>> fileEntry : files) {
            {
              final Function1<String, String> _function_3 = (String it) -> {
                String _xtrycatchfinallyexpression = null;
                try {
                  String _key = fileEntry.getKey();
                  String _plus = (_key + File.separator);
                  _xtrycatchfinallyexpression = (_plus + it);
                } catch (final Throwable _t) {
                  if (_t instanceof IOException) {
                    final IOException exc = (IOException)_t;
                    throw new RuntimeException("Canonical path cannot be calculated", exc);
                  } else {
                    throw Exceptions.sneakyThrow(_t);
                  }
                }
                return _xtrycatchfinallyexpression;
              };
              final List<String> filesPaths = ListExtensions.<String, String>map(fileEntry.getValue(), _function_3);
              Iterables.<String>addAll(absoluteFilePaths, filesPaths);
            }
          }
        }
        absoluteFolderPaths.clear();
        Iterables.<String>addAll(absoluteFolderPaths, newAbsoluteFolderPaths);
      }
    }
    List<String> _filterByFileExtensions = WildcardPathFilterUtils.filterByFileExtensions(absoluteFilePaths);
    final Pair<List<String>, List<String>> result = Pair.<List<String>, List<String>>of(absoluteFolderPaths, _filterByFileExtensions);
    return result;
  }
  
  private static String adaptWildcard(final String projectRelativeWildcardPath, final boolean addFileExtensionPattern) {
    String _xifexpression = null;
    if (((!projectRelativeWildcardPath.endsWith("**")) && addFileExtensionPattern)) {
      _xifexpression = (projectRelativeWildcardPath + WildcardPathFilterUtils.MATCHING_FILE_EXT);
    } else {
      _xifexpression = projectRelativeWildcardPath;
    }
    return _xifexpression;
  }
  
  private static Iterable<String> matchFolders(final String absoluteProjectPath, final String matchString) {
    final Function1<Path, String> _function = (Path it) -> {
      return it.toString();
    };
    return IterableExtensions.<Path, String>map(FileMatcher.scanDirectories(Paths.get(absoluteProjectPath), matchString), _function);
  }
  
  private static Iterable<String> matchFiles(final String absoluteProjectPath, final String matchString) {
    final Function1<Path, String> _function = (Path it) -> {
      return it.toString();
    };
    return IterableExtensions.<Path, String>map(FileMatcher.scanFiles(Paths.get(absoluteProjectPath), matchString), _function);
  }
}
