/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.assistants;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.n4JS.AnnotableElement;
import org.eclipse.n4js.n4JS.Annotation;
import org.eclipse.n4js.n4JS.Argument;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.AssignmentExpression;
import org.eclipse.n4js.n4JS.Block;
import org.eclipse.n4js.n4JS.BooleanLiteral;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ExpressionStatement;
import org.eclipse.n4js.n4JS.FieldAccessor;
import org.eclipse.n4js.n4JS.FormalParameter;
import org.eclipse.n4js.n4JS.FunctionDefinition;
import org.eclipse.n4js.n4JS.FunctionExpression;
import org.eclipse.n4js.n4JS.FunctionOrFieldAccessor;
import org.eclipse.n4js.n4JS.GetterDeclaration;
import org.eclipse.n4js.n4JS.IndexedAccessExpression;
import org.eclipse.n4js.n4JS.ModifiableElement;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.N4ClassDefinition;
import org.eclipse.n4js.n4JS.N4ClassifierDeclaration;
import org.eclipse.n4js.n4JS.N4EnumDeclaration;
import org.eclipse.n4js.n4JS.N4EnumLiteral;
import org.eclipse.n4js.n4JS.N4FieldDeclaration;
import org.eclipse.n4js.n4JS.N4GetterDeclaration;
import org.eclipse.n4js.n4JS.N4InterfaceDeclaration;
import org.eclipse.n4js.n4JS.N4MemberDeclaration;
import org.eclipse.n4js.n4JS.N4MethodDeclaration;
import org.eclipse.n4js.n4JS.N4Modifier;
import org.eclipse.n4js.n4JS.N4SetterDeclaration;
import org.eclipse.n4js.n4JS.N4TypeDeclaration;
import org.eclipse.n4js.n4JS.NewExpression;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.ParameterizedCallExpression;
import org.eclipse.n4js.n4JS.PropertyAssignment;
import org.eclipse.n4js.n4JS.PropertyNameValuePair;
import org.eclipse.n4js.n4JS.SetterDeclaration;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.StringLiteral;
import org.eclipse.n4js.n4JS.TypeDefiningElement;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.n4JS.VersionedElement;
import org.eclipse.n4js.n4idl.N4IDLGlobals;
import org.eclipse.n4js.transpiler.TransformationAssistant;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.es.assistants.DelegationAssistant;
import org.eclipse.n4js.transpiler.es.transform.ClassDeclarationTransformation;
import org.eclipse.n4js.transpiler.es.transform.EnumDeclarationTransformation;
import org.eclipse.n4js.transpiler.es.transform.InterfaceDeclarationTransformation;
import org.eclipse.n4js.transpiler.im.DelegatingMember;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ReferencingElementExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryInternal;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.utils.TranspilerUtils;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TAnnotableElement;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TAnnotationArgument;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TInterface;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.TObjectPrototype;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypingStrategy;
import org.eclipse.n4js.ts.types.util.SuperInterfacesIterable;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.n4js.utils.N4JSLanguageUtils;
import org.eclipse.n4js.utils.ResourceNameComputer;
import org.eclipse.n4js.validation.JavaScriptVariantHelper;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * An {@link TransformationAssistant assistant} to create call expressions for invoking <code>$makeClass</code>,
 * <code>$makeInterface</code>, and such. Used by
 * <ul>
 * <li>{@link ClassDeclarationTransformation},
 * <li>{@link InterfaceDeclarationTransformation},
 * <li>{@link EnumDeclarationTransformation}
 * </ul>
 */
@SuppressWarnings("all")
public class BootstrapCallAssistant extends TransformationAssistant {
  @Inject
  private DelegationAssistant delegationAssistant;
  
  @Inject
  private TypeAssistant typeAssistant;
  
  @Inject
  private ResourceNameComputer resourceNameComputer;
  
  @Inject
  private JavaScriptVariantHelper jsVariantHelper;
  
  /**
   * Create a <code>$makeClass</code> call, including all its required arguments.
   * <pre>
   * $makeClass(C, Object,
   *     {
   *         m: {
   *             value: function m___n4(){}
   *         }
   *     },
   *     {},
   *     function(instanceProto, staticProto) {}
   * );
   * </pre>
   */
  public ExpressionStatement createMakeClassCall(final N4ClassDeclaration classDecl, final SymbolTableEntry superClassSTE) {
    ParameterizedCallExpression __CallExpr = TranspilerBuilderBlocks._CallExpr();
    final Procedure1<ParameterizedCallExpression> _function = (ParameterizedCallExpression it) -> {
      it.setTarget(TranspilerBuilderBlocks._IdentRef(this.steFor_$makeClass()));
      EList<Argument> _arguments = it.getArguments();
      IdentifierRef_IM __IdentRef = TranspilerBuilderBlocks._IdentRef(this.findSymbolTableEntryForElement(classDecl, false));
      ReferencingElementExpression_IM ___NSSafe_IdentRef = this.__NSSafe_IdentRef(superClassSTE);
      ArrayLiteral _createDirectlyImplementedInterfacesArgument = this.createDirectlyImplementedInterfacesArgument(classDecl);
      ObjectLiteral _createMemberDefinitions = this.createMemberDefinitions(classDecl, false);
      ObjectLiteral _createMemberDefinitions_1 = this.createMemberDefinitions(classDecl, true);
      FunctionExpression _createN4TypeMetaInfoFactoryFunction = this.createN4TypeMetaInfoFactoryFunction(classDecl, superClassSTE);
      final Function1<Expression, Argument> _function_1 = (Expression it_1) -> {
        return TranspilerBuilderBlocks._Argument(it_1);
      };
      List<Argument> _map = ListExtensions.map(Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__IdentRef, ___NSSafe_IdentRef, _createDirectlyImplementedInterfacesArgument, _createMemberDefinitions, _createMemberDefinitions_1, _createN4TypeMetaInfoFactoryFunction)), _function_1);
      Iterables.<Argument>addAll(_arguments, _map);
    };
    ParameterizedCallExpression _doubleArrow = ObjectExtensions.<ParameterizedCallExpression>operator_doubleArrow(__CallExpr, _function);
    return TranspilerBuilderBlocks._ExprStmnt(_doubleArrow);
  }
  
  /**
   * Unfortunately, <code>$makeClass</code> and <code>$makeInterface</code> are inconsistent. Member definitions are
   * not passed to <code>$makeInterface</code>, but are instead processed by the generated code directly. This
   * processing code is created here.
   * <p>
   * TODO align $makeInterface with $makeClass
   */
  public Statement[] createInterfaceMemberDefinitionSection(final N4InterfaceDeclaration ifcDecl) {
    final ArrayList<Statement> result = CollectionLiterals.<Statement>newArrayList();
    final SymbolTableEntry ifcSTE = this.findSymbolTableEntryForElement(ifcDecl, true);
    final SymbolTableEntryInternal $methodsSTE = this.steFor_$methods();
    final SymbolTableEntryOriginal objectSTE = this.getSymbolTableEntryOriginal(RuleEnvironmentExtensions.objectType(this.getState().G), true);
    final SymbolTableEntryOriginal definePropertiesSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.objectType(this.getState().G), "defineProperties", false, true, true);
    AssignmentExpression __AssignmentExpr = TranspilerBuilderBlocks._AssignmentExpr();
    final Procedure1<AssignmentExpression> _function = (AssignmentExpression it) -> {
      it.setLhs(TranspilerBuilderBlocks._PropertyAccessExpr(ifcSTE, $methodsSTE));
      it.setRhs(this.createMemberDefinitions(ifcDecl, false));
    };
    AssignmentExpression _doubleArrow = ObjectExtensions.<AssignmentExpression>operator_doubleArrow(__AssignmentExpr, _function);
    ExpressionStatement __ExprStmnt = TranspilerBuilderBlocks._ExprStmnt(_doubleArrow);
    result.add(__ExprStmnt);
    final ObjectLiteral staticMemberDefinitions = this.createMemberDefinitions(ifcDecl, true);
    boolean _isEmpty = staticMemberDefinitions.getPropertyAssignments().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      ExpressionStatement __ExprStmnt_1 = TranspilerBuilderBlocks._ExprStmnt(
        TranspilerBuilderBlocks._CallExpr(
          TranspilerBuilderBlocks._PropertyAccessExpr(objectSTE, definePropertiesSTE), 
          TranspilerBuilderBlocks._IdentRef(ifcSTE), staticMemberDefinitions));
      result.add(__ExprStmnt_1);
    }
    return ((Statement[])Conversions.unwrapArray(result, Statement.class));
  }
  
  /**
   * Create a <code>$makeInterface</code> call, including all its required arguments.
   * <pre>
   * $makeInterface(I,
   *     function(instanceProto, staticProto) {}
   * );
   * </pre>
   */
  public ExpressionStatement createMakeInterfaceCall(final N4InterfaceDeclaration ifcDecl) {
    ParameterizedCallExpression __CallExpr = TranspilerBuilderBlocks._CallExpr();
    final Procedure1<ParameterizedCallExpression> _function = (ParameterizedCallExpression it) -> {
      it.setTarget(TranspilerBuilderBlocks._IdentRef(this.steFor_$makeInterface()));
      EList<Argument> _arguments = it.getArguments();
      IdentifierRef_IM __IdentRef = TranspilerBuilderBlocks._IdentRef(this.findSymbolTableEntryForElement(ifcDecl, false));
      FunctionExpression _createN4TypeMetaInfoFactoryFunction = this.createN4TypeMetaInfoFactoryFunction(ifcDecl, null);
      final Function1<Expression, Argument> _function_1 = (Expression it_1) -> {
        return TranspilerBuilderBlocks._Argument(it_1);
      };
      List<Argument> _map = ListExtensions.map(Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__IdentRef, _createN4TypeMetaInfoFactoryFunction)), _function_1);
      Iterables.<Argument>addAll(_arguments, _map);
    };
    ParameterizedCallExpression _doubleArrow = ObjectExtensions.<ParameterizedCallExpression>operator_doubleArrow(__CallExpr, _function);
    return TranspilerBuilderBlocks._ExprStmnt(_doubleArrow);
  }
  
  /**
   * Create a <code>$makeEnum</code> call, including all its required arguments.
   * <pre>
   * $makeEnum(E, false,
   *     [
   *         ['Literal0', 'Value0'],
   *         ['Literal1', 'Value1']
   *     ],
   *     function(instanceProto, staticProto) {}
   * );
   * </pre>
   */
  public ExpressionStatement createMakeEnumCall(final N4EnumDeclaration enumDecl) {
    final boolean isStringBased = AnnotationDefinition.STRING_BASED.hasAnnotation(enumDecl);
    if (isStringBased) {
      throw new IllegalArgumentException("must not create $makeEnum() call for @StringBased enums (they are no longer represented in output code)");
    }
    final ArrayLiteral enumLiteralArray = TranspilerBuilderBlocks._ArrLit();
    EList<N4EnumLiteral> _literals = enumDecl.getLiterals();
    for (final N4EnumLiteral literal : _literals) {
      EList<ArrayElement> _elements = enumLiteralArray.getElements();
      String _elvis = null;
      String _value = literal.getValue();
      if (_value != null) {
        _elvis = _value;
      } else {
        String _name = literal.getName();
        _elvis = _name;
      }
      ArrayElement __ArrayElement = TranspilerBuilderBlocks._ArrayElement(
        TranspilerBuilderBlocks._ArrLit(
          TranspilerBuilderBlocks._StringLiteral(literal.getName()), 
          TranspilerBuilderBlocks._StringLiteral(_elvis)));
      _elements.add(__ArrayElement);
    }
    ParameterizedCallExpression __CallExpr = TranspilerBuilderBlocks._CallExpr();
    final Procedure1<ParameterizedCallExpression> _function = (ParameterizedCallExpression it) -> {
      it.setTarget(TranspilerBuilderBlocks._IdentRef(this.steFor_$makeEnum()));
      EList<Argument> _arguments = it.getArguments();
      IdentifierRef_IM __IdentRef = TranspilerBuilderBlocks._IdentRef(this.findSymbolTableEntryForElement(enumDecl, false));
      BooleanLiteral __BooleanLiteral = TranspilerBuilderBlocks._BooleanLiteral(isStringBased);
      FunctionExpression _createN4TypeMetaInfoFactoryFunction = this.createN4TypeMetaInfoFactoryFunction(enumDecl, null);
      final Function1<Expression, Argument> _function_1 = (Expression it_1) -> {
        return TranspilerBuilderBlocks._Argument(it_1);
      };
      List<Argument> _map = ListExtensions.map(Collections.<Expression>unmodifiableList(CollectionLiterals.<Expression>newArrayList(__IdentRef, __BooleanLiteral, enumLiteralArray, _createN4TypeMetaInfoFactoryFunction)), _function_1);
      Iterables.<Argument>addAll(_arguments, _map);
    };
    ParameterizedCallExpression _doubleArrow = ObjectExtensions.<ParameterizedCallExpression>operator_doubleArrow(__CallExpr, _function);
    return TranspilerBuilderBlocks._ExprStmnt(_doubleArrow);
  }
  
  private ObjectLiteral createMemberDefinitions(final N4ClassifierDeclaration classDecl, final boolean _static) {
    final Function1<N4MemberDeclaration, Boolean> _function = (N4MemberDeclaration it) -> {
      return Boolean.valueOf((((Boolean.valueOf(it.isStatic()) == Boolean.valueOf(_static)) && (!this.isAbstract(it))) && (!it.isConstructor())));
    };
    Iterable<N4MemberDeclaration> membersToProcess = IterableExtensions.<N4MemberDeclaration>filter(classDecl.getOwnedMembers(), _function);
    if ((classDecl instanceof N4InterfaceDeclaration)) {
      final Function1<N4MemberDeclaration, Boolean> _function_1 = (N4MemberDeclaration it) -> {
        return Boolean.valueOf((((it instanceof FunctionOrFieldAccessor) && (((FunctionOrFieldAccessor) it).getBody() != null)) || ((it instanceof DelegatingMember) && (!((DelegatingMember) it).isDelegationTargetIsAbstract()))));
      };
      membersToProcess = IterableExtensions.<N4MemberDeclaration>filter(membersToProcess, _function_1);
    }
    final ArrayList<N4FieldDeclaration> fields = CollectionLiterals.<N4FieldDeclaration>newArrayList();
    final ArrayListMultimap<String, FieldAccessor> accessors = ArrayListMultimap.<String, FieldAccessor>create();
    final ArrayList<N4MethodDeclaration> methods = CollectionLiterals.<N4MethodDeclaration>newArrayList();
    for (final N4MemberDeclaration m : membersToProcess) {
      boolean _matched = false;
      if (m instanceof N4FieldDeclaration) {
        _matched=true;
        fields.add(((N4FieldDeclaration)m));
      }
      if (!_matched) {
        if (m instanceof FieldAccessor) {
          _matched=true;
          accessors.put(m.getName(), ((FieldAccessor)m));
        }
      }
      if (!_matched) {
        if (m instanceof N4MethodDeclaration) {
          _matched=true;
          methods.add(((N4MethodDeclaration)m));
        }
      }
      if (!_matched) {
        String _name = m.eClass().getName();
        String _plus = ("unsupported type of member: " + _name);
        throw new IllegalStateException(_plus);
      }
    }
    List<PropertyNameValuePair> _xifexpression = null;
    if (((_static && (classDecl instanceof VersionedElement)) && ((VersionedElement) classDecl).hasDeclaredVersion())) {
      PropertyNameValuePair _createEmptyObjectMemberForName = this.createEmptyObjectMemberForName(N4IDLGlobals.MIGRATIONS_STATIC_FIELD);
      _xifexpression = Collections.<PropertyNameValuePair>unmodifiableList(CollectionLiterals.<PropertyNameValuePair>newArrayList(_createEmptyObjectMemberForName));
    } else {
      _xifexpression = Collections.<PropertyNameValuePair>unmodifiableList(CollectionLiterals.<PropertyNameValuePair>newArrayList());
    }
    final List<PropertyNameValuePair> staticMigrationsField = _xifexpression;
    final Function1<N4MethodDeclaration, PropertyNameValuePair> _function_2 = (N4MethodDeclaration it) -> {
      return this.createMemberDefinitionForMethod(it);
    };
    List<PropertyNameValuePair> _map = ListExtensions.<N4MethodDeclaration, PropertyNameValuePair>map(methods, _function_2);
    final Function1<String, PropertyNameValuePair> _function_3 = (String accName) -> {
      return this.createMemberDefinitionForAccessor(accessors.get(accName));
    };
    Iterable<PropertyNameValuePair> _map_1 = IterableExtensions.<String, PropertyNameValuePair>map(IterableExtensions.<String>toSet(accessors.keys()), _function_3);
    Iterable<PropertyNameValuePair> _plus_1 = Iterables.<PropertyNameValuePair>concat(_map, _map_1);
    final Function1<N4FieldDeclaration, PropertyNameValuePair> _function_4 = (N4FieldDeclaration it) -> {
      return this.createMemberDefinitionForField(it);
    };
    List<PropertyNameValuePair> _map_2 = ListExtensions.<N4FieldDeclaration, PropertyNameValuePair>map(fields, _function_4);
    Iterable<PropertyNameValuePair> _plus_2 = Iterables.<PropertyNameValuePair>concat(_plus_1, _map_2);
    return TranspilerBuilderBlocks._ObjLit(
      ((PropertyAssignment[])Conversions.unwrapArray(IterableExtensions.<PropertyNameValuePair>filterNull(Iterables.<PropertyNameValuePair>concat(_plus_2, staticMigrationsField)), PropertyAssignment.class)));
  }
  
  private PropertyNameValuePair createEmptyObjectMemberForName(final String name) {
    ObjectLiteral __ObjLit = TranspilerBuilderBlocks._ObjLit();
    Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("value", __ObjLit);
    BooleanLiteral __TRUE = TranspilerBuilderBlocks._TRUE();
    Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("writable", __TRUE);
    BooleanLiteral __FALSE = TranspilerBuilderBlocks._FALSE();
    Pair<String, Expression> _mappedTo_2 = Pair.<String, Expression>of("configurable", __FALSE);
    BooleanLiteral __FALSE_1 = TranspilerBuilderBlocks._FALSE();
    Pair<String, Expression> _mappedTo_3 = Pair.<String, Expression>of("enumerable", __FALSE_1);
    return TranspilerBuilderBlocks._PropertyNameValuePair(name, 
      TranspilerBuilderBlocks._ObjLit(_mappedTo, _mappedTo_1, _mappedTo_2, _mappedTo_3));
  }
  
  private PropertyNameValuePair createMemberDefinitionForField(final N4FieldDeclaration fieldDecl) {
    IdentifierRef_IM _undefinedRef = this.undefinedRef();
    Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("value", _undefinedRef);
    BooleanLiteral __TRUE = TranspilerBuilderBlocks._TRUE();
    Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("writable", __TRUE);
    return TranspilerBuilderBlocks._PropertyNameValuePair(
      fieldDecl.getName(), 
      TranspilerBuilderBlocks._ObjLit(_mappedTo, _mappedTo_1));
  }
  
  private PropertyNameValuePair createMemberDefinitionForAccessor(final List<FieldAccessor> accDecls) {
    final GetterDeclaration getter = IterableExtensions.<GetterDeclaration>head(Iterables.<GetterDeclaration>filter(accDecls, GetterDeclaration.class));
    final SetterDeclaration setter = IterableExtensions.<SetterDeclaration>head(Iterables.<SetterDeclaration>filter(accDecls, SetterDeclaration.class));
    if (((getter == null) && (setter == null))) {
      throw new IllegalArgumentException("must provide a getter, a setter or both");
    }
    String _elvis = null;
    String _name = null;
    if (getter!=null) {
      _name=getter.getName();
    }
    if (_name != null) {
      _elvis = _name;
    } else {
      String _name_1 = setter.getName();
      _elvis = _name_1;
    }
    final String name = _elvis;
    ObjectLiteral __ObjLit = TranspilerBuilderBlocks._ObjLit();
    final Procedure1<ObjectLiteral> _function = (ObjectLiteral it) -> {
      if ((getter != null)) {
        EList<PropertyAssignment> _propertyAssignments = it.getPropertyAssignments();
        Expression _xifexpression = null;
        if ((getter instanceof DelegatingMember)) {
          _xifexpression = this.delegationAssistant.createDelegationCode(((DelegatingMember)getter));
        } else {
          _xifexpression = this.createMemberFunction(getter);
        }
        PropertyNameValuePair __PropertyNameValuePair = TranspilerBuilderBlocks._PropertyNameValuePair("get", _xifexpression);
        _propertyAssignments.add(__PropertyNameValuePair);
      }
      if ((setter != null)) {
        EList<PropertyAssignment> _propertyAssignments_1 = it.getPropertyAssignments();
        Expression _xifexpression_1 = null;
        if ((setter instanceof DelegatingMember)) {
          _xifexpression_1 = this.delegationAssistant.createDelegationCode(((DelegatingMember)setter));
        } else {
          _xifexpression_1 = this.createMemberFunction(setter);
        }
        PropertyNameValuePair __PropertyNameValuePair_1 = TranspilerBuilderBlocks._PropertyNameValuePair("set", _xifexpression_1);
        _propertyAssignments_1.add(__PropertyNameValuePair_1);
      }
    };
    ObjectLiteral _doubleArrow = ObjectExtensions.<ObjectLiteral>operator_doubleArrow(__ObjLit, _function);
    return TranspilerBuilderBlocks._PropertyNameValuePair(name, _doubleArrow);
  }
  
  private PropertyNameValuePair createMemberDefinitionForMethod(final N4MethodDeclaration methDecl) {
    Expression _xifexpression = null;
    if ((methDecl instanceof DelegatingMember)) {
      _xifexpression = this.delegationAssistant.createDelegationCode(((DelegatingMember)methDecl));
    } else {
      _xifexpression = this.createMemberFunction(methDecl);
    }
    return TranspilerBuilderBlocks._PropertyNameValuePair(
      methDecl.getName(), 
      TranspilerBuilderBlocks._ObjLit(
        TranspilerBuilderBlocks._PropertyNameValuePair("value", _xifexpression)));
  }
  
  private FunctionExpression createMemberFunction(final FunctionOrFieldAccessor template) {
    final FunctionExpression result = TranspilerBuilderBlocks._FunExpr(template.isAsync());
    result.setName(this.getMemberFunctionName(template));
    EList<FormalParameter> _fpars = result.getFpars();
    List<FormalParameter> _switchResult = null;
    boolean _matched = false;
    if (template instanceof FunctionDefinition) {
      _matched=true;
      _switchResult = ((FunctionDefinition)template).getFpars();
    }
    if (!_matched) {
      if (template instanceof SetterDeclaration) {
        _matched=true;
        FormalParameter _fpar = ((SetterDeclaration)template).getFpar();
        _switchResult = Collections.<FormalParameter>unmodifiableList(CollectionLiterals.<FormalParameter>newArrayList(_fpar));
      }
    }
    if (!_matched) {
      _switchResult = Collections.<FormalParameter>unmodifiableList(CollectionLiterals.<FormalParameter>newArrayList());
    }
    Iterables.<FormalParameter>addAll(_fpars, _switchResult);
    result.set_lok(template.get_lok());
    result.setBody(template.getBody());
    boolean _xifexpression = false;
    if ((template instanceof FunctionDefinition)) {
      _xifexpression = ((FunctionDefinition)template).isGenerator();
    } else {
      _xifexpression = false;
    }
    result.setGenerator(_xifexpression);
    boolean _isEmpty = template.getAnnotations().isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      result.setAnnotationList(TranspilerBuilderBlocks._ExprAnnoList(((Annotation[])Conversions.unwrapArray(template.getAnnotations(), Annotation.class))));
    }
    return result;
  }
  
  private FunctionExpression createN4TypeMetaInfoFactoryFunction(final N4TypeDeclaration typeDecl, final SymbolTableEntry superClassSTE) {
    final VariableDeclaration varDeclMetaClass = this.createMetaClassVariable(typeDecl, superClassSTE);
    FormalParameter __Fpar = TranspilerBuilderBlocks._Fpar("instanceProto");
    FormalParameter __Fpar_1 = TranspilerBuilderBlocks._Fpar("staticProto");
    Block __Block = TranspilerBuilderBlocks._Block();
    final Procedure1<Block> _function = (Block it) -> {
      EList<Statement> _statements = it.getStatements();
      VariableStatement __VariableStatement = TranspilerBuilderBlocks._VariableStatement(varDeclMetaClass);
      ExpressionStatement __SnippetAsStmnt = TranspilerBuilderBlocks._SnippetAsStmnt("return metaClass;");
      Iterables.<Statement>addAll(_statements, Collections.<Statement>unmodifiableList(CollectionLiterals.<Statement>newArrayList(__VariableStatement, __SnippetAsStmnt)));
    };
    Block _doubleArrow = ObjectExtensions.<Block>operator_doubleArrow(__Block, _function);
    return TranspilerBuilderBlocks._FunExpr(false, null, new FormalParameter[] { __Fpar, __Fpar_1 }, _doubleArrow);
  }
  
  private VariableDeclaration createMetaClassVariable(final N4TypeDeclaration typeDecl, final SymbolTableEntry superClassSTE) {
    final SymbolTableEntry typeSTE = this.findSymbolTableEntryForElement(typeDecl, true);
    final Type type = this.getState().info.getOriginalDefinedType(typeDecl);
    final Expression n4superType = this.createN4SuperTypeRef(typeDecl, superClassSTE);
    Iterable<TInterface> _xifexpression = null;
    if ((type instanceof TClassifier)) {
      final Function1<TInterface, Boolean> _function = (TInterface it) -> {
        boolean _isStructurallyTyped = this.isStructurallyTyped(it);
        return Boolean.valueOf((!_isStructurallyTyped));
      };
      _xifexpression = IterableExtensions.<TInterface>filter(SuperInterfacesIterable.of(((TClassifier)type)), _function);
    } else {
      _xifexpression = CollectionLiterals.<TInterface>emptyList();
    }
    final Iterable<TInterface> allImplementedInterfaces = _xifexpression;
    Iterable<N4MemberDeclaration> _xifexpression_1 = null;
    if ((typeDecl instanceof N4ClassifierDeclaration)) {
      final Function1<N4MemberDeclaration, Boolean> _function_1 = (N4MemberDeclaration it) -> {
        boolean _isConsumedFromInterface = this.getState().info.isConsumedFromInterface(it);
        return Boolean.valueOf((!_isConsumedFromInterface));
      };
      _xifexpression_1 = IterableExtensions.<N4MemberDeclaration>filter(((N4ClassifierDeclaration)typeDecl).getOwnedMembers(), _function_1);
    } else {
      _xifexpression_1 = Collections.<N4MemberDeclaration>unmodifiableList(CollectionLiterals.<N4MemberDeclaration>newArrayList());
    }
    final Iterable<N4MemberDeclaration> ownedMembers = _xifexpression_1;
    Iterable<N4MemberDeclaration> _xifexpression_2 = null;
    if ((typeDecl instanceof N4ClassifierDeclaration)) {
      final Function1<N4MemberDeclaration, Boolean> _function_2 = (N4MemberDeclaration it) -> {
        return Boolean.valueOf(this.getState().info.isConsumedFromInterface(it));
      };
      _xifexpression_2 = IterableExtensions.<N4MemberDeclaration>filter(((N4ClassifierDeclaration)typeDecl).getOwnedMembers(), _function_2);
    } else {
      _xifexpression_2 = Collections.<N4MemberDeclaration>unmodifiableList(CollectionLiterals.<N4MemberDeclaration>newArrayList());
    }
    final Iterable<N4MemberDeclaration> consumedMembers = _xifexpression_2;
    SymbolTableEntryInternal _switchResult = null;
    boolean _matched = false;
    if (typeDecl instanceof N4ClassDefinition) {
      _matched=true;
      _switchResult = this.steFor_N4Class();
    }
    if (!_matched) {
      if (typeDecl instanceof N4InterfaceDeclaration) {
        _matched=true;
        _switchResult = this.steFor_N4Interface();
      }
    }
    if (!_matched) {
      if (typeDecl instanceof N4EnumDeclaration) {
        _matched=true;
        _switchResult = this.steFor_N4EnumType();
      }
    }
    if (!_matched) {
      String _name = typeDecl.eClass().getName();
      String _plus = ("cannot handle declarations of type " + _name);
      throw new IllegalArgumentException(_plus);
    }
    final SymbolTableEntryInternal metaClassSTE = _switchResult;
    final String origin = this.resourceNameComputer.generateProjectDescriptor(this.getState().resource.getURI());
    final String fqn = this.resourceNameComputer.getFullyQualifiedTypeName_WITH_LEGACY_SUPPORT(type);
    VariableDeclaration __VariableDeclaration = TranspilerBuilderBlocks._VariableDeclaration("metaClass");
    final Procedure1<VariableDeclaration> _function_3 = (VariableDeclaration it) -> {
      StringLiteral __StringLiteralForSTE = TranspilerBuilderBlocks._StringLiteralForSTE(typeSTE);
      Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("name", __StringLiteralForSTE);
      StringLiteral __StringLiteral = TranspilerBuilderBlocks._StringLiteral(origin);
      Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("origin", __StringLiteral);
      StringLiteral __StringLiteral_1 = TranspilerBuilderBlocks._StringLiteral(fqn);
      Pair<String, Expression> _mappedTo_2 = Pair.<String, Expression>of("fqn", __StringLiteral_1);
      Pair<String, Expression> _mappedTo_3 = Pair.<String, Expression>of("n4superType", n4superType);
      final Function1<TInterface, String> _function_4 = (TInterface it_1) -> {
        return this.resourceNameComputer.getFullyQualifiedTypeName_WITH_LEGACY_SUPPORT(it_1);
      };
      final Function1<String, StringLiteral> _function_5 = (String it_1) -> {
        return TranspilerBuilderBlocks._StringLiteral(it_1);
      };
      ArrayLiteral __ArrLit = TranspilerBuilderBlocks._ArrLit(
        ((Expression[])Conversions.unwrapArray(IterableExtensions.<String, StringLiteral>map(IterableExtensions.<TInterface, String>map(allImplementedInterfaces, _function_4), _function_5), Expression.class)));
      Pair<String, Expression> _mappedTo_4 = Pair.<String, Expression>of("allImplementedInterfaces", __ArrLit);
      final Function1<N4MemberDeclaration, Expression> _function_6 = (N4MemberDeclaration it_1) -> {
        return this.createMemberDescriptor(it_1);
      };
      ArrayLiteral __ArrLit_1 = TranspilerBuilderBlocks._ArrLit(
        ((Expression[])Conversions.unwrapArray(IterableExtensions.<N4MemberDeclaration, Expression>map(ownedMembers, _function_6), Expression.class)));
      Pair<String, Expression> _mappedTo_5 = Pair.<String, Expression>of("ownedMembers", __ArrLit_1);
      final Function1<N4MemberDeclaration, Expression> _function_7 = (N4MemberDeclaration it_1) -> {
        return this.createMemberDescriptor(it_1);
      };
      ArrayLiteral __ArrLit_2 = TranspilerBuilderBlocks._ArrLit(
        ((Expression[])Conversions.unwrapArray(IterableExtensions.<N4MemberDeclaration, Expression>map(consumedMembers, _function_7), Expression.class)));
      Pair<String, Expression> _mappedTo_6 = Pair.<String, Expression>of("consumedMembers", __ArrLit_2);
      ArrayLiteral __ArrLit_3 = TranspilerBuilderBlocks._ArrLit(this.createRuntimeAnnotations(typeDecl));
      Pair<String, Expression> _mappedTo_7 = Pair.<String, Expression>of("annotations", __ArrLit_3);
      it.setExpression(TranspilerBuilderBlocks._NewExpr(TranspilerBuilderBlocks._IdentRef(metaClassSTE), TranspilerBuilderBlocks._ObjLit(_mappedTo, _mappedTo_1, _mappedTo_2, _mappedTo_3, _mappedTo_4, _mappedTo_5, _mappedTo_6, _mappedTo_7)));
    };
    return ObjectExtensions.<VariableDeclaration>operator_doubleArrow(__VariableDeclaration, _function_3);
  }
  
  private ArrayLiteral createDirectlyImplementedInterfacesArgument(final N4ClassDeclaration typeDecl) {
    final List<SymbolTableEntryOriginal> directlyImplementedInterfaces = this.typeAssistant.getSuperInterfacesSTEs(typeDecl);
    final Function1<SymbolTableEntryOriginal, Boolean> _function = (SymbolTableEntryOriginal ifcSTE) -> {
      final IdentifiableElement tIfc = ifcSTE.getOriginalTarget();
      if ((tIfc instanceof TInterface)) {
        return Boolean.valueOf(((!TypeUtils.isBuiltIn(((Type)tIfc))) && (!(this.inN4JSD(((Type)tIfc)) && (!AnnotationDefinition.N4JS.hasAnnotation(((TAnnotableElement)tIfc)))))));
      }
      return Boolean.valueOf(false);
    };
    final Iterable<SymbolTableEntryOriginal> directlyImplementedInterfacesFiltered = IterableExtensions.<SymbolTableEntryOriginal>filter(directlyImplementedInterfaces, _function);
    final Function1<SymbolTableEntryOriginal, IdentifierRef_IM> _function_1 = (SymbolTableEntryOriginal it) -> {
      return TranspilerBuilderBlocks._IdentRef(it);
    };
    return TranspilerBuilderBlocks._ArrLit(((Expression[])Conversions.unwrapArray(IterableExtensions.<SymbolTableEntryOriginal, IdentifierRef_IM>map(directlyImplementedInterfacesFiltered, _function_1), Expression.class)));
  }
  
  /**
   * Creates the member descriptor for the n4type meta-information. Don't confuse with the member definition created
   * by <code>#createMemberDefinitionForXYZ()</code> methods.
   */
  private Expression createMemberDescriptor(final N4MemberDeclaration memberDecl) {
    final SymbolTableEntry memberSTE = this.findSymbolTableEntryForElement(memberDecl, true);
    SymbolTableEntryInternal _switchResult = null;
    boolean _matched = false;
    if (memberDecl instanceof N4MethodDeclaration) {
      _matched=true;
      _switchResult = this.steFor_N4Method();
    }
    if (!_matched) {
      if (memberDecl instanceof FieldAccessor) {
        _matched=true;
        _switchResult = this.steFor_N4Accessor();
      }
    }
    if (!_matched) {
      if (memberDecl instanceof N4FieldDeclaration) {
        _matched=true;
        _switchResult = this.steFor_N4DataField();
      }
    }
    if (!_matched) {
      _switchResult = this.steFor_N4Member();
    }
    final SymbolTableEntryInternal n4MemberClassSTE = _switchResult;
    StringLiteral __StringLiteralForSTE = TranspilerBuilderBlocks._StringLiteralForSTE(memberSTE);
    Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("name", __StringLiteralForSTE);
    Pair<String, Expression> _xifexpression = null;
    if ((memberDecl instanceof FieldAccessor)) {
      BooleanLiteral __BooleanLiteral = TranspilerBuilderBlocks._BooleanLiteral((memberDecl instanceof GetterDeclaration));
      _xifexpression = Pair.<String, Expression>of("getter", __BooleanLiteral);
    }
    BooleanLiteral __BooleanLiteral_1 = TranspilerBuilderBlocks._BooleanLiteral(memberDecl.isStatic());
    Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("isStatic", __BooleanLiteral_1);
    Pair<String, Expression> _xifexpression_1 = null;
    if ((memberDecl instanceof N4MethodDeclaration)) {
      Pair<String, Expression> _xblockexpression = null;
      {
        final String memberName = ((N4MethodDeclaration)memberDecl).getName();
        final boolean memberIsSymbol = ((memberName != null) && memberName.startsWith(N4JSLanguageUtils.SYMBOL_IDENTIFIER_PREFIX));
        Expression _xifexpression_2 = null;
        if ((!memberIsSymbol)) {
          _xifexpression_2 = TranspilerBuilderBlocks._StringLiteralForSTE(memberSTE);
        } else {
          _xifexpression_2 = this.typeAssistant.getMemberNameAsSymbol(memberName);
        }
        final Expression memberExpr = _xifexpression_2;
        Pair<String, Expression> _xifexpression_3 = null;
        boolean _isStatic = ((N4MethodDeclaration)memberDecl).isStatic();
        if (_isStatic) {
          IndexedAccessExpression __IndexAccessExpr = TranspilerBuilderBlocks._IndexAccessExpr(TranspilerBuilderBlocks._Snippet("staticProto"), memberExpr);
          _xifexpression_3 = Pair.<String, Expression>of("jsFunction", __IndexAccessExpr);
        } else {
          IndexedAccessExpression __IndexAccessExpr_1 = TranspilerBuilderBlocks._IndexAccessExpr(TranspilerBuilderBlocks._Snippet("instanceProto"), memberExpr);
          _xifexpression_3 = Pair.<String, Expression>of("jsFunction", __IndexAccessExpr_1);
        }
        _xblockexpression = _xifexpression_3;
      }
      _xifexpression_1 = _xblockexpression;
    }
    ArrayLiteral __ArrLit = TranspilerBuilderBlocks._ArrLit(this.createRuntimeAnnotations(memberDecl));
    Pair<String, Expression> _mappedTo_2 = Pair.<String, Expression>of("annotations", __ArrLit);
    return TranspilerBuilderBlocks._NewExpr(TranspilerBuilderBlocks._IdentRef(n4MemberClassSTE), TranspilerBuilderBlocks._ObjLit(_mappedTo, _xifexpression, _mappedTo_1, _xifexpression_1, _mappedTo_2));
  }
  
  private NewExpression[] createRuntimeAnnotations(final AnnotableElement annElem) {
    TAnnotableElement _switchResult = null;
    boolean _matched = false;
    if (annElem instanceof DelegatingMember) {
      _matched=true;
      IdentifiableElement _originalTarget = ((DelegatingMember)annElem).getDelegationTarget().getOriginalTarget();
      _switchResult = ((TAnnotableElement) _originalTarget);
    }
    if (!_matched) {
      if (annElem instanceof N4TypeDeclaration) {
        _matched=true;
        _switchResult = this.getState().info.getOriginalDefinedType(((N4TypeDeclaration)annElem));
      }
    }
    if (!_matched) {
      if (annElem instanceof N4MemberDeclaration) {
        _matched=true;
        _switchResult = this.getState().info.getOriginalDefinedMember(((N4MemberDeclaration)annElem));
      }
    }
    if (!_matched) {
      if (annElem instanceof TypeDefiningElement) {
        _matched=true;
        Type _xblockexpression = null;
        {
          final EObject original = this.getState().tracer.getOriginalASTNode(annElem);
          Type _xifexpression = null;
          if ((original instanceof TypeDefiningElement)) {
            _xifexpression = ((TypeDefiningElement)original).getDefinedType();
          }
          _xblockexpression = _xifexpression;
        }
        _switchResult = _xblockexpression;
      }
    }
    final TAnnotableElement tAnnElem = _switchResult;
    return this.createRuntimeAnnotations(tAnnElem);
  }
  
  private NewExpression[] createRuntimeAnnotations(final TAnnotableElement tAnnElem) {
    if ((tAnnElem == null)) {
      return new NewExpression[] {};
    }
    final Function1<TAnnotation, Boolean> _function = (TAnnotation it) -> {
      final AnnotationDefinition.RetentionPolicy retention = AnnotationDefinition.find(it.getName()).retention;
      return Boolean.valueOf(((retention == AnnotationDefinition.RetentionPolicy.RUNTIME) || (retention == AnnotationDefinition.RetentionPolicy.RUNTIME_TYPEFIELD)));
    };
    final Iterable<TAnnotation> runtimeAnnotations = IterableExtensions.<TAnnotation>filter(tAnnElem.getAnnotations(), _function);
    final Function1<TAnnotation, NewExpression> _function_1 = (TAnnotation it) -> {
      return this.createRuntimeAnnotation(it);
    };
    return ((NewExpression[])Conversions.unwrapArray(IterableExtensions.<TAnnotation, NewExpression>map(runtimeAnnotations, _function_1), NewExpression.class));
  }
  
  private NewExpression createRuntimeAnnotation(final TAnnotation ann) {
    NewExpression _xblockexpression = null;
    {
      final SymbolTableEntryInternal n4AnnotationSTE = this.steFor_N4Annotation();
      StringLiteral __StringLiteral = TranspilerBuilderBlocks._StringLiteral(ann.getName());
      Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("name", __StringLiteral);
      final Function1<TAnnotationArgument, String> _function = (TAnnotationArgument it) -> {
        return it.getArgAsString();
      };
      final Function1<String, StringLiteral> _function_1 = (String it) -> {
        return TranspilerBuilderBlocks._StringLiteral(it);
      };
      ArrayLiteral __ArrLit = TranspilerBuilderBlocks._ArrLit(
        ((Expression[])Conversions.unwrapArray(ListExtensions.<String, StringLiteral>map(ListExtensions.<TAnnotationArgument, String>map(ann.getArgs(), _function), _function_1), Expression.class)));
      Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("details", __ArrLit);
      _xblockexpression = TranspilerBuilderBlocks._NewExpr(TranspilerBuilderBlocks._IdentRef(n4AnnotationSTE), TranspilerBuilderBlocks._ObjLit(_mappedTo, _mappedTo_1));
    }
    return _xblockexpression;
  }
  
  /**
   * Create reference to n4type meta-object of the super type of given typeDefinition or 'undefined' if unavailable.
   */
  private Expression createN4SuperTypeRef(final N4TypeDeclaration typeDecl, final SymbolTableEntry superClassSTE) {
    if ((typeDecl instanceof N4ClassDeclaration)) {
      final TClass type = this.getState().info.getOriginalDefinedType(((N4ClassDeclaration)typeDecl));
      final TClassifier superType = RuleEnvironmentExtensions.getDeclaredOrImplicitSuperType(this.getState().G, type);
      TClass _n4ObjectType = RuleEnvironmentExtensions.n4ObjectType(this.getState().G);
      boolean _tripleEquals = (superType == _n4ObjectType);
      if (_tripleEquals) {
        return TranspilerBuilderBlocks._Snippet("N4Object.n4type");
      } else {
        if ((superType instanceof TClass)) {
          if ((((TClass)superType).isExternal() && (!AnnotationDefinition.N4JS.hasAnnotation(superType)))) {
            return this.undefinedRef();
          }
          final SymbolTableEntryOriginal superTypeSTE = this.getSymbolTableEntryOriginal(superType, true);
          final SymbolTableEntryOriginal n4typeSTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.n4ObjectType(this.getState().G), "n4type", false, true, true);
          return this.__NSSafe_PropertyAccessExpr(superTypeSTE, n4typeSTE);
        } else {
          if ((superType instanceof TObjectPrototype)) {
            return this.undefinedRef();
          }
        }
      }
    }
    return this.undefinedRef();
  }
  
  private boolean isStructurallyTyped(final TN4Classifier n4Classifier) {
    final TypingStrategy ts = n4Classifier.getTypingStrategy();
    return ((ts == TypingStrategy.STRUCTURAL) || (ts == TypingStrategy.STRUCTURAL_FIELDS));
  }
  
  /**
   * In the output code, methods and field accessors (not fields) are represented as function expressions.
   * This method returns the name of this internal function expression for the given member.
   */
  private String getMemberFunctionName(final FunctionOrFieldAccessor m) {
    String _switchResult = null;
    boolean _matched = false;
    if (m instanceof N4GetterDeclaration) {
      _matched=true;
      String _firstUpper = Strings.toFirstUpper(((N4GetterDeclaration)m).getName());
      String _plus = ("get" + _firstUpper);
      _switchResult = BootstrapCallAssistant.getMemberFunctionName(_plus);
    }
    if (!_matched) {
      if (m instanceof N4SetterDeclaration) {
        _matched=true;
        String _firstUpper = Strings.toFirstUpper(((N4SetterDeclaration)m).getName());
        String _plus = ("set" + _firstUpper);
        _switchResult = BootstrapCallAssistant.getMemberFunctionName(_plus);
      }
    }
    if (!_matched) {
      if (m instanceof N4MethodDeclaration) {
        _matched=true;
        _switchResult = BootstrapCallAssistant.getMemberFunctionName(((N4MethodDeclaration)m).getName());
      }
    }
    if (!_matched) {
      String _name = m.eClass().getName();
      String _plus = ("unsupported subclass of N4MemberDeclaration: " + _name);
      throw new IllegalArgumentException(_plus);
    }
    return _switchResult;
  }
  
  /**
   * Same as {@link #getMemberFunctionName(FunctionOrFieldAccessor)}, but operates on raw method name.
   * 
   * @param name
   *            raw method name.
   */
  private static String getMemberFunctionName(final String name) {
    boolean _isLegalIdentifier = TranspilerUtils.isLegalIdentifier(name);
    boolean _not = (!_isLegalIdentifier);
    if (_not) {
      String _sanitizeIdentifierName = TranspilerUtils.sanitizeIdentifierName(name);
      return (_sanitizeIdentifierName + N4JSLanguageConstants.METHOD_STACKTRACE_SUFFIX);
    }
    return (name + N4JSLanguageConstants.METHOD_STACKTRACE_SUFFIX);
  }
  
  private boolean inN4JSD(final Type type) {
    return this.jsVariantHelper.isExternalMode(type);
  }
  
  private boolean isAbstract(final ModifiableElement element) {
    return element.getDeclaredModifiers().contains(N4Modifier.ABSTRACT);
  }
}
