/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.N4JSLanguageConstants;
import org.eclipse.n4js.n4JS.ArrayElement;
import org.eclipse.n4js.n4JS.ArrayLiteral;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.N4ClassDeclaration;
import org.eclipse.n4js.n4JS.ObjectLiteral;
import org.eclipse.n4js.n4JS.Statement;
import org.eclipse.n4js.n4JS.StringLiteral;
import org.eclipse.n4js.organize.imports.DIUtility;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.assistants.TypeAssistant;
import org.eclipse.n4js.transpiler.im.ReferencingElementExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.transpiler.utils.TranspilerUtils;
import org.eclipse.n4js.ts.typeRefs.ParameterizedTypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.types.TAnnotation;
import org.eclipse.n4js.ts.types.TAnnotationArgument;
import org.eclipse.n4js.ts.types.TAnnotationTypeRefArgument;
import org.eclipse.n4js.ts.types.TClass;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TMethod;
import org.eclipse.n4js.ts.types.TN4Classifier;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;

/**
 * Generates DI meta information for the injector.
 * Information is attached to the compiled type in the '$di' property.
 * 
 * TODO the DI code below makes heavy use of TModule (probably) without true need; refactor this
 */
@SuppressWarnings("all")
public class DependencyInjectionTransformation extends Transformation {
  @Inject
  private TypeAssistant typeAssistant;
  
  @Override
  public void assertPreConditions() {
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<N4ClassDeclaration> _function = (N4ClassDeclaration classDecl) -> {
      final TClass tClass = this.getState().info.getOriginalDefinedType(classDecl);
      if ((tClass != null)) {
        final SymbolTableEntryOriginal superClassSTE = this.typeAssistant.getSuperClassSTE(classDecl);
        final Statement codeForDI = this.generateCodeForDI(tClass, classDecl, superClassSTE);
        if ((codeForDI != null)) {
          this.insertAfter(TranspilerUtils.orContainingExportDeclaration(classDecl), codeForDI);
        }
      }
    };
    this.<N4ClassDeclaration>collectNodes(this.getState().im, N4ClassDeclaration.class, false).forEach(_function);
  }
  
  /**
   * Generates hooks for DI mechanisms. Mainly N4Injector (part of n4js libraries)
   * depends on those hooks.
   * Note that those in many cases require proper types to be imported, which makes those
   * hooks indirectly depended on behavior of the script transpiler policy that decides to
   * remove "unused" imported symbols. That one has to leave imported symbols that are refereed to
   * in code generated here.
   * 
   * Also, note second parameter passed is name or alias of the super type.
   * passing it from caller, even if we don't use it, is a shortcut to avoid computing it again here.
   */
  private Statement generateCodeForDI(final TClass it, final N4ClassDeclaration classDecl, final SymbolTableEntry superClassSTE) {
    final Pair<String, Expression>[] propertiesForDI = this.createPropertiesForDI(it, classDecl, superClassSTE);
    boolean _isEmpty = ((List<Pair<String, Expression>>)Conversions.doWrapArray(propertiesForDI)).isEmpty();
    if (_isEmpty) {
      return null;
    }
    final SymbolTableEntry classSTE = this.findSymbolTableEntryForElement(classDecl, true);
    final SymbolTableEntryOriginal objectSTE = this.getSymbolTableEntryOriginal(RuleEnvironmentExtensions.objectType(this.getState().G), true);
    final SymbolTableEntryOriginal definePropertySTE = this.getSymbolTableEntryForMember(RuleEnvironmentExtensions.objectType(this.getState().G), "defineProperty", false, true, true);
    ObjectLiteral __ObjLit = TranspilerBuilderBlocks._ObjLit(
      this.createPropertiesForDI(it, classDecl, superClassSTE));
    Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("value", __ObjLit);
    return TranspilerBuilderBlocks._ExprStmnt(
      TranspilerBuilderBlocks._CallExpr(
        TranspilerBuilderBlocks._PropertyAccessExpr(objectSTE, definePropertySTE), 
        TranspilerBuilderBlocks._IdentRef(classSTE), 
        TranspilerBuilderBlocks._StringLiteral(N4JSLanguageConstants.DI_PROP_NAME), 
        TranspilerBuilderBlocks._ObjLit(_mappedTo)));
  }
  
  private Pair<String, Expression>[] createPropertiesForDI(final TClass it, final N4ClassDeclaration classDecl, final SymbolTableEntry superClassSTE) {
    final ArrayList<Pair<String, Expression>> result = CollectionLiterals.<Pair<String, Expression>>newArrayList();
    boolean _isBinder = DIUtility.isBinder(it);
    if (_isBinder) {
      ArrayLiteral _generateBindingPairs = this.generateBindingPairs(it);
      Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("bindings", ((Expression) _generateBindingPairs));
      result.add(_mappedTo);
      ArrayLiteral _generateMethodBindings = this.generateMethodBindings(it);
      Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("methodBindings", ((Expression) _generateMethodBindings));
      result.add(_mappedTo_1);
      Pair<String, Expression>[] _injectionPointsMetaInfo = this.injectionPointsMetaInfo(it, superClassSTE);
      Iterables.<Pair<String, Expression>>addAll(result, ((Iterable<? extends Pair<String, Expression>>)Conversions.doWrapArray(_injectionPointsMetaInfo)));
    } else {
      boolean _isDIComponent = DIUtility.isDIComponent(it);
      if (_isDIComponent) {
        boolean _hasParentInjector = DIUtility.hasParentInjector(it);
        if (_hasParentInjector) {
          final SymbolTableEntryOriginal parentDIC_STE = this.getSymbolTableEntryOriginal(DIUtility.findParentDIC(it), true);
          ReferencingElementExpression_IM ___NSSafe_IdentRef = this.__NSSafe_IdentRef(parentDIC_STE);
          Pair<String, Expression> _mappedTo_2 = Pair.<String, Expression>of("parent", ((Expression) ___NSSafe_IdentRef));
          result.add(_mappedTo_2);
        }
        ArrayLiteral _generateBinders = this.generateBinders(it);
        Pair<String, Expression> _mappedTo_3 = Pair.<String, Expression>of("binders", ((Expression) _generateBinders));
        result.add(_mappedTo_3);
        Pair<String, Expression>[] _injectionPointsMetaInfo_1 = this.injectionPointsMetaInfo(it, superClassSTE);
        Iterables.<Pair<String, Expression>>addAll(result, ((Iterable<? extends Pair<String, Expression>>)Conversions.doWrapArray(_injectionPointsMetaInfo_1)));
      } else {
        boolean _isInjectedClass = DIUtility.isInjectedClass(it);
        if (_isInjectedClass) {
          Pair<String, Expression>[] _injectionPointsMetaInfo_2 = this.injectionPointsMetaInfo(it, superClassSTE);
          Iterables.<Pair<String, Expression>>addAll(result, ((Iterable<? extends Pair<String, Expression>>)Conversions.doWrapArray(_injectionPointsMetaInfo_2)));
        }
      }
    }
    return ((Pair<String, Expression>[])Conversions.unwrapArray(result, Pair.class));
  }
  
  /**
   * Generate DI hooks for scopes, super type, injected ctor, injected fields
   */
  private Pair<String, Expression>[] injectionPointsMetaInfo(final TClass it, final SymbolTableEntry superClassSTE) {
    Pair<String, Expression> _xifexpression = null;
    boolean _isSingleton = DIUtility.isSingleton(it);
    if (_isSingleton) {
      StringLiteral __StringLiteral = TranspilerBuilderBlocks._StringLiteral("Singleton");
      _xifexpression = Pair.<String, Expression>of("scope", __StringLiteral);
    }
    Pair<String, Expression> _xifexpression_1 = null;
    boolean _hasSuperType = DIUtility.hasSuperType(it);
    if (_hasSuperType) {
      ReferencingElementExpression_IM ___NSSafe_IdentRef = this.__NSSafe_IdentRef(superClassSTE);
      _xifexpression_1 = Pair.<String, Expression>of("superType", ___NSSafe_IdentRef);
    }
    Pair<String, Expression> _xifexpression_2 = null;
    if (((it.getOwnedCtor() != null) && AnnotationDefinition.INJECT.hasAnnotation(it.getOwnedCtor()))) {
      ArrayLiteral _methodInjectedParams = this.methodInjectedParams(it.getOwnedCtor());
      _xifexpression_2 = Pair.<String, Expression>of("injectCtorParams", _methodInjectedParams);
    }
    ArrayLiteral _fieldInjection = this.fieldInjection(it);
    Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("fieldsInjectedTypes", _fieldInjection);
    return new Pair[] { _xifexpression, _xifexpression_1, _xifexpression_2, _mappedTo };
  }
  
  /**
   * Generate injection info for method annotated with {@link AnnotationDefinition#INJECT}.
   * If method has no method parameters returned value is empty string,
   * otherwise description of parameters is returned.
   */
  private ArrayLiteral methodInjectedParams(final TMethod it) {
    final ArrayLiteral result = TranspilerBuilderBlocks._ArrLit();
    EList<TFormalParameter> _fpars = it.getFpars();
    for (final TFormalParameter fpar : _fpars) {
      {
        final SymbolTableEntryOriginal fparSTE = this.getSymbolTableEntryOriginal(fpar, true);
        EList<ArrayElement> _elements = result.getElements();
        StringLiteral __StringLiteralForSTE = TranspilerBuilderBlocks._StringLiteralForSTE(fparSTE);
        Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("name", ((Expression) __StringLiteralForSTE));
        Pair<String, Expression>[] _generateTypeInfo = this.generateTypeInfo(fpar.getTypeRef());
        Iterable<Pair<String, Expression>> _plus = Iterables.<Pair<String, Expression>>concat(Collections.<Pair<String, Expression>>unmodifiableList(CollectionLiterals.<Pair<String, Expression>>newArrayList(_mappedTo)), ((Iterable<? extends Pair<String, Expression>>)Conversions.doWrapArray(_generateTypeInfo)));
        ArrayElement __ArrayElement = TranspilerBuilderBlocks._ArrayElement(
          TranspilerBuilderBlocks._ObjLit(((Pair<String, Expression>[])Conversions.unwrapArray(_plus, Pair.class))));
        _elements.add(__ArrayElement);
      }
    }
    return result;
  }
  
  /**
   * Generate injection info for fields annotated with {@link AnnotationDefinition#INJECT}.
   */
  private ArrayLiteral fieldInjection(final TClass it) {
    final ArrayLiteral result = TranspilerBuilderBlocks._ArrLit();
    Iterable<TField> _ownedInejctedFields = this.getOwnedInejctedFields(it);
    for (final TField field : _ownedInejctedFields) {
      {
        final SymbolTableEntryOriginal fieldSTE = this.getSymbolTableEntryOriginal(field, true);
        EList<ArrayElement> _elements = result.getElements();
        StringLiteral __StringLiteralForSTE = TranspilerBuilderBlocks._StringLiteralForSTE(fieldSTE);
        Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("name", ((Expression) __StringLiteralForSTE));
        Pair<String, Expression>[] _generateTypeInfo = this.generateTypeInfo(field.getTypeRef());
        Iterable<Pair<String, Expression>> _plus = Iterables.<Pair<String, Expression>>concat(Collections.<Pair<String, Expression>>unmodifiableList(CollectionLiterals.<Pair<String, Expression>>newArrayList(_mappedTo)), ((Iterable<? extends Pair<String, Expression>>)Conversions.doWrapArray(_generateTypeInfo)));
        ArrayElement __ArrayElement = TranspilerBuilderBlocks._ArrayElement(
          TranspilerBuilderBlocks._ObjLit(((Pair<String, Expression>[])Conversions.unwrapArray(_plus, Pair.class))));
        _elements.add(__ArrayElement);
      }
    }
    return result;
  }
  
  /**
   * Generate injection info from {@link AnnotationDefinition#BIND} annotations on the provided class.
   */
  private ArrayLiteral generateBindingPairs(final TClass it) {
    final ArrayLiteral result = TranspilerBuilderBlocks._ArrLit();
    Iterable<Pair<TN4Classifier, TN4Classifier>> _bindingPairs = this.getBindingPairs(it);
    for (final Pair<TN4Classifier, TN4Classifier> binding : _bindingPairs) {
      {
        final SymbolTableEntryOriginal keySTE = this.getSymbolTableEntryOriginal(binding.getKey(), true);
        final SymbolTableEntryOriginal valueSTE = this.getSymbolTableEntryOriginal(binding.getValue(), true);
        EList<ArrayElement> _elements = result.getElements();
        ReferencingElementExpression_IM ___NSSafe_IdentRef = this.__NSSafe_IdentRef(keySTE);
        Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("from", ___NSSafe_IdentRef);
        ReferencingElementExpression_IM ___NSSafe_IdentRef_1 = this.__NSSafe_IdentRef(valueSTE);
        Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("to", ___NSSafe_IdentRef_1);
        ArrayElement __ArrayElement = TranspilerBuilderBlocks._ArrayElement(
          TranspilerBuilderBlocks._ObjLit(_mappedTo, _mappedTo_1));
        _elements.add(__ArrayElement);
      }
    }
    return result;
  }
  
  /**
   * Generate injection info for methods annotated with {@link AnnotationDefinition#PROVIDES}.
   * Returned information contains returned type, name and formal parameters of the method.
   */
  private ArrayLiteral generateMethodBindings(final TClass it) {
    final ArrayLiteral result = TranspilerBuilderBlocks._ArrLit();
    Iterable<TMethod> _ownedProviderMethods = this.getOwnedProviderMethods(it);
    for (final TMethod method : _ownedProviderMethods) {
      EList<ArrayElement> _elements = result.getElements();
      Pair<String, Expression> _head = IterableExtensions.<Pair<String, Expression>>head(((Iterable<Pair<String, Expression>>)Conversions.doWrapArray(this.generateTypeInfo(method.getReturnTypeRef(), "to"))));
      StringLiteral __StringLiteral = TranspilerBuilderBlocks._StringLiteral(method.getName());
      Pair<String, Expression> _mappedTo = Pair.<String, Expression>of("name", __StringLiteral);
      ArrayLiteral _methodInjectedParams = this.methodInjectedParams(method);
      Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of("args", _methodInjectedParams);
      ArrayElement __ArrayElement = TranspilerBuilderBlocks._ArrayElement(
        TranspilerBuilderBlocks._ObjLit(_head, _mappedTo, _mappedTo_1));
      _elements.add(__ArrayElement);
    }
    return result;
  }
  
  private ArrayLiteral generateBinders(final TClass it) {
    final ArrayLiteral result = TranspilerBuilderBlocks._ArrLit();
    List<Type> _resolveBinders = DIUtility.resolveBinders(it);
    for (final Type binderType : _resolveBinders) {
      {
        final SymbolTableEntryOriginal binderTypeSTE = this.getSymbolTableEntryOriginal(binderType, true);
        EList<ArrayElement> _elements = result.getElements();
        ArrayElement __ArrayElement = TranspilerBuilderBlocks._ArrayElement(
          this.__NSSafe_IdentRef(binderTypeSTE));
        _elements.add(__ArrayElement);
      }
    }
    return result;
  }
  
  /**
   * Generate type information for DI. Mainly FQN of the {@link TypeRef}, or composed information
   * if given type is generic.
   */
  private Pair<String, Expression>[] generateTypeInfo(final TypeRef typeRef) {
    return this.generateTypeInfo(typeRef, "type");
  }
  
  private Pair<String, Expression>[] generateTypeInfo(final TypeRef typeRef, final String propertyName) {
    boolean _isProviderType = DIUtility.isProviderType(typeRef);
    boolean _not = (!_isProviderType);
    if (_not) {
      final SymbolTableEntryOriginal declaredTypeSTE = this.getSymbolTableEntryOriginal(typeRef.getDeclaredType(), true);
      ReferencingElementExpression_IM ___NSSafe_IdentRef = this.__NSSafe_IdentRef(declaredTypeSTE);
      Pair<String, Expression> _mappedTo = Pair.<String, Expression>of(propertyName, ___NSSafe_IdentRef);
      return new Pair[] { _mappedTo };
    } else {
      if ((typeRef instanceof ParameterizedTypeRef)) {
        final SymbolTableEntryOriginal declaredTypeSTE_1 = this.getSymbolTableEntryOriginal(((ParameterizedTypeRef)typeRef).getDeclaredType(), true);
        ReferencingElementExpression_IM ___NSSafe_IdentRef_1 = this.__NSSafe_IdentRef(declaredTypeSTE_1);
        Pair<String, Expression> _mappedTo_1 = Pair.<String, Expression>of(propertyName, ___NSSafe_IdentRef_1);
        ObjectLiteral __ObjLit = TranspilerBuilderBlocks._ObjLit(
          this.generateTypeInfo(IterableExtensions.<TypeRef>head(Iterables.<TypeRef>filter(((ParameterizedTypeRef)typeRef).getTypeArgs(), TypeRef.class))));
        Pair<String, Expression> _mappedTo_2 = Pair.<String, Expression>of("typeVar", __ObjLit);
        return new Pair[] { _mappedTo_1, _mappedTo_2 };
      } else {
        Type _declaredType = null;
        if (typeRef!=null) {
          _declaredType=typeRef.getDeclaredType();
        }
        String _name = null;
        if (_declaredType!=null) {
          _name=_declaredType.getName();
        }
        String _plus = ("cannot generate type info for " + _name);
        throw new IllegalStateException(_plus);
      }
    }
  }
  
  /**
   * Get list of {@link Pair}s of first and second argument of the {@link AnnotationDefinition#BIND} annotation.
   */
  private Iterable<Pair<TN4Classifier, TN4Classifier>> getBindingPairs(final TClass clazz) {
    final Function1<TAnnotation, Pair<TN4Classifier, TN4Classifier>> _function = (TAnnotation it) -> {
      TAnnotationArgument _head = IterableExtensions.<TAnnotationArgument>head(it.getArgs());
      Type _declaredType = ((TAnnotationTypeRefArgument) _head).getTypeRef().getDeclaredType();
      TAnnotationArgument _last = IterableExtensions.<TAnnotationArgument>last(it.getArgs());
      Type _declaredType_1 = ((TAnnotationTypeRefArgument) _last).getTypeRef().getDeclaredType();
      return Pair.<TN4Classifier, TN4Classifier>of(((TN4Classifier) _declaredType), ((TN4Classifier) _declaredType_1));
    };
    return IterableExtensions.<TAnnotation, Pair<TN4Classifier, TN4Classifier>>map(AnnotationDefinition.BIND.getAllAnnotations(clazz), _function);
  }
  
  private Iterable<TMethod> getOwnedProviderMethods(final TClass clazz) {
    final Function1<TMethod, Boolean> _function = (TMethod it) -> {
      return Boolean.valueOf(AnnotationDefinition.PROVIDES.hasAnnotation(it));
    };
    return IterableExtensions.<TMethod>filter(Iterables.<TMethod>filter(clazz.getOwnedMembers(), TMethod.class), _function);
  }
  
  private Iterable<TField> getOwnedInejctedFields(final TN4Classifier clazz) {
    final Function1<TField, Boolean> _function = (TField it) -> {
      return Boolean.valueOf(AnnotationDefinition.INJECT.hasAnnotation(it));
    };
    return IterableExtensions.<TField>filter(Iterables.<TField>filter(clazz.getOwnedMembers(), TField.class), _function);
  }
}
