/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.transpiler.es.transform;

import com.google.common.collect.Iterables;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.n4js.AnnotationDefinition;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.ImportDeclaration;
import org.eclipse.n4js.n4JS.ParenExpression;
import org.eclipse.n4js.n4JS.ScriptElement;
import org.eclipse.n4js.n4JS.StringLiteral;
import org.eclipse.n4js.n4JS.VariableDeclaration;
import org.eclipse.n4js.n4JS.VariableStatement;
import org.eclipse.n4js.transpiler.Transformation;
import org.eclipse.n4js.transpiler.TranspilerBuilderBlocks;
import org.eclipse.n4js.transpiler.im.IdentifierRef_IM;
import org.eclipse.n4js.transpiler.im.ParameterizedPropertyAccessExpression_IM;
import org.eclipse.n4js.transpiler.im.SymbolTableEntry;
import org.eclipse.n4js.transpiler.im.SymbolTableEntryOriginal;
import org.eclipse.n4js.ts.types.IdentifiableElement;
import org.eclipse.n4js.ts.types.TEnum;
import org.eclipse.n4js.ts.types.TEnumLiteral;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.typesystem.RuleEnvironmentExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Transforms references to literals of <code>@StringBased</code> enums by replacing them with a plain string literal.
 * References to ordinary enums (i.e. not string-based) are not touched by this transformation.
 */
@SuppressWarnings("all")
public class EnumAccessTransformation extends Transformation {
  private final Map<TEnum, VariableDeclaration> literalsConstants = CollectionLiterals.<TEnum, VariableDeclaration>newHashMap();
  
  private TMember member_StringBasedEnum_literals;
  
  @Override
  public void assertPreConditions() {
    this.member_StringBasedEnum_literals = RuleEnvironmentExtensions.n4StringBasedEnumType(this.getState().G).findOwnedMember("literals", false, true);
    this.assertNotNull("member of built-in type not found: StringBasedEnum#literals", this.member_StringBasedEnum_literals);
  }
  
  @Override
  public void assertPostConditions() {
  }
  
  @Override
  public void analyze() {
  }
  
  @Override
  public void transform() {
    final Consumer<ParameterizedPropertyAccessExpression_IM> _function = (ParameterizedPropertyAccessExpression_IM it) -> {
      this.transformEnumAccess(it);
    };
    this.<ParameterizedPropertyAccessExpression_IM>collectNodes(this.getState().im, ParameterizedPropertyAccessExpression_IM.class, true).forEach(_function);
  }
  
  private void transformEnumAccess(final ParameterizedPropertyAccessExpression_IM expr) {
    final SymbolTableEntry propSTE = expr.getProperty_IM();
    IdentifiableElement _xifexpression = null;
    if ((propSTE instanceof SymbolTableEntryOriginal)) {
      _xifexpression = ((SymbolTableEntryOriginal)propSTE).getOriginalTarget();
    }
    final IdentifiableElement prop = _xifexpression;
    boolean _matched = false;
    if (prop instanceof TEnumLiteral) {
      _matched=true;
      this.transformEnumLiteralAccess(this.resolveOriginalStringBasedEnum(expr), expr, ((TEnumLiteral)prop));
    }
    if (!_matched) {
      if ((prop == this.member_StringBasedEnum_literals)) {
        _matched=true;
        this.transformEnumLiteralsConstantAccess(this.resolveOriginalStringBasedEnum(expr), expr);
      }
    }
  }
  
  /**
   * Assumes {@code originalEnum} was resolved to string based enum, e,g, <code>@StringBased enum Color {RED : "red", SOME}</code>,
   * transforms access to <code>Color.RED</code> to <code>"red"</code> and <code>Color.SOME</code> to <code>"SOME"</code>.It is low level transformation, assumes
   * caller did all necessary checks and did provide proper data. Replacement is applied only if none of the provided parameters is {@code null}.
   * 
   * @param originalEnum string based enum from AST for which literal access is generated.
   * @param expr expression which will have value generated.
   * @param prop enum literal from which value is generated.
   */
  private void transformEnumLiteralAccess(final TEnum originalEnum, final ParameterizedPropertyAccessExpression_IM expr, final TEnumLiteral prop) {
    if ((((originalEnum != null) && (expr != null)) && (prop != null))) {
      this.replace(expr, this.enumLiteralToStringLiteral(prop));
    }
  }
  
  /**
   * Assumes {@code originalEnum} was resolved to string based enum, e,g, <code>@StringBased enum Color {RED : "red", SOME}</code>,
   * transforms access to <code>Color.literals</code> to <code>["red", "SOME"</code>. It is low level transformation, assumes
   * caller did all necessary checks and did provide proper data. Replacement is applied only if none of the provided parameters is {@code null}.
   * 
   * @param originalEnum string based enum from AST for which literals access is generated.
   * @param expr expression which will have values generated.
   */
  private void transformEnumLiteralsConstantAccess(final TEnum originalEnum, final ParameterizedPropertyAccessExpression_IM expr) {
    if (((originalEnum != null) && (expr != null))) {
      this.replace(expr, this.getReferenceToLiteralsConstant(originalEnum));
    }
  }
  
  /**
   * Resolves left hand side of the {@code ParameterizedPropertyAccessExpression_IM} to the original {@link AnnotationDefinition#STRING_BASED} enum.
   * 
   * @return resolved original string based enum or null.
   */
  private TEnum resolveOriginalStringBasedEnum(final ParameterizedPropertyAccessExpression_IM pex) {
    final SymbolTableEntry targetEnumSTE = this.resolveOriginalExpressionTarget(pex.getTarget());
    if ((targetEnumSTE instanceof SymbolTableEntryOriginal)) {
      final IdentifiableElement orginal = ((SymbolTableEntryOriginal)targetEnumSTE).getOriginalTarget();
      if ((orginal instanceof TEnum)) {
        boolean _isStringBased = this.isStringBased(((TEnum)orginal));
        if (_isStringBased) {
          return ((TEnum)orginal);
        }
      }
    }
    return null;
  }
  
  /**
   * Resolve target of the {@code ParameterizedPropertyAccessExpression_IM} when left hand side is an simple access or nested expression, e.g. for
   * <ul>
   *   <li> <code> Enum.EnumLiteral </code> </li>
   *   <li> <code> Enum.literals </code> </li>
   *   <li> <code> (((((Enum)))).EnumLiteral </code> </li>
   *   <li> <code> (((((Enum)))).literals </code> </li>
   * </ul>
   *  resolves to {@code SymbolTableEntry} of the Enum.
   */
  private SymbolTableEntry resolveOriginalExpressionTarget(final Expression ex) {
    SymbolTableEntry _switchResult = null;
    boolean _matched = false;
    if (ex instanceof IdentifierRef_IM) {
      _matched=true;
      _switchResult = ((IdentifierRef_IM)ex).getRewiredTarget();
    }
    if (!_matched) {
      if (ex instanceof ParameterizedPropertyAccessExpression_IM) {
        _matched=true;
        _switchResult = ((ParameterizedPropertyAccessExpression_IM)ex).getProperty_IM();
      }
    }
    if (!_matched) {
      if (ex instanceof ParenExpression) {
        _matched=true;
        _switchResult = this.resolveOriginalExpressionTarget(((ParenExpression)ex).getExpression());
      }
    }
    if (!_matched) {
      _switchResult = null;
    }
    return _switchResult;
  }
  
  private IdentifierRef_IM getReferenceToLiteralsConstant(final TEnum tEnum) {
    VariableDeclaration vdecl = this.literalsConstants.get(tEnum);
    if ((vdecl == null)) {
      vdecl = this.createLiteralsConstant(tEnum);
      this.literalsConstants.put(tEnum, vdecl);
    }
    return TranspilerBuilderBlocks._IdentRef(this.findSymbolTableEntryForElement(vdecl, false));
  }
  
  private VariableDeclaration createLiteralsConstant(final TEnum tEnum) {
    final String name = this.findUniqueNameForLiteralsConstant(tEnum);
    final Function1<TEnumLiteral, StringLiteral> _function = (TEnumLiteral it) -> {
      return this.enumLiteralToStringLiteral(it);
    };
    final VariableDeclaration vdecl = TranspilerBuilderBlocks._VariableDeclaration(name, TranspilerBuilderBlocks._ArrLit(((Expression[])Conversions.unwrapArray(ListExtensions.<TEnumLiteral, StringLiteral>map(tEnum.getLiterals(), _function), Expression.class))));
    this.getState().info.markAsToHoist(vdecl);
    final VariableStatement vstmnt = TranspilerBuilderBlocks._VariableStatement(vdecl);
    final ImportDeclaration lastImport = IterableExtensions.<ImportDeclaration>last(Iterables.<ImportDeclaration>filter(this.getState().im.getScriptElements(), ImportDeclaration.class));
    if ((lastImport != null)) {
      this.insertAfter(lastImport, vstmnt);
    } else {
      boolean _isEmpty = this.getState().im.getScriptElements().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        this.insertBefore(IterableExtensions.<ScriptElement>head(this.getState().im.getScriptElements()), vstmnt);
      } else {
        this.getState().im.getScriptElements().add(vstmnt);
      }
    }
    this.createSymbolTableEntryIMOnly(vdecl);
    return vdecl;
  }
  
  private String findUniqueNameForLiteralsConstant(final TEnum tEnum) {
    final Function1<VariableDeclaration, String> _function = (VariableDeclaration it) -> {
      return it.getName();
    };
    final Set<String> names = IterableExtensions.<String>toSet(IterableExtensions.<VariableDeclaration, String>map(this.literalsConstants.values(), _function));
    String _elvis = null;
    String _name = null;
    if (tEnum!=null) {
      _name=tEnum.getName();
    }
    if (_name != null) {
      _elvis = _name;
    } else {
      _elvis = "Unnamed";
    }
    String newName = ("$enumLiteralsOf" + _elvis);
    boolean _contains = names.contains(newName);
    if (_contains) {
      int cnt = 1;
      while (names.contains((newName + Integer.valueOf(cnt)))) {
        cnt++;
      }
      newName = (newName + Integer.valueOf(cnt));
    }
    return newName;
  }
  
  private boolean isStringBased(final TEnum tEnum) {
    return AnnotationDefinition.STRING_BASED.hasAnnotation(tEnum);
  }
  
  private StringLiteral enumLiteralToStringLiteral(final TEnumLiteral enumLiteral) {
    String _elvis = null;
    String _value = null;
    if (enumLiteral!=null) {
      _value=enumLiteral.getValue();
    }
    if (_value != null) {
      _elvis = _value;
    } else {
      String _name = enumLiteral.getName();
      _elvis = _name;
    }
    return TranspilerBuilderBlocks._StringLiteral(_elvis);
  }
}
