/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.collect.Iterables;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * Simple POJO for representing a difference.
 */
@Data
@SuppressWarnings("all")
public class Diff<T extends Object> {
  /**
   * The initial, ordered state of the relevant items.
   */
  private final T[] oldItems;
  
  /**
   * The initial, ordered state of all items.
   */
  private final T[] oldAllItems;
  
  /**
   * The items that have been added.
   */
  private final T[] addedItems;
  
  /**
   * The removed items.
   */
  private final T[] deletedItems;
  
  /**
   * Map of edited items. Keys are old state, values are the new state.
   */
  private final Map<T, T> editedItems;
  
  /**
   * The final, ordered state of the relevant items.
   */
  private final T[] newItems;
  
  /**
   * The final, ordered state of all items.
   */
  private final T[] newAllItems;
  
  /**
   * Returns with {@code true} if the diff contains no addition, deletion and no edition
   * and the state of all and relevant items equals with the new state of all and relevant items.
   * Otherwise returns with {@code false}.
   */
  public boolean isEmpty() {
    return ((((((List<T>)Conversions.doWrapArray(this.addedItems)).isEmpty() && ((List<T>)Conversions.doWrapArray(this.deletedItems)).isEmpty()) && this.editedItems.isEmpty()) && Arrays.equals(this.newItems, this.oldItems)) && Arrays.equals(this.newAllItems, this.oldAllItems));
  }
  
  @Override
  public String toString() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Diff:");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("-----------------------------");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("Old items:     ");
    String _string = Iterables.toString(((Iterable<?>)Conversions.doWrapArray(this.oldItems)));
    _builder.append(_string, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("Old all items: ");
    String _string_1 = Iterables.toString(((Iterable<?>)Conversions.doWrapArray(this.oldAllItems)));
    _builder.append(_string_1, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("Added items:   ");
    String _string_2 = Iterables.toString(((Iterable<?>)Conversions.doWrapArray(this.addedItems)));
    _builder.append(_string_2, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("Deleted items: ");
    String _string_3 = Iterables.toString(((Iterable<?>)Conversions.doWrapArray(this.deletedItems)));
    _builder.append(_string_3, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("Edited items:  ");
    _builder.append(this.editedItems, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("New items:     ");
    String _string_4 = Iterables.toString(((Iterable<?>)Conversions.doWrapArray(this.newItems)));
    _builder.append(_string_4, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("New all items: ");
    String _string_5 = Iterables.toString(((Iterable<?>)Conversions.doWrapArray(this.newAllItems)));
    _builder.append(_string_5, "\t\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("-----------------------------");
    _builder.newLine();
    return _builder.toString();
  }
  
  public Diff(final T[] oldItems, final T[] oldAllItems, final T[] addedItems, final T[] deletedItems, final Map<T, T> editedItems, final T[] newItems, final T[] newAllItems) {
    super();
    this.oldItems = oldItems;
    this.oldAllItems = oldAllItems;
    this.addedItems = addedItems;
    this.deletedItems = deletedItems;
    this.editedItems = editedItems;
    this.newItems = newItems;
    this.newAllItems = newAllItems;
  }
  
  @Override
  @Pure
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.oldItems== null) ? 0 : Arrays.deepHashCode(this.oldItems));
    result = prime * result + ((this.oldAllItems== null) ? 0 : Arrays.deepHashCode(this.oldAllItems));
    result = prime * result + ((this.addedItems== null) ? 0 : Arrays.deepHashCode(this.addedItems));
    result = prime * result + ((this.deletedItems== null) ? 0 : Arrays.deepHashCode(this.deletedItems));
    result = prime * result + ((this.editedItems== null) ? 0 : this.editedItems.hashCode());
    result = prime * result + ((this.newItems== null) ? 0 : Arrays.deepHashCode(this.newItems));
    result = prime * result + ((this.newAllItems== null) ? 0 : Arrays.deepHashCode(this.newAllItems));
    return result;
  }
  
  @Override
  @Pure
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    Diff<?> other = (Diff<?>) obj;
    if (this.oldItems == null) {
      if (other.oldItems != null)
        return false;
    } else if (!Arrays.deepEquals(this.oldItems, other.oldItems))
      return false;
    if (this.oldAllItems == null) {
      if (other.oldAllItems != null)
        return false;
    } else if (!Arrays.deepEquals(this.oldAllItems, other.oldAllItems))
      return false;
    if (this.addedItems == null) {
      if (other.addedItems != null)
        return false;
    } else if (!Arrays.deepEquals(this.addedItems, other.addedItems))
      return false;
    if (this.deletedItems == null) {
      if (other.deletedItems != null)
        return false;
    } else if (!Arrays.deepEquals(this.deletedItems, other.deletedItems))
      return false;
    if (this.editedItems == null) {
      if (other.editedItems != null)
        return false;
    } else if (!this.editedItems.equals(other.editedItems))
      return false;
    if (this.newItems == null) {
      if (other.newItems != null)
        return false;
    } else if (!Arrays.deepEquals(this.newItems, other.newItems))
      return false;
    if (this.newAllItems == null) {
      if (other.newAllItems != null)
        return false;
    } else if (!Arrays.deepEquals(this.newAllItems, other.newAllItems))
      return false;
    return true;
  }
  
  @Pure
  public T[] getOldItems() {
    return this.oldItems;
  }
  
  @Pure
  public T[] getOldAllItems() {
    return this.oldAllItems;
  }
  
  @Pure
  public T[] getAddedItems() {
    return this.addedItems;
  }
  
  @Pure
  public T[] getDeletedItems() {
    return this.deletedItems;
  }
  
  @Pure
  public Map<T, T> getEditedItems() {
    return this.editedItems;
  }
  
  @Pure
  public T[] getNewItems() {
    return this.newItems;
  }
  
  @Pure
  public T[] getNewAllItems() {
    return this.newAllItems;
  }
}
