/**
 * Copyright (c) 2017 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.utils;

import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.List;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.xtext.resource.IContainer;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.resource.IResourceDescription;
import org.eclipse.xtext.resource.IResourceDescriptions;
import org.eclipse.xtext.resource.impl.ResourceDescriptionsProvider;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class contains helper methods for working with Xtext.
 */
@Singleton
@SuppressWarnings("all")
public class XtextUtilN4 {
  @Inject
  private IContainer.Manager containerManager;
  
  @Inject
  private ResourceDescriptionsProvider resourceDescriptionsProvider;
  
  @Inject
  private IResourceDescription.Manager descriptionManager;
  
  /**
   * Return all EObjectDescriptions in the index that are visible from an EObject.
   *  @param o
   * 			the EObject
   * <p>
   * See Lorenzo's book page 260.
   */
  public Iterable<IEObjectDescription> getVisibleEObjectDescriptions(final EObject o) {
    final Function1<IContainer, Iterable<IEObjectDescription>> _function = (IContainer container) -> {
      return container.getExportedObjects();
    };
    return Iterables.<IEObjectDescription>concat(ListExtensions.<IContainer, Iterable<IEObjectDescription>>map(this.getVisibleContainers(o), _function));
  }
  
  /**
   * Return all EObjectDescriptions of a certain type in the index that are visible from an EObject.
   * @param o
   * 			the EObject.
   * @param type
   * 			the type of EObjectDescriptions to filter.
   * <p>
   * See Lorenzo's book page 260.
   */
  public Iterable<IEObjectDescription> getVisibleEObjectDescriptions(final EObject o, final EClass type) {
    final Function1<IContainer, Iterable<IEObjectDescription>> _function = (IContainer container) -> {
      return container.getExportedObjectsByType(type);
    };
    return Iterables.<IEObjectDescription>concat(ListExtensions.<IContainer, Iterable<IEObjectDescription>>map(this.getVisibleContainers(o), _function));
  }
  
  /**
   * Return all visible containers from within a given resource.
   * @param resource
   * 					the resource
   */
  public List<IContainer> getVisibleContainers(final Resource resource) {
    final IResourceDescriptions index = this.resourceDescriptionsProvider.getResourceDescriptions(resource.getResourceSet());
    final IResourceDescription resourceDescription = this.descriptionManager.getResourceDescription(resource);
    return this.containerManager.getVisibleContainers(resourceDescription, index);
  }
  
  private List<IContainer> getVisibleContainers(final EObject o) {
    List<IContainer> _xblockexpression = null;
    {
      final IResourceDescriptions index = this.resourceDescriptionsProvider.getResourceDescriptions(o.eResource());
      final IResourceDescription resourceDescription = index.getResourceDescription(o.eResource().getURI());
      _xblockexpression = this.containerManager.getVisibleContainers(resourceDescription, index);
    }
    return _xblockexpression;
  }
}
