/**
 * Copyright (c) 2016 NumberFour AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   NumberFour AG - Initial API and implementation
 */
package org.eclipse.n4js.n4jsx;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import com.google.inject.Provider;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.n4js.n4JS.Expression;
import org.eclipse.n4js.n4JS.JSXElement;
import org.eclipse.n4js.resource.N4JSResource;
import org.eclipse.n4js.scoping.N4JSScopeProvider;
import org.eclipse.n4js.ts.typeRefs.FunctionTypeExprOrRef;
import org.eclipse.n4js.ts.typeRefs.TypeRef;
import org.eclipse.n4js.ts.typeRefs.TypeTypeRef;
import org.eclipse.n4js.ts.types.TClassifier;
import org.eclipse.n4js.ts.types.TField;
import org.eclipse.n4js.ts.types.TFormalParameter;
import org.eclipse.n4js.ts.types.TFunction;
import org.eclipse.n4js.ts.types.TGetter;
import org.eclipse.n4js.ts.types.TMember;
import org.eclipse.n4js.ts.types.TModule;
import org.eclipse.n4js.ts.types.Type;
import org.eclipse.n4js.ts.types.TypeVariable;
import org.eclipse.n4js.ts.utils.TypeUtils;
import org.eclipse.n4js.typesystem.N4JSTypeSystem;
import org.eclipse.n4js.typesystem.utils.RuleEnvironment;
import org.eclipse.n4js.typesystem.utils.RuleEnvironmentExtensions;
import org.eclipse.n4js.typesystem.utils.TypeSystemHelper;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.resource.IEObjectDescription;
import org.eclipse.xtext.scoping.IScope;
import org.eclipse.xtext.scoping.IScopeProvider;
import org.eclipse.xtext.util.IResourceScopeCache;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * This helper provides utilities for looking up React definitions such as React.Component or React.Element or
 * for calculating types related to React (e.g. of props property) etc.
 */
@SuppressWarnings("all")
public class ReactHelper {
  @Inject
  private N4JSTypeSystem ts;
  
  @Inject
  private TypeSystemHelper tsh;
  
  @Inject
  private IResourceScopeCache resourceScopeCacheHelper;
  
  @Inject
  private IScopeProvider scopeProvider;
  
  public final static String REACT_PROJECT_ID = "react";
  
  public final static String REACT_COMPONENT = "Component";
  
  public final static String REACT_ELEMENT = "Element";
  
  public final static String REACT_NAMESPACE_NAME = StringExtensions.toFirstUpper(ReactHelper.REACT_PROJECT_ID);
  
  public final static String REACT_ELEMENT_FACTORY_FUNCTION_NAME = "createElement";
  
  private final static String REACT_KEY = ("KEY__" + ReactHelper.REACT_PROJECT_ID);
  
  /**
   * Returns the {@link TModule} which contains the definitions of the React
   * JSX backend (e.g. {@code Component}, {@code createElement}).
   * 
   * Returns {@code null} if no valid React implementation can be found in the index.
   */
  public TModule getJsxBackendModule(final Resource resource) {
    final String key = ((ReactHelper.REACT_KEY + ".") + "TMODULE");
    final Provider<TModule> _function = () -> {
      final IScope scope = ((N4JSScopeProvider) this.scopeProvider).getScopeForImplicitImports(((N4JSResource) resource));
      final Iterable<IEObjectDescription> matchingDescriptions = scope.getElements(QualifiedName.create(ReactHelper.REACT_PROJECT_ID));
      final Function1<IEObjectDescription, TModule> _function_1 = (IEObjectDescription desc) -> {
        if ((desc == null)) {
          return null;
        }
        EObject _resolve = EcoreUtil2.resolve(desc.getEObjectOrProxy(), resource);
        return ((TModule) _resolve);
      };
      final Function1<TModule, Boolean> _function_2 = (TModule module) -> {
        return Boolean.valueOf(this.isValidReactModule(module));
      };
      return IterableExtensions.<TModule>head(IterableExtensions.<TModule>filter(IterableExtensions.<IEObjectDescription, TModule>map(matchingDescriptions, _function_1), _function_2));
    };
    return this.resourceScopeCacheHelper.<TModule>get(key, resource, _function);
  }
  
  /**
   * Returns the preferred name of the namespace used to import the JSX backend.
   */
  public String getJsxBackendNamespaceName() {
    return ReactHelper.REACT_NAMESPACE_NAME;
  }
  
  /**
   * Returns the name of the element factory function to use with React as JSX backend.
   */
  public String getJsxBackendElementFactoryFunctionName() {
    return ReactHelper.REACT_ELEMENT_FACTORY_FUNCTION_NAME;
  }
  
  /**
   * Calculate the type that an JSX element is binding to, usually class/function type
   * 
   * @param jsxElem the input JSX element
   * @return the typeref that the JSX element is binding to and null if not found
   */
  public TypeRef getJsxElementBindingType(final JSXElement jsxElem) {
    final Expression expr = jsxElem.getJsxElementName().getExpression();
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(expr);
    final TypeRef exprResult = this.ts.type(G, expr);
    return exprResult;
  }
  
  /**
   * Returns the element factory function for JSX elements which can be extracted
   * from the given {@link Resource}.
   */
  public TFunction getJsxBackendElementFactoryFunction(final Resource resource) {
    final TModule module = this.getJsxBackendModule(resource);
    if ((module != null)) {
      return this.lookUpReactElementFactoryFunction(module);
    }
    return null;
  }
  
  /**
   * Look up React.Element in the index.
   * 
   * @param context the EObject serving the context to look for React.Element.
   */
  public TClassifier lookUpReactElement(final EObject context) {
    final TClassifier reactElement = this.lookUpReactClassifier(context, ReactHelper.REACT_ELEMENT);
    return reactElement;
  }
  
  /**
   * Look up React.Component in the index.
   * 
   * @param context the EObject serving the context to look for React.Component.
   */
  public TClassifier lookUpReactComponent(final EObject context) {
    final TClassifier reactComponent = this.lookUpReactClassifier(context, ReactHelper.REACT_COMPONENT);
    return reactComponent;
  }
  
  /**
   * Calculate the "props" type of an JSX element. It is either the first type parameter of React.Component class or
   * the type of the first parameter of a functional React component
   * 
   * @param jsxElement the input JSX element
   * @return the typeref if exists and null otherwise
   */
  public TypeRef getPropsType(final JSXElement jsxElem) {
    final TypeRef exprTypeRef = this.getJsxElementBindingType(jsxElem);
    if ((exprTypeRef == null)) {
      return null;
    }
    final RuleEnvironment G = RuleEnvironmentExtensions.newRuleEnvironment(jsxElem);
    if (((exprTypeRef instanceof TypeTypeRef) && ((TypeTypeRef) exprTypeRef).isConstructorRef())) {
      final Type tclass = this.tsh.getStaticType(G, ((TypeTypeRef) exprTypeRef));
      final TClassifier tComponentClassifier = this.lookUpReactClassifier(jsxElem, ReactHelper.REACT_COMPONENT);
      if (((tComponentClassifier == null) || tComponentClassifier.getTypeVars().isEmpty())) {
        return null;
      }
      final TypeVariable reactComponentProps = tComponentClassifier.getTypeVars().get(0);
      this.tsh.addSubstitutions(G, TypeUtils.createTypeRef(tclass));
      final TypeRef reactComponentPropsTypeRef = this.ts.substTypeVariables(G, 
        TypeUtils.createTypeRef(reactComponentProps));
      return reactComponentPropsTypeRef;
    } else {
      if ((exprTypeRef instanceof FunctionTypeExprOrRef)) {
        int _length = ((Object[])Conversions.unwrapArray(((FunctionTypeExprOrRef)exprTypeRef).getFpars(), Object.class)).length;
        boolean _greaterThan = (_length > 0);
        if (_greaterThan) {
          final TFormalParameter tPropsParam = ((FunctionTypeExprOrRef)exprTypeRef).getFpars().get(0);
          return tPropsParam.getTypeRef();
        }
      }
    }
    return null;
  }
  
  /**
   * Return the type of a field or return type of a getter.
   * 
   * @param member MUST be either a field or getter (otherwise an exception is thrown).
   */
  public TypeRef typeRefOfFieldOrGetter(final TMember member, final TypeRef context) {
    if (((member instanceof TField) || (member instanceof TGetter))) {
      return this.ts.tau(member, context);
    }
    String _plus = (member + " must be either a TField or TGetter");
    throw new IllegalArgumentException(_plus);
  }
  
  /**
   * Lookup React component/element type. For increased efficiency, the found results are cached.
   * 
   * @param context the EObject serving the context to look for React classifiers.
   * @param reactClassifierName the name of React classifier.
   */
  private TClassifier lookUpReactClassifier(final EObject context, final String reactClassifierName) {
    final Resource resource = context.eResource();
    final String key = ((ReactHelper.REACT_KEY + ".") + reactClassifierName);
    final Provider<TClassifier> _function = () -> {
      final TModule tModule = this.getJsxBackendModule(resource);
      if ((tModule == null)) {
        return null;
      }
      final Function1<TClassifier, Boolean> _function_1 = (TClassifier it) -> {
        String _name = it.getName();
        return Boolean.valueOf(Objects.equal(_name, reactClassifierName));
      };
      final TClassifier tClassifier = IterableExtensions.<TClassifier>findFirst(Iterables.<TClassifier>filter(tModule.getTopLevelTypes(), TClassifier.class), _function_1);
      return tClassifier;
    };
    return this.resourceScopeCacheHelper.<TClassifier>get(key, resource, _function);
  }
  
  /**
   * Looks up the element factory function to use, assuming the react implementation in use
   * is to be found in the given {@code module}.
   * 
   * Returns {@code null} if no factory function can be found in the given module.
   */
  private TFunction lookUpReactElementFactoryFunction(final TModule module) {
    if ((module != null)) {
      EList<Type> _topLevelTypes = module.getTopLevelTypes();
      for (final Type currTopLevelType : _topLevelTypes) {
        if (((currTopLevelType instanceof TFunction) && ReactHelper.REACT_ELEMENT_FACTORY_FUNCTION_NAME.equals(currTopLevelType.getName()))) {
          return ((TFunction) currTopLevelType);
        }
      }
    }
    return null;
  }
  
  /**
   * Returns {@code true} if the given {@code module} is a {@link TModule}
   * that represents a valid implementation of React.
   * 
   * Returns {@code false} otherwise.
   * 
   * Note that this requires type definitions to be available for the given {@link TModule}.
   */
  private boolean isValidReactModule(final TModule module) {
    TFunction _lookUpReactElementFactoryFunction = this.lookUpReactElementFactoryFunction(module);
    return (_lookUpReactElementFactoryFunction != null);
  }
}
